/******************************************************************************\
 gnofin/preferences.c   $Revision: 1.10 $
 Copyright (C) 1999-2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

//#define ENABLE_DEBUG_TRACE

#include "common.h"
#include <gtk/gtkwindow.h>
#include <gtk/gtkctree.h>
#include <gtk/gtkhpaned.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkframe.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-stock.h>
#include "preferences.h"
#include "config-saver.h"
#include "dialogs.h"


/******************************************************************************
 * Data structure
 */

typedef struct {
  gchar  *label;
  GSList *pages;
} Category;

typedef struct {
  Category                *cat;
  gchar                   *label;
  GtkWidget               *frame;
  GtkWidget               *scrolled_win;
  GtkWidget               *widget;
  PreferencesPageMakeFunc  make;
  PreferencesPageInitFunc  init;
  PreferencesPageApplyFunc apply;
  gpointer                 data;
  guint			   dirty : 1;
} Page;

static GSList *categories = NULL;


/******************************************************************************
 * Data Helpers
 */

static gint
match_category (const Category *cat, const gchar *label)
{
  return strcmp (cat->label, label);
}

static gint
match_page (const Page *page, const gchar *label)
{
  return strcmp (page->label, label);
}

static gint
compare_categories (const Category *a, const Category *b)
{
  return strcmp (a->label, b->label);
}

static gint
compare_pages (const Page *a, const Page *b)
{
  return strcmp (a->label, b->label);
}

static Category *
make_category (const gchar *label)
{
  Category *cat;

  trace ("");
  g_assert (label);

  cat = g_new0 (Category, 1);
  cat->label = g_strdup (label);

  categories = g_slist_insert_sorted (categories, cat,
  				     (GCompareFunc) compare_categories);
  return cat;
}

static Page *
make_page (Category *cat, const PreferencesPage *info)
{
  Page *page;

  trace ("");
  g_assert (cat);
  g_assert (info);

  page = g_new0 (Page, 1);
  page->cat = cat;
  page->label = g_strdup (info->label);
  page->make = info->make;
  page->init = info->init;
  page->apply = info->apply;
  page->data = info->user_data;

  cat->pages = g_slist_insert_sorted (cat->pages, page,
  				     (GCompareFunc) compare_pages);
  return page;
}

static inline void
set_can_apply (GnomeDialog *dialog, gboolean enable)
{
  trace ("");
  gnome_dialog_set_sensitive (dialog, 0, enable);
  gnome_dialog_set_sensitive (dialog, 1, enable);
}


/******************************************************************************
 * UI
 */

typedef struct {
  GtkCTree  *ctree;
  GtkPaned  *paned;
  Page      *current_page;
  GtkWidget *blank_page;
} DialogInfo;

static gboolean block_page_changed = FALSE;


static inline DialogInfo *
get_dialog_info (GnomeDialog *dialog)
{
  return (DialogInfo *) gtk_object_get_data (GTK_OBJECT (dialog), "dialog_info");
}

static gboolean
on_dialog_apply (GnomeDialog *dialog)
{
  DialogInfo *info;
  Page *page;

  trace ("");

  info = get_dialog_info (dialog);
  page = info->current_page;

  if (page && page->apply && page->apply (GTK_WINDOW (dialog), page->data))
  {
    set_can_apply (dialog, FALSE);
    config_save_all ();
    page->dirty = 0;
    return TRUE;
  }
  return FALSE;
}

static void
on_dialog_clicked (GnomeDialog *dialog, gint button_number)
{
  trace ("");

  switch (button_number)
  {
  case 0:
    if (on_dialog_apply (dialog))
      gtk_widget_hide (GTK_WIDGET (dialog));
    break;
  case 1:
    on_dialog_apply (dialog);
    break;
  case 2:
    gtk_widget_hide (GTK_WIDGET (dialog));
    break;
  }
}

static inline void
remove_page_widget (GtkPaned *paned)
{
  trace ("");
  g_return_if_fail (paned);

  if (paned->child2)
  {
    GtkWidget *widget = paned->child2;

    gtk_widget_ref (widget);
    gtk_container_remove (GTK_CONTAINER (paned), widget);
    gtk_widget_hide_all (widget);
  }
}

static void
on_tree_select_row (GtkCTree *ctree, GtkCTreeNode *row, gint column, GtkObject *dialog)
{
  DialogInfo *info;
  GtkPaned *paned;
  Page *page;

  trace ("");

  info = get_dialog_info (GNOME_DIALOG (dialog));
  paned = info->paned;
  page = info->current_page;

  if (page && page->dirty)
  {
  /* CANT DO THIS WHILE IN CTREE POINTER GRAB!!
    switch (dialog_question_yes_no_cancel 
    	   (GTK_WINDOW (dialog),
	  _("The current page has been modified.  Would you like to apply these changes before continuing?")))
    {
    case DIALOG_YES:
      on_dialog_apply (GNOME_DIALOG (dialog));
      break;
    case DIALOG_NO:
      set_can_apply (GNOME_DIALOG (dialog), FALSE);
      break;
    case DIALOG_CANCEL:
      return;
    }
    */
    g_print ("Warning: discarding modifications to previous page!!\n");
    set_can_apply (GNOME_DIALOG (dialog), FALSE);
    page->dirty = 0;
  }

  page = (Page *) gtk_ctree_node_get_row_data (ctree, row);
  if (page)
  {
    info->current_page = page;

    if ((page->frame == NULL) || (paned->child2 != page->frame))
    {
      remove_page_widget (paned);

      if (page->frame == NULL)
      {
        GtkViewport *viewport;

        page->scrolled_win = gtk_scrolled_window_new (NULL, NULL);
	page->frame = gtk_frame_new (page->label);
	page->widget = page->make (page->data);

	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (page->scrolled_win),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (page->frame), page->scrolled_win);

	gtk_scrolled_window_add_with_viewport (
		GTK_SCROLLED_WINDOW (page->scrolled_win), page->widget);
	viewport = GTK_VIEWPORT (GTK_BIN (page->scrolled_win)->child);
	gtk_viewport_set_shadow_type (viewport, GTK_SHADOW_NONE);
      }

      /* Initialize page to be shown */
      block_page_changed = TRUE;
      page->init (page->data);
      block_page_changed = FALSE;

      /* Show page and add to right-side panel */
      gtk_widget_show_all (page->frame);
      gtk_paned_add2 (paned, page->frame);
    }
  }
  else
  {
    remove_page_widget (paned);

    gtk_widget_show (info->blank_page);
    gtk_paned_add2 (paned, info->blank_page);
  }
}

static void
build_category_list (GtkCTree *ctree, GtkCTreeNode *parent, Category *cat)
{
  GSList *node;
  GtkCTreeNode *sibling = NULL;

  trace ("");

  for (node=cat->pages; node; node=node->next)
  {
    Page *page = LIST_DEREF (Page, node);

    sibling = gtk_ctree_insert_node (ctree, parent, NULL, &page->label, 0,
    				     NULL, NULL, NULL, NULL, TRUE, FALSE);
    gtk_ctree_node_set_shift (ctree, sibling, 0, 0, -10);
    gtk_ctree_node_set_row_data (ctree, sibling, page);
  }
}

static void
refresh_dialog (GnomeDialog *dialog)
{
  DialogInfo *info;
  GSList *node;
  GtkCTree *ctree;
  GtkCList *clist;
  GtkCTreeNode *parent;

  trace ("");

  info = get_dialog_info (dialog);
  ctree = info->ctree;
  clist = GTK_CLIST (info->ctree);

  gtk_clist_freeze (clist);
  gtk_clist_clear (clist);

  for (node=categories; node; node=node->next)
  {
    Category *cat = LIST_DEREF (Category, node);

    parent = gtk_ctree_insert_node (ctree, NULL, NULL, &cat->label, 5, 
				    NULL, NULL, NULL, NULL, FALSE, TRUE);
    build_category_list (ctree, parent, cat);
  }

  /* Select first preferences page */
  parent = gtk_ctree_node_nth (ctree, 1);
  if (parent)
    gtk_ctree_select (ctree, parent);

  gtk_clist_thaw (clist);
}

static GnomeDialog *
make_dialog (void)
{
  DialogInfo *info;
  GnomeDialog *dialog;
  GtkWidget *hpaned;
  GtkWidget *scrolled_win;
  GtkWidget *ctree;

  trace ("");

  dialog = GNOME_DIALOG (gnome_dialog_new (_("Preferences"),
  					   GNOME_STOCK_BUTTON_OK,
					   GNOME_STOCK_BUTTON_APPLY,
					   GNOME_STOCK_BUTTON_CLOSE,
					   NULL));
  gnome_dialog_close_hides (dialog, TRUE);
  gtk_window_set_policy (GTK_WINDOW (dialog), TRUE, TRUE, FALSE);
  gtk_window_set_default_size (GTK_WINDOW (dialog), 500, 400);

  gtk_signal_connect (GTK_OBJECT (dialog), "clicked",
  		      GTK_SIGNAL_FUNC (on_dialog_clicked), NULL);

  hpaned = gtk_hpaned_new ();
  gtk_box_pack_start (GTK_BOX (dialog->vbox), hpaned, TRUE, TRUE, 0);

  scrolled_win = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_win),
  				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);
  gtk_paned_add1 (GTK_PANED (hpaned), scrolled_win);

  ctree = gtk_ctree_new (1, 0);
  gtk_widget_set_usize (ctree, 130, -1);
  gtk_container_add (GTK_CONTAINER (scrolled_win), ctree);
  gtk_clist_set_column_auto_resize (GTK_CLIST (ctree), 0, TRUE);
  gtk_clist_set_selection_mode (GTK_CLIST (ctree), GTK_SELECTION_BROWSE);
  gtk_ctree_set_line_style (GTK_CTREE (ctree), GTK_CTREE_LINES_NONE);
  gtk_ctree_set_expander_style (GTK_CTREE (ctree), GTK_CTREE_EXPANDER_TRIANGLE);

  gtk_signal_connect (GTK_OBJECT (ctree), "tree_select_row",
		      GTK_SIGNAL_FUNC (on_tree_select_row), dialog);

  info = g_new0 (DialogInfo, 1);
  info->paned = GTK_PANED (hpaned);
  info->ctree = GTK_CTREE (ctree);
  info->blank_page = gtk_label_new ("");

  gtk_object_set_data (GTK_OBJECT (dialog), "dialog_info", info);

  return dialog;
}


/******************************************************************************
 * Interface
 */

gpointer
preferences_page_register (const PreferencesPage *info)
{
  Category *cat;
  GSList *node;

  trace ("");
  g_return_val_if_fail (info, NULL);
  g_return_val_if_fail (info->category, NULL);
  g_return_val_if_fail (info->label, NULL);
  g_return_val_if_fail (info->make, NULL);
  g_return_val_if_fail (info->init, NULL);
  g_return_val_if_fail (info->apply, NULL);

  node = g_slist_find_custom (categories, (gpointer) info->category, (GCompareFunc) match_category);
  if (node == NULL)
    cat = make_category (info->category);
  else
    cat = LIST_DEREF (Category, node);

  node = g_slist_find_custom (cat->pages, (gpointer) info->label, (GCompareFunc) match_page);
  if (node)
    return NULL;  /* category/label identifier is already in use */
  else  
    return make_page (cat, info);
}

void
preferences_page_unregister (gpointer key)
{
  Page *page = (Page *) key;

  trace ("");
  g_return_if_fail (page);
  g_return_if_fail (page->cat);

  page->cat->pages = g_slist_remove (page->cat->pages, page);

  if (page->widget)
    /* FIXME */
  
  g_free (page->label);
  g_free (page);
}

void
preferences_page_changed (gpointer key)
{
  if (!block_page_changed)
  {
    GnomeDialog *dialog;
    Page *page = (Page *) key;

    trace ("");
    g_return_if_fail (page);

    dialog = GNOME_DIALOG (gtk_widget_get_toplevel (page->frame));
    set_can_apply (dialog, TRUE);

    page->dirty = 1;
  }
}

void
preferences_dialog_run (GtkWindow *parent)
{
  static GnomeDialog *dialog = NULL;

  trace ("");

  if (!dialog)
    dialog = make_dialog ();

  if (GTK_WIDGET_VISIBLE (dialog))
    return;

  refresh_dialog (dialog);

  gnome_dialog_set_parent (dialog, parent);
  gnome_dialog_set_sensitive (dialog, 0, FALSE);
  gnome_dialog_set_sensitive (dialog, 1, FALSE);
  gnome_dialog_set_sensitive (dialog, 2, TRUE);

  gtk_widget_show_all (GTK_WIDGET (dialog));
}
