/******************************************************************************\
 gnofin/date-preferences-page.c   $Revision: 1.8 $
 Copyright (C) 1999-2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

//#define ENABLE_DEBUG_TRACE

#include "common.h"
#include <libgnomeui/gnome-stock.h>
#include <libgnomeui/gnome-uidefs.h>
#include <gtk/gtk.h>
#include "preferences.h"
#include "preferences-pages.h"
#include "xml-io.h"
#include "numeric-parser.h"
#include "dialogs.h"
#include "ui-record-list.h"


static gchar *clist_titles[] =
{
  N_("Field"),
  N_("Align"),
  N_("Label"),
};

static const struct
{
  guint type;
  const gchar *name;
}
fields[] =
{
  { 0,                            N_("(UNDEFINED)")     },
  { RECORD_FIELD_TYPE,            N_("TYPE")            },
  { RECORD_FIELD_DATE,            N_("DATE")            },
  { RECORD_FIELD_NUMBER,          N_("NUMBER")          },
  { RECORD_FIELD_LINKED_ACC_NAME, N_("TRANSFER_FROM")   },
  { RECORD_FIELD_CATEGORY,        N_("CATEGORY")        },
  { RECORD_FIELD_PAYEE,           N_("PAYEE")           },
  { RECORD_FIELD_MEMO,            N_("MEMO")            },
  { RECORD_FIELD_CLEARED,         N_("CLEARED_STATUS")  },
  { RECORD_FIELD_AMOUNT,          N_("AMOUNT")          },
  { RECORD_FIELD_CLEARED_BAL,     N_("CLEARED_BALANCE") },
  { RECORD_FIELD_OVERALL_BAL,     N_("OVERALL_BALANCE") },
};

static const struct
{
  guint type;
  const gchar *name;
}
aligns[] =
{
  { GTK_JUSTIFY_LEFT,   N_("Left")   },
  { GTK_JUSTIFY_RIGHT,  N_("Right")  },
  { GTK_JUSTIFY_CENTER, N_("Center") },
};

typedef struct {
  gpointer       key;
  GtkCList      *clist;
  GtkOptionMenu *field;
  GtkOptionMenu *align;
  GtkEntry      *label;
  guint          selection;
  guint          selection_changed : 1;
} PageContext;


/******************************************************************************
 * Helpers
 */

static const gchar *
stringize_field (guint field)
{
  guint i;

  trace ("");

  for (i=0; i < sizeof_array (fields); ++i)
  {
    if (fields[i].type == field)
      return _(fields[i].name);
  }
  return "";
}

static const gchar *
stringize_align (GtkJustification type)
{
  guint i;

  trace ("");

  for (i=0; i < sizeof_array (aligns); ++i)
  {
    if (aligns[i].type == type)
      return _(aligns[i].name);
  }
  return "";
}

static guint
lookup_field (const gchar *name)
{
  guint i;

  trace ("");

  for (i=0; i < sizeof_array (fields); ++i)
  {
    if (strcmp (name, _(fields[i].name)) == 0)
      return i;
  }
  return (guint) -1;
}

static guint
lookup_align (const gchar *name)
{
  guint i;

  trace ("");

  for (i=0; i < sizeof_array (aligns); ++i)
  {
    if (strcmp (name, _(aligns[i].name)) == 0)
      return i;
  }
  return (guint) -1;
}

static inline guint
parse_field (const gchar *name)
{
  guint i;

  trace ("");

  i = lookup_field (name);
  return fields[i].type;
}

static inline GtkJustification
parse_align (const gchar *name)
{
  guint i;

  trace ("");

  i = lookup_align (name);
  return aligns[i].type;
}


/******************************************************************************
 * Signals
 */

static void
on_insert_clicked (GtkButton *button, PageContext *page)
{
  const gchar *text[3];

  trace ("");

  text[0] = stringize_field (0);
  text[1] = stringize_align (GTK_JUSTIFY_LEFT);
  text[2] = "";

  gtk_clist_append (page->clist, (gchar **) text);
  gtk_clist_select_row (page->clist, page->clist->rows - 1, 0);
  gtk_clist_moveto (page->clist, page->selection, 0, 0.0, 0.0);

  preferences_page_changed (page->key);
}

static void
on_remove_clicked (GtkButton *button, PageContext *page)
{
  trace ("");

  if (page->selection != (guint) -1)
  {
    gtk_clist_remove (page->clist, page->selection);
    preferences_page_changed (page->key);
  }
}

static void
on_move_up_clicked (GtkButton *button, PageContext *page)
{
  trace ("");

  if (page->selection > 0)
  {
    gtk_clist_row_move (page->clist, page->selection, page->selection - 1);
    // page->selection has been decremented by row_move handler
    gtk_clist_moveto (page->clist, page->selection, 0, 0.5, 0.0);
  }
}

static void
on_move_down_clicked (GtkButton *button, PageContext *page)
{
  trace ("");

  if (page->selection < (page->clist->rows - 1))
  {
    gtk_clist_row_move (page->clist, page->selection, page->selection + 1);
    // page->selection has been incremented by row_move handler
    gtk_clist_moveto (page->clist, page->selection, 0, 0.5, 0.0);
  }
}

static void
on_select_row (GtkCList *clist, gint row, gint col, GdkEvent *e, PageContext *page)
{
  gchar *text;
  guint i;

  trace ("");

  page->selection = row;
  page->selection_changed = 1;

  gtk_clist_get_text (clist, row, 0, &text);
  i = lookup_field (text);
  gtk_option_menu_set_history (page->field, i);

  gtk_clist_get_text (clist, row, 1, &text);
  i = lookup_align (text);
  gtk_option_menu_set_history (page->align, i);

  gtk_clist_get_text (clist, row, 2, &text);
  gtk_entry_set_text (page->label, text);

  page->selection_changed = 0;
}

static void
on_unselect_row (GtkCList *clist, guint row, guint col, GdkEvent *e, PageContext *page)
{
  trace ("");

  page->selection = (guint) -1;
}

static void
on_row_move (GtkCList *clist, gint src, gint dest, PageContext *page)
{
  trace ("");

  page->selection = dest;
  preferences_page_changed (page->key);
}

static void
on_label_changed (GtkEntry *entry, PageContext *page)
{
  trace ("");

  if (!page->selection_changed)
  {
    const gchar *text = gtk_entry_get_text (entry);
    gtk_clist_set_text (page->clist, page->selection, 2, text);

    preferences_page_changed (page->key);
  }
}

static void
on_field_activated (GtkMenuItem *item, PageContext *page)
{
  trace ("");
  
  if (!page->selection_changed)
  {
    gchar *text;
    GtkWidget *label;

    label = GTK_BIN (page->field)->child;
    gtk_label_get (GTK_LABEL (label), &text);
    gtk_clist_set_text (page->clist, page->selection, 0, text);

    preferences_page_changed (page->key);
  }
}

static void
on_align_activated (GtkMenuItem *item, PageContext *page)
{
  trace ("");
  
  if (!page->selection_changed)
  {
    gchar *text;
    GtkWidget *label;

    label = GTK_BIN (page->align)->child;
    gtk_label_get (GTK_LABEL (label), &text);
    gtk_clist_set_text (page->clist, page->selection, 1, text);

    preferences_page_changed (page->key);
  }
}


/******************************************************************************
 * Initialization
 */

static void
build_field_menu (PageContext *page)
{
  GtkMenu *menu;
  GtkWidget *item;
  guint i;

  trace ("");

  menu = GTK_MENU (gtk_menu_new ());

  for (i=0; i < sizeof_array (fields); ++i)
  {
    item = gtk_menu_item_new_with_label (_(fields[i].name));

    gtk_signal_connect (GTK_OBJECT (item), "activate",
    			GTK_SIGNAL_FUNC (on_field_activated), page);

    gtk_widget_show (item);
    gtk_menu_append (menu, item);
  }

  gtk_option_menu_set_menu (page->field, GTK_WIDGET (menu));
}

static void
build_align_menu (PageContext *page)
{
  GtkMenu *menu;
  GtkWidget *item;
  guint i;

  trace ("");

  menu = GTK_MENU (gtk_menu_new ());

  for (i=0; i < sizeof_array (aligns); ++i)
  {
    item = gtk_menu_item_new_with_label (_(aligns[i].name));

    gtk_signal_connect (GTK_OBJECT (item), "activate",
    			GTK_SIGNAL_FUNC (on_align_activated), page);

    gtk_widget_show (item);
    gtk_menu_append (menu, item);
  }

  gtk_option_menu_set_menu (page->align, GTK_WIDGET (menu));
}

static void
refresh_page (PageContext *page)
{
  guint i, ncols;
  UI_RecordListColumnInfo *cinfo = NULL;

  trace ("");

  ncols = ui_record_list_get_column_info (&cinfo);

  gtk_clist_freeze (page->clist);
  gtk_clist_clear (page->clist);

  for (i=0; i < ncols; ++i)
  {
    gchar *text[3];

    text[0] = (gchar *) stringize_field (cinfo[i].field);
    text[1] = (gchar *) stringize_align (cinfo[i].align);
    text[2] = cinfo[i].label;

    gtk_clist_insert (page->clist, i, text);
    gtk_clist_set_row_data (page->clist, i, GUINT_TO_POINTER (cinfo[i].width));
  }

  gtk_clist_columns_autosize (page->clist);
  gtk_clist_thaw (page->clist);
}

static GtkWidget *
create_pixmap_button (const gchar *pixmap_name, const gchar *text)
{
  GtkWidget *button;
  GtkWidget *pixmap;
  GtkWidget *hbox;
  GtkWidget *label;

  trace ("");

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox);

  pixmap = gnome_stock_pixmap_widget (NULL, pixmap_name);
  gtk_widget_show (pixmap);
  gtk_box_pack_start (GTK_BOX (hbox), pixmap, FALSE, FALSE, 0);

  label = gtk_label_new (text);
  gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
  gtk_widget_show (label);
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 2);

  button = gtk_button_new ();
  gtk_container_add (GTK_CONTAINER (button), hbox);

  return button;
}

static GtkWidget *
make_page (PageContext *page)
{
  GtkWidget *toplevel;
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkWidget *label;
  GtkWidget *scrolled_win;
  GtkWidget *clist;
  GtkWidget *button;
  GtkWidget *entry;
  GtkWidget *optmenu;
  GtkWidget *frame;
  GtkWidget *table;

  trace ("");

  vbox = gtk_vbox_new (FALSE, 10);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 10);

  toplevel = vbox;

  /* Page description
   */
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

  label = gtk_label_new (_("Specify the layout of the columns in the transaction list."));
  gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

  /* Scrolled list
   */
  hbox = gtk_hbox_new (FALSE, 5);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);

  scrolled_win = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_win),
  				  GTK_POLICY_AUTOMATIC,
  				  GTK_POLICY_AUTOMATIC);
  gtk_widget_set_usize (scrolled_win, -1, 120);
  gtk_box_pack_start (GTK_BOX (hbox), scrolled_win, TRUE, TRUE, 0);

# ifdef ENABLE_NLS
  {
    gint i;
    for (i=0; i < sizeof_array (clist_titles); ++i)
      clist_titles[i] = _(clist_titles[i]);
  }
# endif

  clist = gtk_clist_new_with_titles (sizeof_array (clist_titles), clist_titles);
  gtk_clist_columns_autosize (GTK_CLIST (clist));
  gtk_clist_set_selection_mode (GTK_CLIST (clist), GTK_SELECTION_BROWSE);
  gtk_clist_set_reorderable (GTK_CLIST (clist), TRUE);
  gtk_clist_column_titles_passive (GTK_CLIST (clist));
  gtk_container_add (GTK_CONTAINER (scrolled_win), clist);

  gtk_signal_connect (GTK_OBJECT (clist), "select_row",
  		      GTK_SIGNAL_FUNC (on_select_row), page);
  gtk_signal_connect (GTK_OBJECT (clist), "unselect_row",
  		      GTK_SIGNAL_FUNC (on_unselect_row), page);
  gtk_signal_connect (GTK_OBJECT (clist), "row_move",
  		      GTK_SIGNAL_FUNC (on_row_move), page);

  page->clist = GTK_CLIST (clist);

  /* Button panel
   */
  hbox = gtk_hbox_new (TRUE, GNOME_PAD_SMALL);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

  button = create_pixmap_button (GNOME_STOCK_PIXMAP_ADD, _("Insert"));
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
  		      GTK_SIGNAL_FUNC (on_insert_clicked), page);
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, TRUE, 0);

  button = create_pixmap_button (GNOME_STOCK_PIXMAP_REMOVE, _("Remove"));
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
  		      GTK_SIGNAL_FUNC (on_remove_clicked), page);
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, TRUE, 0);

  button = create_pixmap_button (GNOME_STOCK_PIXMAP_UP, _("Up"));
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
  		      GTK_SIGNAL_FUNC (on_move_up_clicked), page);
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, TRUE, 0);

  button = create_pixmap_button (GNOME_STOCK_PIXMAP_DOWN, _("Down"));
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
  		      GTK_SIGNAL_FUNC (on_move_down_clicked), page);
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, TRUE, 0);

  /* Editor
   */
  frame = gtk_frame_new (_("Parameters"));
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);

  table = gtk_table_new (2, 5, FALSE);
  gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
  gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
  gtk_container_set_border_width (GTK_CONTAINER (table), GNOME_PAD_SMALL);
  gtk_container_add (GTK_CONTAINER (frame), table);

  optmenu = gtk_option_menu_new ();
  gtk_table_attach_defaults (GTK_TABLE (table), optmenu, 0, 3, 0, 1);
  page->field = GTK_OPTION_MENU (optmenu);

  optmenu = gtk_option_menu_new ();
  gtk_table_attach_defaults (GTK_TABLE (table), optmenu, 3, 5, 0, 1);
  page->align = GTK_OPTION_MENU (optmenu);

  entry = gtk_entry_new ();
  gtk_table_attach_defaults (GTK_TABLE (table), entry, 0, 5, 1, 2);
  page->label = GTK_ENTRY (entry);
  
  gtk_signal_connect (GTK_OBJECT (entry), "changed",
  		      GTK_SIGNAL_FUNC (on_label_changed), page);

  build_field_menu (page);
  build_align_menu (page);

  return toplevel;
}

static void 
init_page (PageContext *page)
{
  trace ("");

  refresh_page (page);
}

static gboolean
apply_page (GtkWindow *parent, PageContext *page)
{
  UI_RecordListColumnInfo *cinfo;
  guint i, n, width;
  gchar *text;

  trace ("");

  n = page->clist->rows;
  cinfo = g_new0 (UI_RecordListColumnInfo, n);

  for (i=0; i < n; ++i)
  {
    gtk_clist_get_text (page->clist, i, 0, &text);
    cinfo[i].field = parse_field (text);

    if (cinfo[i].field == 0)
    {
      dialog_error (parent, _("One or more columns have an undefined field type."));
      g_free (cinfo);
      return FALSE;
    }

    gtk_clist_get_text (page->clist, i, 1, &text);
    cinfo[i].align = parse_align (text);

    gtk_clist_get_text (page->clist, i, 2, &text);
    cinfo[i].label = text;

    width = GPOINTER_TO_UINT (gtk_clist_get_row_data (page->clist, i));
    cinfo[i].width = (width > 0) ? width : 80;
  }

  ui_record_list_set_column_info (cinfo, n);
  g_free (cinfo);

  return TRUE;
}

void
preferences_page_list_columns_init (void)
{
  PreferencesPage page = {0};
  PageContext *context;

  trace ("");

  context = g_new0 (PageContext, 1);

  page.category = _("Appearance");
  page.label = _("List Columns");
  page.make = (PreferencesPageMakeFunc) make_page;
  page.init = (PreferencesPageInitFunc) init_page;
  page.apply = (PreferencesPageApplyFunc) apply_page;
  page.user_data = context;
  
  context->key = preferences_page_register (&page);
}
