/******************************************************************************\
 gnofin/gnofin-defaults.c   $Revision: 1.5.2.1 $
 Copyright (C) 1999-2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

#include "common.h"
#include <libgnome/gnome-config.h>
#include "gnofin-defaults.h"
#include "data-if.h"
#include "config-saver.h"


/*****************************************************************************
 * Configuration
 */

typedef struct {
  GSList *record_types;
  GSList *categories;
} Config;

static const RecordTypeInfo default_record_types[] =
{
  { N_("ATM"),  N_("ATM Cash"),               0, 0, RECORD_TYPE_SIGN_NEG },
  { N_("CHK"),  N_("Check"),                  1, 0, RECORD_TYPE_SIGN_NEG },
  { N_("CC"),   N_("Check Card"),             0, 0, RECORD_TYPE_SIGN_NEG },
  { N_("DEP"),  N_("Deposit"),                0, 0, RECORD_TYPE_SIGN_POS },
  { N_("EFT"),  N_("Electronic Transaction"), 0, 0, RECORD_TYPE_SIGN_ANY },  
  { N_("FEE"),  N_("Bank Fee"),               0, 0, RECORD_TYPE_SIGN_NEG },
  { N_("TXFR"), N_("Transfer"),               0, 1, RECORD_TYPE_SIGN_ANY },
};
#define NDEFTYPES sizeof_array (default_record_types)

static const gchar *default_categories[] =
{
  N_("Deposit"),
  N_("Bonus"),
  N_("Paycheck"),
  N_("Dividend Income"),
  N_("Gift Received"),
  N_("Interest Earned"),
  N_("Refund"),
  N_("Bank Charge"),
  N_("Books"),
  N_("Auto"),
  N_("Auto:Payment"),
  N_("Auto:Repairs"),
  N_("Clothing"),
  N_("Computer"),
  N_("Computer:Hardware"),
  N_("Computer:Software"),
  N_("Computer:Service"),
  N_("Credit"),
  N_("Donation"),
  N_("Taxes"),
  N_("Taxes:State"),
  N_("Taxes:Income"),
  N_("Taxes:Property"),
  N_("Taxes:Auto"),
  N_("Taxes:Preparation"),
  N_("Furniture"),
  N_("Gasoline"),
  N_("Groceries"),
  N_("Home Repair"),
  N_("Insurance"),
  N_("Investment"),
  N_("Medical"),
  N_("Miscellaneous"),
  N_("Office"),
  N_("Mortgage"),
  N_("Pets"),
  N_("Rent"),
  N_("Savings"),
  N_("Stock Transaction"),
  N_("Utilities"),
  N_("Utilities:Power"),
  N_("Utilities:Water"),
  N_("Utilities:Telephone"),
  N_("Utilities:Cellphone"),
  N_("Utilities:Cable"),
  N_("Utilities:Internet")
};
#define NDEFCATS sizeof_array (default_categories)

#define CAT  "Defaults"
#define KEY  "/" PACKAGE "/" CAT "/"

static void
load_config (Config *config)
{
  gchar path[512];
  guint i, num;

  trace ("");
  g_return_if_fail (config);
  g_return_if_fail (config->record_types == NULL);
  g_return_if_fail (config->categories == NULL);

  /* Load default record types
   */
  g_snprintf (path, sizeof path, KEY "num_types=%d", NDEFTYPES);
  num = gnome_config_get_int (path);
  for (i=0; i<num; ++i)
  {
    RecordTypeInfo *typ = g_new0 (RecordTypeInfo, 1);

    g_snprintf (path, sizeof path, KEY "record_type[%d].name=%s", i,
	        i < NDEFTYPES ? _(default_record_types[i].name) : "");
    typ->name = gnome_config_get_string (path);

    if (strlen (typ->name) == 0)
      break;

    g_snprintf (path, sizeof path, KEY "record_type[%d].description=%s", i,
	        i < NDEFTYPES ? _(default_record_types[i].description) : "");
    typ->description = gnome_config_get_string (path);

    g_snprintf (path, sizeof path, KEY "record_type[%d].numbered=%d", i,
	        i < NDEFTYPES ? default_record_types[i].numbered : 0);
    typ->numbered = gnome_config_get_int (path);

    g_snprintf (path, sizeof path, KEY "record_type[%d].linked=%d", i,
	        i < NDEFTYPES ? default_record_types[i].linked : 0);
    typ->linked = gnome_config_get_int (path);

    g_snprintf (path, sizeof path, KEY "record_type[%d].sign=%d", i,
	        i < NDEFTYPES ? default_record_types[i].sign : RECORD_TYPE_SIGN_ANY);
    typ->sign = gnome_config_get_int (path);

    config->record_types = g_slist_prepend (config->record_types, typ);
  }
  config->record_types = g_slist_reverse (config->record_types);

  /* Load default categories
   */
  g_snprintf (path, sizeof path, KEY "num_categories=%d", NDEFCATS);
  num = gnome_config_get_int (path);
  for (i=0; i<num; ++i)
  {
    gchar *cat;

    g_snprintf (path, sizeof path, KEY "category[%d]=%s", i,
    		i < NDEFCATS ? _(default_categories[i]) : "");
    cat = gnome_config_get_string (path);

    config->categories = g_slist_prepend (config->categories, cat);
  }
  config->categories = g_slist_reverse (config->categories);
}

static void
save_config (const Config *config)
{
  GSList *node;
  gchar path[512];
  guint i, num;

  trace ("");
  g_return_if_fail (config);

  /* Save default record types
   */
  num = g_slist_length (config->record_types);
  gnome_config_set_int (KEY "num_types", num);
  for (i=0, node=config->record_types; node; node=node->next, ++i)
  {
    RecordTypeInfo *typ = LIST_DEREF (RecordTypeInfo, node);

    g_snprintf (path, sizeof path, KEY "record_type[%d].name", i);
    gnome_config_set_string (path, typ->name);

    g_snprintf (path, sizeof path, KEY "record_type[%d].description", i);
    gnome_config_set_string (path, typ->description);

    g_snprintf (path, sizeof path, KEY "record_type[%d].numbered", i);
    gnome_config_set_int (path, typ->numbered);

    g_snprintf (path, sizeof path, KEY "record_type[%d].linked", i);
    gnome_config_set_int (path, typ->linked);

    g_snprintf (path, sizeof path, KEY "record_type[%d].sign", i);
    gnome_config_set_int (path, typ->sign);
  }

  /* Save default categories
   */
  num = g_slist_length (config->categories);
  gnome_config_set_int (KEY "num_categories", num);
  for (i=0, node=config->categories; node; node=node->next, ++i)
  {
    g_snprintf (path, sizeof path, KEY "category[%d]", i);
    gnome_config_set_string (path, node->data);
  }
}

static Config *
get_config (void)
{
  static Config config = {0};
  static gboolean init = FALSE;

  if (!init)
  {
    load_config (&config);
    config_saver_register (CAT, (ConfigSaveFunc) save_config, &config);
    init = TRUE;
  }
  return &config;
}


/******************************************************************************
 * Interface
 */

void
install_default_record_types (Bankbook *book)
{
  Config *config = get_config ();
  GSList *node;

  trace ("");
  g_return_if_fail (book);

  for (node=config->record_types; node; node=node->next)
    if_bankbook_insert_record_type (book, LIST_DEREF (RecordTypeInfo, node));
}

void
install_default_categories (Bankbook *book)
{
  Config *config = get_config ();
  GSList *node;

  trace ("");
  g_return_if_fail (book);

  for (node=config->categories; node; node=node->next)
    if_bankbook_insert_category (book, _(LIST_DEREF (const gchar, node)));
}

const GSList *
get_default_record_types (void)
{
  trace ("");
  return get_config ()->record_types;
}

const GSList *
get_default_categories (void)
{
  trace ("");
  return get_config ()->categories;
}

void
set_default_record_types (GSList *types)
{
  Config *config = get_config ();

  trace ("");

  /* Free old defaults */
  {
    GSList *node;
    for (node=config->record_types; node; node=node->next)
    {
      RecordTypeInfo *typ = LIST_DEREF (RecordTypeInfo, node);

      g_free (typ->name);
      g_free (typ->description);
      g_free (typ);
    }
    g_slist_free (config->record_types);
  }

  /* Assume reference to passed-in record type list */
  config->record_types = types;
}

void
set_default_categories (GSList *cats)
{
  trace ("");
  g_warning ("Unimplemented");
}

void
restore_factory_default_record_types (void)
{
  trace ("");
  g_warning ("Unimplemented");
}

void
restore_factory_default_categories (void)
{
  trace ("");
  g_warning ("Unimplemented");
}

void
restore_factory_defaults (void)
{
  trace ("");
  restore_factory_default_record_types ();
  restore_factory_default_categories ();
}
