/* A custom widget for selecting resources
 *
 * The code is based on the TICTACTOE and DIAL examples
 * http://www.gtk.org/tutorial/app-codeexamples.html
 * http://www.gtk.org/tutorial/sec-gtkdial.html
 *
 * Adaptation for Gnocatan: 2004 Roland Clobus
 *
 */

#include "config.h"
#include <gtk/gtksignal.h>
#include <gtk/gtktable.h>
#include <gtk/gtktooltips.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkspinbutton.h>
#include <string.h>
#include <stdio.h>
#include <glib.h>

#include "resource-table.h"

#include "callback.h"
/* The signals */
enum {
	CHANGE,
	LAST_SIGNAL
};

static void resource_table_class_init(ResourceTableClass *klass);
static void resource_table_init(ResourceTable *rt);
static void resource_table_update(ResourceTable *rt);

static void less_resource_cb(GtkButton *widget, gpointer user_data);
static void more_resource_cb(GtkButton *widget, gpointer user_data);
static void value_changed_cb(GtkSpinButton *widget, gpointer user_data);

/* All signals */
static guint resource_table_signals[LAST_SIGNAL] = { 0 };

/* Register the class */
GType resource_table_get_type(void)
{
	static GType rt_type = 0;

	if (!rt_type) {
		static const GTypeInfo rt_info = {
			sizeof(ResourceTableClass),
			NULL, /* base_init */
			NULL, /* base_finalize */
			(GClassInitFunc)resource_table_class_init,
			NULL, /* class_finalize */
			NULL, /* class_data */
			sizeof(ResourceTable),
			0,
			(GInstanceInitFunc)resource_table_init,
			NULL
		};
		rt_type = g_type_register_static(
				GTK_TYPE_TABLE, "ResourceTable",
				&rt_info, 0);
	}
	return rt_type;
}

/* Register the signals.
 * ResourceTable will emit two signals:
 * 'change'         when any change in the amount occurs.
 */
static void resource_table_class_init(ResourceTableClass *klass)
{
	resource_table_signals[CHANGE] = g_signal_new(
			"change",
			G_TYPE_FROM_CLASS(klass),
			G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
			G_STRUCT_OFFSET(ResourceTableClass, change),
			NULL,
			NULL,
			g_cclosure_marshal_VOID__VOID,
			G_TYPE_NONE,
			0);
}

/* Initialise the composite widget */
static void resource_table_init(ResourceTable *rt)
{
	int i;
	
	for (i = 0; i < NO_RESOURCE; i++) {
		rt->row[i].hand = 0;
		rt->row[i].bank = 0;
		rt->row[i].amount = 0;
		rt->row[i].hand_widget = NULL;
		rt->row[i].bank_widget = NULL;
		rt->row[i].amount_widget = NULL;
		rt->row[i].less_widget = NULL;
		rt->row[i].more_widget = NULL;
	}
	rt->total_target = 0;
	rt->total_current = 0;
	rt->total_widget = NULL;

	rt->with_bank = FALSE;
	rt->with_total = FALSE;	
}

/* Create a new ResourceTable */
GtkWidget* resource_table_new(
		const gchar *title, 
		gboolean with_bank,
		gboolean with_total)
{
	ResourceTable *rt;

	GtkTooltips *tooltips;
	gchar temp[256]; /* @TODO Fixed length?? */
	GtkWidget *widget;
	gint i;
	gint row;

	rt = g_object_new(resource_table_get_type(), NULL);

	rt->with_bank = with_bank;
	/* Don't set rt->with_total yet, wait for _wait_total */

	rt->bank_offset = with_bank ? 1 : 0;
	gtk_table_resize(GTK_TABLE(rt), 
			NO_RESOURCE + 1 + with_total ? 1 : 0, 
			5 + rt->bank_offset);
	gtk_table_set_row_spacings(GTK_TABLE(rt), 3);
	gtk_table_set_col_spacings(GTK_TABLE(rt), 5);

	tooltips = gtk_tooltips_new();
	sprintf(temp, "<b>%s</b>", title);
	widget = gtk_label_new(NULL);
	gtk_label_set_markup(GTK_LABEL(widget), temp);
	gtk_widget_show(widget);
	gtk_table_attach_defaults(GTK_TABLE(rt), widget, 
			0, 5 + rt->bank_offset, 0, 1);
	gtk_misc_set_alignment(GTK_MISC(widget), 0, 0.5);

	row = 1;
	for (i = 0; i < NO_RESOURCE; i++) {
		widget = gtk_label_new(resource_name(i, TRUE));
		gtk_widget_show(widget);
		gtk_table_attach_defaults(GTK_TABLE(rt), widget,
				0, 1, row, row + 1);
		gtk_misc_set_alignment(GTK_MISC(widget), 0.0, 0.5);

		widget = rt->row[i].hand_widget = 
				gtk_entry_new();
		gtk_widget_show(widget);
		gtk_table_attach_defaults(GTK_TABLE(rt), widget,
				1, 2, row, row + 1);
		gtk_entry_set_width_chars(GTK_ENTRY(widget), 3);
		gtk_widget_set_sensitive(widget, FALSE);
		gtk_entry_set_alignment(GTK_ENTRY(widget), 1.0);
		
		rt->row[i].hand = resource_asset(i);

		widget = rt->row[i].less_widget =
				gtk_button_new_with_label(_("<less"));
		gtk_widget_set_sensitive(widget, FALSE);
		g_signal_connect(G_OBJECT(widget), "clicked",
				G_CALLBACK(less_resource_cb), &rt->row[i]);
		gtk_widget_show(widget);
		gtk_table_attach_defaults(GTK_TABLE(rt), widget, 
				2, 3, row, row + 1);

		if (with_bank) {
			widget = rt->row[i].bank_widget = 
					gtk_entry_new();
			gtk_widget_show(widget);
			gtk_table_attach_defaults(GTK_TABLE(rt), widget,
					3, 4, row, row + 1);
			gtk_entry_set_width_chars(GTK_ENTRY(widget), 3);
			gtk_widget_set_sensitive(widget, FALSE);
			gtk_entry_set_alignment(GTK_ENTRY(widget), 1.0);
		}
		widget = rt->row[i].more_widget =
				gtk_button_new_with_label(_("more>"));
		gtk_widget_set_sensitive(widget, FALSE);
		g_signal_connect(G_OBJECT(widget), "clicked",
				G_CALLBACK(more_resource_cb), &rt->row[i]);
		gtk_widget_show(widget);
		gtk_table_attach_defaults(GTK_TABLE(rt), widget, 
				3 + rt->bank_offset, 4 + rt->bank_offset, 
				row, row + 1);

		widget = rt->row[i].amount_widget =
				gtk_spin_button_new_with_range(0, 99, 1);
		gtk_widget_show(widget);
		gtk_table_attach_defaults(GTK_TABLE(rt), widget, 
				4 + rt->bank_offset, 5 + rt->bank_offset, 
				row, row + 1);
		gtk_entry_set_width_chars(GTK_ENTRY(widget), 3);
		gtk_entry_set_alignment(GTK_ENTRY(widget), 1.0);
		g_signal_connect(G_OBJECT(widget), "value-changed",
				G_CALLBACK(value_changed_cb),
				&rt->row[i]);

		rt->row[i].limit = rt->row[i].hand;
		row++;
	}
	resource_table_update(rt);
	return GTK_WIDGET(rt);
}

void resource_table_set_total(ResourceTable *rt, const gchar *text, gint total) {
	GtkWidget *widget;
	gint row;
	gint i;

	g_assert(IS_RESOURCETABLE(rt));
	rt->with_total = TRUE;
	rt->total_target = total;
	rt->total_current = 0;
	row = NO_RESOURCE + 1;

	widget = gtk_label_new(text);
	gtk_widget_show(widget);
	gtk_table_attach_defaults(GTK_TABLE(rt), widget, 
			0, 4 + rt->bank_offset, 
			row, row + 1);
	gtk_misc_set_alignment(GTK_MISC(widget), 1.0, 0.5);
	
	widget = rt->total_widget =
			gtk_spin_button_new_with_range(0, total, 1);
	gtk_widget_show(widget);
	gtk_table_attach_defaults(GTK_TABLE(rt), widget, 
			4 + rt->bank_offset, 5 + rt->bank_offset, 
			row, row + 1);
	gtk_entry_set_width_chars(GTK_ENTRY(widget), 3);
	gtk_widget_set_sensitive(widget, FALSE);
	gtk_entry_set_alignment(GTK_ENTRY(widget), 1.0);
	
	for (i = 0; i < NO_RESOURCE; i++) {
		rt->row[i].limit = MIN(rt->row[i].limit, total);
		gtk_spin_button_set_range(
			GTK_SPIN_BUTTON(rt->row[i].amount_widget), 0, rt->row[i].limit);
		gtk_widget_set_sensitive(rt->row[i].amount_widget, rt->row[i].limit > 0);
	}
	resource_table_update(rt);
}

/* Update the display to the current state */
static void resource_table_update(ResourceTable *rt)
{
	gchar buff[16];
	gint i;
	struct _ResourceRow *row;

	g_assert(IS_RESOURCETABLE(rt));
	rt->total_current = 0;
	for (i = 0; i < NO_RESOURCE; i++)
		rt->total_current += rt->row[i].amount;

	for (i = 0; i < NO_RESOURCE; i++) {
		row = &rt->row[i];

		sprintf(buff, "%d", row->amount);
		gtk_entry_set_text(GTK_ENTRY(row->amount_widget), buff);
	
		if (rt->with_total) {
			gtk_widget_set_sensitive(row->less_widget, row->amount > 0);
			if (rt->total_current >= rt->total_target)
				gtk_widget_set_sensitive(row->more_widget, FALSE);
			else
				gtk_widget_set_sensitive(row->more_widget, row->amount < row->limit);
			sprintf(buff, "%d", rt->total_current);
			gtk_entry_set_text(GTK_ENTRY(rt->total_widget), buff);
		}
		if (rt->with_bank)
			sprintf(buff, "%d", row->hand + row->amount);
		else
			sprintf(buff, "%d", row->hand - row->amount);

		gtk_entry_set_text(GTK_ENTRY(row->hand_widget), buff);
	
		if (rt->with_bank) {
			sprintf(buff, "%d", row->bank - row->amount);
			gtk_entry_set_text(GTK_ENTRY(row->bank_widget), buff);
		}
	}
}

static void less_resource_cb(GtkButton *widget, gpointer user_data) {
	struct _ResourceRow *row = user_data;
	ResourceTable *rt =
		RESOURCETABLE(gtk_widget_get_parent(GTK_WIDGET(widget)));

	row->amount--;
	resource_table_update(rt);
	g_signal_emit(G_OBJECT(rt), resource_table_signals[CHANGE], 0);
}

static void more_resource_cb(GtkButton *widget, gpointer user_data) {
	struct _ResourceRow *row = user_data;
	ResourceTable *rt =
		RESOURCETABLE(gtk_widget_get_parent(GTK_WIDGET(widget)));

	row->amount++;
	resource_table_update(rt);
	g_signal_emit(G_OBJECT(rt), resource_table_signals[CHANGE], 0);
}

static void value_changed_cb(GtkSpinButton *widget, gpointer user_data) {
	struct _ResourceRow *row = user_data;
	ResourceTable *rt =
		RESOURCETABLE(gtk_widget_get_parent(GTK_WIDGET(widget)));

	row->amount = gtk_spin_button_get_value_as_int(widget);
	resource_table_update(rt);
	g_signal_emit(G_OBJECT(rt), resource_table_signals[CHANGE], 0);
}

void resource_table_get_amount(ResourceTable *rt, gint *amount)
{
	gint i;

	g_assert(IS_RESOURCETABLE(rt));
	for (i = 0; i < NO_RESOURCE; i++)
		amount[i] = rt->row[i].amount;
}

gboolean resource_table_is_total_reached(ResourceTable *rt)
{
	g_assert(IS_RESOURCETABLE(rt));
	return (rt->total_current == rt->total_target);
}
