with Ada.Calendar; use Ada.Calendar;
with Ada.Text_IO; use Ada.Text_IO;
with Tries;
with Ada.Numerics.Discrete_Random;
with HTables; use HTables;

procedure Test_Trie is
   package Char_Random is new Ada.Numerics.Discrete_Random (Character);
   use Char_Random;

   procedure Free (Data : in out Integer);
   package Integer_Tries is new Tries (Integer, Integer'First);
   use Integer_Tries;

   type Header_Num is range 1 .. 20_000;
   function Hash is new HTables.Hash (Header_Num);
   package Integer_Hash is new Simple_HTable
     (Header_Num, Integer, Free, Integer'First, String, Hash, "=");
   use Integer_Hash;

   Length_Inserted : Natural := 0;
   Random_Items_Count : constant := 30_000;
   Tree : Trie_Tree;

   procedure Dump is new Integer_Tries.Dump (Integer'Image);

   procedure Free (Data : in out Integer) is
      pragma Unreferenced (Data);
   begin
      null;
   end Free;

   procedure Insert_And_Dump
     (Index : String; Value : Integer; Scenario : Integer);
   --  Insert a new entry in the tree, and dump it

   procedure Check_Iter (Prefix : String);
   --  Prints all items starting with Prefix

   procedure Insert_And_Dump
     (Index : String; Value : Integer; Scenario : Integer)
   is
   begin
      New_Line;
      Put_Line ("Inserting " & Index & "=>" & Value'Img
                & " (scenario" & Scenario'Img & ")");
      Insert (Tree, Index, Value);
      Length_Inserted := Length_Inserted + Index'Length;
      Dump (Tree);
      Put_Line ("Length inserted=" & Length_Inserted'Img);
   end Insert_And_Dump;

   procedure Check_Iter (Prefix : String) is
      Iter : Integer_Tries.Iterator;
   begin
      New_Line;
      Put_Line ("All items starting with """ & Prefix & '"');
      Iter := Start (Tree, Prefix);
      loop
         exit when Get (Iter) = Integer'First;
         Put (Get_Key (Iter) & Get (Iter)'Img & ' ');
         Next (Iter);
      end loop;
      New_Line;
   end Check_Iter;

begin
   Insert_And_Dump ("MANU",        1, 1);
   Insert_And_Dump ("FOO",         2, 4);
   Insert_And_Dump ("MAN",         3, 2);
   Insert_And_Dump ("MAN",         4, 3);
   Insert_And_Dump ("MANUEL",      5, 4);
   Insert_And_Dump ("MANUTENTION", 6, 4);
   Insert_And_Dump ("MANUELAA",    7, 4);
   Insert_And_Dump ("MANUTENTAL",  8, 1);

   Put_Line ("Value for MANU is " & Get (Tree, "MANU")'Img);

   Check_Iter ("MANU");
   Check_Iter ("F");
   Check_Iter ("");
   Check_Iter ("MANUWW");
   Check_Iter ("TI");

   New_Line;
   Put_Line ("Clear the tree");
   Clear (Tree);
   Dump (Tree);

   New_Line;
   Put_Line ("Inserting" & Random_Items_Count'Img & " items at random");
   declare
      Gen : Generator;
      Len : Integer;
      Start : constant Time := Clock;
   begin
      Reset (Gen);
      Length_Inserted := 0;

      for Count in 1 .. Random_Items_Count loop
         Len := Character'Pos (Random (Gen)) + 5;
         Length_Inserted := Length_Inserted + Len;
         declare
            S : String (1 .. Len);
         begin
            for L in S'Range loop
               S (L) := Random (Gen);
            end loop;
            Insert (Tree, S, 1);
         end;
      end loop;

      Put_Line ("Time=" & Duration'Image (Clock - Start));

      Dump (Tree, Size_Only => True);
      Put_Line ("Length_Inserted=" & Length_Inserted'Img);
   end;

   New_Line;
   Put_Line ("Inserting" & Random_Items_Count'Img & " items in hash table");
   declare
      Table : HTable;
      Gen : Generator;
      Len : Integer;
      Start : constant Time := Clock;
   begin
      Reset (Gen);
      Length_Inserted := 0;

      for Count in 1 .. Random_Items_Count loop
         Len := Character'Pos (Random (Gen)) + 5;
         Length_Inserted := Length_Inserted + Len;
         declare
            S : String (1 .. Len);
         begin
            for L in S'Range loop
               S (L) := Random (Gen);
            end loop;
            Set (Table, S, 1);
         end;
      end loop;

      Put_Line ("Time=" & Duration'Image (Clock - Start));
   end;

end Test_Trie;

