/**
 * GMyth Library
 *
 * @file gmyth/gmyth_programinfo.c
 * 
 * @brief <p> GMythProgramInfo representing the program info, with the
 * configuration data to the actual remote file in the TV chain.
 *
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia.
 * @author Rosfran Borges <rosfran.borges@indt.org.br>
 * @author Leonardo Sobral Cunha <leonardo.cunha@indt.org.br>
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "gmyth_programinfo.h"
#include "gmyth_util.h"
#include "gmyth_debug.h"

static void     gmyth_program_info_class_init(GMythProgramInfoClass *
                                              klass);
static void     gmyth_program_info_init(GMythProgramInfo * object);

static void     gmyth_program_info_dispose(GObject * object);
static void     gmyth_program_info_finalize(GObject * object);

G_DEFINE_TYPE(GMythProgramInfo, gmyth_program_info, G_TYPE_OBJECT)
    static const gchar *gmyth_program_info_non_null_value(const GString *
                                                          str);

    static void     gmyth_program_info_class_init(GMythProgramInfoClass *
                                                  klass)
{
    GObjectClass   *gobject_class = G_OBJECT_CLASS(klass);

    gobject_class->dispose = gmyth_program_info_dispose;
    gobject_class->finalize = gmyth_program_info_finalize;
}

static void
gmyth_program_info_init(GMythProgramInfo * gmyth_program_info)
{
    gmyth_program_info->chancommfree = 0;

    /** A flag informing if the program has video or not. */
    gmyth_program_info->isVideo = FALSE;
    gmyth_program_info->lenMins = 0;

    gmyth_program_info->stars = 0.0f;
    gmyth_program_info->repeat = 0;

    gmyth_program_info->hasAirDate = FALSE;

    gmyth_program_info->spread = 0;
    gmyth_program_info->startCol = 0;

    gmyth_program_info->recpriority2 = 0;
    gmyth_program_info->reactivate = 0;

    gmyth_program_info->recordid = 0;
    gmyth_program_info->parentid = 0;

    /** The backend video source id associated to this program.*/
    gmyth_program_info->sourceid = 0;
    /** the backend input id associated to this program.*/
    gmyth_program_info->inputid = 0;
    /** The backend card id associated to this program.*/
    gmyth_program_info->cardid = 0;
    gmyth_program_info->shareable = FALSE;
    gmyth_program_info->duplicate = FALSE;

    gmyth_program_info->findid = 0;

    gmyth_program_info->programflags = 0;
    gmyth_program_info->transcoder = 0;

    gmyth_program_info->recpriority = 0;

    /** The file size of the recorded program.*/
    gmyth_program_info->filesize = -1;
}

static void
gmyth_program_info_dispose(GObject * object)
{
    GMythProgramInfo *gmyth_program_info = GMYTH_PROGRAM_INFO(object);

    /** The program start time. */
    g_free(gmyth_program_info->startts);

    /** The program end time. */
    g_free(gmyth_program_info->endts);

    /** The recording schedule start time. */
    g_free(gmyth_program_info->recstartts);

    /** The recording schedule end time */
    g_free(gmyth_program_info->recendts);

    /** The program title. */
    if (gmyth_program_info->title != NULL) {
        g_string_free(gmyth_program_info->title, TRUE);
        gmyth_program_info->title = NULL;
    }

    /** The program subtitle. */
    if (gmyth_program_info->subtitle != NULL) {
        g_string_free(gmyth_program_info->subtitle, TRUE);
        gmyth_program_info->subtitle = NULL;
    }

    /** The program description. */
    if (gmyth_program_info->description != NULL) {
        g_string_free(gmyth_program_info->description, TRUE);
        gmyth_program_info->description = NULL;
    }

    /** The program category. */
    if (gmyth_program_info->category != NULL) {
        g_string_free(gmyth_program_info->category, TRUE);
        gmyth_program_info->category = NULL;
    }

    if (gmyth_program_info->chanstr != NULL) {
        g_string_free(gmyth_program_info->chanstr, TRUE);
        gmyth_program_info->chanstr = NULL;
    }

    if (gmyth_program_info->chansign != NULL) {
        g_string_free(gmyth_program_info->chansign, TRUE);
        gmyth_program_info->chansign = NULL;
    }

    /** The associated channel name. */
    if (gmyth_program_info->channame != NULL) {
        g_string_free(gmyth_program_info->channame, TRUE);
        gmyth_program_info->channame = NULL;
    }

    if (gmyth_program_info->chanOutputFilters != NULL) {
        g_string_free(gmyth_program_info->chanOutputFilters, TRUE);
        gmyth_program_info->chanOutputFilters = NULL;
    }

    if (gmyth_program_info->seriesid != NULL) {
        g_string_free(gmyth_program_info->seriesid, TRUE);
        gmyth_program_info->chanOutputFilters = NULL;
    }

    /** The program unique id. */
    if (gmyth_program_info->program_id != NULL) {
        g_string_free (gmyth_program_info->program_id, TRUE);
        gmyth_program_info->program_id = NULL;
    }

    if (gmyth_program_info->catType != NULL) {
        g_string_free(gmyth_program_info->catType, TRUE);
        gmyth_program_info->catType = NULL;
    }

    if (gmyth_program_info->sortTitle != NULL) {
        g_string_free(gmyth_program_info->sortTitle, TRUE);
        gmyth_program_info->sortTitle = NULL;
    }

    if (gmyth_program_info->year != NULL) {
        g_string_free(gmyth_program_info->year, TRUE);
        gmyth_program_info->year = NULL;
    }

    g_free(gmyth_program_info->originalAirDate);

    g_free(gmyth_program_info->lastmodified);

    g_free(gmyth_program_info->lastInUseTime);

    if (gmyth_program_info->schedulerid != NULL) {
        g_string_free(gmyth_program_info->schedulerid, TRUE);
        gmyth_program_info->schedulerid = NULL;
    }

    if (gmyth_program_info->recgroup != NULL) {
        g_string_free(gmyth_program_info->recgroup, TRUE);
        gmyth_program_info->recgroup = NULL;
    }

    if (gmyth_program_info->playgroup != NULL) {
        g_string_free(gmyth_program_info->playgroup, TRUE);
        gmyth_program_info->playgroup = NULL;
    }

    /** The file name of the recorded program.*/
    if (gmyth_program_info->pathname != NULL) {
        g_string_free(gmyth_program_info->pathname, TRUE);
        gmyth_program_info->pathname = NULL;
    }

    if (gmyth_program_info->hostname != NULL) {
        g_string_free(gmyth_program_info->hostname, TRUE);
        gmyth_program_info->hostname = NULL;
    }

    G_OBJECT_CLASS(gmyth_program_info_parent_class)->dispose(object);
}

static void
gmyth_program_info_finalize(GObject * object)
{
    g_signal_handlers_destroy(object);

    G_OBJECT_CLASS(gmyth_program_info_parent_class)->finalize(object);
}

/**
 * Creates a new instance of GMythProgramInfo.
 * 
 * @return a new instance of GMythProgramInfo.
 */
GMythProgramInfo *
gmyth_program_info_new(void)
{
    GMythProgramInfo *program_info =
        GMYTH_PROGRAM_INFO(g_object_new(GMYTH_PROGRAM_INFO_TYPE, NULL));

    return program_info;
}

/**
 * Converts an instance of a GMythProgramInfo, to a GMythStringList.
 * 
 * @param prog A GMythProgramInfo instance.
 * @param slist The GMythStringList to be passed to this function, in order to
 * 							give the responsibility of the string list creation to the
 * 						  API user.
 * 
 * @return a GMythStringList with the program info fields.
 */
GMythStringList *
gmyth_program_info_to_string_list(GMythProgramInfo * prog,
                                  GMythStringList * slist)
{
    g_return_val_if_fail(prog != NULL, NULL);
    g_return_val_if_fail(slist != NULL, NULL);

    gmyth_string_list_append_string(slist, prog->title);    /* 0 */
    gmyth_string_list_append_string(slist, prog->subtitle); /* 1 */
    gmyth_string_list_append_string(slist, prog->description);  /* 2 */
    gmyth_string_list_append_string(slist, prog->category); /* 3 */
    gmyth_string_list_append_int (slist, prog->channel_id);   /* 4 */
    gmyth_string_list_append_string(slist, prog->chanstr);  /* 5 */
    gmyth_string_list_append_string(slist, prog->chansign); /* 6 */
    gmyth_string_list_append_string(slist, prog->channame); /* 7 */
    gmyth_string_list_append_string(slist, prog->pathname); /* 8 */
    gmyth_string_list_append_int64(slist, 0);   /* 9 */

    // fixme
    // gmyth_string_list_append_int64 (slist, 100/*prog->filesize*/); /* 9 
    // 
    // 
    // */
    // gmyth_string_list_append_int (slist, 0); /* 10 */

    if (prog->startts)
        gmyth_string_list_append_int(slist, prog->startts->tv_sec); /* 11 */// DATETIME_TO_LIST(startts)
    else
        gmyth_string_list_append_int(slist, 0);

    if (prog->endts)
        gmyth_string_list_append_int(slist, prog->endts->tv_sec);   /* 12 */// DATETIME_TO_LIST(endts)
    else
        gmyth_string_list_append_int(slist, 0);

    gmyth_string_list_append_int(slist, prog->duplicate);   /* 13 */
    gmyth_string_list_append_int(slist, prog->shareable);   /* 14 */
    gmyth_string_list_append_int(slist, prog->findid);  /* 15 */
    gmyth_string_list_append_string(slist, prog->hostname); /* 16 */
    gmyth_string_list_append_int(slist, prog->sourceid);    /* 17 */
    gmyth_string_list_append_int(slist, prog->cardid);  /* 18 */
    gmyth_string_list_append_int(slist, prog->inputid); /* 19 */
    gmyth_string_list_append_int(slist, prog->recpriority); /* 20 */
    gmyth_string_list_append_int(slist, 0 /* prog->recstatus */ );  /* 21 */
    gmyth_string_list_append_int(slist, prog->recordid);    /* 22 */
    gmyth_string_list_append_int(slist, 0 /* prog->rectype */ );    /* 23 */
    gmyth_string_list_append_int(slist, 0 /* prog->dupin */ );  /* 24 */
    gmyth_string_list_append_int(slist, 0 /* prog->dupmethod */ );  /* 25 */
    gmyth_string_list_append_int(slist, prog->recstartts != NULL ? prog->recstartts->tv_sec : 0);   /* 26 
     */// DATETIME_TO_LIST(recstartts)
    gmyth_string_list_append_int(slist, prog->recendts != NULL ? prog->recendts->tv_sec : 0);   /* 27 
     */// DATETIME_TO_LIST(recendts)
    gmyth_string_list_append_int(slist, prog->repeat);  /* 28 */
    gmyth_string_list_append_int(slist, prog->programflags);    /* 29 */
    gmyth_string_list_append_char_array(slist, "Default");  /* 30 */// prog->(recgroup 
                                                            // 
    // 
    // != "") ?
    // recgroup :
    // "Default")
    gmyth_string_list_append_int(slist, prog->chancommfree);    /* 31 */
    gmyth_string_list_append_string(slist, prog->chanOutputFilters);    /* 32 
                                                                         */
    gmyth_string_list_append_string(slist, prog->seriesid); /* 33 */
    gmyth_string_list_append_string(slist, prog->program_id);    /* 34 */
    gmyth_string_list_append_char_array(slist, ""); /* 35 */
    gmyth_string_list_append_int(slist, prog->lastmodified != NULL ? prog->lastmodified->tv_sec : 0);   /* 36 
     */// DATETIME_TO_LIST(lastmodified)
    gmyth_string_list_append_int(slist, 0); /* 37 */// FLOAT_TO_LIST(stars)
    gmyth_string_list_append_int(slist, prog->originalAirDate != NULL ? prog->originalAirDate->tv_sec : 0); /* 38 
     */// DATETIME_TO_LIST(QDateTime(originalAirDate))
    gmyth_string_list_append_int(slist, prog->hasAirDate);  /* 39 */
    gmyth_string_list_append_char_array(slist, "Default");  /* 40 */// prog->(playgroup 
                                                            // 
    // 
    // != "") ?
    // playgroup : 
    // "Default")
    gmyth_string_list_append_int(slist, prog->recpriority2);    /* 41 */

    return slist;
}

/**
 * Converts an instance of a GMythStringList, to a GMythProgramInfo.
 * 
 * @param slist The GMythStringList got from the MythTV backend server.
 * @param pos   The position in the GMythStringList to start getting the fields.
 * 
 * @return a GMythProgramInfo representing the string list got from network.
 */
GMythProgramInfo *
gmyth_program_info_from_string_list_from_pos(GMythStringList * slist,
                                             guint pos)
{
    GMythProgramInfo *prog = gmyth_program_info_new();

    g_return_val_if_fail(slist != NULL &&
                         gmyth_string_list_get_string(slist, pos) != NULL,
                         NULL);

    prog->title = gmyth_string_list_get_string(slist, pos);
    prog->subtitle = gmyth_string_list_get_string(slist, pos + 1);
    prog->description = gmyth_string_list_get_string(slist, pos + 2);
    prog->category = gmyth_string_list_get_string(slist, pos + 3);
    prog->channel_id = gmyth_string_list_get_int (slist, pos + 4);
    prog->channame = gmyth_string_list_get_string(slist, pos + 5);
    prog->chanstr = gmyth_string_list_get_string(slist, pos + 6);
    prog->chansign = gmyth_string_list_get_string(slist, pos + 7);
    prog->pathname = gmyth_string_list_get_string(slist, pos + 8);

    prog->filesize = gmyth_string_list_get_int64(slist, pos + 9);

    gmyth_debug("Prog info: [ %s, %s, %s, %s, %s, %s, %s, %s, %d ]\n",
                gmyth_program_info_non_null_value(prog->title),
                gmyth_program_info_non_null_value(prog->subtitle),
                gmyth_program_info_non_null_value(prog->description),
                gmyth_program_info_non_null_value(prog->category),
                gmyth_program_info_non_null_value(prog->channame),
                gmyth_program_info_non_null_value(prog->chanstr),
                gmyth_program_info_non_null_value(prog->chansign),
                gmyth_program_info_non_null_value(prog->pathname),
                gmyth_string_list_get_int(slist, pos + 11));

    prog->startts = gmyth_util_string_to_time_val((gmyth_util_time_to_isoformat((time_t) gmyth_string_list_get_int(slist, pos + 11)))->str);    // DATETIME_TO_LIST(startts)
    prog->endts = gmyth_util_string_to_time_val((gmyth_util_time_to_isoformat((time_t) gmyth_string_list_get_int(slist, pos + 12)))->str);  // DATETIME_TO_LIST(endts)
    prog->duplicate = gmyth_string_list_get_int(slist, pos + 13);
    prog->shareable = gmyth_string_list_get_int(slist, pos + 14);
    prog->findid = gmyth_string_list_get_int(slist, pos + 15);
    prog->hostname = gmyth_string_list_get_string(slist, pos + 16);
    prog->sourceid = gmyth_string_list_get_int(slist, pos + 17);
    prog->cardid = gmyth_string_list_get_int(slist, pos + 18);
    prog->inputid = gmyth_string_list_get_int(slist, pos + 19);
    prog->recpriority = gmyth_string_list_get_int(slist, pos + 20);
    prog->reactivate = gmyth_string_list_get_int(slist, pos + 21);
    prog->recordid = gmyth_string_list_get_int(slist, pos + 22);
    gmyth_string_list_get_int(slist, pos + 23);
    gmyth_string_list_get_int(slist, pos + 24);
    gmyth_string_list_get_int(slist, pos + 25);
    prog->recstartts = gmyth_util_string_to_time_val((gmyth_util_time_to_isoformat((time_t) gmyth_string_list_get_int(slist, pos + 26)))->str); // DATETIME_TO_LIST(recstartts)
    prog->recendts = gmyth_util_string_to_time_val((gmyth_util_time_to_isoformat((time_t) gmyth_string_list_get_int(slist, pos + 27)))->str);   // DATETIME_TO_LIST(recendts)
    prog->repeat = gmyth_string_list_get_int(slist, pos + 28);
    prog->programflags = gmyth_string_list_get_int(slist, pos + 29);
    prog->recgroup = gmyth_string_list_get_string(slist, pos + 30); // prog->(recgroup 
                                                                    // 
    // 
    // !=
    // "") 
    // ?
    // recgroup 
    // :
    // "Default")
    prog->chancommfree = gmyth_string_list_get_int(slist, pos + 31);
    prog->chanOutputFilters =
        gmyth_string_list_get_string(slist, pos + 32);
    prog->seriesid = gmyth_string_list_get_string(slist, pos + 33);
    prog->program_id = gmyth_string_list_get_string(slist, pos + 34);
    gmyth_string_list_get_string(slist, pos + 35);
    prog->lastmodified = gmyth_util_string_to_time_val((gmyth_util_time_to_isoformat((time_t) gmyth_string_list_get_int(slist, pos + 36)))->str);   // DATETIME_TO_LIST(lastmodified)
    gmyth_string_list_get_int(slist, pos + 37); // FLOAT_TO_LIST(stars)
    prog->originalAirDate = gmyth_util_string_to_time_val((gmyth_util_time_to_isoformat((time_t) gmyth_string_list_get_int(slist, pos + 38)))->str);    // DATETIME_TO_LIST(QDateTime(originalAirDate))
    prog->hasAirDate = gmyth_string_list_get_int(slist, pos + 39);
    prog->playgroup = gmyth_string_list_get_string(slist, pos + 40);    // prog->(playgroup 
                                                                        // 
    // 
    // != 
    // "") 
    // ? 
    // playgroup 
    // : 
    // "Default")
    prog->recpriority2 = gmyth_string_list_get_int(slist, pos + 41);

    return prog;
}

/**
 * Converts an instance of a GMythStringList, to a GMythProgramInfo.
 * 
 * @param slist The GMythStringList got from the MythTV backend server.
 * 
 * @return a GMythProgramInfo representing the string list got from network.
 */
GMythProgramInfo *
gmyth_program_info_from_string_list(GMythStringList * slist)
{
    GMythProgramInfo *prog = NULL;

    g_return_val_if_fail(slist != NULL, NULL);

    prog = gmyth_program_info_from_string_list_from_pos(slist, 0);

    return prog;
}

/**
 * Converts an instance of a GMythStringList, to a GMythProgramInfo ( NEXT_PROGRAM_INFO ).
 * 
 * @param slist The GMythStringList got from the MythTV backend server.
 * 
 * @return a GMythProgramInfo representing the string list got from network.
 */
GMythProgramInfo *
gmyth_program_info_from_string_list_next_prog(GMythStringList * slist)
{
    GMythProgramInfo *prog = gmyth_program_info_new();

    g_return_val_if_fail(slist != NULL, NULL);

    /*
     * {Home Improvement[]:[]No, No, Godot[]:[] US sitcom about a
     * DIY-obsessed father of three. When Tim tries to sell off his extra 
     * ticket for an ice hockey game, he is arrested as a tout.[]:[]
     * Sitcom[]:[]2007-04-18T15:30:00[]:[]2007-04-18T16:00:00[]:[]ABC1[]:[]
     * /home/hmelo/.mythtv/channels/abc1.jpg[]:[]abc1[]:[]2000[]:[]25725844[]:[] 
     * } 
     */

    prog->title = gmyth_string_list_get_string(slist, 0);
    prog->subtitle = gmyth_string_list_get_string(slist, 1);
    prog->description = gmyth_string_list_get_string(slist, 2);
    prog->category = gmyth_string_list_get_string(slist, 3);
    prog->startts = gmyth_util_string_to_time_val(gmyth_string_list_get_char_array(slist, 4));  // DATETIME_TO_LIST(startts)
    prog->endts = gmyth_util_string_to_time_val(gmyth_string_list_get_char_array(slist, 5));    // DATETIME_TO_LIST(endts)
    prog->channame = gmyth_string_list_get_string(slist, 6);
    prog->chansign = gmyth_string_list_get_string(slist, 7);
    prog->chanstr = gmyth_string_list_get_string(slist, 8);
    prog->channel_id = gmyth_string_list_get_int (slist, 9);
    prog->filesize = gmyth_string_list_get_int64(slist, 10);

    gmyth_debug
        ("NEXT program info: [ %s, %s, %s, %s, %s, %s, %s, %s ]\n",
         gmyth_program_info_non_null_value(prog->title),
         gmyth_program_info_non_null_value(prog->subtitle),
         gmyth_program_info_non_null_value(prog->description),
         gmyth_program_info_non_null_value(prog->category),
         gmyth_program_info_non_null_value(prog->channame),
         gmyth_program_info_non_null_value(prog->chanstr),
         gmyth_program_info_non_null_value(prog->chansign),
         gmyth_program_info_non_null_value(prog->pathname));

    return prog;
}

static const gchar *
gmyth_program_info_non_null_value(const GString * str)
{
    return (str != NULL && str->str != NULL
            && strlen(str->str) > 0 ? str->str : " ");
}

/**
 * Prints out an instance of a GMythProgramInfo.
 * 
 * @param prog A GMythProgramInfo instance.
 * 
 * @return a string representing the program info.
 */
const gchar    *
gmyth_program_info_to_string(const GMythProgramInfo * prog)
{
    return
        g_strdup_printf
        ("Title: %s, Subtitle: %s, Description: %s, Category: %s, Channel ID: %d, "
         "Channel Name: %s, Chan str: %s, Channel Sign: %s, Path Name: %s, File Size: %lld, \n"
         "Start TS: %s, End TS: %s, Duplicate: %d, Shareable: %d, Find ID: %d, Hostname: %s, "
         "Source ID: %d, Vard ID: %d, Input ID: %d, Rec Priority: %d, Reactivate: %d, \n"
         "Record ID: %d, Rec Start TS: %s, Rec End TS: %s, Repeat: %d, Program Flags: %d, "
         "Rec Group: %s, Channel Comm Free: %d, Channel Output Filters: %s, Series ID: %s, \n"
         "Program ID: %s, Last Modified Date: %s, Original Air Date: %s, Has Air Date: %d, "
         "Play Group: %s.\n",
         gmyth_program_info_non_null_value(prog->title),
         gmyth_program_info_non_null_value(prog->subtitle),
         gmyth_program_info_non_null_value(prog->description),
         gmyth_program_info_non_null_value(prog->category),
         prog->channel_id,
         gmyth_program_info_non_null_value(prog->channame),
         gmyth_program_info_non_null_value(prog->chanstr),
         gmyth_program_info_non_null_value(prog->chansign),
         gmyth_program_info_non_null_value(prog->pathname), prog->filesize,
         gmyth_util_time_to_string_from_time_val(prog->startts),
         gmyth_util_time_to_string_from_time_val(prog->endts),
         prog->duplicate, prog->shareable, prog->findid,
         gmyth_program_info_non_null_value(prog->hostname), prog->sourceid,
         prog->cardid, prog->inputid, prog->recpriority, prog->reactivate,
         prog->recordid,
         gmyth_util_time_to_string_from_time_val(prog->recstartts),
         gmyth_util_time_to_string_from_time_val(prog->recendts),
         prog->repeat, prog->programflags,
         gmyth_program_info_non_null_value(prog->recgroup),
         prog->chancommfree,
         gmyth_program_info_non_null_value(prog->chanOutputFilters),
         gmyth_program_info_non_null_value(prog->seriesid),
         gmyth_program_info_non_null_value(prog->program_id),
         gmyth_util_time_to_string_from_time_val(prog->lastmodified),
         gmyth_util_time_to_string_from_time_val(prog->originalAirDate),
         prog->hasAirDate,
         gmyth_program_info_non_null_value(prog->playgroup));
}

/**
 * Say if an instance of a GMythProgramInfo is equals to another one.
 * 
 * @param prog The first GMythProgramInfo instance.
 * @param prog The second GMythProgramInfo instance.
 * 
 * @return <code>true</code>, if the program infos are equals.
 */
gboolean
gmyth_program_info_is_equals(const GMythProgramInfo * prog1,
                             const GMythProgramInfo * prog2)
{
    if ((strcmp(gmyth_program_info_non_null_value(prog1->title),
                gmyth_program_info_non_null_value(prog2->title)) == 0)
        ||
        (strcmp
         (gmyth_program_info_non_null_value(prog1->pathname),
          gmyth_program_info_non_null_value(prog2->pathname)) == 0))
        return TRUE;
    else
        return FALSE;

}
