/* gmoo - a gtk+ based graphical MOO/MUD/MUSH/... client
 * Copyright (C) 1999-2000 Gert Scholten
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "perlscript.h"
#ifdef PERL

#undef PACKAGE
#undef _

#include <EXTERN.h>
#include <perl.h>
#include <XSUB.h>

#include <string.h>
#include <glib.h>

#include "script.h"

PerlInterpreter *perl_interp;
void perl_register_functions();

void gm_perl_init(const char *programname) {
    char *perl_args[] = { "", "-e", "0" };
    char load_file[]=
        "sub __load_file__() {\n"
        "  (my $file_name) = @_;\n"
        "  open FH, $file_name or return 2;\n"
        "  local($/) = undef;\n"
        "  $file = <FH>;\n"
        "  close FH;\n"
        "  eval $file;\n"
        "  eval $file if $@;\n"
        "  return 1 if $@;\n"
        "  return 0;\n"
        "}\n";
    if(debug) printf("\tIniting perl interpreter...\n");

    perl_interp = perl_alloc();
    perl_construct(perl_interp);
    perl_parse(perl_interp, NULL, 3, perl_args, NULL);
    perl_eval_pv(load_file, TRUE);

    perl_register_functions();
}

void gm_perl_exit() {
    if(debug) printf("Exiting perl interpreter\n");
    perl_destruct(perl_interp);
    perl_free(perl_interp);
}

int gm_perl_is_perl_file(const char *filename) {
    char *s;
    if((s = strrchr(filename, '.'))) {
        if(g_strcasecmp(s + 1, PERL_EXTENTION) == 0) {
            return TRUE;
        }
    }
    return FALSE;
}

void gm_perl_load_file(const char *filename) {
    if(debug) printf("\tLoading Perl file: %s\n", filename);
    gm_perl_exec("__load_file__", filename);
}

char *escape_single_quotes(const char *args) {
    char *ret = g_malloc(strlen(args) * 2 + 1);
    int i, j = 0;

    for(i = 0; args[i]; i++) {
        if(args[i] == '\'' || args[i] == '\\')
            ret[j++] = '\\';
        ret[j] = args[i];
        j++;
    }
    ret[j] = '\0';

    return ret;
}

void gm_perl_exec(const char *_command, const char *argstr) {
    char *args;
    char *command;
    if(argstr) {
        args = escape_single_quotes(argstr);
        command = g_strdup_printf("&%s('%s')", _command, args);
    } else {
        args = NULL;
        command = g_strdup_printf("&%s()", _command);
    }
    if(debug) printf("Perl: %s\n", command);
    perl_eval_pv(command, TRUE);

    g_free(command);
    g_free(args);
}


void gm_perl_do_macro(int id, const char *macro_name, const char *argstr) {
    char *args    = escape_single_quotes(argstr);
    char *command = g_strdup_printf("&%s(%d, '%s')", macro_name, id, args);

    perl_eval_pv(command, FALSE);
    
    g_free(command);
    g_free(args);
}

/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/

XS(c_perl_print);
XS(c_perl_println);
XS(c_perl_msg);
XS(c_perl_write);
XS(c_perl_writeln);
XS(c_perl_name);
XS(c_perl_hostname);
XS(c_perl_port);
XS(c_perl_connected);
XS(c_perl_width);
XS(c_perl_height);
XS(c_perl_register_open);
XS(c_perl_register_close);
XS(c_perl_register_connect);
XS(c_perl_register_disconnect);

XS(c_perl_register_macro);
XS(c_perl_register_start);
XS(c_perl_register_end);
XS(c_perl_version);
XS(c_perl_msgbox);
XS(c_perl_input);

#define register(nspace, name, fname) newXS(nspace "::" name, fname, nspace);

void perl_register_functions() {
    register("WORLD", "print",     c_perl_print);
    register("WORLD", "println",   c_perl_println);
    register("WORLD", "tell",      c_perl_println);
    register("WORLD", "msg",       c_perl_msg);
    register("WORLD", "write",     c_perl_write);
    register("WORLD", "writeln",   c_perl_writeln);
    register("WORLD", "send",      c_perl_writeln);
    register("WORLD", "name",      c_perl_name);
    register("WORLD", "hostname",  c_perl_hostname);
    register("WORLD", "port",      c_perl_port);
    register("WORLD", "connected", c_perl_connected);
    register("WORLD", "width",     c_perl_width);
    register("WORLD", "height",    c_perl_height);
    register("WORLD", "register_open",       c_perl_register_open);
    register("WORLD", "register_close",      c_perl_register_close);
    register("WORLD", "register_connect",    c_perl_register_connect);
    register("WORLD", "register_disconnect", c_perl_register_disconnect);

    register("CLIENT", "register_macro", c_perl_register_macro);
    register("CLIENT", "register_start", c_perl_register_start);
    register("CLIENT", "register_end",   c_perl_register_end);
    register("CLIENT", "version",        c_perl_version);
    register("CLIENT", "msgbox",         c_perl_msgbox);
    register("CLIENT", "input",          c_perl_input);
}

#define ARG_S(i) SvPV(ST(i), junk);
#define ARG_I(i) SvIV(ST(i));

#define return_int(i) XSRETURN_IV(i)
#define return_str(s) XSRETURN_PV(s)
#define return_undef  XSRETURN_UNDEF

XS(c_perl_print) {
    int id; char *text;
    int junk; dXSARGS; 
    
    if(items != 2) XSRETURN(0);
    id   = ARG_I(0);
    text = ARG_S(1);

    return_int(gm_script_do_print(id, text));
}

XS(c_perl_println) {
    int id; char *text;
    int junk; dXSARGS; 
    
    if(items != 2) XSRETURN(0);
    id   = ARG_I(0);
    text = ARG_S(1);

    return_int(gm_script_do_println(id, text));
}

XS(c_perl_msg) {
    int id; char *text;
    int junk; dXSARGS; 
    
    if(items != 2) XSRETURN(0);
    id   = ARG_I(0);
    text = ARG_S(1);

    return_int(gm_script_do_msg(id, text));
}

XS(c_perl_write) {
    int id; char *text;
    int junk; dXSARGS; 
    
    if(items != 2) XSRETURN(0);
    id   = ARG_I(0);
    text = ARG_S(1);

    return_int(gm_script_do_write(id, text));
}

XS(c_perl_writeln) {
    int id; char *text;
    int junk; dXSARGS; 
    
    if(items != 2) XSRETURN(0);
    id   = ARG_I(0);
    text = ARG_S(1);

    return_int(gm_script_do_writeln(id, text));
}

XS(c_perl_name) {
    int id; char *s;
    dXSARGS; 
    
    if(items != 1) XSRETURN(0);
    id   = ARG_I(0);

    if(gm_script_do_name(id, &s)) {
	return_str(s);
    }
    return_undef;
}

XS(c_perl_hostname) {
    int id; char *s;
    dXSARGS; 
    
    if(items != 1) XSRETURN(0);
    id   = ARG_I(0);

    if(gm_script_do_hostname(id, &s)) {
	return_str(s);
    }
    return_undef;
}

XS(c_perl_port) {
    int id; int i;
    dXSARGS; 
    
    if(items != 1) XSRETURN(0);
    id   = ARG_I(0);

    if(gm_script_do_port(id, &i)) {
	return_int(i);
    }
    return_undef;
}

XS(c_perl_connected) {
    int id; int i;
    dXSARGS; 
    
    if(items != 1) XSRETURN(0);
    id   = ARG_I(0);

    if(gm_script_do_connected(id, &i)) {
	return_int(i);
    }
    return_undef;
}

XS(c_perl_width) {
    int id; int i;
    dXSARGS; 
    
    if(items != 1) XSRETURN(0);
    id   = ARG_I(0);

    if(gm_script_do_width(id, &i)) {
	return_int(i);
    }
    return_undef;
}

XS(c_perl_height) {
    int id; int i;
    dXSARGS; 
    
    if(items != 1) XSRETURN(0);
    id   = ARG_I(0);

    if(gm_script_do_height(id, &i)) {
	return_int(i);
    }
    return_undef;
}

XS(c_perl_register_open) {
    char *name;
    int junk; dXSARGS; 

    if(items != 1) XSRETURN(0);
    name = ARG_S(0);

    return_int(gm_script_do_register_open(PERL_TYPE, name));
}

XS(c_perl_register_close) {
    char *name;
    int junk; dXSARGS; 

    if(items != 1) XSRETURN(0);
    name = ARG_S(0);

    return_int(gm_script_do_register_close(PERL_TYPE, name));
}

XS(c_perl_register_connect) {
    char *name;
    int junk; dXSARGS; 

    if(items != 1) XSRETURN(0);
    name = ARG_S(0);

    return_int(gm_script_do_register_connect(PERL_TYPE, name));
}

XS(c_perl_register_disconnect) {
    char *name;
    int junk; dXSARGS; 

    if(items != 1) XSRETURN(0);
    name = ARG_S(0);

    return_int(gm_script_do_register_disconnect(PERL_TYPE, name));
}



XS(c_perl_register_macro) {
    char *mname, *fname, *desc = "";
    int junk; dXSARGS; 

    if(items != 2 && items != 3) XSRETURN(0);
    mname = ARG_S(0);
    fname = ARG_S(1);
    if(items == 3) desc = ARG_S(2);

    return_int(gm_script_do_register_macro(PERL_TYPE, mname, fname, desc));
}

XS(c_perl_register_start) {
    char *name;
    int junk; dXSARGS; 

    if(items != 1) XSRETURN(0);
    name = ARG_S(0);

    return_int(gm_script_do_register_start(PERL_TYPE, name));
}

XS(c_perl_register_end) {
    char *name;
    int junk; dXSARGS; 

    if(items != 1) XSRETURN(0);
    name = ARG_S(0);

    return_int(gm_script_do_register_end(PERL_TYPE, name));
}

XS(c_perl_version) {
    dXSARGS;
    if(items != 0) XSRETURN(0);

    return_str(VERSION);
}

XS(c_perl_msgbox) {
    char *title, *type, *buttons, *text; int i;
    int def = -1;
    int junk; dXSARGS;

    if(items != 4 && items != 5) XSRETURN(0);

    title = ARG_S(0);
    type = ARG_S(1);
    buttons = ARG_S(2);
    text = ARG_S(3);
    if(items == 5) def = ARG_I(4);

    if(gm_script_do_msgbox(title, type, buttons, text, def, &i)) {
	return_int(i);
    }
    return_undef;
}

XS(c_perl_input) {
    char *title, *type, *desc, *def = "";
    int type_int;
    static char *ret_str = NULL; int ret_int;
    int junk; dXSARGS;

    if(items != 3 && items != 4) XSRETURN(0);

    title = ARG_S(0);
    type = ARG_S(1);
    desc = ARG_S(2);
    if(items == 4) def = ARG_S(3);

    g_free(ret_str); ret_str = NULL; /* static variable agains memleaks :( */

    if(gm_script_do_input(title, desc, type, &type_int, def,
			  &ret_str, &ret_int)) {
	if(type_int == TYPE_STR) {
	    return_str(ret_str);
	} else if(type_int == TYPE_INT) {
	    return_int(ret_int);
	}
    }
    return_undef;
}


#endif /* PERL */
