/* gmoo - a gtk+ based graphical MOO/MUD/MUSH/... client
 * Copyright (C) 1999-2000 Gert Scholten
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <string.h>
#include <stdio.h>

#include "config.h"

#include "history.h"
#include "settings.h"

GList *go_up  (world *w, GList *lis, const char *keyt);
GList *go_down(world *w, GList *lis, const char *keyt);

void history_load(world *w);
void history_store(world *w);


void gm_history_new(world *w) {
    w->history = g_malloc(sizeof(history));
    w->history->list = NULL;
    w->history->count = 0;
    w->history->current = NULL;
    history_load(w);
}

void gm_history_destroy(world *w) {
    history_store(w);
    g_free(w->history->current);
    w->history->list = g_list_first(w->history->list);
    g_list_foreach(w->history->list, (GFunc) g_free, NULL);
    g_list_free(w->history->list);
    g_free(w->history);
}


void gm_history_add (world *w, const char *text) {
    GList *l;
    w->history->list = g_list_first(w->history->list);
    if(w->history->list && w->history->list->data == NULL) {
        w->history->list = g_list_remove_link(w->history->list,
                                              w->history->list);
    }
    w->history->list = g_list_first(g_list_prepend(w->history->list,
                                                   g_strdup(text)));
    g_free(w->history->current);
    w->history->current = NULL;

    if(w->history->count >= settings->history_size) {
        while((l = g_list_nth(w->history->list, settings->history_size))) {
            g_free(l->data);
            w->history->list = g_list_remove_link(w->history->list, l);
        }
    } else {
        w->history->count++;
    }
}

#define get_text() gtk_editable_get_chars(GTK_EDITABLE(w->input), 0, gtk_text_get_length(GTK_TEXT(w->input)))
#define delete_text() gtk_editable_delete_text(GTK_EDITABLE(w->input), 0, gtk_text_get_length(GTK_TEXT(w->input)));
#define set_text(t) delete_text(); gtk_text_insert(GTK_TEXT(w->input), NULL, NULL, NULL, t, strlen(t))

void gm_history_up(world *w) {
    if(!settings->history || !w->history->list) {
	return;
    }
    if(w->history->current == NULL) {
	w->history->list = g_list_prepend(w->history->list, NULL);
	w->history->current = get_text();
	w->history->list = go_up(w, g_list_first(w->history->list),
				 w->history->current);
	if(!w->history->list->data) {
	    w->history->list =
		g_list_remove_link(w->history->list, 
				   w->history->list);
	    g_free(w->history->current);
	    w->history->current = NULL;
	    return;
	}
    } else {
	if(w->history->list->next)
	    w->history->list = go_up(w, w->history->list,
				     w->history->current);
    }
    set_text((char *) w->history->list->data);
}

void gm_history_down(world *w) {
    GList *l;

    if(!settings->history || !w->history->list || !w->history->current) {
        return;
    }
    if(w->history->list->prev->data == NULL) {
        set_text(w->history->current);
        g_free(w->history->current);
        w->history->current = NULL;
        w->history->list = g_list_first(w->history->list);
        w->history->list = g_list_remove_link(w->history->list,
                                              w->history->list);
    } else {
        l = go_down(w, w->history->list, w->history->current);
        if(l == w->history->list) {
            w->history->list = g_list_next(g_list_first(w->history->list));
            return gm_history_down(w);
        }
        w->history->list = l;
        set_text((char *) w->history->list->data);
    }
}

int compare(world *w, const char *key, const char *str, int len) {
    if(settings->history_sensitive) {
        return strncmp(key, str, len) == 0 ? TRUE : FALSE;
    }
    return strncasecmp(key, str, len) == 0 ? TRUE : FALSE;
}

GList *go_up(world *w, GList *list, const char *key) {
    GList *ret;
    int len;
    if(!settings->history_selective) {
        return g_list_next(list);
    }
    ret = list;
    len = strlen(key);
    while((list = g_list_next(list))) {
        if(len <= strlen((char *) list->data) &&
           compare(w, key, (char *) list->data, len)) {
            ret = list;
            break;
        }
    }
    return ret;
}

GList *go_down(world *w, GList *list, const char *key) {
    GList *ret;
    int len;
    if(!settings->history_selective) {
        return g_list_previous(list);
    }
    ret = list;
    len = strlen(key);
    while((list = g_list_previous(list)) && list->data) {
        if(len <= strlen((char *) list->data) &&
           compare(w, key, (char *) list->data, len)) {
            ret = list;
            break;
        }
    }
    return ret;
}

#define history_file(world) \
g_strconcat(gm_settings_get_worlds_dir(), "/", world->p->name, \
	    "/" WORLD_HISTORY_FILE, NULL)

void replace(char *s, char old, char new) {
    int i;
    for(i = 0; s[i]; i++) {
	if(s[i] == old) s[i] = new;
    }
}

void read_history(world *w, FILE *file) {
    int len, maxlen = 0;
    char *buf = NULL;
    while(fscanf(file, "%d ", &len) != EOF) {
	len++;
	if(len > maxlen) buf = g_realloc(buf, (maxlen = len) + 1);
	fgets(buf, len, file);
	fgetc(file);
	replace(buf, '\x01', '\n');
	gm_history_add(w, buf);
    }
    g_free(buf);
}

void history_load(world *w) {
    char *filename = history_file(w);
    FILE *file;
    if(settings->history_store) {
        if((file = fopen(filename, "r"))) {
            read_history(w, file);
            fclose(file);
        }
    }
    g_free(filename);
}

void write_history(const char *line, FILE *file) {
    char *s;
    int len;

    if(!line || !file) return;
    len = strlen(line);
    s = g_strdup(line);
    replace(s, '\n', '\x01');
    fprintf(file, "%d ", len);
    fputs(s, file);
    fputc('\n', file);
    g_free(s);
}

void history_store(world *w) {
    char *filename = history_file(w);
    FILE *file;
    GList *l;
    if(settings->history_store && w->history && w->history->count) {
        if((file = fopen(filename, "w"))) {
            for(l = g_list_last(w->history->list); l; l = g_list_previous(l)) {
                write_history(l->data, file);
            }
            fclose(file);
        }
    } else {
        remove(filename);
    }
    g_free(filename);
}

