/*
  Copyright (c) 2005-2006 by Jakob Schroeter <js@camaya.net>
  This file is part of the gloox library. http://camaya.net/gloox

  This software is distributed under a license. The full license
  agreement can be found in the file LICENSE in this distribution.
  This software may not be copied, modified, sold or distributed
  other than expressed in the named license agreement.

  This software is distributed without any warranty.
*/

/*! @mainpage gloox API Documentation
 *
 * @section contents Contents
 * @ref intro_sec <br>
 * @ref handlers_sec <br>
 * @ref comp_sec <br>
 * @ref client_sec <br>
 * @ref block_conn_sec <br>
 * @ref roster_sec <br>
 * @ref privacy_sec <br>
 * @ref auth_sec <br>
 * @ref msg_sec <br>
 * @ref jeps_sec <br>
 * @ref filetransfer_sec <br>
 * <br>
 *
 * @section intro_sec Introduction
 *
 * The design of gloox follows the so-called observer pattern, which basically means that everything is
 * event-driven. There are two ways you can connect to the Jabber/XMPP network using gloox, either as
 * client or as component. A third way, as server, is not supported by gloox, even though it might be
 * possible to get something going.
 *
 * @note Section 11.5 of the XMPP specification (RFC 3290) requires that only UTF-8 is used as encoding
 * for any traffic sent over the wire. Since gloox cannot know which encoding is used in any given input,
 * it is a requirement that any input to gloox is valid UTF-8.
 *
 * @section handlers_sec Event Handlers
 *
 * The most important tools of gloox are the event handlers. Currently, there exist 4 handlers for
 * the basic protocol as defined in the RFCs, as well as 8 handlers for events generated by the included
 * JEP-implementations. Additionally, a log handler, a generic tag handler and a handler for connection
 * events are available.
 *
 * Basically these handlers are virtual interfaces from which you derive a class and implement a few
 * virtual functions. Then you register such an object with the respective protocol implementation. A
 * short example:
 * @code
 * class MyClass : public PresenceHandler
 * {
 *   public:
 *     // reimplemented from PresenceHandler
 *     virtual void handlePresence( Stanza *stanza );
 *
 *   [...]
 * };
 *
 * void MyClass::handlePresence( Stanza *stanza )
 * {
 *   // extract further information from the stanza
 * }
 * @endcode
 *
 * Somewhere else you do something like this:
 * @code
 * OtherClass::doSomething()
 * {
 *   Client *client = new Client( ... );
 *   [...]
 *   MyClass *handler = new MyClass( ... );
 *   client->registerPresenceHandler( handler );
 * }
 * @endcode
 *
 * Now, everytime a presence stanza (not subscription stanza) is received, handlePresence() is called
 * with the current stanza as argument. You can then use the extensive getters of the Stanza class to
 * extract stanza data.
 *
 * This works similar for all the other event handlers.
 * Another example, this time using the connection event handler (class @link gloox::ConnectionListener
 * ConnectionListener @endlink):
 * @code
 * class MyClass : public ConnectionListener
 * {
 *   public:
 *     virtual void onConnect();
 *
 *     virtual bool onTLSConnect( ... );
 * };
 *
 * void MyClass::onConnect()
 * {
 *   // do something when the connection is established
 * }
 *
 * bool MyClass::onTLSConnect( const CertInfo& info )
 * {
 *   // decide whether you trust the certificate, examine the CertInfo structure
 *   return true; // if you trust it, otherwise return false
 * }
 * @endcode
 *
 * @note The ConnectionListener interface is a peculiarity. You MUST re-implement
 * @link gloox::ConnectionListener::onTLSConnect() ConnectionListener::onTLSConnect() @endlink if
 * you want to be able to connect successfully to TLS/SSL enabled servers. Even though gloox tries
 * to verify the server's certificate it does not automatically trust a server. The client programmer
 * and/or user have to decide whether to trust a server or not. This trust is expressed by the return
 * value of onTLSConnect(). @b False means you don't trust the server/certificate and as a consequence
 * the connection is dropped immediately.
 *
 * @section comp_sec Components
 *
 * A component in the Jabber/XMPP network is an add-on to a server which runs externally
 * to the actual server software, but can have similar privileges. Components use a protocol described in
 * JEP-0114 to connect and authenticate to a server.
 *
 * The @link gloox::Component Component @endlink class supports this protocol and can be used to create
 * a new Jabber component. It's as simple as:
 * @code
 * Component *comp = new Component( ... );
 * comp->connect();
 * @endcode
 *
 * @section client_sec Clients
 *
 * A client can be an end-user's chat client, a bot, or a similar entity not tied to a particular
 * server. The @link gloox::Client Client @endlink class implements the necessary functionality to
 * connect to an XMPP server. Usage is, again, pretty simple:
 * @code
 * class MyClass : public ConnectionListener, PresenceHandler
 * {
 *   public:
 *     void doSomething();
 *
 *     virtual void handlePresence( ... );
 *
 *     virtual void onConnect();
 *
 *     virtual bool onTLSConnect( const CertInfo& info );
 * };
 *
 * void MyClass::doSomething()
 * {
 *   JID jid( "jid@server/resource" );
 *   Client *client = new Client( jid, "password" );
 *   client->registerConnectionListener( this );
 *   client->registerPresenceHandler( this );
 *   client->connect();
 * }
 *
 * void MyClass::onConnect()
 * {
 *   // connection established, auth done (see API docs for exceptions)
 * }
 *
 * bool MyClass::onTLSConnect( const CertInfo& info )
 * {
 *   // examine certificate info
 * }
 *
 * void MyClass::handlePresence( Stanza *stanza )
 * {
 *   // presence info
 * }
 * @endcode
 *
 * @note gloox does not (and will not) support the style of connection which is usually used on port
 * 5223, i.e. SSL encryption before any XML is sent, because it's a legacy method and not standard XMPP.
 *
 * @note @link gloox::Client::connect() Client::connect() @endlink by default blocks until the
 * connection ends (either @link gloox::Client::disconnect() Client::disconnect() @endlink is called
 * or the server closes the connection).
 *
 * @section block_conn_sec Blocking vs. Non-blocking Connections
 *
 * For some kind of bots a blocking connection (the default behaviour) is ideal. All the bot does is
 * react to events coming from the server. However, for end user clients or anything with a GUI this
 * is far from perfect.
 *
 * In these cases non-blocking connections can be used. If
 * @link gloox::ClientBase::connect() ClientBase::connect( false ) @endlink is
 * called, the function returnes immediately after the connection has been established. It is then
 * the resposibility of the programmer to initiate receiving of data from the socket.
 *
 * The easiest way is to call @link gloox::ClientBase::recv() ClientBase::recv() @endlink
 * periodically with the desired timeout (in seconds) as parameter. The default value of -1
 * means the call blocks until any data was received, which is then parsed automatically.
 *
 * As an alternative to periodic polling you can use
 * @link gloox::ClientBase::fileDescriptor() ClientBase::fileDescriptor() @endlink to get a hold
 * of the raw file descriptor used for the connection. You can then use select() on it and use
 * @link gloox::ClientBase::recv() ClientBase::recv() @endlink when select indicates that data is
 * available. You should @b not recv() any data from the file descriptor directly as there is no
 * way to feed that back into the parser.
 *
 * @section roster_sec Roster Management
 *
 * Among others, RFC 3921 defines the protocol to manage one's contact list (roster). In gloox, the
 * @link gloox::RosterManager RosterManager @endlink class implements this functionality. A few
 * easy-to-use functions are available to subscribe to or unsubscribe from the presence of remote
 * entities. It is also possible to add a contact to a roster without actually subscribing to the
 * contacts presence. Additionally, the interface @link gloox::RosterListener RosterListener @endlink
 * offers many callbacks for various roster-related events.
 *
 * If you create a Client object as shown above, you also get a RosterManager for free.
 * @link gloox::Client::rosterManager() Client::rosterManager() @endlink returns a pointer to the
 * object.
 *
 * @section privacy_sec Privacy Lists
 *
 * Also defined in RFC 3921: Privacy Lists. A Privacy List can be used to explicitely block or allow
 * sending of stanzas from and to contacts, respectively. You can define rules based on JID, stanza type,
 * etc. Needless to say that gloox implements Privacy Lists as well. ;) The
 * @link gloox::PrivacyManager PrivacyManager @endlink class and the
 * @link gloox::PrivacyListHandler PrivacyListHandler @endlink virtual interface allow for full
 * flexibility in Privacy List handling.
 *
 * @code
 * PrivacyManager *p = new PrivacyManager( ... );
 * [...]
 * PrivacyListHandler::PrivacyList list;
 * PrivacyItem item( PrivacyItem::TYPE_JID, PrivacyItem::ACTION_DENY,
 *                   PrivacyItem::PACKET_MESSAGE, "me@there.com" );
 * list.push_back( item );
 *
 * PrivacyItem item2( PrivacyItem::TYPE_JID, PrivacyItem::ACTION_ALLOW,
 *                    PrivacyItem::PACKET_IQ, "me@example.org" );
 * list.push_back( item2 );
 *
 * p->store( "myList", list );
 * @endcode
 *
 * @section auth_sec Authentication
 *
 * gloox supports old-style IQ-based authentication defined in JEP-0078 as well as several SASL mechanisms.
 * See the documentation of the @link gloox::Client Client @endlink class for more information.
 *
 * @section msg_sec Sending and Receiving of Chat Messages
 *
 * For Messaging it is recommended to use the MessageSession interface. It handles sending and receiving
 * of messages as well as message events and chat states (such as typing notifications, etc.). See
 * @link gloox::MessageSession MessageSession @endlink for more details.
 *
 * @section jeps_sec Protocol Enhancements
 *
 * The Jabber Software Foundation has published a number of extensions to the core protocols, called
 * Jabber Enhancement Proposals (JEPs). A couple of these JEPs are implemented in gloox:
 *
 * @li JEP-0004 @link gloox::DataForm Data Forms @endlink
 * @li JEP-0012 @link gloox::LastActivity  Last Activity @endlink
 * @li JEP-0013 @link gloox::FlexibleOffline Flexible Offline Message Retrieval @endlink
 * @li JEP-0022 Message Events (see @link gloox::MessageSession MessageSession @endlink for examples)
 * @li JEP-0030 @link gloox::Disco Service Discovery @endlink
 * @li JEP-0047 @link gloox::InBandBytestreamManager In-Band Bytestreams @endlink
 * @li JEP-0048 @link gloox::BookmarkStorage Bookmark Storage @endlink
 * @li JEP-0049 @link gloox::PrivateXML Private XML Storage @endlink
 * @li JEP-0050 @link gloox::Adhoc Ad-hoc Commands @endlink
 * @li JEP-0054 @link gloox::VCardManager vcard-temp @endlink
 * @li JEP-0077 @link gloox::Registration In-Band Registration @endlink
 * @li JEP-0078 Non-SASL Authentication (automatically used if the server does not support SASL)
 * @li JEP-0083 Nested Roster Groups (automatically used if supported by the server. see
 * @link gloox::RosterManager::delimiter() RosterManager @endlink)
 * @li JEP-0085 Chat State Notifications (see @link gloox::MessageSession MessageSession @endlink for
 * examples)
 * @li JEP-0092 Software Version (integrated into @link gloox::Disco Service Discovery @endlink)
 * @li JEP-0114 @link gloox::Component Jabber Component Protocol @endlink
 * @li JEP-0138 Stream Compression (used automatically if gloox is compiled with zlib and if the server
 * supports it)
 * @li JEP-0145 @link gloox::Annotations Annotations @endlink
 *
 * @section filetransfer_sec File Transfer
 *
 * For file transfer there is currently an implementation of JEP-0047 (In-Band Bytestreams). However,
 * this protocol is probably not suited for offering file transfer to end-users. No other file
 * transfer protocols are currently supported. See
 * @link gloox::InBandBytestreamManager InBandBytestreamManager @endlink for a starting point.
 */

#ifndef GLOOX_H__
#define GLOOX_H__

#include "macros.h"

#include <string>
#include <list>
#include <map>

/**
 * @brief The namespace for the gloox library.
 *
 * @author Jakob Schroeter <js@camaya.net>
 * @since 0.3
 */
namespace gloox
{

  const std::string XMLNS_CLIENT            = "jabber:client";
  const std::string XMLNS_COMPONENT_ACCEPT  = "jabber:component:accept";
  const std::string XMLNS_COMPONENT_CONNECT = "jabber:component:connect";

  const std::string XMLNS_DISCO_INFO        = "http://jabber.org/protocol/disco#info";
  const std::string XMLNS_DISCO_ITEMS       = "http://jabber.org/protocol/disco#items";
  const std::string XMLNS_ADHOC_COMMANDS    = "http://jabber.org/protocol/commands";
  const std::string XMLNS_COMPRESSION       = "http://jabber.org/protocol/compress";
  const std::string XMLNS_OFFLINE           = "http://jabber.org/protocol/offline";
  const std::string XMLNS_CHAT_STATES       = "http://jabber.org/protocol/chatstates";
  const std::string XMLNS_AMP               = "http://jabber.org/protocol/amp";
  const std::string XMLNS_IBB               = "http://jabber.org/protocol/ibb";
  const std::string XMLNS_CHATNEG           = "http://jabber.org/protocol/chatneg";
  const std::string XMLNS_XHTML_IM          = "http://jabber.org/protocol/xhtml-im";
  const std::string XMLNS_ROSTER            = "jabber:iq:roster";
  const std::string XMLNS_VERSION           = "jabber:iq:version";
  const std::string XMLNS_REGISTER          = "jabber:iq:register";
  const std::string XMLNS_PRIVACY           = "jabber:iq:privacy";
  const std::string XMLNS_AUTH              = "jabber:iq:auth";
  const std::string XMLNS_PRIVATE_XML       = "jabber:iq:private";
  const std::string XMLNS_LAST              = "jabber:iq:last";
  const std::string XMLNS_SEARCH            = "jabber:iq:search";
  const std::string XMLNS_X_DATA            = "jabber:x:data";
  const std::string XMLNS_X_EVENT           = "jabber:x:event";
  const std::string XMLNS_X_OOB             = "jabber:x:oob";
  const std::string XMLNS_VCARD_TEMP        = "vcard-temp";
  const std::string XMLNS_BOOKMARKS         = "storage:bookmarks";
  const std::string XMLNS_ANNOTATIONS       = "storage:rosternotes";
  const std::string XMLNS_ROSTER_DELIMITER  = "roster:delimiter";

  const std::string XMLNS_XMPP_STREAM       = "urn:ietf:params:xml:ns:xmpp-streams";
  const std::string XMLNS_XMPP_STANZAS      = "urn:ietf:params:xml:ns:xmpp-stanzas";
  const std::string XMLNS_STREAM_TLS        = "urn:ietf:params:xml:ns:xmpp-tls";
  const std::string XMLNS_STREAM_SASL       = "urn:ietf:params:xml:ns:xmpp-sasl";
  const std::string XMLNS_STREAM_BIND       = "urn:ietf:params:xml:ns:xmpp-bind";
  const std::string XMLNS_STREAM_SESSION    = "urn:ietf:params:xml:ns:xmpp-session";
  const std::string XMLNS_STREAM_IQAUTH     = "http://jabber.org/features/iq-auth";
  const std::string XMLNS_STREAM_IQREGISTER = "http://jabber.org/features/iq-register";
  const std::string XMLNS_STREAM_ACK        = "http://jabber.org/features/ack";
  const std::string XMLNS_STREAM_COMPRESS   = "http://jabber.org/features/compress";

  const int XMPP_STREAM_VERSION_MAJOR       = 1;
  const int XMPP_STREAM_VERSION_MINOR       = 0;
  const std::string GLOOX_VERSION           = "0.8.6-sic";

  /**
   * This describes the possible states of a stream.
   */
  enum ConnectionState
  {
    StateDisconnected,              /**< The client is in disconnected state. */
    StateConnecting,                /**< The client is currently trying to establish a connection. */
    StateConnected                  /**< The client is connected to the server but authentication is not
                                     * (yet) done. */
  };

  /**
   * This describes connection error conditions.
   */
  enum ConnectionError
  {
    ConnNoError,                    /**< Not really an error. Everything went just fine. */
    ConnStreamError,                /**< A stream error occured. The stream has been closed. */
    ConnStreamClosed,               /**< The stream has been closed graciously. */
    ConnIoError,                    /**< An I/O error occured. */
    ConnOutOfMemory,                /**< Out of memory. Uhoh. */
    ConnNoSupportedAuth,            /**< The auth mechanisms the server offers are not supported
                                     * or the server offered no auth mechanisms at all. */
    ConnTlsFailed,                  /**< The server's certificate could not be verified. */
    ConnAuthenticationFailed,       /**< Authentication failed. Username/password wrong or account does
                                     * not exist. */
    ConnUserDisconnected,           /**< The user (or higher-level protocol) requested a disconnect. */
    ConnNotConnected                /**< There is no active connection. */
  };

  /**
   * Supported Stream Features.
   */
  enum StreamFeature
  {
    StreamFeatureBind             =    1, /**< The server supports resource binding. */
    StreamFeatureSession          =    2, /**< The server supports sessions. */
    StreamFeatureSaslDigestMd5    =    4, /**< The server supports SASL DIGEST-MD5 mechanism. */
    StreamFeatureSaslPlain        =    8, /**< The server supports SASL PLAIN mechanism. */
    StreamFeatureSaslAnonymous    =   16, /**< The server supports SASL ANONYMOUS mechanism. */
    StreamFeatureSaslExternal     =   32, /**< The server supports SASL EXTERNAL mechanism. */
    StreamFeatureStartTls         =   64, /**< The server supports &lt;starttls&gt;. */
    StreamFeatureIqRegister       =  128, /**< The server supports JEP-0077 (In-Band
                                           * Registration). */
    StreamFeatureIqAuth           =  256, /**< The server supports JEP-0078 (Non-SASL
                                           * Authentication). */
    StreamFeatureAck              =  512, /**< The server supports JEPAck (experimental). */
    StreamFeatureCompressZlib     = 1024, /**< The server supports JEP-0138 (Stream
                                           * Compression) (Zlib)*/
    StreamFeatureCompressDclz     = 2048  /**< The server supports JEP-0138 (Stream
                                           * Compression) (LZW/DCLZ)*/
  };

  /**
   * This decribes stream error conditions as defined in RFC 3920 Sec. 4.7.3.
   */
  enum StreamError
  {
    StreamErrorUndefined,           /**< An undefined/unknown error occured. Also used if a diconnect was
                                     * user-initiated. Also set before and during a established connection
                                     * (where obviously no error occured). */
    StreamErrorBadFormat,           /**< The entity has sent XML that cannot be processed;
                                     * this error MAY be used instead of the more specific XML-related
                                     * errors, such as &lt;bad-namespace-prefix/&gt;, &lt;invalid-xml/&gt;,
                                     * &lt;restricted-xml/&gt;, &lt;unsupported-encoding/&gt;, and
                                     * &lt;xml-not-well-formed/&gt;, although the more specific errors are
                                     * preferred. */
    StreamErrorBadNamespacePrefix,  /**< The entity has sent a namespace prefix that is unsupported, or has
                                     * sent no namespace prefix on an element that requires such a prefix
                                     * (see XML Namespace Names and Prefixes (Section 11.2)). */
    StreamErrorConflict,            /**< The server is closing the active stream for this entity because a
                                     * new stream has been initiated that conflicts with the existing
                                     * stream. */
    StreamErrorConnectionTimeout,   /**< The entity has not generated any traffic over the stream for some
                                     * period of time (configurable according to a local service policy).*/
    StreamErrorHostGone,            /**< the value of the 'to' attribute provided by the initiating entity
                                     * in the stream header corresponds to a hostname that is no longer
                                     * hosted by the server.*/
    StreamErrorHostUnknown,         /**< The value of the 'to' attribute provided by the initiating entity
                                     * in the stream header does not correspond to a hostname that is hosted
                                     * by the server. */
    StreamErrorImproperAddressing,  /**< A stanza sent between two servers lacks a 'to' or 'from' attribute
                                     * (or the attribute has no value). */
    StreamErrorInternalServerError, /**< the server has experienced a misconfiguration or an
                                     * otherwise-undefined internal error that prevents it from servicing the
                                     * stream. */
    StreamErrorInvalidFrom,         /**< The JID or hostname provided in a 'from' address does not match an
                                     * authorized JID or validated domain negotiated between servers via SASL
                                     * or dialback, or between a client and a server via authentication and
                                     * resource binding.*/
    StreamErrorInvalidId,           /**< The stream ID or dialback ID is invalid or does not match an ID
                                     * previously provided. */
    StreamErrorInvalidNamespace,    /**< The streams namespace name is something other than
                                     * "http://etherx.jabber.org/streams" or the dialback namespace name is
                                     * something other than "jabber:server:dialback" (see XML Namespace Names
                                     * and Prefixes (Section 11.2)). */
    StreamErrorInvalidXml,          /**< The entity has sent invalid XML over the stream to a server that
                                     * performs validation (see Validation (Section 11.3)). */
    StreamErrorNotAuthorized,       /**< The entity has attempted to send data before the stream has been
                                     * authenticated, or otherwise is not authorized to perform an action
                                     * related to stream negotiation; the receiving entity MUST NOT process
                                     * the offending stanza before sending the stream error. */
    StreamErrorPolicyViolation,     /**< The entity has violated some local service policy; the server MAY
                                     * choose to specify the policy in the &lt;text/&gt;  element or an
                                     * application-specific condition element. */
    StreamErrorRemoteConnectionFailed,/**< The server is unable to properly connect to a remote entity that is
                                     * required for authentication or authorization. */
    StreamErrorResourceConstraint,  /**< the server lacks the system resources necessary to service the
                                     * stream. */
    StreamErrorRestrictedXml,       /**< The entity has attempted to send restricted XML features such as a
                                     * comment, processing instruction, DTD, entity reference, or unescaped
                                     * character (see Restrictions (Section 11.1)). */
    StreamErrorSeeOtherHost,        /**< The server will not provide service to the initiating entity but is
                                     * redirecting traffic to another host; the server SHOULD specify the
                                     * alternate hostname or IP address (which MUST be a valid domain
                                     * identifier) as the XML character data of the &lt;see-other-host/&gt;
                                     * element. */
    StreamErrorSystemShutdown,      /**< The server is being shut down and all active streams are being
                                     * closed. */
    StreamErrorUndefinedCondition,  /**< The error condition is not one of those defined by the other
                                     * conditions in this list; this error condition SHOULD be used only in
                                     * conjunction with an application-specific condition. */
    StreamErrorUnsupportedEncoding, /**< The initiating entity has encoded the stream in an encoding that is
                                     * not supported by the server (see Character Encoding (Section 11.5)). */
    StreamErrorUnsupportedStanzaType,/**< The initiating entity has sent a first-level child of the stream
                                     * that is not supported by the server. */
    StreamErrorUnsupportedVersion,  /**< The value of the 'version' attribute provided by the initiating
                                     * entity in the stream header specifies a version of XMPP that is not
                                     * supported by the server; the server MAY specify the version(s) it
                                     * supports in the &lt;text/&gt; element. */
    StreamErrorXmlNotWellFormed     /**< The initiating entity has sent XML that is not well-formed as
                                     * defined by [XML]. */
  };

  /**
   * Describes the possible stanza types.
   */
  enum StanzaType
  {
    StanzaUndefined,                /**< Undefined. */
    StanzaIq,                       /**< An Info/Query stanza. */
    StanzaMessage,                  /**< A message stanza. */
    StanzaS10n,                     /**< A presence/subscription stanza. */
    StanzaPresence                  /**< A presence stanza. */
  };

  /**
   * Describes the possible stanza-sub-types.
   */
  enum StanzaSubType
  {
    StanzaSubUndefined,             /**< Undefined. */
    StanzaIqGet,                    /**< The stanza is a request for information or requirements. */
    StanzaIqSet,                    /**<
                                     * The stanza provides required data, sets new values, or
                                     * replaces existing values.
                                     */
    StanzaIqResult,                 /**< The stanza is a response to a successful get or set request. */
    StanzaIqError,                  /**<
                                     * An error has occurred regarding processing or
                                     * delivery of a previously-sent get or set (see Stanza Errors
                                     * (Section 9.3)).
                                     */
    StanzaPresenceUnavailable,      /**<
                                     * Signals that the entity is no longer available for
                                     * communication.
                                     */
    StanzaPresenceAvailable,        /**<
                                     * Signals to the server that the sender is online and available
                                     * for communication.
                                     */
    StanzaPresenceProbe,            /**<
                                     * A request for an entity's current presence; SHOULD be
                                     * generated only by a server on behalf of a user.
                                     */
    StanzaPresenceError,            /**<
                                     * An error has occurred regarding processing or delivery of
                                     * a previously-sent presence stanza.
                                     */
    StanzaS10nSubscribe,            /**<
                                     * The sender wishes to subscribe to the recipient's
                                     * presence.
                                     */
    StanzaS10nSubscribed,           /**<
                                     * The sender has allowed the recipient to receive
                                     * their presence.
                                     */
    StanzaS10nUnsubscribe,          /**<
                                     * The sender is unsubscribing from another entity's
                                     * presence.
                                     */
    StanzaS10nUnsubscribed,         /**<
                                     * The subscription request has been denied or a
                                     * previously-granted subscription has been cancelled.
                                     */
    StanzaMessageChat,              /**<
                                     * The message is sent in the context of a one-to-one chat
                                     * conversation.  A compliant client SHOULD present the message in an
                                     * interface enabling one-to-one chat between the two parties,
                                     * including an appropriate conversation history.
                                     */
    StanzaMessageError,             /**<
                                     * An error has occurred related to a previous message sent
                                     * by the sender (for details regarding stanza error syntax, refer to
                                     * [XMPP-CORE]).  A compliant client SHOULD present an appropriate
                                     * interface informing the sender of the nature of the error.
                                     */
    StanzaMessageGroupchat,         /**<
                                     * The message is sent in the context of a multi-user
                                     * chat environment (similar to that of [IRC]).  A compliant client
                                     * SHOULD present the message in an interface enabling many-to-many
                                     * chat between the parties, including a roster of parties in the
                                     * chatroom and an appropriate conversation history.  Full definition
                                     * of XMPP-based groupchat protocols is out of scope for this memo.
                                     */
    StanzaMessageHeadline,          /**<
                                     * The message is probably generated by an automated
                                     * service that delivers or broadcasts content (news, sports, market
                                     * information, RSS feeds, etc.).  No reply to the message is
                                     * expected, and a compliant client SHOULD present the message in an
                                     * interface that appropriately differentiates the message from
                                     * standalone messages, chat sessions, or groupchat sessions (e.g.,
                                     * by not providing the recipient with the ability to reply).
                                     */
    StanzaMessageNormal             /**<
                                     * The message is a single message that is sent outside the
                                     * context of a one-to-one conversation or groupchat, and to which it
                                     * is expected that the recipient will reply.  A compliant client
                                     * SHOULD present the message in an interface enabling the recipient
                                     * to reply, but without a conversation history.
                                     */
  };

  /**
   * Describes types of stanza errors.
   */
  enum StanzaErrorType
  {
    StanzaErrorTypeUndefined,       /**< No error. */
    StanzaErrorTypeCancel,          /**< Do not retry (the error is unrecoverable). */
    StanzaErrorTypeContinue,        /**< Proceed (the condition was only a warning). */
    StanzaErrorTypeModify,          /**< Retry after changing the data sent. */
    StanzaErrorTypeAuth,            /**< Retry after providing credentials. */
    StanzaErrorTypeWait             /**< Retry after waiting (the error is temporary). */
  };

  /**
   * Describes the defined stanza error conditions of RFC 3920.
   */
  enum StanzaError
  {
    StanzaErrorUndefined = 0,       /**< No stanza error occured. */
    StanzaErrorBadRequest,          /**< The sender has sent XML that is malformed or that cannot be
                                     * processed (e.g., an IQ stanza that includes an unrecognized value
                                     * of the 'type' attribute); the associated error type SHOULD be
                                     * "modify". */
    StanzaErrorConflict,            /**< Access cannot be granted because an existing resource or session
                                     * exists with the same name or address; the associated error type
                                     * SHOULD be "cancel". */
    StanzaErrorFeatureNotImplemented,/**< The feature requested is not implemented by the recipient or server
                                     * and therefore cannot be processed; the associated error type SHOULD be
                                     * "cancel". */
    StanzaErrorForbidden,           /**< The requesting entity does not possess the required permissions to
                                     * perform the action; the associated error type SHOULD be "auth". */
    StanzaErrorGone,                /**< The recipient or server can no longer be contacted at this address
                                     * (the error stanza MAY contain a new address in the XML character data
                                     * of the &lt;gone/&gt; element); the associated error type SHOULD be
                                     * "modify". */
    StanzaErrorInternalServerError, /**< The server could not process the stanza because of a
                                     * misconfiguration or an otherwise-undefined internal server error; the
                                     * associated error type SHOULD be "wait". */
    StanzaErrorItemNotFound,        /**< The addressed JID or item requested cannot be found; the associated
                                     * error type SHOULD be "cancel". */
    StanzaErrorJidMalformed,        /**< The sending entity has provided or communicated an XMPP address
                                     * (e.g., a value of the 'to' attribute) or aspect thereof (e.g., a
                                     * resource identifier) that does not adhere to the syntax defined in
                                     * Addressing Scheme (Section 3); the associated error type SHOULD be
                                     * "modify". */
    StanzaErrorNotAcceptable,       /**< The recipient or server understands the request but is refusing to
                                     * process it because it does not meet criteria defined by the recipient
                                     * or server (e.g., a local policy regarding acceptable words in
                                     * messages); the associated error type SHOULD be "modify". */
    StanzaErrorNotAllowed,          /**< The recipient or server does not allow any entity to perform the
                                     * action; the associated error type SHOULD be "cancel". */
    StanzaErrorNotAuthorized,       /**< The sender must provide proper credentials before being allowed to
                                     * perform the action, or has provided improper credentials; the
                                     * associated error type SHOULD be "auth". */
    StanzaErrorPaymentRequired,     /**< The requesting entity is not authorized to access the requested
                                     * service because payment is required; the associated error type SHOULD
                                     * be "auth". */
    StanzaErrorRecipientUnavailable,/**< The intended recipient is temporarily unavailable; the associated
                                     * error type SHOULD be "wait" (note: an application MUST NOT return this
                                     * error if doing so would provide information about the intended
                                     * recipient's network availability to an entity that is not authorized
                                     * to know such information). */
    StanzaErrorRedirect,            /**< The recipient or server is redirecting requests for this information
                                     * to another entity, usually temporarily (the error stanza SHOULD
                                     * contain the alternate address, which MUST be a valid JID, in the XML
                                     * character data of the &lt;redirect/&gt; element); the associated
                                     * error type SHOULD be "modify". */
    StanzaErrorRegistrationRequired,/**< The requesting entity is not authorized to access the requested
                                     * service because registration is required; the associated error type
                                     * SHOULD be "auth". */
    StanzaErrorRemoteServerNotFound,/**< A remote server or service specified as part or all of the JID of
                                     * the intended recipient does not exist; the associated error type
                                     * SHOULD be "cancel". */
    StanzaErrorRemoteServerTimeout, /**< A remote server or service specified as part or all of the JID of
                                     * the intended recipient (or required to fulfill a request) could not be
                                     * contacted within a reasonable amount of time; the associated error
                                     * type SHOULD be "wait". */
    StanzaErrorResourceConstraint,  /**< The server or recipient lacks the system resources necessary to
                                     * service the request; the associated error type SHOULD be "wait". */
    StanzaErrorServiceUnavailable,  /**< The server or recipient does not currently provide the requested
                                     * service; the associated error type SHOULD be "cancel". */
    StanzaErrorSubscribtionRequired,/**< The requesting entity is not authorized to access the requested
                                     * service because a subscription is required; the associated error type
                                     * SHOULD be "auth". */
    StanzaErrorUndefinedCondition,  /**< The error condition is not one of those defined by the other
                                     * conditions in this list; any error type may be associated with this
                                     * condition, and it SHOULD be used only in conjunction with an
                                     * application-specific condition. */
    StanzaErrorUnexpectedRequest    /**< The recipient or server understood the request but was not expecting
                                     * it at this time (e.g., the request was out of order); the associated
                                     * error type SHOULD be "wait". */
  };

  /**
   * Describes the possible 'available presence' types.
   */
  enum Presence
  {
    PresenceUnknown,                /**< Unknown status. */
    PresenceAvailable,              /**< The entity or resource is online and available. */
    PresenceChat,                   /**< The entity or resource is actively interested in chatting. */
    PresenceAway,                   /**< The entity or resource is temporarily away. */
    PresenceDnd,                    /**< The entity or resource is busy (dnd = "Do Not Disturb"). */
    PresenceXa,                     /**< The entity or resource is away for an extended period (xa =
                                     * "eXtended Away"). */
    PresenceUnavailable             /**< The entity or resource if offline. */
  };

  /**
   * Describes the verification results of a certificate.
   */
  enum CertStatus
  {
    CertOk               =  0,      /**< The certificate is valid and trusted. */
    CertInvalid          =  1,      /**< The certificate is not trusted. */
    CertSignerUnknown    =  2,      /**< The certificate hasn't got a known issuer. */
    CertRevoked          =  4,      /**< The certificate has been revoked. */
    CertExpired          =  8,      /**< The certificate has expired. */
    CertNotActive        = 16,      /**< The certifiacte is not yet active. */
    CertWrongPeer        = 32,      /**< The certificate has not been issued for the
                                     * peer we're connected to. */
    CertSignerNotCa      = 64       /**< The signer is not a CA. */
  };

  /**
   * Describes the certificate presented by the peer.
   */
  struct CertInfo
  {
    int status;                     /**< Bitwise or'ed CertStatus or CertOK. */
    bool chain;                     /**< Determines whether the cert chain verified ok. */
    std::string issuer;             /**< The name of the issuing entity.*/
    std::string server;             /**< The server the certificate has been issued for. */
    int date_from;                  /**< The date from which onwards the certificate is valid. */
    int date_to;                    /**< The date up to which the certificate is valid. */
    std::string protocol;           /**< The encryption protocol used for the connection. */
    std::string cipher;             /**< The cipher used for the connection. */
    std::string mac;                /**< The MAC used for the connection. */
    std::string compression;        /**< The compression used for the connection. */
  };

  /**
   * Describes the defined SASL error conditions.
   */
  enum AuthenticationError
  {
    AuthErrorUndefined,             /**< No error occurred, or error condition is unknown. */
    SaslAborted,                    /**< The receiving entity acknowledges an &lt;abort/&gt; element sent
                                     * by the initiating entity; sent in reply to the &lt;abort/&gt;
                                     * element. */
    SaslIncorrectEncoding,          /**< The data provided by the initiating entity could not be processed
                                     * because the [BASE64] encoding is incorrect (e.g., because the encoding
                                     * does not adhere to the definition in Section 3 of [BASE64]); sent in
                                     * reply to a &lt;response/&gt; element or an &lt;auth/&gt; element with
                                     * initial response data. */
    SaslInvalidAuthzid,             /**< The authzid provided by the initiating entity is invalid, either
                                     * because it is incorrectly formatted or because the initiating entity
                                     * does not have permissions to authorize that ID; sent in reply to a
                                     * &lt;response/&gt; element or an &lt;auth/&gt; element with initial
                                     * response data.*/
    SaslInvalidMechanism,           /**< The initiating entity did not provide a mechanism or requested a
                                     * mechanism that is not supported by the receiving entity; sent in reply
                                     * to an &lt;auth/&gt; element. */
    SaslMechanismTooWeak,           /**< The mechanism requested by the initiating entity is weaker than
                                     * server policy permits for that initiating entity; sent in reply to a
                                     * &lt;response/&gt; element or an &lt;auth/&gt; element with initial
                                     * response data. */
    SaslNotAuthorized,              /**< The authentication failed because the initiating entity did not
                                     * provide valid credentials (this includes but is not limited to the
                                     * case of an unknown username); sent in reply to a &lt;response/&gt;
                                     * element or an &lt;auth/&gt; element with initial response data. */
    SaslTemporaryAuthFailure,       /**< The authentication failed because of a temporary error condition
                                     * within the receiving entity; sent in reply to an &lt;auth/&gt; element
                                     * or &lt;response/&gt; element. */
    NonSaslConflict,                /**< JEP-0078: Resource Conflict */
    NonSaslNotAcceptable,           /**< JEP-0078: Required Information Not Provided */
    NonSaslNotAuthorized            /**< JEP-0078: Incorrect Credentials */
  };

  /**
   * Identifies log sources.
   */
  enum LogArea
  {
    LogAreaClassParser     = 0x00001, /**< Log messages from Parser. */
    LogAreaClassConnection = 0x00002, /**< Log messages from Connection. */
    LogAreaClassClient     = 0x00004, /**< Log messages from Client. */
    LogAreaClassClientbase = 0x00008, /**< Log messages from ClientBase. */
    LogAreaClassComponent  = 0x00010, /**< Log messages from Component. */
    LogAreaClassDns        = 0x00020, /**< Log messages from DNS. */
    LogAreaAllClasses      = 0x01FFF, /**< All log messages from all the classes. */
    LogAreaXmlIncoming     = 0x02000, /**< Incoming XML. */
    LogAreaXmlOutgoing     = 0x04000, /**< Outgoing XML. */
    LogAreaUser            = 0x80000, /**< User-defined sources. */
    LogAreaAll             = 0xFFFFF  /**< All log sources. */
  };

  /**
   * Describes a log message's severity.
   */
  enum LogLevel
  {
    LogLevelDebug,                  /**< Debug messages. */
    LogLevelWarning,                /**< Non-crititcal warning messages. */
    LogLevelError                   /**< Critical, unrecoverable errors. */
  };

  /**
   * The possible Message Events according to JEP-0022.
   */
  enum MessageEventType
  {
    MessageEventCancel    = 0,      /**< Cancels the 'Composing' event. */
    MessageEventOffline   = 1,      /**< Indicates that the message has been stored offline by the
                                     * intended recipient's server. */
    MessageEventDelivered = 2,      /**< Indicates that the message has been delivered to the
                                     * recipient. */
    MessageEventDisplayed = 4,      /**< Indicates that the message has been displayed */
    MessageEventComposing = 8       /**< Indicates that a reply is being composed. */
  };

  /**
   * The possible Chat States according to JEP-0085.
   */
  enum ChatStateType
  {
    ChatStateActive       =  1,     /**< User is actively participating in the chat session. */
    ChatStateComposing    =  2,     /**< User is composing a message. */
    ChatStatePaused       =  4,     /**< User had been composing but now has stopped. */
    ChatStateInactive     =  8,     /**< User has not been actively participating in the chat session. */
    ChatStateGone         = 16      /**< User has effectively ended their participation in the chat
                                     * session. */
  };

  /**
   * Describes the possible error conditions for resource binding.
   */
  enum ResourceBindError
  {
    RbErrorUnknownError,            /**< An unknown error occured. */
    RbErrorBadRequest,              /**< Resource identifier cannot be processed. */
    RbErrorNotAllowed,              /**< Client is not allowed to bind a resource. */
    RbErrorConflict                 /**< Resource identifier is in use. */
  };

  /**
   * Describes the possible error conditions for session establishemnt.
   */
  enum SessionCreateError
  {
    ScErrorUnknownError,            /**< An unknown error occured. */
    ScErrorInternalServerError,     /**< Internal server error. */
    ScErrorForbidden,               /**< Username or resource not allowed to create session. */
    ScErrorConflict                 /**< Server informs newly-requested session of resource
                                     * conflict. */
  };

  /**
   * Currently implemented message session filters.
   */
  enum MessageSessionFilter
  {
    FilterMessageEvents    = 1,     /**< Message Events (JEP-0022) */
    FilterChatStates       = 2      /**< Chat State Notifications (JEP-0085) */
  };

  /**
   * A list of strings.
   */
  typedef std::list<std::string> StringList;

  /**
   * A map of strings.
   */
  typedef std::map<std::string, std::string> StringMap;

}

extern "C"
{
  const char* gloox_version();
}

#endif // GLOOX_H__
