/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <guichaz@yahoo.fr>
 */

/******************
 * The files list *
 ******************/

#include <string.h>             /* strcmp(), strchr() */
#include <sys/stat.h>           /* stat() */
#include <dirent.h>             /* DIR, opendir(), readdir(), closedir() */
#include <stdlib.h>             /* qsort() */
#include <stdio.h>              /* remove(), perror(), stdin, getdelim() */
#include <sys/types.h>          /* size_t */

#include "gliv.h"
#include "files_list.h"
#include "options.h"
#include "loading.h"
#include "str_utils.h"
#include "foreach_file.h"
#include "next_image.h"
#include "messages.h"
#include "windows.h"
#include "cursors.h"
#include "collection.h"
#include "formats.h"
#include "timestamp.h"

#ifndef HAVE_GETDELIM
#include "../lib/getdelim.h"
#endif

extern options_struct *options;
extern GlivImage *current_image;

static GList *files_list = NULL;
static gint list_length = 0;
static GList *list_end = NULL;
static DECLARE_TIMESTAMP(timestamp);    /* Last modification. */

/* The node to delete when no longer displayed. */
static GList *to_destroy_node = NULL;
static gint to_destroy_number;

gint get_list_length(void)
{
    return list_length;
}

GList *get_list_head(void)
{
    return files_list;
}

GList *get_list_end(void)
{
    if (list_end == NULL) {
        /* Find the end, since we lost it. */

        if (current_image == NULL)
            list_end = g_list_last(files_list);
        else
            /* We know the end is after current_image->node. */
            list_end = g_list_last(current_image->node);
    }

    return list_end;
}

/* Used when we insert in the head. */
static void reverse(void)
{
    list_end = files_list;
    files_list = g_list_reverse(files_list);
}

/*** Additions. ***/

static gboolean is_loadable(const gchar * filename)
{
    loader_t loader;

    if (options->force)
        return TRUE;

    loader = get_loader(filename);
    return loader == LOADER_PIXBUF || loader == LOADER_DECOMP_PIXBUF;
}

/* Returns the number of files added. */
gint add_file_to_list(const gchar * filename)
{
    if (is_loadable(filename)) {
        files_list = g_list_prepend(files_list, clean_filename(filename));
        list_length++;

        if (list_length == 1)
            list_end = files_list;

        return 1;
    }

    return 0;
}

/* Returns the number of files added. */
static gint add_to_list(const gchar * name, gboolean recursive)
{
    gint nb_inserted = 0;
    struct stat st;
    struct dirent *dir_ent;
    DIR *dir;
    gchar *full_path;

    if (stat(name, &st) < 0) {
        perror(name);
        return 0;
    }

    if (S_ISDIR(st.st_mode)) {
        if (recursive)
            /* Traverse recursively the directory. */
            return foreach_file(name, add_file_to_list);

        /* Add every file in the directory. */
        dir = opendir(name);
        if (dir == NULL) {
            perror(name);
            return 0;
        }

        for (dir_ent = readdir(dir); dir_ent != NULL; dir_ent = readdir(dir)) {
            full_path = g_build_filename(name, dir_ent->d_name, NULL);

            stat(full_path, &st);

            if (S_ISDIR(st.st_mode) == FALSE)
                /* We have a file. */
                nb_inserted += add_file_to_list(full_path);

            g_free(full_path);
        }

        closedir(dir);
    } else {
        loader_t loader = get_loader(name);
        if (loader == LOADER_DOT_GLIV || loader == LOADER_DECOMP_DOT_GLIV)
            /* A .gliv collection. */
            nb_inserted = load_dot_gliv(name, TRUE);
        else if (loader != LOADER_NONE || options->force)
            /* Add the file. */
            nb_inserted = add_file_to_list(name);
    }

    return nb_inserted;
}

/*** Deletion ***/

void remove_from_list(GList * node)
{
    if (node == list_end)
        list_end = node->prev;

    g_free(node->data);
    files_list = g_list_delete_link(files_list, node);
    list_length--;
    touch(&timestamp);
}

/*** Sorting ***/

/* To shuffle the list, we simply sort with a random compare func. */
static gint random_compar(gconstpointer unused1, gconstpointer unused2)
{
    return g_random_int_range(-1, 3);
}

/* We want children to be after their parent, or the alphabetical order. */
G_GNUC_PURE static gint filename_compar(gconstpointer a, gconstpointer b)
{
    const gchar *file1, *file2, *ptr1, *ptr2;
    gboolean ptr1_has_dir, ptr2_has_dir;

    ptr1 = file1 = (const gchar *) a;
    ptr2 = file2 = (const gchar *) b;

    if (file1[0] != file2[0])
        /* Comparing an absolute filename, and a relative one. */
        return file1[0] == '/' ? -1 : 1;

    /* Skip identical characters in the paths. */
    while (*ptr1 != '\0' && *ptr1 == *ptr2) {
        ptr1++;
        ptr2++;
    }

    if (*ptr1 == *ptr2)
        /* The filenames were equal. */
        return 0;

    /* Go back to the first different dir. */
    while (*ptr1 != '/' || *ptr2 != '/') {
        ptr1--;
        ptr2--;
    }

    /* Skip the common '/'. */
    ptr1++;
    ptr2++;

    ptr1_has_dir = (strchr(ptr1, '/') != NULL);
    ptr2_has_dir = (strchr(ptr2, '/') != NULL);

    if (ptr1_has_dir == ptr2_has_dir)
        /*
         * Either the files are in the same directory,
         * or they are not parent.
         */
        return strcmp(ptr1, ptr2);

    /* One of the directory is parent of the other one. */
    return ptr1_has_dir ? -1 : 1;
}

static void reorder_list(gboolean shuffle)
{
    GCompareFunc compare_func;

    compare_func = shuffle ? random_compar : filename_compar;
    files_list = g_list_sort(files_list, compare_func);
    list_end = NULL;
}

/* Called by the menu. */
gboolean reorder_files(gboolean shuffle)
{
    GList *node, *current_node;
    gint new_id = 0;

    if (files_list == NULL)
        return FALSE;

    current_node = current_image->node;

    reorder_list(shuffle);

    /* Search the new id for current_image. */
    for (node = files_list; node != NULL; node = node->next) {
        if (current_node == node)
            break;
        new_id++;
    }

    after_reorder(new_id);
    return FALSE;
}

G_GNUC_PURE static gint compar(gconstpointer a, gconstpointer b)
{
    return filename_compar(*((const gchar **) a), *((const gchar **) b));
}

/*
 * Used to build the images menus.
 */
gchar **get_sorted_files_array(void)
{
    gchar **array, **array_ptr;
    GList *list_ptr;

    if (list_length == 0)
        return NULL;

    array_ptr = array = g_new(gchar *, list_length + 1);

    /* Fill the array. */
    for (list_ptr = files_list; list_ptr != NULL; list_ptr = list_ptr->next) {
        *array_ptr = list_ptr->data;
        array_ptr++;
    }

    *array_ptr = NULL;
    qsort(array, list_length, sizeof(gchar *), compar);

    return array;
}

/*** Initialization ***/

static void list_initialized(gboolean recursive,
                             gboolean shuffle, gboolean sort)
{
    if (sort || shuffle)
        reorder_list(shuffle);
    else
        /* We inserted in the head. */
        reverse();

    touch(&timestamp);
}

void init_list(gchar ** names, gint num, gboolean recursive,
               gboolean shuffle, gboolean sort)
{
    while (num) {
        add_to_list(*names, recursive);
        names++;
        num--;
    }

    if (files_list == NULL) {
        DIALOG_MSG(_("No image found"));
        return;
    }

    list_initialized(recursive, shuffle, sort);
}

void read_null_filenames(gboolean recursive, gboolean shuffle, gboolean sort)
{
    gchar *filename = NULL;
    size_t len = 0;

    while (!feof(stdin) && getdelim(&filename, &len, '\0', stdin) > 0)
        add_to_list(filename, recursive);

    list_initialized(recursive, shuffle, sort);
}


/*** Misc. operations. ***/

/* To avoid a gcc warning. */
G_GNUC_PURE static gint strcmp_compar(gconstpointer a, gconstpointer b)
{
    return strcmp(a, b);
}

/* Move backward a node to the head, or to the next image. */
void place_next(const gchar * filename)
{
    GList *node;

    /* Where to start the search. */
    node = (current_image == NULL) ? files_list : current_image->node->next;

    /* The search. */
    node = g_list_find_custom(node, filename, strcmp_compar);

    if (node == list_end)
        list_end = list_end->prev;

    files_list = g_list_remove_link(files_list, node);

    if (current_image == NULL) {
        /* Move to the head. */
        node->next = files_list;
        if (files_list)
            files_list->prev = node;

        files_list = node;
    } else {
        /* Move to the next image. */
        node->prev = current_image->node;
        node->next = current_image->node->next;

        if (node->prev != NULL)
            node->prev->next = node;

        if (node->next != NULL)
            node->next->prev = node;
    }
}

/* Returns the number of files inserted. */
gint insert_after_current(gchar ** names, gint nb, gboolean recursive,
                          gboolean shuffle, gboolean sort, gboolean do_stat)
{
    gint nb_inserted = 0;

    /* Backup the current list. */
    GList *files_list_old = files_list, *last;
    gint list_length_old = list_length;

    /* Reinitialize the current list. */
    files_list = NULL;
    list_length = 0;
    for (; nb != 0; names++, nb--)
        nb_inserted += do_stat ? add_to_list(*names, recursive) :
            add_file_to_list(*names);

    if (files_list == NULL) {
        /* No files were added. */
        files_list = files_list_old;
        list_length = list_length_old;
        return 0;
    }

    touch(&timestamp);
    if (sort || shuffle)
        reorder_list(shuffle);
    else
        /* We inserted in the head. */
        reverse();

    if (files_list_old == NULL)
        /* The previous list was empty => nothing to merge. */
        return nb_inserted;

    if (current_image != NULL) {
        /* Insert after the current image. */

        /* Search the last inserted. */
        for (last = files_list; last->next != NULL; last = last->next);

        /* Merge the insertion end. */
        last->next = current_image->node->next;
        if (last->next)
            last->next->prev = last;

        /* Merge the insertion beginning. */
        current_image->node->next = files_list;
        files_list->prev = current_image->node;

        files_list = files_list_old;
    } else {
        /* Insert at the end. */
        GList *old_last = g_list_last(files_list_old);
        old_last->next = files_list;
        files_list->prev = old_last;
        files_list = files_list_old;
    }

    list_length += list_length_old;

    return nb_inserted;
}

static gboolean show_remove_dialog(const gchar * msg)
{
    GtkMessageDialog *dialog;
    gint res;

    dialog = GTK_MESSAGE_DIALOG(gtk_message_dialog_new(get_current_window(),
                                                       GTK_DIALOG_MODAL,
                                                       GTK_MESSAGE_QUESTION,
                                                       GTK_BUTTONS_OK_CANCEL,
                                                       "%s", msg));

    res = run_modal_dialog(GTK_DIALOG(dialog));
    gtk_widget_destroy(GTK_WIDGET(dialog));

    return res == GTK_RESPONSE_ACCEPT || res == GTK_RESPONSE_OK ||
        res == GTK_RESPONSE_YES;
}

gboolean confirm_remove_current(void)
{
    gchar *filename, *msg;

    if (current_image == NULL)
        return FALSE;

    filename = current_image->node->data;
    msg = g_strdup_printf(_("Do you really want to delete this file?\n%s\n"),
                          filename_to_utf8(filename));

    if (show_remove_dialog(msg)) {
        to_destroy_node = current_image->node;
        to_destroy_number = current_image->number;

        if (remove(filename) < 0)
            perror(filename);
    }

    g_free(msg);
    return FALSE;
}

/* Called when the displayed image is changed. */
void delete_selected_image(void)
{
    if (to_destroy_node == NULL)
        /* Nothing to destroy. */
        return;

    unload(to_destroy_node);

    if (current_image->number >= to_destroy_number)
        current_image->number--;

    remove_from_list(to_destroy_node);

    to_destroy_node = NULL;
}

timestamp_t get_list_timestamp(void)
{
    return timestamp;
}
