/*
 *  
 *  $Id: i2d.cpp 3839 2011-05-11 08:47:59Z tovar $
 *  Ginkgo CADx Project
 *
 *  Code addapted from DCMTK
 *
 *
 *  Copyright (C) 2001-2007, OFFIS
 *
 *  This software and supporting documentation were developed by
 *
 *    Kuratorium OFFIS e.V.
 *    Healthcare Information and Communication Systems
 *    Escherweg 2
 *    D-26121 Oldenburg, Germany
 *
 *  THIS SOFTWARE IS MADE AVAILABLE,  AS IS,  AND OFFIS MAKES NO  WARRANTY
 *  REGARDING  THE  SOFTWARE,  ITS  PERFORMANCE,  ITS  MERCHANTABILITY  OR
 *  FITNESS FOR ANY PARTICULAR USE, FREEDOM FROM ANY COMPUTER DISEASES  OR
 *  ITS CONFORMITY TO ANY SPECIFICATION. THE ENTIRE RISK AS TO QUALITY AND
 *  PERFORMANCE OF THE SOFTWARE IS WITH THE USER.
 *
 *  Module:  dcmdata
 *
 *  Author:  Michael Onken
 *
 *  Purpose: Implements utility for converting standard image formats to DICOM
 *
 *  Last Update:      $Author: onken $
 *  Update Date:      $Date: 2008-01-16 16:32:31 $
 *  CVS/RCS Revision: $Revision: 1.2 $
 *  Status:           $State: Exp $
 *
 *  CVS/RCS Log at end of file
 *
 */

#include <sstream>
#ifdef verify
#define MACRO_QUE_ESTORBA verify
#undef verify
#endif

#include <dcmtk/config/osconfig.h>
#include "i2d.h"
#include <dcmtk/dcmdata/dcpxitem.h>

#include "dcmtk/dcmdata/dctk.h"
//#include "dcmtk/dcmdata/dcdebug.h"
#include "dcmtk/dcmdata/cmdlnarg.h"
#include "dcmtk/ofstd/ofconapp.h"
#include "dcmtk/dcmdata/dcuid.h"     /* for dcmtk version name */
#include "dcmtk/dcmjpeg/djencode.h"  /* for dcmjpeg encoders */
#include "dcmtk/dcmjpeg/djrplol.h"   /* for DJ_RPLossless */
#include "dcmtk/dcmjpeg/djrploss.h"   /* for DJ_RPLossy */
#include "dcmtk/dcmjpeg/dipijpeg.h"  /* for dcmimage JPEG plugin */
#include "dcmtk/dcmimage/diregist.h"  /* include to support color images */

#include <main/controllers/controladorpermisos.h>
#include <main/controllers/controladorlog.h>

#include <ctime>
#include <cstdlib>

#ifdef MACRO_QUE_ESTORBA
#define verify MACRO_QUE_ESTORBA
#endif

Image2Dcm::Image2Dcm() : m_overrideKeys(NULL), m_templateFile(""),
m_readStudyLevel(OFFalse), m_readSeriesLevel(OFFalse), m_studySeriesFile(),
m_incInstNoFromFile(OFFalse), m_disableAttribChecks(OFFalse),
m_inventMissingType2Attribs(OFTrue), m_inventMissingType1Attribs(OFFalse),
m_insertLatin1(OFFalse), m_insertUTF8(OFTrue), m_debug(OFFalse), m_logStream(NULL)
{
	;
}

OFCondition Image2Dcm::convert(I2DImgSource *inputPlug,
			I2DOutputPlug *outPlug,
			DcmDataset*& resultDset,
			E_TransferSyntax& proposedTS)

{
	if (!inputPlug || !outPlug)
		return EC_IllegalParameter;

	OFCondition cond;
	if (m_debug)
		printMessage(m_logStream, "Image2Dcm: Starting conversion");

	// If specified, copy DICOM template file to export file
	if (m_templateFile.length() != 0) {
		DcmFileFormat dcmff;
		cond = dcmff.loadFile(m_templateFile.c_str());
		if (cond.bad())
			return cond;
		// remove problematic attributes from dataset
		cleanupTemplate(dcmff.getDataset());
		// copy from input file
		resultDset = new DcmDataset(*(dcmff.getDataset()));
	} else // otherwise, start with an empty DICOM file
		resultDset = new DcmDataset();
	if (!resultDset)
		return EC_MemoryExhausted;

	// Read patient and study or series information if desired and write to export file
	if (m_readStudyLevel || m_readSeriesLevel) {
		cond = applyStudyOrSeriesFromFile(resultDset);
		if (cond.bad()) {
			delete resultDset;
			resultDset = NULL;
			return cond;
		}
	}

	OFString value;

	// Increment instance number
	if (m_incInstNoFromFile) {
		cond = incrementInstanceNumber(resultDset);
		if (cond.bad()) {
			delete resultDset;
			resultDset = NULL;
			return cond;
		}
		OFString s;
		if (cond.good())
		{
			cond = DcmDate::getCurrentDate(value);
		}
		if (cond.good())
		{
			cond = resultDset->putAndInsertOFStringArray(DCM_InstanceCreationDate, value);
		}
		if (cond.good())
		{
			cond = DcmTime::getCurrentTime(value);
		}
		if (cond.good())
		{
			cond = resultDset->putAndInsertOFStringArray(DCM_InstanceCreationTime, value);
		}
		if (cond.bad()) {
			delete resultDset;
			resultDset = NULL;
			return cond;
		}
	}
	else {
		cond = resultDset->findAndGetOFString(DCM_InstanceCreationDate, value);
		if (cond.bad()) {
			cond = DcmDate::getCurrentDate(value);
			if (cond.good()) {
				cond = resultDset->putAndInsertOFStringArray(DCM_InstanceCreationDate, value);
			}
		}
		if (cond.bad()) {
			delete resultDset;
			resultDset = NULL;
			return cond;
		}

		cond = resultDset->findAndGetOFString(DCM_InstanceCreationTime, value);

		if (cond.bad()) {
			cond = DcmTime::getCurrentTime(value);
			if (cond.good()) {
				cond = resultDset->putAndInsertOFStringArray(DCM_InstanceCreationTime, value);
			}
		}
		if (cond.bad()) {
			delete resultDset;
			resultDset = NULL;
			return cond;
		}
	}

	// Insert Latin 1 as standard character set if desired
	if (m_insertLatin1)
		cond = insertLatin1(resultDset);
	if (cond.bad())
		return cond;

	if (m_insertUTF8)
		cond = insertUTF8(resultDset);
	if (cond.bad())
		return cond;

	// Generate and insert UIDs as necessary
	generateUIDs(resultDset);

	// Read and insert pixel data

	cond = readAndInsertPixelData(inputPlug, resultDset, proposedTS);
	if (cond.bad()) {
		delete resultDset;
		resultDset = NULL;
		return cond;
	}

	// Insert SOP Class specific attributes (and values)
	cond = outPlug->convert(*resultDset);
	if (cond.bad()) {
		delete resultDset;
		resultDset = NULL;
		return cond;
	}

	 // At last, apply override keys on dataset
	applyOverrideKeys(resultDset);

	// Do some very basic attribute checking (e. g. existence (type 2) and values (type 1))
	if (!m_disableAttribChecks) {
		OFString err;
		err = isValid(*resultDset);
		err += outPlug->isValid(*resultDset);
		if (!err.empty()) {
			delete resultDset;
			resultDset = NULL;
			return makeOFCondition(OFM_dcmdata, 18, OF_error, err.c_str());
		}
	}

	return EC_Normal;
}

OFCondition Image2Dcm::updateOverrideKeys(DcmDataset* dset) {
	if (dset == NULL) {
		return EC_IllegalParameter;
	}

	applyOverrideKeys(dset);

	return EC_Normal;
}

OFCondition Image2Dcm::insertLatin1(DcmDataset *outputDset)
{
	if (outputDset == NULL)
		return EC_IllegalParameter;
	return outputDset->putAndInsertOFStringArray(DCM_SpecificCharacterSet, "ISO_IR 100");
}

OFCondition Image2Dcm::insertUTF8(DcmDataset *outputDset)
{
	if (outputDset == NULL)
		return EC_IllegalParameter;
	return outputDset->putAndInsertOFStringArray(DCM_SpecificCharacterSet, "ISO_IR 192");
}

void Image2Dcm::cleanupTemplate(DcmDataset *targetDset)
{
	if (!targetDset)
		return;
	// Remove any existing image pixel module attribute
	targetDset->findAndDeleteElement(DcmTagKey(0x0028, 0x7FE0)); // Pixel Data Provider URL (JPIP)
	targetDset->findAndDeleteElement(DCM_PhotometricInterpretation);
	targetDset->findAndDeleteElement(DCM_SamplesPerPixel);
	targetDset->findAndDeleteElement(DCM_Rows);
	targetDset->findAndDeleteElement(DCM_Columns);
	targetDset->findAndDeleteElement(DCM_BitsAllocated);
	targetDset->findAndDeleteElement(DCM_BitsStored);
	targetDset->findAndDeleteElement(DCM_HighBit);
	targetDset->findAndDeleteElement(DCM_PixelRepresentation);
	targetDset->findAndDeleteElement(DCM_PixelData);
	targetDset->findAndDeleteElement(DCM_PlanarConfiguration);
	targetDset->findAndDeleteElement(DCM_PixelAspectRatio);
	targetDset->findAndDeleteElement(DCM_SmallestImagePixelValue);
	targetDset->findAndDeleteElement(DCM_LargestImagePixelValue);
	targetDset->findAndDeleteElement(DCM_RedPaletteColorLookupTableDescriptor);
	targetDset->findAndDeleteElement(DCM_GreenPaletteColorLookupTableDescriptor);
	targetDset->findAndDeleteElement(DCM_BluePaletteColorLookupTableDescriptor);
	targetDset->findAndDeleteElement(DCM_RedPaletteColorLookupTableData);
	targetDset->findAndDeleteElement(DCM_GreenPaletteColorLookupTableData);
	targetDset->findAndDeleteElement(DCM_BluePaletteColorLookupTableData);
	targetDset->findAndDeleteElement(DCM_ICCProfile);
	// Remove SOP Class / Instance information
	targetDset->findAndDeleteElement(DCM_SOPClassUID);
	targetDset->findAndDeleteElement(DCM_SOPInstanceUID);

}

OFCondition Image2Dcm::applyStudyOrSeriesFromFile(DcmDataset *targetDset)
{
	if (m_debug)
		printMessage(m_logStream, "Image2Dcm: Applying study and/or series information from file");
	if ((!m_readSeriesLevel && !m_readStudyLevel) || (m_studySeriesFile.length() == 0))
		return EC_IllegalCall;
	DcmFileFormat dcmff;
	OFString errMsg;
	OFCondition cond;

	// Open DICOM file to read patient/study/series information from
	cond = dcmff.loadFile(m_studySeriesFile.c_str());
	if (cond.bad()) {
		errMsg = "Error: Unable to open study / series file ";
		errMsg += m_studySeriesFile;
		return makeOFCondition(OFM_dcmdata, 18, OF_error, errMsg.c_str());
	}

	DcmDataset *srcDset = NULL;
	srcDset = dcmff.getDataset();
	if (srcDset == NULL)
		return EC_IllegalCall;

	// Patient level attributes (type 2 - if value cannot be read, insert empty value
	OFString value;
	srcDset->findAndGetOFString(DCM_PatientName, value);
	cond = targetDset->putAndInsertOFStringArray(DCM_PatientName, value);
	if (cond.bad())
		return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable write Patients Name to file");
	value.clear();

	srcDset->findAndGetOFString(DCM_PatientID, value);
	cond = targetDset->putAndInsertOFStringArray(DCM_PatientID, value);
	if (cond.bad())
		return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable write Patient ID to file");
	value.clear();

	srcDset->findAndGetOFString(DCM_PatientSex, value);
	cond = targetDset->putAndInsertOFStringArray(DCM_PatientSex, value);
	if (cond.bad())
		return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable write Patient's Sex to file");
	value.clear();

	srcDset->findAndGetOFString(DCM_PatientBirthDate, value);
	cond = targetDset->putAndInsertOFStringArray(DCM_PatientBirthDate, value);
	if (cond.bad())
		return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable write Patient's Birth Date to file");
	value.clear();

	srcDset->findAndGetOFString(DCM_SpecificCharacterSet, value);
	cond = targetDset->putAndInsertOFStringArray(DCM_SpecificCharacterSet, value);
	if (cond.bad())
		return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable write Specific Character Set to file");
	value.clear();

	srcDset->findAndGetOFString(DCM_InstanceCreationDate, value);
	cond = targetDset->putAndInsertOFStringArray(DCM_InstanceCreationDate, value);
	if (cond.bad())
		return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable write Instance Creation Date to file");
	value.clear();

	srcDset->findAndGetOFString(DCM_InstanceCreationTime, value);
	cond = targetDset->putAndInsertOFStringArray(DCM_InstanceCreationTime, value);
	if (cond.bad())
		return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable write Instance Creation Time to file");
	value.clear();

	// Study level attributes (type 2 except Study Instance UID)

	cond = srcDset->findAndGetOFString(DCM_StudyInstanceUID, value);
	if (cond.bad())
		return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable to read Study Instance UID (type 1) from file");
	cond = targetDset->putAndInsertOFStringArray(DCM_StudyInstanceUID, value);
	if (cond.bad())
		return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable write Study Instance UID to file");
	value.clear();

	srcDset->findAndGetOFString(DCM_StudyDate, value);
	cond = targetDset->putAndInsertOFStringArray(DCM_StudyDate, value);
	if (cond.bad())
		return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable write Study Date to file");
	value.clear();

	srcDset->findAndGetOFString(DCM_StudyTime, value);
	cond = targetDset->putAndInsertOFStringArray(DCM_StudyTime, value);
	if (cond.bad())
		return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable write Study Time to file");
	value.clear();

	srcDset->findAndGetOFString(DCM_ReferringPhysicianName, value);
	cond = targetDset->putAndInsertOFStringArray(DCM_ReferringPhysicianName, value);
	if (cond.bad())
		return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable write Referring Physician's Name to file");
	value.clear();

	srcDset->findAndGetOFString(DCM_StudyID, value);
	cond = targetDset->putAndInsertOFStringArray(DCM_StudyID, value);
	if (cond.bad())
		return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable write Study ID to file");
	value.clear();

	srcDset->findAndGetOFString(DCM_AccessionNumber, value);
	cond = targetDset->putAndInsertOFStringArray(DCM_AccessionNumber, value);
	if (cond.bad())
		return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable write Accession Number to file");
	value.clear();

	// Series Level attributes (type 2 except Series Instance UID which is type 1)
	if (m_readSeriesLevel) {
		// General Series Module attributes
		cond = srcDset->findAndGetOFString(DCM_SeriesInstanceUID, value);
		if (cond.bad())
			return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable to read Series Instance UID (type 1) from file");
		cond = targetDset->putAndInsertOFStringArray(DCM_SeriesInstanceUID, value);
		if (cond.bad())
			return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable write Series Instance UID to file");
		value.clear();

		srcDset->findAndGetOFString(DCM_SeriesNumber, value);
		cond = targetDset->putAndInsertOFStringArray(DCM_SeriesNumber, value);
		if (cond.bad())
			return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable write Series Number to file");
		value.clear();

		// General Equipment Module attributes
		srcDset->findAndGetOFString(DCM_Manufacturer, value);
		cond = targetDset->putAndInsertOFStringArray(DCM_Manufacturer, value);
		if (cond.bad())
			return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable write Manufacturer to file");
		value.clear();
	}

	return EC_Normal;
}

OFCondition Image2Dcm::incrementInstanceNumber(DcmDataset *targetDset)
{
	// Read and increment Instance Number if desired
	if (m_incInstNoFromFile) {
		if (m_debug)
			printMessage(m_logStream, "Image2Dcm: Trying to read and increment instance number");
		Sint32 instanceNumber;
		if (targetDset->findAndGetSint32(DCM_InstanceNumber, instanceNumber).good()) {
			instanceNumber++;
			std::ostringstream os;
			os << OFstatic_cast(long, instanceNumber);
			OFCondition cond = targetDset->putAndInsertOFStringArray(DCM_InstanceNumber, os.str().c_str());
			if (cond.bad())
				return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable write Instance Number to dataset");
		} else
			return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable to read Instance Number from dataset");
	}
	return EC_Normal;
}

OFCondition Image2Dcm::generateUIDs(DcmDataset *dset)
{
	OFString value;
	OFCondition cond;

	if (m_debug)
		printMessage(m_logStream, "Image2Dcm: Generate and insert new UIDs if necessary");
	// Generate and write Series Instance UID if not already present
	if (!m_readSeriesLevel) {
		cond = dset->findAndGetOFString(DCM_SeriesInstanceUID, value);
		if (cond.bad() || (value.length() == 0)) {
			char newUID[100];
			dcmGenerateUniqueIdentifier(newUID, SITE_SERIES_UID_ROOT);
			cond = dset->putAndInsertOFStringArray(DCM_SeriesInstanceUID, newUID);
			if (cond.bad())
				return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable write Series Instance UID to file");
		}
		value.clear();
	}

	// Generate and write Study Instance UID if not already present
	if (!m_readStudyLevel) {
		cond = dset->findAndGetOFString(DCM_StudyInstanceUID, value);
		if (cond.bad() || (value.length() == 0)) {
			char newUID[100];
			dcmGenerateUniqueIdentifier(newUID, SITE_STUDY_UID_ROOT);
			cond = dset->putAndInsertOFStringArray(DCM_StudyInstanceUID, newUID);
			if (cond.bad())
				return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable write Study Instance UID to file");
		}
		value.clear();

		cond = dset->findAndGetOFString(DCM_AccessionNumber, value);
		if (cond.bad() || (value.length() == 0)) {
			std::stringstream is;

			int random_integer;
			const int lowest  = 0;
			const int highest = 9999;
			const int range   = (highest-lowest)+1;

			const time_t timestamp = std::time(NULL);
			std::srand((unsigned int)timestamp);

			random_integer = lowest + int(range * rand()/(RAND_MAX + 1.0));

			is << (unsigned long) timestamp;
			is.width(4);
			is.fill('0');
			is << (unsigned int) random_integer;
			cond = dset->putAndInsertString(DCM_AccessionNumber, is.str().c_str());
			if (cond.bad()) {
				return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable write Accession Number");
			}
		}
		value.clear();
	}

	// Generate SOP Instance UID if not already present
	cond = dset->findAndGetOFString(DCM_SOPInstanceUID, value);
	if (cond.bad() || (value.length() == 0)) {
		char newUID[100];
		dcmGenerateUniqueIdentifier(newUID, SITE_INSTANCE_UID_ROOT);
		cond = dset->putAndInsertOFStringArray(DCM_SOPInstanceUID, newUID);
		if (cond.bad())
			return makeOFCondition(OFM_dcmdata, 18, OF_error, "Unable write SOP Instance UID to file");
	}

	return EC_Normal;
}

void Image2Dcm::setISOLatin1(OFBool insertLatin1)
{
	m_insertLatin1 = insertLatin1;
}

void Image2Dcm::setUTF8(OFBool insertUTF8)
{
	m_insertUTF8 = insertUTF8;
}

OFCondition Image2Dcm::readAndInsertPixelData(I2DImgSource* imgSource,
			DcmDataset* dset,
			E_TransferSyntax& outputTS)
{
	imgSource->setDebugMode(m_debug);

	Uint16 samplesPerPixel, rows, cols, bitsAlloc, bitsStored, highBit, pixelRepr, planConf;
	Uint16 pixAspectH = 1;
	Uint16 pixAspectV = 1;
	OFString photoMetrInt;
	outputTS = EXS_Unknown;
	char* pixData = NULL;
	Uint32 length;

	OFCondition cond = imgSource->readPixelData(rows, cols,
				samplesPerPixel, photoMetrInt, bitsAlloc, bitsStored, highBit, pixelRepr,
				planConf, pixAspectH, pixAspectV, pixData, length, outputTS);
	if (cond.bad())
		return cond;

	if (m_debug)
		printMessage(m_logStream, "Image2Dcm: Store imported pixel data to DICOM file");

	cond = dset->putAndInsertUint16(DCM_SamplesPerPixel, samplesPerPixel);
	if (cond.bad())
		return cond;

	cond = dset->putAndInsertOFStringArray(DCM_PhotometricInterpretation, photoMetrInt);
	if (cond.bad())
		return cond;

	cond = dset->putAndInsertOFStringArray(DCM_ConversionType, "WSD");
	if (cond.bad())
		return cond;
/*
	cond = dset->putAndInsertOFStringArray(DCM_ImagerPixelSpacing, "1.000000\\1.000000");
	if (cond.bad())
		return cond;

	cond = dset->putAndInsertOFStringArray(DCM_PixelSpacing, "1.000000\\1.000000");
	if (cond.bad())
		return cond;
*/
	cond = dset->putAndInsertOFStringArray(DCM_ImagePositionPatient, "0.000000\\0.000000\\0.000000");
	if (cond.bad())
		return cond;

	cond = dset->putAndInsertOFStringArray(DCM_ImageOrientationPatient, "1.000000\\0.000000\\0.000000\\0.000000\\1.000000\\0.000000");
	if (cond.bad())
		return cond;


	// Should only be written if Samples per Pixel > 1
	if (samplesPerPixel > 1) {
		cond = dset->putAndInsertUint16(DCM_PlanarConfiguration, planConf);
		if (cond.bad())
			return cond;
	}

	cond = dset->putAndInsertUint16(DCM_Rows, rows);
	if (cond.bad())
		return cond;

	cond = dset->putAndInsertUint16(DCM_Columns, cols);
	if (cond.bad())
		return cond;

	cond = dset->putAndInsertUint16(DCM_BitsAllocated, bitsAlloc);
	if (cond.bad())
		return cond;

	cond = dset->putAndInsertUint16(DCM_BitsStored, bitsStored);
	if (cond.bad())
		return cond;

	cond = dset->putAndInsertUint16(DCM_HighBit, highBit);
	if (cond.bad())
		return cond;

	cond = dset->putAndInsertUint16(DCM_PixelRepresentation, pixelRepr);
	if (cond.bad())
		return cond;


	if (imgSource->Recompress()) 
	{
		// create initial pixel sequence
		DcmElement* element = newDicomElement(DcmTag(DCM_PixelData, EVR_OW));
		element->putUint8Array((const Uint8*)pixData, length);
		cond = dset->insert(element);
		if (cond.bad()) {
			delete element;
			return cond;
		}

		//lo pasamos a jpeg lossless
		// create representation parameters for lossy and lossless
		OFCmdUnsignedInt opt_selection_value = 6;
		//este numero implica cierta perdida... si fuera 0 seria lossless real
		OFCmdUnsignedInt opt_point_transform = 3;

		E_TransferSyntax opt_oxfer;
		GNC::GCS::Permisos::EstadoPermiso codificacion = GNC::GCS::ControladorPermisos::Instance()->Get("core.importacion", "codec");
		switch (codificacion.ObtenerValor<int>()) {
			case 0:
				{
					//baseline
					opt_oxfer = EXS_JPEGProcess1TransferSyntax;
					DJ_RPLossy rp_lossy((int)90);
					const DcmRepresentationParameter *rp = &rp_lossy;
					dset->chooseRepresentation(opt_oxfer, rp);
				}
				break;
			case 1:
				{
					//progresivo
					opt_oxfer = EXS_JPEGProcess10_12TransferSyntax;
					DJ_RPLossy rp_lossy((int)90);
					const DcmRepresentationParameter *rp = &rp_lossy;
					dset->chooseRepresentation(opt_oxfer, rp);
				}
				break;
			case 2:
			default:
				{
					//lossless
					GNC::GCS::Permisos::EstadoPermiso estado = GNC::GCS::ControladorPermisos::Instance()->Get("core.importacion", "quality");
					if (estado) {
						opt_point_transform = std::min<int>(estado.ObtenerValor<int>(), 14);
						opt_point_transform = std::max<int>(estado.ObtenerValor<int>(), 0);
					}
					opt_oxfer = EXS_JPEGProcess14SV1TransferSyntax;
					DJ_RPLossless rp_lossless((int)opt_selection_value, (int)opt_point_transform);
					const DcmRepresentationParameter *rp = &rp_lossless;
					dset->chooseRepresentation(opt_oxfer, rp);
				}
				break;
		}

		if(!dset->canWriteXfer(opt_oxfer))
			return OFCondition(EC_UnsupportedEncoding);

		// force meta-header to refresh SOP Class/Instance UIDs.
		delete dset->remove(DCM_MediaStorageSOPClassUID);
		delete dset->remove(DCM_MediaStorageSOPInstanceUID);
		outputTS = opt_oxfer;

		return cond;
	} else {
		if (imgSource->IsCompressed()) {
			DcmPixelSequence *pixelSequence = NULL;

			if (m_debug)
				printMessage(m_logStream, "Image2Dcm: Store imported pixel data to DICOM file");
			// create initial pixel sequence
			pixelSequence = new DcmPixelSequence(DcmTag(DCM_PixelData, EVR_OB));
			if (pixelSequence == NULL)
				return EC_MemoryExhausted;
			// insert empty offset table into sequence
			DcmPixelItem *offsetTable = new DcmPixelItem(DcmTag(DCM_Item, EVR_OB));
			if (offsetTable == NULL) {
				delete pixelSequence;
				pixelSequence = NULL;
				return EC_MemoryExhausted;
			}
			cond = pixelSequence->insert(offsetTable);
			if (cond.bad()) {
				delete offsetTable;
				offsetTable = NULL;
				delete pixelSequence;
				pixelSequence = NULL;
				return cond;
			}

			// insert frame into pixel sequence
			DcmOffsetList dummyList;
			cond = pixelSequence->storeCompressedFrame(dummyList, (Uint8*) pixData, length, 0);
			// storeCompressedFrame(..) does a deep copy, so the pixdata memory can be freed now
			delete[] pixData;
			if (cond.bad()) {
				delete pixelSequence;
				return cond;
			}
			cond = dset->insert(pixelSequence);
			if (cond.bad())
				delete pixelSequence;

			if (m_debug)
				printMessage(m_logStream, "Image2Dcm: Inserting Image Pixel module information");
			return dset->putAndInsertUint16(DCM_PixelRepresentation, pixelRepr);
		} else {
			//little endian to little endian...
			// create initial pixel sequence
			DcmElement* element = newDicomElement(DcmTag(DCM_PixelData, EVR_OW));
			element->putUint8Array((const Uint8*)pixData, length);
			cond = dset->insert(element);
			if (cond.bad()) {
				delete element;
				return cond;
			}
			outputTS = EXS_LittleEndianExplicit;

			return cond;
		}
	}
}

OFString Image2Dcm::isValid(DcmDataset& dset) const
{
	if (m_debug)
		printMessage(m_logStream, "Image2Dcm: Checking validity of DICOM output dataset");
	OFString dummy, err;
	OFCondition cond;
	// General Patient module attributes
	err += checkAndInventType2Attrib(DCM_PatientName, &dset);
	err += checkAndInventType2Attrib(DCM_PatientSex, &dset);
	err += checkAndInventType2Attrib(DCM_PatientBirthDate, &dset);
	err += checkAndInventType2Attrib(DCM_PatientID, &dset);

	// General Study module attributes
	err += checkAndInventType1Attrib(DCM_StudyInstanceUID, &dset);
	err += checkAndInventType2Attrib(DCM_StudyDate, &dset);
	err += checkAndInventType2Attrib(DCM_StudyTime, &dset);
	err += checkAndInventType2Attrib(DCM_ReferringPhysicianName, &dset);
	err += checkAndInventType2Attrib(DCM_StudyID, &dset);
	err += checkAndInventType2Attrib(DCM_AccessionNumber, &dset);

	// General Series module attributes
	err += checkAndInventType1Attrib(DCM_SeriesInstanceUID, &dset);
	err += checkAndInventType2Attrib(DCM_SeriesNumber, &dset);
	err += checkAndInventType2Attrib(DCM_InstanceNumber, &dset);

	// General Image module attributes
	/* Patient Orientation is of type 2C and must be written if not
		Image Orientation (Patient) (0020,0037) and Image Position (Patient)
		are required for the IOD. The current output IODs (SC, new SC, VLP)
		therefore need Patient Orientation. Make sure any new output plugin
		takes care about this attribute
	 */
	err += checkAndInventType2Attrib(DCM_PatientOrientation, &dset);


	// Image Pixel Module
	err += checkAndInventType1Attrib(DCM_Rows, &dset);
	err += checkAndInventType1Attrib(DCM_Columns, &dset);
	err += checkAndInventType1Attrib(DCM_SamplesPerPixel, &dset);
	err += checkAndInventType1Attrib(DCM_PhotometricInterpretation, &dset);
	err += checkAndInventType1Attrib(DCM_BitsAllocated, &dset);
	err += checkAndInventType1Attrib(DCM_BitsStored, &dset);
	err += checkAndInventType1Attrib(DCM_HighBit, &dset);
	err += checkAndInventType1Attrib(DCM_PixelRepresentation, &dset);
	err += checkAndInventType1Attrib(DCM_SOPInstanceUID, &dset);

	return err;
}

void Image2Dcm::setSeriesFrom(const OFString& file)
{
	m_readSeriesLevel = OFTrue;
	m_studySeriesFile = file;
}

void Image2Dcm::setStudyFrom(const OFString& file)
{
	m_readStudyLevel = OFTrue;
	m_studySeriesFile = file;
}

void Image2Dcm::setValidityChecking(OFBool doChecks,
			OFBool insertMissingType2,
			OFBool inventMissingType1)
{
	m_disableAttribChecks = !doChecks;
	m_inventMissingType2Attribs = insertMissingType2;
	m_inventMissingType1Attribs = inventMissingType1;
}

void Image2Dcm::setTemplateFile(const OFString& file)
{
	m_templateFile = file;
}

void Image2Dcm::setIncrementInstanceNumber(OFBool incInstNo)
{
	m_incInstNoFromFile = incInstNo;
}

void Image2Dcm::setOverrideKeys(DcmDataset* dset)
{
	if (m_overrideKeys != NULL) {
		delete m_overrideKeys;
	}
	m_overrideKeys = dset;
}

void Image2Dcm::applyOverrideKeys(DcmDataset *outputDset)

{
	if ((m_overrideKeys == NULL) || (outputDset == NULL)) {
		return; /* nothing to do */
	}
	if (m_debug)
		printMessage(m_logStream, "Image2Dcm: Applying override keys");
	/* copy the override keys */
	DcmDataset keys(*m_overrideKeys);

	/* put the override keys into dset replacing existing tags */
	unsigned long elemCount = keys.card();
	OFCondition cond;
	for (unsigned long i = 0; i < elemCount; i++) {
		DcmElement *elem = keys.remove((unsigned long) 0);
		const DcmTag& tag = elem->getTag();
		cond = outputDset->insert(elem, true, false);
		if (cond.bad()) {
			std::ostringstream os;
			os << "Error al insertar Tag: " << tag.toString().c_str() << " : " << cond.text();
			LOG_ERROR("Dicomizacion", os.str());
		}
		else {
			//std::cout << "Insertado  " << tag.toString().c_str() << std::endl;
		}
	}
}

OFString Image2Dcm::checkAndInventType1Attrib(const DcmTagKey& key,
			DcmDataset* targetDset,
			const OFString& defaultValue) const
{
	OFString err;
	OFBool exists = targetDset->tagExists(key);
	if (!exists) {
		OFString err = "Image2Dcm: Missing type 1 attribute: ";
		err += DcmTag(key).getTagName();
		err += "\n";
		return err;
	}
	DcmElement *elem;
	OFCondition cond = targetDset->findAndGetElement(key, elem);
	if (cond.bad() || !elem || (elem->getLength() == 0)) {
		if (!m_inventMissingType1Attribs) {
			err += "Image2Dcm: Empty value for type 1 attribute: ";
			err += DcmTag(key).getTagName();
			err += "\n";
			return err;
		}
		//holds element to insert in item
		DcmElement *elem = NULL;
		DcmTag tag(key);
		OFBool wasError = OFFalse;
		//if dicom element could be created, insert in to item and modify to value
		if (newDicomElement(elem, tag).good()) {
			if (targetDset->insert(elem, OFTrue).good()) {
				if (elem->putString(defaultValue.c_str()).good()) {
					if (m_debug) {
						OFString msg = "Image2Dcm: Inserting missing type 1 attribute ";
						msg += tag.getTagName();
						msg += " with value ";
						msg += defaultValue;
						printMessage(m_logStream, msg);
						return err;
					}
				} else wasError = OFTrue;
			} else wasError = OFTrue;
		} else wasError = OFTrue;
		if (wasError) {
			err += "Unable to insert type 1 attribute ";
			err += tag.getTagName();
			err += " with value ";
			err += defaultValue;
			err += "\n";
		}
	}
	return err;
}

OFString Image2Dcm::checkAndInventType2Attrib(const DcmTagKey& key,
			DcmDataset* targetDset) const
{
	OFString err;
	OFBool exists = targetDset->tagExists(key);
	if (!exists) {
		if (m_inventMissingType2Attribs) {
			DcmTag tag(key);
			if (m_debug)
				printMessage(m_logStream, "Image2Dcm: Inserting missing type 2 attribute: ", tag.getTagName());
			targetDset->insertEmptyElement(tag);
		} else {
			err = "Image2Dcm: Missing type 2 attribute: ";
			err += DcmTag(key).getTagName();
			err += "\n";
			return err;
		}
	}
	return err;
}

Image2Dcm::~Image2Dcm()
{
	if (m_debug)
		printMessage(m_logStream, "Image2Dcm: Freeing memory");
	if (m_overrideKeys != NULL) {
		delete m_overrideKeys;
	}
	m_overrideKeys = NULL;
}


/*
 * CVS/RCS Log:
 * $Log: i2d.cc,v $
 * Revision 1.2  2008-01-16 16:32:31  onken
 * Fixed some empty or doubled log messages in libi2d files.
 *
 * Revision 1.1  2008-01-16 14:36:02  onken
 * Moved library "i2dlib" from /dcmdata/libsrc/i2dlib to /dcmdata/libi2d
 *
 * Revision 1.1  2007/11/08 15:55:17  onken
 * Initial checkin of img2dcm application and corresponding library i2dlib.
 *
 *
 */
