
import datetime
import os
from gettext import gettext as _

import gobject
import gtk
import dbus

from gimmie_base import Item, ItemSource, DisabledItemSource, Topic
from gimmie_util import bookmarks, icon_factory, ToolMenuButton, ImplementMe
from gimmie_gaim import gaim_reader, gaim_dbus

try:
    from gimmie_gmail import gmail_reader
except ImportError, err:
    print "Unable to load GMail contact support:", err
    gmail_reader = None


#
#  People
#

class RecentConversations(ItemSource):
    '''
    A recent conversation source that supports filtering by last chat date.
    This uses the Gaim account reader so only chats with buddies in the buddy
    list will be displayed.

    This is a bug, but using the GaimReader makes things much faster, since we
    don\'t traverse the filesystem, and provides a single search result for the
    same buddy.
    '''
    def __init__(self, name, icon = None):
        ItemSource.__init__(self, name=name, icon=icon, uri="source://People/Recent")

    def do_reload(self):
        buddies = gaim_reader.get_all_buddies()
        self.set_items(buddies)


class GmailSource(ItemSource):
    '''
    A list of Gmail contacts
    '''
    GMAIL_CONTACTS_UPDATE_INTERVAL = 3600000 # 1 hour in milliseconds
    
    def __init__(self, name, icon = None):
        ItemSource.__init__(self, name, icon, uri="source://People/Gmail", filter_by_date=False)
        self._DIRTY = True

        gmail_reader.connect("reload", lambda x: self.emit("reload"))

        # Set up timer to update contacts at regular intervals.
        gobject.timeout_add(GmailSource.GMAIL_CONTACTS_UPDATE_INTERVAL, lambda: self.do_reload())

    def get_items(self):
        if self._DIRTY:
            self._reload()
        return ItemSource.get_items(self)

    def _reload(self):
        items = [gmail_reader.get_account_settings()]
        contacts = gmail_reader.get_contacts()
        if contacts:
            items += contacts
        
        self.set_items(items)

        self._DIRTY = False

    def do_reload(self):
        # We call _reload in get_items to load on-demand.
        self._DIRTY = True

        # Return True to continue timeout
        return True


class GroupSource(ItemSource):
    '''
    A list of all people in a GaimGroup, filtered by active accounts
    '''
    def __init__(self, group, icon = "gaim"):
        uri = "source://People/Group/%s" % group.get_name()
        ItemSource.__init__(self, group.get_name(), icon=icon, uri=uri, filter_by_date=False)
        self.group = group

    def do_reload(self):
        buddies = [b for b in self.group.get_buddies() if b.get_account().get_auto_login()]
        self.set_items(buddies)


class EverybodySource(ItemSource):
    '''
    A list of "everyone", filtered by active accounts
    '''
    def __init__(self, name, icon = None):
        ItemSource.__init__(self, name, icon, uri="source://People/Everybody", filter_by_date=False)

        # Listen for changes in the Gaim files
        gaim_reader.connect("reload", lambda x: self.emit("reload"))

    def do_reload(self):
        everyone = gaim_reader.get_all_buddies()
        self.set_items(everyone)


class OnlineBuddySource(ItemSource):
    '''
    A list of all currently online buddies.
    '''
    def __init__(self, name, icon = None):
        ItemSource.__init__(self, name, icon, uri="source://People/Online", filter_by_date=False)
        self._DIRTY = True

        # Listen for changes in the Gaim files.  This isn't a good indicator of
        # presence changes, but it works sometimes.
        gaim_reader.connect("reload", lambda x: self.emit("reload"))

    def get_items(self):
        if self._DIRTY:
            self._reload()
        return ItemSource.get_items(self)

    def _reload(self):
        online = []
        try:
            online = [x for x in gaim_reader.get_all_buddies() if x.get_is_online()]
        except (dbus.DBusException, TypeError):
            print " !!! Error accessing Gaim2 D-BUS interface.  Is Gaim2 running?"
        self.set_items(online)
        self._DIRTY = False

    def do_reload(self):
        # We call _reload in get_items to load on-demand.
        self._DIRTY = True


class PeopleTopic(Topic):
    '''
    Lists recently talked to people, people online now, and all contacts, as
    well as listing out Gaim groups.  Placeholders for future
    Gmail/Friendster/LinkedIn contact integration.
    '''
    def __init__(self):
        Topic.__init__(self, _("People"), uri="topic://People")

        from gimmie_running import RunningChats
        self.set_running_source_factory(lambda: RunningChats())

        # Reload the sidebar when Gaim's files change
        gaim_reader.connect("reload", lambda x: self.emit("reload"))

    def do_reload(self):
        # FIXME: Prune for groups at the current zoom level?
        source_list = [RecentConversations(_("Recent People"), "im"),
                       OnlineBuddySource(_("Online Now"), "gaim"),
                       EverybodySource(_("Everybody"), "gnome-globe"),
                       None]

        if gmail_reader:
            source_list += [GmailSource(_("Gmail"), "stock_mail")]
        else:
            source_list += [DisabledItemSource(_("GMail"), "stock_mail")]
            
        source_list += [DisabledItemSource(_("Friendster"), "stock_people"),
                        None]

        for group in gaim_reader.get_groups():
            for buddy in group.get_buddies():
                # Only include groups which have at least one active buddy
                if buddy.get_account().get_auto_login():
                    source = GroupSource(group)
                    source_list.append(source)
                    break

        self.set_sidebar_source_list(source_list)

    def get_hint_color(self):
        return gtk.gdk.color_parse("pink")

    def _show_accounts_editor(self):
        ImplementMe()

    def make_availability_menu(self):
        menu = gtk.Menu()

        item = gtk.RadioMenuItem(None, _("Available"))
        item.set_active(True)
        item.show()
        menu.append(item)

        item = gtk.RadioMenuItem(item, _("Offline"))
        item.show()
        menu.append(item)

        away_menu = gtk.Menu()
        away_item = gtk.MenuItem(_("Away Messages..."))
        away_item.show()
        away_menu.append(away_item)

        item = gtk.MenuItem(_("Away"))
        item.show()
        item.set_submenu(away_menu)
        menu.append(item)

        item = gtk.SeparatorMenuItem()
        item.show()
        menu.append(item)

        item = gtk.MenuItem(_("Accounts..."))
        item.connect("activate", lambda w: self._show_accounts_editor())
        item.show()
        menu.append(item)

        item = gtk.SeparatorMenuItem()
        item.show()
        menu.append(item)

        for acct in gaim_reader.get_accounts():
            item = gtk.CheckMenuItem(acct.get_name())
            if acct.get_auto_login():
                item.set_active(True)
            item.show()
            menu.append(item)

        # FIXME: Everything here isn't implemented yet.
        for mi in menu:
            mi.set_sensitive(False)
        return menu

    def _gaim_get_default_account_obj(self):
        acct = gaim_dbus.GaimAccountsFindConnected("", "")
        if not acct:
            acct = gaim_dbus.GaimAccountsFindAny("", "")
        return acct

    def _add_person(self, w):
        if gaim_dbus:
            try:
                acct = self._gaim_get_default_account_obj()
                gaim_dbus.GaimBlistRequestAddBuddy(acct, "", "", "")
            except dbus.DBusException:
                print " !!! Gaim2 D-BUS interface not available.  Is Gaim2 running?"

    def _add_group(self, w):
        if gaim_dbus:
            try:
                gaim_dbus.GaimBlistRequestAddGroup()
            except dbus.DBusException:
                print " !!! Gaim2 D-BUS interface not available.  Is Gaim2 running?"

    def get_toolbar_items(self, tooltips):
        tools = []

        img = icon_factory.load_image("stock_person", gtk.ICON_SIZE_LARGE_TOOLBAR)
        btn = gtk.ToolButton(img, _("New Person"))
        btn.set_tooltip(tooltips, _("Add a new contact person"))
        btn.set_is_important(True)
        btn.connect("clicked", self._add_person)
        tools.append(btn)

        ### Groups are lame
        '''
        img = icon_factory.load_image("stock_people", gtk.ICON_SIZE_LARGE_TOOLBAR)
        btn = gtk.ToolButton(img, _("New Group"))
        btn.set_tooltip(tooltips, _("Add a new group of people to the sidebar"))
        btn.set_is_important(True)
        btn.connect("clicked", self._add_group)
        tools.append(btn)
        '''

        img = icon_factory.load_image(gtk.STOCK_YES, gtk.ICON_SIZE_BUTTON)
        btn = ToolMenuButton(img, _("Available"))
        btn.set_tooltip(tooltips, _("Set your online status"))
        btn.set_is_important(True)
        btn.set_menu(self.make_availability_menu())
        tools.append(btn)

        return tools

    def get_context_menu_items(self):
        items = []

        img = icon_factory.load_image(gtk.STOCK_YES, gtk.ICON_SIZE_LARGE_TOOLBAR)
        btn = gtk.ImageMenuItem(_("Available"))
        btn.set_image(img)
        btn.set_submenu(self.make_availability_menu())
        btn.show_all()
        items.append(btn)
        
        return items

