
# -----------------------------------------------------------------------------
#
# (c) 2009 The University of Glasgow
#
# This file is part of the GHC build system.
#
# To understand how the build system works and how to modify it, see
#      http://hackage.haskell.org/trac/ghc/wiki/Building/Architecture
#      http://hackage.haskell.org/trac/ghc/wiki/Building/Modifying
#
# -----------------------------------------------------------------------------

# ToDo List.
#
#   * remove old Makefiles, add new stubs for building in subdirs
#     * docs/Makefile
#     * docs/docbook-cheat-sheet/Makefile
#     * docs/man/Makefile
#     * docs/storage-mgmt/Makefile
#     * docs/vh/Makefile
#     * rts/dotnet/Makefile
#     * utils/Makefile
#   * add Makefiles for the rest of the utils/ programs that aren't built
#     by default (need to exclude them from 'make all' too)

# Possible cleanups:
#
#   * per-source-file dependencies instead of one .depend file?
#   * eliminate undefined variables, and use --warn-undefined-variables?
#   * put outputs from different ways in different subdirs of distdir/build,
#     then we don't have to use -osuf/-hisuf.  We would have to install
#     them in different places too, so we'd need ghc-pkg support for packages
#     of different ways.
#   * make PACKAGES_STAGE1 generated by './configure' or './boot'?
#   * we should use a directory of package.conf files rather than a single
#     file for the inplace package database, so that we can express
#     dependencies more accurately.  Otherwise it's possible to get into
#     a state where the package database is out of date, and the build
#     system doesn't know.

# Approximate build order.
#
# The actual build order is defined by dependencies, and the phase
# ordering used to ensure correct ordering of makefile-generation; see
#    http://hackage.haskell.org/trac/ghc/wiki/Building/Architecture/Idiom/PhaseOrdering
#
#     * With bootstrapping compiler:
#           o Build utils/ghc-cabal
#           o Build utils/ghc-pkg
#           o Build utils/hsc2hs
#     * For each package:
#	    o configure, generate package-data.mk and inplace-pkg-info
#           o register each package into inplace/lib/package.conf
#     * build libffi
#     * With bootstrapping compiler:
#	    o Build libraries/{filepath,hpc,Cabal}
#           o Build compiler (stage 1)
#     * With stage 1:
#           o Build libraries/*
#	    o Build rts
#           o Build utils/* (except haddock)
#           o Build compiler (stage 2)
#     * With stage 2:
#           o Build utils/haddock
#           o Build compiler (stage 3) (optional)
#     * With haddock:
#           o libraries/*
#           o compiler

.PHONY: default all haddock

# We need second expansion for the way we handle directories, so that
#     | $$$$(dir $$$$@)/.
# expands to the directory of a rule that uses a % pattern.
.SECONDEXPANSION:

default : all

# Catch make if it runs away into an infinite loop
ifeq      "$(MAKE_RESTARTS)" ""
else ifeq "$(MAKE_RESTARTS)" "1"
else
$(error Make has restarted itself $(MAKE_RESTARTS) times; is there a makefile bug?)
endif

ifneq "$(CLEANING)" "YES"
CLEANING = NO
endif

# -----------------------------------------------------------------------------
# Misc GNU make utils

nothing=
space=$(nothing) $(nothing)
comma=,

# Cancel all suffix rules.  Ideally we'd like to have 'make -r' turned on
# by default, because that disables all the implicit rules, but there doesn't
# seem to be a good way to do that.  This turns off all the old-style suffix
# rules, which does half the job and speeds up make quite a bit:
.SUFFIXES:

# -----------------------------------------------------------------------------
# Makefile debugging
#
# to see the effective value used for a Makefile variable, do
#  make show VALUE=MY_VALUE
#

show:
	@echo '$(VALUE)="$($(VALUE))"'

# -----------------------------------------------------------------------------
# Include subsidiary build-system bits

include mk/tree.mk

ifeq "$(findstring clean,$(MAKECMDGOALS))" ""
include mk/config.mk
ifeq "$(ProjectVersion)" ""
$(error Please run ./configure first)
endif
endif

include mk/ways.mk

# (Optional) build-specific configuration
include mk/custom-settings.mk

ifeq "$(findstring clean,$(MAKECMDGOALS))" ""
ifeq "$(GhcLibWays)" ""
$(error $$(GhcLibWays) is empty, it must contain at least one way)
endif
endif

ifeq "$(phase)" ""
phase = final
endif

# -----------------------------------------------------------------------------
# Utility definitions

include rules/prof.mk
include rules/trace.mk
include rules/make-command.mk
include rules/pretty_commands.mk

# -----------------------------------------------------------------------------
# Macros for standard targets

include rules/all-target.mk
include rules/clean-target.mk

# -----------------------------------------------------------------------------
# The inplace tree

$(eval $(call clean-target,inplace,,inplace/bin inplace/lib))

# -----------------------------------------------------------------------------
# Whether to build dependencies or not

# When we're just doing 'make clean' or 'make show', then we don't need
# to build dependencies.

ifneq "$(findstring clean,$(MAKECMDGOALS))" ""
NO_INCLUDE_DEPS = YES
NO_INCLUDE_PKGDATA = YES
endif
ifneq "$(findstring bootstrapping-files,$(MAKECMDGOALS))" ""
NO_INCLUDE_DEPS = YES
NO_INCLUDE_PKGDATA = YES
endif
ifeq "$(findstring show,$(MAKECMDGOALS))" "show"
NO_INCLUDE_DEPS = YES
# We want package-data.mk for show
endif

# -----------------------------------------------------------------------------
# Ways

include rules/way-prelims.mk

$(foreach way,$(ALL_WAYS),\
  $(eval $(call way-prelims,$(way))))

# -----------------------------------------------------------------------------
# Compilation Flags

include rules/distdir-way-opts.mk

# -----------------------------------------------------------------------------
# Finding source files and object files

include rules/hs-sources.mk
include rules/c-sources.mk
include rules/includes-sources.mk
include rules/hs-objs.mk
include rules/c-objs.mk
include rules/cmm-objs.mk

# -----------------------------------------------------------------------------
# Suffix rules

# Suffix rules cause "make clean" to fail on Windows (trac #3233)
# so we don't make any when cleaning.
ifneq "$(CLEANING)" "YES"

include rules/hs-suffix-rules-srcdir.mk
include rules/hs-suffix-rules.mk
include rules/hi-rule.mk

$(foreach way,$(ALL_WAYS),\
  $(eval $(call hi-rule,$(way))))

include rules/c-suffix-rules.mk
include rules/cmm-suffix-rules.mk

endif # CLEANING=YES

# -----------------------------------------------------------------------------
# Building package-data.mk files from .cabal files

include rules/package-config.mk

# -----------------------------------------------------------------------------
# Building dependencies

include rules/dependencies.mk
include rules/build-dependencies.mk
include rules/include-dependencies.mk

# -----------------------------------------------------------------------------
# Build package-data.mk files

include rules/build-package-data.mk

# -----------------------------------------------------------------------------
# Build and install a program

include rules/build-prog.mk
include rules/shell-wrapper.mk

# -----------------------------------------------------------------------------
# Build a perl script

include rules/build-perl.mk

# -----------------------------------------------------------------------------
# Build a package

include rules/build-package.mk
include rules/build-package-way.mk
include rules/haddock.mk
include rules/tags-package.mk
include rules/extra-packages.mk

# -----------------------------------------------------------------------------
# Registering hand-written package descriptions (used in rts)

include rules/manual-package-config.mk

# -----------------------------------------------------------------------------
# Docs

include rules/docbook.mk

# -----------------------------------------------------------------------------
# Making bindists

include rules/bindist.mk

# -----------------------------------------------------------------------------
# Directories

# Don't try to delete directories:
.PRECIOUS: %/.

# Create build directories on demand.  NB. the | below: this indicates
# that $(MKDIRHIER) is an order-only dependency, which means that this
# rule fires after building mkdirhier, but we won't try to recreate
# directories if mkdirhier changes.
%/. : | $(MKDIRHIER)
	"$(MKDIRHIER)" $@

# -----------------------------------------------------------------------------
# Lax dependencies

ifeq "$(LAX_DEPENDENCIES)" "YES"
LAX_DEPS_FOLLOW = |
else
LAX_DEPS_FOLLOW =
endif

# This is a bit of a hack. When LAX_DEPS_FOLLOW is | some rules end up
# looking like
#     target: a | b | c
# The first | signals the start of the order-only dependencies, but make
# treats the second | as a dependency. So we need to tell make how to
# build that dependency.

.PHONY: |
| :
	@:

# -----------------------------------------------------------------------------
# Properties of packages

# These lists say "if this package is built, here's a property it has"
# They do not say "this package will be built"; see $(PACKAGES_xx) for that

# Packages that are built but not installed
PKGS_THAT_ARE_INTREE_ONLY := haskeline transformers terminfo utf8-string xhtml

PKGS_THAT_ARE_DPH := \
    dph/dph-base \
    dph/dph-prim-interface dph/dph-prim-seq dph/dph-prim-par \
    dph/dph-lifted-base \
    dph/dph-lifted-boxed dph/dph-lifted-copy dph/dph-lifted-vseg \
    vector primitive random

# Packages that, if present, must be built by the stage2 compiler,
# because they use TH and/or annotations, or depend on other stage2
# packages:
PKGS_THAT_BUILD_WITH_STAGE2 := $(PKGS_THAT_ARE_DPH) haskell98 haskell2010

# Packages that we shouldn't build if we don't have TH (e.g. because
# we're building a profiled compiler):
PKGS_THAT_USE_TH := $(PKGS_THAT_ARE_DPH)

# Packages that are built by stage0, in addition to stage1.  These
# packages are dependencies of GHC, that we do not assume the stage0
# compiler already has installed (or up-to-date enough).
#
# We assume that the stage0 compiler has a suitable bytestring package,
# so we don't have to include it below.
PKGS_THAT_BUILD_WITH_STAGE0 = Cabal/Cabal hpc binary bin-package-db hoopl

# $(EXTRA_PACKAGES)  is another classification, of packages built but
#                    not installed
#                    It is set in rules/extra-package.mk, 
#                    by $(call extra-packages) a little further down 
#                    this ghc.mk 



# ----------------------------------------------------------------------------
# Packages to build
# The lists of packages that we *actually* going to build in each stage:
#
#  $(PACKAGES_STAGE0) 
#  $(PACKAGES_STAGE1)
#  $(PACKAGES_STAGE2)
#
# These are automatically derived from
#    (a) the set of packages in this source tree
#    (b) the predicates above, e.g. $(PKGS_THAT_BUILD_WITH_STAGE2)
#    (c) which platform we're on, and a few other things


# no processing to do on this one: it really is the list of packages
# to build with stage 0.
PACKAGES_STAGE0 = $(PKGS_THAT_BUILD_WITH_STAGE0)

define addPackageGeneral
# args: $1 = PACKAGES variable, $2 = package, $3 = condition
    ifeq "$3" ""
        $1 += $2
    else
        ifeq "$$(CLEANING)" "YES"
            $1 += $2
        else
            ifeq $3
                $1 += $2
            endif
        endif
    endif
endef

define addPackage # args: $1 = package, $2 = condition
ifneq "$(filter $1,$(PKGS_THAT_USE_TH)) $(GhcProfiled)" "$1 YES"
ifeq "$(filter $1,$(PKGS_THAT_BUILD_WITH_STAGE2))" "$1"
ifneq "$(BuildingCrossCompiler)" "YES"
$(call addPackageGeneral,PACKAGES_STAGE2,$1,$2)
endif
else
$(call addPackageGeneral,PACKAGES_STAGE1,$1,$2)
endif
endif
endef

# Add all the packages. Note that we need to add them in dependency
# order, as this is the order that they get configured in.

$(eval $(call addPackage,ghc-prim))
ifeq "$(CLEANING)" "YES"
$(eval $(call addPackage,integer-gmp))
$(eval $(call addPackage,integer-simple))
else
$(eval $(call addPackage,$(INTEGER_LIBRARY)))
endif
$(eval $(call addPackage,base))
$(eval $(call addPackage,filepath))
$(eval $(call addPackage,array))
$(eval $(call addPackage,deepseq))
$(eval $(call addPackage,bytestring))
$(eval $(call addPackage,containers))
$(eval $(call addPackage,old-locale))
$(eval $(call addPackage,old-time))

$(eval $(call addPackage,Win32,($$(Windows),YES)))
$(eval $(call addPackage,time))
$(eval $(call addPackage,unix,($$(Windows),NO)))

$(eval $(call addPackage,directory))
$(eval $(call addPackage,process))
$(eval $(call addPackage,haskell98))
$(eval $(call addPackage,haskell2010))
$(eval $(call addPackage,hpc))
$(eval $(call addPackage,pretty))
$(eval $(call addPackage,template-haskell))
$(eval $(call addPackage,Cabal/Cabal))
$(eval $(call addPackage,binary))
$(eval $(call addPackage,bin-package-db))
$(eval $(call addPackage,hoopl))
$(eval $(call addPackage,transformers))
$(eval $(call addPackage,utf8-string))
$(eval $(call addPackage,xhtml))
$(eval $(call addPackage,terminfo,($$(Windows),NO)))
$(eval $(call addPackage,haskeline))

$(eval $(call extra-packages))

# -------------------------------------------
# Dependencies between package-data.mk files

# We cannot run ghc-cabal to configure a package until we have
# configured and registered all of its dependencies.  So the following
# hack forces all the configure steps to happen in exactly the following order:
#
#  $(PACKAGES_STAGE1) ghc(stage2) $(PACKAGES_STAGE2)
#
# Ideally we should use the correct dependencies here to allow more
# parallelism, but we don't know the dependencies until we've
# generated the package-data.mk files.
define fixed_pkg_dep
libraries/$1/$2/package-data.mk : $$(GHC_PKG_INPLACE) $$(fixed_pkg_prev)
fixed_pkg_prev:=libraries/$1/$2/package-data.mk
endef

ifneq "$(BINDIST)" "YES"
fixed_pkg_prev=
$(foreach pkg,$(PACKAGES_STAGE1),$(eval $(call fixed_pkg_dep,$(pkg),dist-install)))

# the GHC package doesn't live in libraries/, so we add its dependency manually:
compiler/stage2/package-data.mk: $(fixed_pkg_prev)
fixed_pkg_prev:=compiler/stage2/package-data.mk

# and continue with PACKAGES_STAGE2, which depend on GHC:
$(foreach pkg,$(PACKAGES_STAGE2),$(eval $(call fixed_pkg_dep,$(pkg),dist-install)))

ghc/stage1/package-data.mk: compiler/stage1/package-data.mk
ghc/stage2/package-data.mk: compiler/stage2/package-data.mk
# haddock depends on ghc and some libraries, but depending on GHC's
# package-data.mk is sufficient, as that in turn depends on all the
# libraries
utils/haddock/dist/package-data.mk: compiler/stage2/package-data.mk
utils/ghc-pwd/dist-install/package-data.mk: compiler/stage2/package-data.mk
utils/ghc-cabal/dist-install/package-data.mk: compiler/stage2/package-data.mk

utils/ghc-pkg/dist-install/package-data.mk: compiler/stage2/package-data.mk
utils/hsc2hs/dist-install/package-data.mk: compiler/stage2/package-data.mk
utils/compare_sizes/dist-install/package-data.mk: compiler/stage2/package-data.mk
utils/runghc/dist-install/package-data.mk: compiler/stage2/package-data.mk

# add the final package.conf dependency: ghc-prim depends on RTS
libraries/ghc-prim/dist-install/package-data.mk : rts/package.conf.inplace
endif

# --------------------------------
# Misc package-related settings

BOOT_PKG_CONSTRAINTS := \
    $(foreach d,$(PACKAGES_STAGE0),\
        $(foreach p,$(basename $(notdir $(wildcard libraries/$d/*.cabal))),\
            --constraint "$p == $(shell grep -i "^Version:" libraries/$d/$p.cabal | sed "s/[^0-9.]//g")"))

# The actual .a and .so/.dll files: needed for dependencies.
ALL_STAGE1_LIBS  = $(foreach lib,$(PACKAGES_STAGE1),$(libraries/$(lib)_dist-install_v_LIB))
ifeq "$(BuildSharedLibs)" "YES"
ALL_STAGE1_LIBS += $(foreach lib,$(PACKAGES_STAGE1),$(libraries/$(lib)_dist-install_dyn_LIB))
endif
BOOT_LIBS = $(foreach lib,$(PACKAGES_STAGE0),$(libraries/$(lib)_dist-boot_v_LIB))

# ----------------------------------------
# Special magic for the ghc-prim package

# We want the ghc-prim package to include the GHC.Prim module when it
# is registered, but not when it is built, because GHC.Prim is not a
# real source module, it is built-in to GHC.  The old build system did
# this using Setup.hs, but we can't do that here, so we have a flag to
# enable GHC.Prim in the .cabal file (so that the ghc-prim package
# remains compatible with the old build system for the time being).
# GHC.Prim module in the ghc-prim package with a flag:
#
libraries/ghc-prim_CONFIGURE_OPTS += --flag=include-ghc-prim

# And then we strip it out again before building the package:
define libraries/ghc-prim_PACKAGE_MAGIC
libraries/ghc-prim_dist-install_MODULES := $$(filter-out GHC.Prim,$$(libraries/ghc-prim_dist-install_MODULES))
endef

PRIMOPS_TXT = $(GHC_COMPILER_DIR)/prelude/primops.txt

libraries/ghc-prim/dist-install/build/GHC/PrimopWrappers.hs : $(GENPRIMOP_INPLACE) $(PRIMOPS_TXT) | $$(dir $$@)/.
	"$(GENPRIMOP_INPLACE)" --make-haskell-wrappers <$(PRIMOPS_TXT) >$@

# Required so that Haddock documents the primops.
libraries/ghc-prim_dist-install_EXTRA_HADDOCK_SRCS = libraries/ghc-prim/dist-install/build/autogen/GHC/Prim.hs

# ----------------------------------------
# Special magic for the integer package

ifneq "$(CLEANING)" "YES"
ifeq "$(INTEGER_LIBRARY)" "integer-gmp"
libraries/base_dist-install_CONFIGURE_OPTS += --flags=-integer-simple
else ifeq "$(INTEGER_LIBRARY)" "integer-simple"
libraries/base_dist-install_CONFIGURE_OPTS += --flags=integer-simple
else
$(error Unknown integer library: $(INTEGER_LIBRARY))
endif
endif

# ----------------------------------------
# Workarounds for problems building DLLs on Windows

ifeq "$(TargetOS_CPP)" "mingw32"
# We don't build the GHC package the dyn way on Windows, so
# we can't build these packages the dyn way either. See trac #5987
libraries/dph/dph-lifted-base_dist-install_EXCLUDED_WAYS := dyn
libraries/dph/dph-lifted-boxed_dist-install_EXCLUDED_WAYS := dyn
libraries/dph/dph-lifted-copy_dist-install_EXCLUDED_WAYS := dyn
libraries/dph/dph-lifted-vseg_dist-install_EXCLUDED_WAYS := dyn
endif

# ----------------------------------------------
# Checking packages with 'cabal check'

ifeq "$(phase)" "final"
ifeq "$(CHECK_PACKAGES)" "YES"
all: check_packages
endif
endif

# These packages don't pass the Cabal checks because hs-source-dirs
# points outside the source directory. This isn't a real problem in
# these cases, so we just skip checking them.
# NB. these must come before we include the ghc.mk files below, because
# they disable the relevant rules.
# In compiler's case, include-dirs points outside of the source tree
CHECKED_compiler = YES

# -----------------------------------------------------------------------------
# Include build instructions from all subdirs

ifneq "$(BINDIST)" "YES"
BUILD_DIRS += \
   $(GHC_MKDIRHIER_DIR)
endif

ifeq "$(Windows)" "YES"
BUILD_DIRS += \
   $(GHC_TOUCHY_DIR)
endif

BUILD_DIRS += \
   docs/users_guide \
   docs/man \
   $(GHC_UNLIT_DIR) \
   $(GHC_HP2PS_DIR)

ifneq "$(GhcUnregisterised)" "YES"
BUILD_DIRS += \
   $(GHC_SPLIT_DIR)
endif

ifneq "$(BINDIST)" "YES"
BUILD_DIRS += \
   $(GHC_GENPRIMOP_DIR)
endif

ifeq "$(BuildingCrossCompiler)-$(phase)" "YES-final"
MAYBE_GHCI=
else
MAYBE_GHCI=driver/ghci
endif

BUILD_DIRS += \
   driver \
   $(MAYBE_GHCI) \
   driver/ghc \
   driver/haddock \
   libffi \
   includes \
   rts

ifneq "$(BINDIST)" "YES"
BUILD_DIRS += \
   bindisttest \
   $(GHC_GENAPPLY_DIR)
endif

ifneq "$(CLEANING)" "YES"
BUILD_DIRS += \
   $(patsubst %, libraries/%, $(PACKAGES_STAGE1))
endif

ifeq "$(INTEGER_LIBRARY)" "integer-gmp"
BUILD_DIRS += libraries/integer-gmp/gmp
else ifneq "$(findstring clean,$(MAKECMDGOALS))" ""
BUILD_DIRS += libraries/integer-gmp/gmp
endif

ifeq "$(BuildingCrossCompiler)-$(phase)" "YES-final"
MAYBE_COMPILER=
MAYBE_GHCTAGS=
MAYBE_HPC=
MAYBE_RUNGHC=
else
MAYBE_COMPILER=compiler
MAYBE_GHCTAGS=utils/ghctags
MAYBE_HPC=utils/hpc
MAYBE_RUNGHC=utils/runghc
endif

BUILD_DIRS += \
   utils/haddock \
   utils/haddock/doc \
   $(MAYBE_COMPILER) \
   $(GHC_HSC2HS_DIR) \
   $(GHC_PKG_DIR) \
   utils/testremove \
   $(MAYBE_GHCTAGS) \
   utils/ghc-pwd \
   $(GHC_CABAL_DIR) \
   $(MAYBE_HPC) \
   $(MAYBE_RUNGHC) \
   ghc

ifneq "$(BINDIST)" "YES"
ifneq "$(BuildingCrossCompiler)-$(phase)" "YES-final"
BUILD_DIRS += \
   utils/mkUserGuidePart
endif
endif

BUILD_DIRS += utils/count_lines
BUILD_DIRS += utils/compare_sizes

ifneq "$(CLEANING)" "YES"
# After compiler/, because these packages depend on it
BUILD_DIRS += \
   $(patsubst %, libraries/%, $(PACKAGES_STAGE2))
endif

# ----------------------------------------------
# Actually include all the sub-ghc.mk's

# BUILD_DIRS_EXTRA needs to come after BUILD_DIRS, because stuff in
# libraries/dph/ghc.mk refers to stuff defined earlier, in particular
# things like $(libraries/dph/dph-base_dist-install_GHCI_LIB)
include $(patsubst %, %/ghc.mk, $(BUILD_DIRS) $(BUILD_DIRS_EXTRA))

# A useful pseudo-target (must be after the include above, because it needs
# the value of things like $(libraries/base_dist-install_v_LIB).
.PHONY: stage1_libs
stage1_libs : $(ALL_STAGE1_LIBS)

# ----------------------------------------------
# Per-package compiler flags
# 
# If you want to add per-package compiler flags, this 
# is the place to do it.  Do it like this for package <pkg>
#   
#   libraries/<pkg>_dist-boot_HC_OPTS += -Wwarn
#   libraries/<pkg>_dist-install_HC_OPTS += -Wwarn

# Add $(GhcLibHcOpts) to all package builds
$(foreach pkg,$(PACKAGES_STAGE1) $(PACKAGES_STAGE2),$(eval libraries/$(pkg)_dist-install_HC_OPTS += $$(GhcLibHcOpts)))

# Add $(GhcBootLibHcOpts) to all stage0 package builds
$(foreach pkg,$(PACKAGES_STAGE0),$(eval libraries/$(pkg)_dist-boot_HC_OPTS += $$(GhcBootLibHcOpts)))

# -----------------------------------------------
# Haddock-related bits

# Don't run Haddock for the package that will not be installed
$(foreach p,$(PKGS_THAT_ARE_INTREE_ONLY),$(eval libraries/$p_dist-install_DO_HADDOCK = NO))
# We don't haddock the bootstrapping libraries
$(foreach p,$(PACKAGES_STAGE0),$(eval libraries/$p_dist-boot_DO_HADDOCK = NO))

# Build the Haddock contents and index
ifeq "$(HADDOCK_DOCS)" "YES"
libraries/dist-haddock/index.html: inplace/bin/haddock$(exeext) $(ALL_HADDOCK_FILES)
	cd libraries && sh gen_contents_index --inplace
ifeq "$(phase)" "final"
$(eval $(call all-target,library_doc_index,libraries/dist-haddock/index.html))
endif
INSTALL_LIBRARY_DOCS += libraries/dist-haddock/*
endif

# -----------------------------------------------------------------------------
# Bootstrapping libraries

# We need to build a few libraries with the installed GHC, since GHC itself
# and some of the tools depend on them:

ifneq "$(BINDIST)" "YES"

ifneq "$(BOOTSTRAPPING_CONF)" ""
ifeq "$(wildcard $(BOOTSTRAPPING_CONF))" ""
$(shell echo "[]" >$(BOOTSTRAPPING_CONF))
endif
endif

$(eval $(call clean-target,$(BOOTSTRAPPING_CONF),,$(BOOTSTRAPPING_CONF)))

# register the boot packages in strict sequence, because running
# multiple ghc-pkgs in parallel doesn't work (registrations may get
# lost).
fixed_pkg_prev=
$(foreach pkg,$(PACKAGES_STAGE0),$(eval $(call fixed_pkg_dep,$(pkg),dist-boot)))

compiler/stage1/package-data.mk : $(fixed_pkg_prev)
endif

ifneq "$(BINDIST)" "YES"
# Make sure we have all the GHCi libs by the time we've built
# ghc-stage2.  DPH includes a bit of Template Haskell which needs the
# GHCI libs, and we don't have a better way to express that dependency.
#
GHCI_LIBS = $(foreach lib,$(PACKAGES_STAGE1),$(libraries/$(lib)_dist-install_GHCI_LIB)) \
	    $(compiler_stage2_GHCI_LIB)

ifeq "$(UseArchivesForGhci)" "NO"
ghc/stage2/build/tmp/$(ghc_stage2_PROG) : $(GHCI_LIBS)
endif

ifeq "$(UseArchivesForGhci)" "YES"
GHCI_lib_way = v
else
GHCI_lib_way = GHCI
endif

# Deps for TH uses in libraries
$(foreach way, $(GhcLibWays),$(eval \
libraries/vector/dist-install/build/Data/Vector/Fusion/Stream/Monadic.$($(way)_osuf): \
    $(libraries/primitive_dist-install_$(GHCI_lib_way)_LIB) \
  ))
endif

# -----------------------------------------------------------------------------
# Creating a local mingw copy on Windows

ifeq "$(Windows)" "YES"

install : install_mingw
.PHONY: install_mingw
install_mingw : $(INPLACE_MINGW)
	"$(CP)" -Rp $(INPLACE_MINGW) $(prefix)

install : install_perl
.PHONY: install_perl
install_perl : $(INPLACE_PERL)
	"$(CP)" -Rp $(INPLACE_PERL) $(prefix)

endif # Windows

ifneq "$(BINDIST)" "YES"
$(ghc-prim-$(libraries/ghc-prim_dist-install_VERSION)_HADDOCK_FILE): \
    libraries/ghc-prim/dist-install/build/autogen/GHC/Prim.hs
endif # BINDIST

libraries/ghc-prim/dist-install/build/autogen/GHC/Prim.hs: \
                            $(PRIMOPS_TXT) $(GENPRIMOP_INPLACE) \
                          | $$(dir $$@)/.
	"$(GENPRIMOP_INPLACE)" --make-haskell-source < $< > $@

.PHONY: tags
tags: tags_compiler

.PHONY: TAGS
TAGS: TAGS_compiler

# -----------------------------------------------------------------------------
# Installation

install: install_libs install_packages install_libexecs \
         install_libexec_scripts install_bins install_topdirs
ifeq "$(HADDOCK_DOCS)" "YES"
install: install_docs
endif

install_bins: $(INSTALL_BINS)
	$(call INSTALL_DIR,"$(DESTDIR)$(bindir)")
	for i in $(INSTALL_BINS); do \
		$(call INSTALL_PROGRAM,$(INSTALL_BIN_OPTS),$$i,"$(DESTDIR)$(bindir)") ;  \
	done

install_libs: $(INSTALL_LIBS)
	$(call INSTALL_DIR,"$(DESTDIR)$(ghclibdir)")
	for i in $(INSTALL_LIBS); do \
		case $$i in \
		  *.a) \
		    $(call INSTALL_DATA,$(INSTALL_OPTS),$$i,"$(DESTDIR)$(ghclibdir)"); \
		    $(RANLIB) $(DESTDIR)$(ghclibdir)/`basename $$i` ;; \
		  *.dll) \
		    $(call INSTALL_PROGRAM,$(INSTALL_OPTS),$$i,"$(DESTDIR)$(ghclibdir)") ; \
		    $(STRIP_CMD) "$(DESTDIR)$(ghclibdir)"/$$i ;; \
		  *.so) \
		    $(call INSTALL_SHLIB,$(INSTALL_OPTS),$$i,"$(DESTDIR)$(ghclibdir)") ;; \
		  *.dylib) \
		    $(call INSTALL_SHLIB,$(INSTALL_OPTS),$$i,"$(DESTDIR)$(ghclibdir)");; \
		  *) \
		    $(call INSTALL_DATA,$(INSTALL_OPTS),$$i,"$(DESTDIR)$(ghclibdir)"); \
		esac; \
	done

install_libexec_scripts: $(INSTALL_LIBEXEC_SCRIPTS)
ifeq "$(INSTALL_LIBEXEC_SCRIPTS)" ""
	@:
else
	$(call INSTALL_DIR,"$(DESTDIR)$(ghclibexecdir)")
	for i in $(INSTALL_LIBEXEC_SCRIPTS); do \
		$(call INSTALL_SCRIPT,$(INSTALL_OPTS),$$i,"$(DESTDIR)$(ghclibexecdir)"); \
	done
endif

install_libexecs:  $(INSTALL_LIBEXECS)
ifeq "$(INSTALL_LIBEXECS)" ""
	@:
else
	$(call INSTALL_DIR,"$(DESTDIR)$(ghclibexecdir)")
	for i in $(INSTALL_LIBEXECS); do \
		$(call INSTALL_PROGRAM,$(INSTALL_BIN_OPTS),$$i,"$(DESTDIR)$(ghclibexecdir)"); \
	done
# We rename ghc-stage2, so that the right program name is used in error
# messages etc.
	"$(MV)" "$(DESTDIR)$(ghclibexecdir)/ghc-stage$(INSTALL_GHC_STAGE)" "$(DESTDIR)$(ghclibexecdir)/ghc"
endif

install_topdirs: $(INSTALL_TOPDIRS)
	$(call INSTALL_DIR,"$(DESTDIR)$(topdir)")
	for i in $(INSTALL_TOPDIRS); do \
		$(call INSTALL_PROGRAM,$(INSTALL_BIN_OPTS),$$i,"$(DESTDIR)$(topdir)"); \
	done

install_docs: $(INSTALL_DOCS)
	$(call INSTALL_DIR,"$(DESTDIR)$(docdir)")
ifneq "$(INSTALL_DOCS)" ""
	for i in $(INSTALL_DOCS); do \
		$(call INSTALL_DOC,$(INSTALL_OPTS),$$i,"$(DESTDIR)$(docdir)"); \
	done
endif
	$(call INSTALL_DIR,"$(DESTDIR)$(docdir)/html")
	$(call INSTALL_DOC,$(INSTALL_OPTS),docs/index.html,"$(DESTDIR)$(docdir)/html")
ifneq "$(INSTALL_LIBRARY_DOCS)" ""
	$(call INSTALL_DIR,"$(DESTDIR)$(docdir)/html/libraries")
	for i in $(INSTALL_LIBRARY_DOCS); do \
		$(call INSTALL_DOC,$(INSTALL_OPTS),$$i,"$(DESTDIR)$(docdir)/html/libraries/"); \
	done
	$(call INSTALL_DATA,$(INSTALL_OPTS),libraries/prologue.txt,"$(DESTDIR)$(docdir)/html/libraries/")
	$(call INSTALL_SCRIPT,$(INSTALL_OPTS),libraries/gen_contents_index,"$(DESTDIR)$(docdir)/html/libraries/")
endif
ifneq "$(INSTALL_HTML_DOC_DIRS)" ""
	for i in $(INSTALL_HTML_DOC_DIRS); do \
		$(call INSTALL_DIR,"$(DESTDIR)$(docdir)/html/`basename $$i`"); \
		$(call INSTALL_DOC,$(INSTALL_OPTS),$$i/*,"$(DESTDIR)$(docdir)/html/`basename $$i`"); \
	done
endif

INSTALLED_PACKAGE_CONF=$(DESTDIR)$(topdir)/package.conf.d

# Install packages in the right order, so that ghc-pkg doesn't complain.
# Also, install ghc-pkg first.
ifeq "$(Windows)" "NO"
INSTALLED_GHC_REAL=$(DESTDIR)$(ghclibexecdir)/ghc
INSTALLED_GHC_PKG_REAL=$(DESTDIR)$(ghclibexecdir)/ghc-pkg
else
INSTALLED_GHC_REAL=$(DESTDIR)$(bindir)/ghc.exe
INSTALLED_GHC_PKG_REAL=$(DESTDIR)$(bindir)/ghc-pkg.exe
endif

INSTALLED_PKG_DIRS := $(addprefix libraries/,$(PACKAGES_STAGE1))
ifeq "$(BuildingCrossCompiler)" "NO"
INSTALLED_PKG_DIRS := $(INSTALLED_PKG_DIRS) compiler
endif
INSTALLED_PKG_DIRS := $(INSTALLED_PKG_DIRS) $(addprefix libraries/,$(PACKAGES_STAGE2))
ifeq "$(InstallExtraPackages)" "NO"
INSTALLED_PKG_DIRS := $(filter-out $(addprefix libraries/,$(EXTRA_PACKAGES)),\
                                   $(INSTALLED_PKG_DIRS))
endif
INSTALLED_PKG_DIRS := $(filter-out $(addprefix libraries/,$(PKGS_THAT_ARE_INTREE_ONLY)),\
                                   $(INSTALLED_PKG_DIRS))

# Set the INSTALL_DISTDIR_p for each package; compiler is special
$(foreach p,$(filter-out compiler,$(INSTALLED_PKG_DIRS)),\
   $(eval INSTALL_DISTDIR_$p = dist-install))
INSTALL_DISTDIR_compiler = stage2

# Now we can do the installation
install_packages: install_libexecs
install_packages: rts/package.conf.install
	$(call INSTALL_DIR,"$(DESTDIR)$(topdir)")
	$(call removeTrees,"$(INSTALLED_PACKAGE_CONF)")
	$(call INSTALL_DIR,"$(INSTALLED_PACKAGE_CONF)")
	"$(INSTALLED_GHC_PKG_REAL)" --force --global-package-db "$(INSTALLED_PACKAGE_CONF)" update rts/package.conf.install
	$(foreach p, $(INSTALLED_PKG_DIRS),                           \
	    $(call make-command,                                      \
                   CROSS_COMPILE="$(CrossCompilePrefix)"              \
	           "$(GHC_CABAL_INPLACE)" install                     \
	                                  "$(INSTALLED_GHC_REAL)"     \
	                                  "$(INSTALLED_GHC_PKG_REAL)" \
	                                  "$(STRIP_CMD)"              \
	                                  "$(DESTDIR)$(topdir)"       \
	                                  $p $(INSTALL_DISTDIR_$p)    \
	                                  '$(DESTDIR)'                \
	                                  '$(prefix)'                 \
	                                  '$(ghclibdir)'              \
	                                  '$(docdir)/html/libraries'  \
	                                  $(RelocatableBuild)))
# when we install the packages above, ghc-pkg obeys umask when creating
# the package.conf files, but for everything else we specify the
# permissions. We therefore now fix the permissions of package.cache.
# This means "sudo make install" does the right thing even if it runs
# with an 077 umask.
	for f in '$(INSTALLED_PACKAGE_CONF)'/*; do $(CREATE_DATA) "$$f"; done

# -----------------------------------------------------------------------------
# Binary distributions

ifneq "$(CLEANING)" "YES"
# This rule seems to hold some files open on Windows which prevents
# cleaning, perhaps due to the $(wildcard).

$(eval $(call bindist,.,\
    LICENSE \
    README \
    INSTALL \
    configure config.sub config.guess install-sh \
    settings.in \
    packages \
    Makefile \
    mk/config.mk.in \
    $(INPLACE_BIN)/mkdirhier \
    utils/ghc-cabal/dist-install/build/tmp/ghc-cabal \
    utils/ghc-pwd/dist-install/build/tmp/ghc-pwd \
    $(BINDIST_WRAPPERS) \
    $(BINDIST_PERL_SOURCES) \
    $(BINDIST_LIBS) \
    $(BINDIST_HI) \
    $(BINDIST_EXTRAS) \
    $(includes_H_FILES) \
    $(includes_DERIVEDCONSTANTS) \
    $(includes_GHCCONSTANTS) \
    $(libffi_HEADERS) \
    $(INSTALL_LIBEXECS) \
    $(INSTALL_LIBEXEC_SCRIPTS) \
    $(INSTALL_TOPDIRS) \
    $(INSTALL_BINS) \
    $(INSTALL_MANPAGES) \
    $(INSTALL_DOCS) \
    $(INSTALL_LIBRARY_DOCS) \
    $(addsuffix /*,$(INSTALL_HTML_DOC_DIRS)) \
    docs/index.html \
    compiler/stage2/doc \
    $(wildcard libraries/*/dist-install/doc/) \
    $(wildcard libraries/*/*/dist-install/doc/) \
    $(filter-out settings,$(INSTALL_LIBS)) \
    $(filter-out %/project.mk mk/config.mk %/mk/install.mk,$(MAKEFILE_LIST)) \
    mk/project.mk \
    mk/install.mk.in \
    bindist.mk \
    libraries/gen_contents_index \
    libraries/prologue.txt \
    $(wildcard libraries/dph/LICENSE \
               libraries/dph/ghc-packages \
               libraries/dph/ghc-packages2 \
               libraries/dph/ghc-stage2-package) \
 ))
endif
# mk/project.mk gets an absolute path, so we manually include it in
# the bindist with a relative path

BIN_DIST_MK = $(BIN_DIST_PREP_DIR)/bindist.mk

unix-binary-dist-prep:
	$(call removeTrees,bindistprep/)
	"$(MKDIRHIER)" $(BIN_DIST_PREP_DIR)
	set -e; for i in packages LICENSE compiler ghc rts libraries utils docs libffi includes driver mk rules Makefile aclocal.m4 config.sub config.guess install-sh settings.in ghc.mk inplace distrib/configure.ac distrib/README distrib/INSTALL; do ln -s ../../$$i $(BIN_DIST_PREP_DIR)/; done
	echo "HADDOCK_DOCS       = $(HADDOCK_DOCS)"       >> $(BIN_DIST_MK)
	echo "LATEX_DOCS         = $(LATEX_DOCS)"         >> $(BIN_DIST_MK)
	echo "BUILD_DOCBOOK_HTML = $(BUILD_DOCBOOK_HTML)" >> $(BIN_DIST_MK)
	echo "BUILD_DOCBOOK_PS   = $(BUILD_DOCBOOK_PS)"   >> $(BIN_DIST_MK)
	echo "BUILD_DOCBOOK_PDF  = $(BUILD_DOCBOOK_PDF)"  >> $(BIN_DIST_MK)
	echo "BUILD_MAN          = $(BUILD_MAN)"          >> $(BIN_DIST_MK)
	echo "GHC_CABAL_INPLACE  = utils/ghc-cabal/dist-install/build/tmp/ghc-cabal" >> $(BIN_DIST_MK)
	cd $(BIN_DIST_PREP_DIR) && autoreconf
	$(call removeFiles,$(BIN_DIST_PREP_TAR))
# h means "follow symlinks", e.g. if aclocal.m4 is a symlink to a source
# tree then we want to include the real file, not a symlink to it
	cd bindistprep && "$(TAR_CMD)" hcf - -T ../$(BIN_DIST_LIST) | bzip2 -c > ../$(BIN_DIST_PREP_TAR_BZ2)

windows-binary-dist-prep:
	$(call removeTrees,bindistprep/)
	$(MAKE) prefix=$(TOP)/$(BIN_DIST_PREP_DIR) install
	cd bindistprep && "$(TAR_CMD)" cf - $(BIN_DIST_NAME) | bzip2 -c > ../$(BIN_DIST_PREP_TAR_BZ2)

windows-installer:
ifeq "$(ISCC_CMD)" ""
	@echo No ISCC_CMD, so not making installer
else
	"$(ISCC_CMD)" /O. /Fbindistprep/$(WINDOWS_INSTALLER_BASE) - < distrib/ghc.iss
endif

# tryTimes tries to run its third argument multiple times, until it
# succeeds. Don't call it directly; call try10Times instead.
# The first and second argument to tryTimes are lists of values.
# The length of the first argument is the number of times we have
# already tried. The length of the second argument is the number more
# times we will try.
tryTimes = $(if $2, \
                { echo 'Try $(words x $1): $3' ; $3 ; } || \
                $(call tryTimes,x $1,$(wordlist 2,$(words $2),$2),$3), \
                )

# Try to run the argument 10 times. If all 10 fail, fail.
try10Times = $(call tryTimes,,x x x x x x x x x x,$1) { echo Failed; false; }

.PHONY: publish-binary-dist
publish-binary-dist:
	$(call try10Times,$(PublishCp) $(BIN_DIST_TAR_BZ2) $(PublishLocation)/dist)
ifeq "$(mingw32_TARGET_OS)" "1"
	$(call try10Times,$(PublishCp) $(WINDOWS_INSTALLER) $(PublishLocation)/dist)
endif

ifeq "$(mingw32_TARGET_OS)" "1"
DOCDIR_TO_PUBLISH = $(BIN_DIST_INST_DIR)/doc
else
DOCDIR_TO_PUBLISH = $(BIN_DIST_INST_DIR)/share/doc/ghc
endif

.PHONY: publish-docs
publish-docs:
	$(call try10Times,$(PublishCp) -r $(DOCDIR_TO_PUBLISH)/* $(PublishLocation)/docs)

# -----------------------------------------------------------------------------
# Source distributions

# Do it like this:
#
#	$ make
#	$ make sdist
#

# A source dist is built from a complete build tree, because we
# require some extra files not contained in a darcs checkout: the
# output from Happy and Alex, for example.
#
# The steps performed by 'make dist' are as follows:
#   - create a complete link-tree of the current build tree in /tmp
#   - run 'make distclean' on that tree
#   - remove a bunch of other files that we know shouldn't be in the dist
#   - tar up first the extralibs package, then the main source package

#
# Directory in which we're going to build the src dist
#
SRC_DIST_ROOT      = sdistprep
SRC_DIST_BASE_NAME = ghc-$(ProjectVersion)

SRC_DIST_GHC_NAME          = ghc-$(ProjectVersion)-src
SRC_DIST_GHC_ROOT          = $(SRC_DIST_ROOT)/ghc
SRC_DIST_GHC_DIR           = $(SRC_DIST_GHC_ROOT)/$(SRC_DIST_BASE_NAME)
SRC_DIST_GHC_TARBALL       = $(SRC_DIST_ROOT)/$(SRC_DIST_GHC_NAME).tar.bz2

SRC_DIST_TESTSUITE_NAME    = ghc-$(ProjectVersion)-testsuite
SRC_DIST_TESTSUITE_ROOT    = $(SRC_DIST_ROOT)/testsuite-ghc
SRC_DIST_TESTSUITE_DIR     = $(SRC_DIST_TESTSUITE_ROOT)/$(SRC_DIST_BASE_NAME)
SRC_DIST_TESTSUITE_TARBALL = $(SRC_DIST_ROOT)/$(SRC_DIST_TESTSUITE_NAME).tar.bz2

#
# Files to include in source distributions
#
SRC_DIST_GHC_DIRS = mk rules docs distrib bindisttest libffi includes \
    utils docs rts compiler ghc driver libraries ghc-tarballs
SRC_DIST_GHC_FILES += \
    configure.ac config.guess config.sub configure \
    aclocal.m4 README ANNOUNCE HACKING LICENSE Makefile install-sh \
    ghc.spec.in ghc.spec settings.in VERSION \
    boot boot-pkgs packages ghc.mk

VERSION :
	echo $(ProjectVersion) >VERSION

sdist : VERSION

# Use:
#     $(call sdist_ghc_file,compiler,stage2,cmm,Foo/Bar,CmmLex,x)
# to copy the generated file that replaces compiler/cmm/Foo/Bar/CmmLex.x, where
# "stage2" is the dist dir.
define sdist_ghc_file
	"$(CP)" $1/$2/build/$4/$5.hs $(SRC_DIST_GHC_DIR)/$1/$3/$4
	mv $(SRC_DIST_GHC_DIR)/$1/$3/$4/$5.$6 $(SRC_DIST_GHC_DIR)/$1/$3/$4/$5.$6.source
endef

.PHONY: sdist-ghc-prep
sdist-ghc-prep :
	$(call removeTrees,$(SRC_DIST_GHC_ROOT))
	$(call removeFiles,$(SRC_DIST_GHC_TARBALL))
	-mkdir $(SRC_DIST_ROOT)
	mkdir $(SRC_DIST_GHC_ROOT)
	mkdir $(SRC_DIST_GHC_DIR)
	cd $(SRC_DIST_GHC_DIR) && for i in $(SRC_DIST_GHC_DIRS); do mkdir $$i; ( cd $$i && lndir $(TOP)/$$i ); done
	cd $(SRC_DIST_GHC_DIR) && for i in $(SRC_DIST_GHC_FILES); do $(LN_S) $(TOP)/$$i .; done
	cd $(SRC_DIST_GHC_DIR) && $(MAKE) distclean
	$(call removeTrees,$(SRC_DIST_GHC_DIR)/libraries/tarballs/)
	$(call removeTrees,$(SRC_DIST_GHC_DIR)/libraries/stamp/)
	$(call removeTrees,$(SRC_DIST_GHC_DIR)/compiler/stage[123])
	$(call removeFiles,$(SRC_DIST_GHC_DIR)/mk/build.mk)
	$(call sdist_ghc_file,compiler,stage2,cmm,,CmmLex,x)
	$(call sdist_ghc_file,compiler,stage2,cmm,,CmmParse,y)
	$(call sdist_ghc_file,compiler,stage2,parser,,Lexer,x)
	$(call sdist_ghc_file,compiler,stage2,parser,,Parser,y.pp)
	$(call sdist_ghc_file,compiler,stage2,parser,,ParserCore,y)
	$(call sdist_ghc_file,utils/hpc,dist-install,,,HpcParser,y)
	$(call sdist_ghc_file,utils/genprimopcode,dist,,,Lexer,x)
	$(call sdist_ghc_file,utils/genprimopcode,dist,,,Parser,y)
	$(call sdist_ghc_file,utils/haddock,dist,src,Haddock,Lex,x)
	$(call sdist_ghc_file,utils/haddock,dist,src,Haddock,Parse,y)
	cd $(SRC_DIST_GHC_DIR) && "$(FIND)" $(SRC_DIST_GHC_DIRS) \( -name .git -o -name "autom4te*" -o -name "*~" -o -name "\#*" -o -name ".\#*" -o -name "log" -o -name "*-SAVE" -o -name "*.orig" -o -name "*.rej" \) -print | "$(XARGS)" $(XARGS_OPTS) "$(RM)" $(RM_OPTS_REC)

.PHONY: sdist-testsuite-prep
sdist-testsuite-prep :
	$(call removeTrees,$(SRC_DIST_TESTSUITE_ROOT))
	$(call removeFiles,$(SRC_DIST_TESTSUITE_TARBALL))
	-mkdir $(SRC_DIST_ROOT)
	mkdir $(SRC_DIST_TESTSUITE_ROOT)
	mkdir $(SRC_DIST_TESTSUITE_DIR)
	mkdir $(SRC_DIST_TESTSUITE_DIR)/testsuite
	cd $(SRC_DIST_TESTSUITE_DIR)/testsuite && lndir $(TOP)/testsuite
	$(call removeTrees,$(SRC_DIST_TESTSUITE_DIR)/testsuite/.git)

.PHONY: sdist
sdist : sdist-ghc-prep sdist-testsuite-prep
	cd $(SRC_DIST_GHC_ROOT) && "$(TAR_CMD)" chf - $(SRC_DIST_BASE_NAME) 2> src_ghc_log | bzip2 > $(TOP)/$(SRC_DIST_GHC_TARBALL)
	cd $(SRC_DIST_TESTSUITE_ROOT) && "$(TAR_CMD)" chf - $(SRC_DIST_BASE_NAME) 2> src_ghc_log | bzip2 > $(TOP)/$(SRC_DIST_TESTSUITE_TARBALL)

sdist-manifest : $(SRC_DIST_GHC_TARBALL)
	tar tjf $(SRC_DIST_GHC_TARBALL) | sed "s|^ghc-$(ProjectVersion)/||" | sort >sdist-manifest

# Upload the distribution(s)
# Retrying is to work around buggy firewalls that corrupt large file transfers
# over SSH.
ifneq "$(PublishLocation)" ""
publish-sdist :
	$(call try10Times,$(PublishCp) $(SRC_DIST_GHC_TARBALL) $(PublishLocation)/dist)
	$(call try10Times,$(PublishCp) $(SRC_DIST_TESTSUITE_TARBALL) $(PublishLocation)/dist)
endif

ifeq "$(BootingFromHc)" "YES"
# In a normal build we use GHC to compile C files (see
# rules/c-suffix-rules.mk), which passes a number of its own options
# to the C compiler.  So when bootstrapping we have to provide these
# flags explicitly to C compilations.
SRC_CC_OPTS += -DNO_REGS -DUSE_MINIINTERPRETER
SRC_CC_OPTS += -D__GLASGOW_HASKELL__=$(ProjectVersionInt)
SRC_CC_OPTS += $(addprefix -I,$(GHC_INCLUDE_DIRS))
endif

# -----------------------------------------------------------------------------
# sdisting libraries

# Use manually, with e.g.:
#     make sdist_directory

sdist_%:
	inplace/bin/ghc-cabal sdist libraries/$* dist-install

# -----------------------------------------------------------------------------
# Cleaning

.PHONY: clean

CLEAN_FILES += libraries/bootstrapping.conf
CLEAN_FILES += libraries/integer-gmp/cbits/GmpDerivedConstants.h
CLEAN_FILES += libraries/integer-gmp/cbits/mkGmpDerivedConstants

# These four are no longer generated, but we still clean them for a while
# as they may still be in old GHC trees:
CLEAN_FILES += includes/GHCConstants.h
CLEAN_FILES += includes/DerivedConstants.h
CLEAN_FILES += includes/ghcautoconf.h
CLEAN_FILES += includes/ghcplatform.h

clean : clean_files clean_libraries

.PHONY: clean_files
clean_files :
	$(call removeFiles,$(CLEAN_FILES))

.PHONY: clean_libraries
clean_libraries: $(patsubst %,clean_libraries/%_dist-install,$(PACKAGES_STAGE1) $(PACKAGES_STAGE2))
clean_libraries: $(patsubst %,clean_libraries/%_dist-boot,$(PACKAGES_STAGE0))

clean_libraries:
	$(call removeTrees,$(patsubst %, libraries/%/dist, $(PACKAGES_STAGE1) $(PACKAGES_STAGE2)))
	$(call removeFiles,$(wildcard $(patsubst %, libraries/%/*.buildinfo, $(PACKAGES_STAGE1) $(PACKAGES_STAGE2))))
	$(call removeFiles,$(patsubst %, libraries/%/config.log, $(PACKAGES_STAGE1) $(PACKAGES_STAGE2)))
	$(call removeFiles,$(patsubst %, libraries/%/config.status, $(PACKAGES_STAGE1) $(PACKAGES_STAGE2)))
	$(call removeFiles,$(wildcard $(patsubst %, libraries/%/include/Hs*Config.h, $(PACKAGES_STAGE1) $(PACKAGES_STAGE2))))

# We have to define a clean target for each library manually, because the
# libraries/*/ghc.mk files are not included when we're cleaning.
ifeq "$(CLEANING)" "YES"
$(foreach lib,$(PACKAGES_STAGE0),\
  $(eval $(call clean-target,libraries/$(lib),dist-boot,libraries/$(lib)/dist-boot)))
$(foreach lib,$(PACKAGES_STAGE1) $(PACKAGES_STAGE2),\
  $(eval $(call clean-target,libraries/$(lib),dist-install,libraries/$(lib)/dist-install)))
endif

clean : clean_haddock_index
.PHONY: clean_haddock_index
clean_haddock_index:
	$(call removeTrees,libraries/dist-haddock)

clean : clean_bindistprep
.PHONY: clean_bindistprep
clean_bindistprep:
	$(call removeTrees,bindistprep/)

distclean : clean
	$(call removeFiles,config.cache config.status config.log mk/config.h mk/stamp-h)
	$(call removeFiles,mk/config.mk mk/are-validating.mk mk/project.mk)
	$(call removeFiles,mk/config.mk.old mk/project.mk.old)
	$(call removeFiles,settings docs/users_guide/ug-book.xml)
	$(call removeFiles,compiler/ghc.cabal compiler/ghc.cabal.old)
	$(call removeFiles,ghc/ghc-bin.cabal)
	$(call removeFiles,libraries/base/include/HsBaseConfig.h)
	$(call removeFiles,libraries/directory/include/HsDirectoryConfig.h)
	$(call removeFiles,libraries/process/include/HsProcessConfig.h)
	$(call removeFiles,libraries/unix/include/HsUnixConfig.h)
	$(call removeFiles,libraries/old-time/include/HsTimeConfig.h)
	$(call removeTrees,utils/ghc-pwd/dist-boot)
	$(call removeTrees,inplace)
	$(call removeTrees,$(patsubst %, libraries/%/autom4te.cache, $(PACKAGES_STAGE1) $(PACKAGES_STAGE2)))

maintainer-clean : distclean
	$(call removeFiles,configure mk/config.h.in)
	$(call removeTrees,autom4te.cache $(wildcard libraries/*/autom4te.cache))
	$(call removeFiles,ghc.spec)
	$(call removeFiles,$(patsubst %, libraries/%/GNUmakefile, \
	        $(PACKAGES_STAGE1) $(PACKAGES_STAGE2)))
	$(call removeFiles,$(patsubst %, libraries/%/ghc.mk, $(PACKAGES_STAGE1) $(PACKAGES_STAGE2)))
	$(call removeFiles,$(patsubst %, libraries/%/configure, \
	        $(PACKAGES_STAGE1) $(PACKAGES_STAGE2)))
	$(call removeFiles,libraries/base/include/HsBaseConfig.h.in)
	$(call removeFiles,libraries/directory/include/HsDirectoryConfig.h.in)
	$(call removeFiles,libraries/process/include/HsProcessConfig.h.in)
	$(call removeFiles,libraries/unix/include/HsUnixConfig.h.in)
	$(call removeFiles,libraries/old-time/include/HsTimeConfig.h.in)

.PHONY: all_libraries

.PHONY: bootstrapping-files
bootstrapping-files: $(includes_H_CONFIG)
bootstrapping-files: $(includes_DERIVEDCONSTANTS)
bootstrapping-files: $(includes_GHCCONSTANTS)
bootstrapping-files: $(libffi_HEADERS)

.DELETE_ON_ERROR:

# -----------------------------------------------------------------------------
# Numbered phase targets

.PHONY: phase_0_builds
phase_0_builds: $(utils/hsc2hs_dist_depfile_haskell)
phase_0_builds: $(utils/hsc2hs_dist_depfile_c_asm)
phase_0_builds: $(utils/genprimopcode_dist_depfile_haskell)
phase_0_builds: $(utils/genprimopcode_dist_depfile_c_asm)

.PHONY: phase_1_builds
phase_1_builds: $(PACKAGE_DATA_MKS)

