/*
** 1998-08-15 -	A small utility module that deals with icons. Serves two main purposes:
**		1) Provides name-to-pixmap lookups, and thus a form of caching, and
**		2) Allows the search paths used to locate icon pixmaps to be abstracted
**		   somewhat. Nice.
**		This module is mainly used by e.g. dpformat, when it needs to render
**		icon pixmaps.
** 1998-08-30 -	Added function to seek out all icons in a path. Also added a built-in
**		inline pixmap which is used when a named pixmap can't be found.
** 1999-08-28 -	Started using the gdk_pixmap_colormap_create_from_xpm_d() function (whose
**		name really kind of rolls off your tongue, doesn't it?). Simplified things.
** 1999-11-21 -	Rewrote the ico_flush() function, to use g_hash_table_foreach_remove(),
**		which made it a lot shorter and cleaner. Good.
*/

#include "gentoo.h"

#include <stdlib.h>

#include "fileutil.h"
#include "strutil.h"
#include "iconutil.h"

/* ----------------------------------------------------------------------------------------- */

/* This name is reserved to mean "a blank icon". It is very easy
** to think of it as specifying that NO icon should be shown.
*/
#define	ICON_NONE	_("(None)")

#include "graphics/icon_empty.xpm"
#include "graphics/icon_failed.xpm"

/* ----------------------------------------------------------------------------------------- */

struct IconInfo {
	GHashTable	*icon;			/* Provides name-to-pixmap lookup. */
	const gchar	*path;			/* A local copy of the PTID_ICONS path. */
	GdkPixmap	*empty_pixmap;		/* An empty icon. It's just blank! */
	GdkBitmap	*empty_mask;		/* The very empty mask. */
	GdkPixbuf	*empty;
	GdkPixmap	*fail_pixmap;		/* This is returned if a named icon isn't found. */
	GdkBitmap	*fail_mask;		/* Mask for unable-to-load case. */
	GdkPixbuf	*fail;
};

typedef struct {
	gchar		name[PATH_MAX];		/* This is real big. */
	GdkPixmap	*pixmap;
	GdkBitmap	*mask;
	GdkPixbuf	*pixbuf;
} IconGfx;

/* ----------------------------------------------------------------------------------------- */

/* 1998-08-15 -	IconGfx comparison function for hash table. */
static gint cmp_icon(gconstpointer a, gconstpointer b)
{
	return strcmp(((const IconGfx *) a)->name, ((const IconGfx *) b)->name) == 0;
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-03-31 -	Initialize icon utility module. */
IconInfo * ico_initialize(MainInfo *min)
{
	IconInfo	*ico;

	ico = g_malloc(sizeof *ico);

	ico->icon = g_hash_table_new(g_str_hash, cmp_icon);

	ico->empty_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, gtk_widget_get_default_colormap(), &ico->empty_mask, NULL, icon_empty_xpm);
	ico->empty = gdk_pixbuf_new_from_xpm_data((const char **) icon_empty_xpm);

	ico->fail_pixmap  = gdk_pixmap_colormap_create_from_xpm_d(NULL, gtk_widget_get_default_colormap(), &ico->fail_mask, NULL, icon_failed_xpm);
	ico->fail = gdk_pixbuf_new_from_xpm_data((const char **) icon_failed_xpm);

	ico->path = min->cfg.path.path[PTID_ICON]->str;

	return ico;
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-08-30 -	Filter out stuff not ending in ".xpm".
** 1998-09-02 -	Eased up the implementation, using the string utility module. I like one-liners.
*/
static gboolean icon_filter(const gchar *path, const gchar *name)
{
	return stu_has_suffix(name, ".xpm") || stu_has_suffix(name, ".png");
}

/* 1998-08-30 -	This returns a (possibly giant) list of all icon names in the given
**		icon <path>. If <path> is NULL, the current icon path (PTID_ICON) is used.
**		All path components are scanned for files ending in ".xpm"; these are assumed
**		to be icon pixmaps. The returned list has each element's <data> pointer pointing
**		at a dynamically allocated string which is the icon name. This memory tied up
**		by these strings is reclaimed by calling ico_free_all() on the list once you're
**		done with it.
*/
GList * ico_get_all(MainInfo *min, const gchar *path)
{
	GList	*list = NULL;
	gchar	*head;

	if(path == NULL)
		path = min->cfg.path.path[PTID_ICON]->str;

	if(path != NULL)
	{
		list = fut_scan_path(path, icon_filter);
		if((head = g_strdup(ICON_NONE)) != NULL)
			list = g_list_prepend(list, head);
	}
	return list;
}

gboolean ico_no_icon(const gchar *name)
{
	return strcmp(name, ICON_NONE) == 0;
}

/* 1998-08-30 -	Free a list of icon names as returned by ico_get_all() above. */
void ico_free_all(GList *list)
{
	if(list != NULL)
		fut_free_path(list);
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-08-15 -	Internal icon loader. Uses the (new) file utility function 'fut_locate' to
**		search the icon path. A configurable icon path is on the agenda...
** 1999-08-28 -	Removed need for passing a GdkWindow, by using the _colormap_ pixmap creator. Good?
*/
static IconGfx * load(IconInfo *ico, const gchar *name)
{
	const gchar	*fname;
	IconGfx		*ig;

	ig = g_malloc(sizeof *ig);
	stu_strncpy(ig->name, name, sizeof ig->name);
	ig->pixbuf = NULL;
	if(strcmp(name, ICON_NONE) == 0)		/* Is it the special, reserved, blank icon? */
	{
		ig->pixmap = ico->empty_pixmap;
		ig->mask   = ico->empty_mask;
		ig->pixbuf = ico->empty;
		return ig;
	}
	if((fname = fut_locate(ico->path, name)) != NULL)
	{
		if((ig->pixmap = gdk_pixmap_colormap_create_from_xpm(NULL, gtk_widget_get_default_colormap(), &ig->mask, NULL, fname)) != NULL)
		{
			gdk_pixmap_ref(ig->pixmap);
			gdk_bitmap_ref(ig->mask);
			return ig;
		}
	}
	g_free(ig);

	return NULL;
}

/* 1998-08-15 -	Internal icon retrieval function. Does a lookup in the hash table, loads
**		from disk if not found.
*/
static GdkPixmap * get_or_load(IconInfo *ico, const gchar *name, GdkBitmap **mask)
{
	IconGfx	*ig;

	if(ico->icon == NULL)
		ico->icon = g_hash_table_new(g_str_hash, cmp_icon);
	if(ico->icon != NULL)
	{
		if((ig = g_hash_table_lookup(ico->icon, name)) != NULL)
		{
			*mask = ig->mask;
			return ig->pixmap;
		}
		else if((ig = load(ico, name)) != NULL)
		{
			g_hash_table_insert(ico->icon, ig->name, ig);
			*mask = ig->mask;
			return ig->pixmap;
		}
		else
		{
			*mask = ico->fail_mask;
			return ico->fail_pixmap;
		}
	}
	return NULL;
}

static IconGfx * load_pixbuf(IconInfo *ico, const gchar *name)
{
	const gchar	*fname;
	IconGfx		*ig;

	ig = g_malloc(sizeof *ig);
	stu_strncpy(ig->name, name, sizeof ig->name);
	if(strcmp(name, ICON_NONE) == 0)		/* Is it the special, reserved, blank icon? */
	{
		ig->pixmap = ico->empty_pixmap;
		ig->mask   = ico->empty_mask;
		ig->pixbuf = NULL;
		return ig;
	}
	if((fname = fut_locate(ico->path, name)) != NULL)
	{
		if((ig->pixbuf = gdk_pixbuf_new_from_file(fname, NULL)) != NULL)
			return ig;
	}
	return NULL;
}

static GdkPixbuf * get_or_load_pixbuf(IconInfo *ico, const gchar *name)
{
	IconGfx	*ig;

	if(ico->icon == NULL)
		ico->icon = g_hash_table_new(g_str_hash, cmp_icon);
	if((ig = g_hash_table_lookup(ico->icon, name)) != NULL)
		return ig->pixbuf;
	else if((ig = load_pixbuf(ico, name)) != NULL)
		return ig->pixbuf;
	return ico->fail;
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-08-15 -	Retrieve a pixmap corresponding to the given <name>, which is typically
**		an (unqualified) file name, such as "Makefile.xpm". Returns a pointer to
**		a GDK pixmap, and fills in the <mask> pointer with a pointer to the mask
**		needed when rendering this pixmap. Returns NULL on failure.
** 1999-08-28 -	Removed silly GdkWindow argument. Nicer.
** 2008-07-29 -	FIXME DEPRECATED This interface needs to go.
*/
GdkPixmap * ico_icon_get(MainInfo *min, const gchar *name, GdkBitmap **mask)
{
	if((name == NULL) || (*name == '\0') || (mask == NULL))
		return NULL;

	min->ico->path = min->cfg.path.path[PTID_ICON]->str;	/* Make sure we have a fresh path. */

	return get_or_load(min->ico, name, mask);
}

/* 2008-07-29 -	New interface, this returns a GDK Pixbuf which is far more useful in these
 *		times of GTK+ 2 than the old GdkPixmap/GdkBitmap combo.
*/
GdkPixbuf * ico_icon_get_pixbuf(MainInfo *min, const gchar *name)
{
	if(min == NULL || name == NULL || *name == '\0')
		return NULL;
	min->ico->path = min->cfg.path.path[PTID_ICON]->str;

	return get_or_load_pixbuf(min->ico, name);
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-11-21 -	Kill given icon. This is a g_hash_table_foreach_remove() callback.
** 2001-01-01 -	Happy new year! :) Um, now frees the memory, too...
*/
static gboolean kill_icon(gpointer key, gpointer value, gpointer user)
{
	IconGfx	*icon = value;

	gdk_pixmap_unref(icon->pixmap);
	gdk_bitmap_unref(icon->mask);
	g_free(icon);

	return TRUE;			/* Tells glib to actually remove hash entry. */
}

/* 1998-09-02 -	Flush all loaded icons out, thus making the next reference a load for sure. */
void ico_flush(MainInfo *min)
{
	if(min->ico->icon != NULL)
		g_hash_table_foreach_remove(min->ico->icon, kill_icon, NULL);
}
