/*
** 1998-09-12 -	We must have a Copy As command, since it's quite useful. Reuses all
**		copy_XXX() functions defined in the cmd_copy module, of course. Since
**		it so easy, I just had to let this module implement a Clone command, too!
** 1999-01-03 -	No longer passes the stat-structure from the dir listing to the copy module,
**		thus forcing it to do a re-read, which assures that the info (size) is fresh.
** 1999-03-06 -	Adapted for new selection/generic command handling.
** 1999-05-30 -	Added SymLinkAs and SymLinkClone commands. Nice.
*/

#include "gentoo.h"
#include "convutil.h"
#include "fileutil.h"
#include "dirpane.h"
#include "errors.h"
#include "overwrite.h"
#include "progress.h"
#include "strutil.h"
#include "cmd_copy.h"
#include "cmd_delete.h"
#include "cmd_generic.h"

#include "cmd_copyas.h"

#define	CMD_ID	"copy as"

/* ----------------------------------------------------------------------------------------- */

typedef enum { CPA_COPYAS, CPA_CLONE, CPA_SYMLINKAS, CPA_SYMLINKCLONE } CpaAction;

typedef struct {
	GtkWidget	*vbox;
	GtkWidget	*label;
	GtkWidget	*entry;
	CpaAction	action;
	MainInfo	*min;
	gint		ovw_open;
} CpaInfo;

/* ----------------------------------------------------------------------------------------- */

/* 1998-09-12 -	Update body of copy as GUI. */
static void cpa_copy_body(MainInfo *min, DirPane *src, DirRow *row, gpointer user)
{
	gchar		buf[FILENAME_MAX + 32];
	const gchar	*text;
	CpaInfo		*cpa = user;

	text = DP_ROW_NAMED(row);
	if(cpa->action == CPA_COPYAS)
		g_snprintf(buf, sizeof buf, _("Enter Name for Copy of \"%s\""), text);
	else
		g_snprintf(buf, sizeof buf, _("Enter Name for Clone of \"%s\""), text);
	gtk_label_set_text(GTK_LABEL(cpa->label), buf);
	gtk_entry_set_text(GTK_ENTRY(cpa->entry), text);
	gtk_entry_select_region(GTK_ENTRY(cpa->entry), 0, -1);
	gtk_widget_grab_focus(cpa->entry);

	if(cpa->ovw_open == FALSE)
	{
		if(cpa->action == CPA_COPYAS)
			ovw_overwrite_begin(min, _("\"%s\" Already Exists - Proceed With Copy?"), 0U);
		else
			ovw_overwrite_begin(min, _("\"%s\" Already Exists - Continue With Clone?"), 0U);
		pgs_progress_begin(min, cpa->action == CPA_COPYAS ? _("Copying As...") : _("Cloning..."), PFLG_COUNT_RECURSIVE | PFLG_ITEM_VISIBLE | PFLG_BYTE_VISIBLE);
		cpa->ovw_open = TRUE;
	}
}

/* 1999-05-30 -	Update body for symlink operation. */
static void cpa_link_body(MainInfo *min, DirPane *src, DirRow *row, gpointer user)
{
	gchar		buf[FILENAME_MAX + 32];
	const gchar	*text;
	CpaInfo		*cpa = user;

	text = DP_ROW_NAMED(row);
	if(cpa->action == CPA_SYMLINKAS)
		g_snprintf(buf, sizeof buf, _("Enter Name to Link \"%s\" As"), text);
	else if(cpa->action == CPA_SYMLINKCLONE)
		g_snprintf(buf, sizeof buf, _("Enter Name for Link Clone of \"%s\""), text);
	gtk_label_set_text(GTK_LABEL(cpa->label), buf);
	gtk_entry_set_text(GTK_ENTRY(cpa->entry), text);
	gtk_entry_select_region(GTK_ENTRY(cpa->entry), 0, -1);
	gtk_widget_grab_focus(cpa->entry);

	if(cpa->ovw_open == FALSE)
	{
		ovw_overwrite_begin(min, _("\"%s\" Already Exists - Proceed With Symlink?"), OVWF_NO_RECURSE_TEST);
		pgs_progress_begin(min, "Linking...", -1);
		cpa->ovw_open = TRUE;
	}
}

static const gchar * op_name(CpaAction action)
{
	switch(action)
	{
		case CPA_COPYAS:	return "CopyAs";
		case CPA_CLONE:		return "Clone";
		case CPA_SYMLINKAS:	return "SymLinkAs";
		case CPA_SYMLINKCLONE:	return "SymLinkClone";
	}
	return NULL;
}

/* 1998-09-12 -	Perform action for copy as/clone command. */
static gint cpa_action(MainInfo *min, DirPane *src, DirPane *dst, DirRow *row, gpointer user)
{
	const gchar		*text;
	gchar			dest[PATH_MAX], *file;
	CpaInfo			*cpa = user;
	OvwRes			ores;

	if(!fut_cd(src->dir.path, NULL, 0))
		return 0;

	text = gtk_entry_get_text(GTK_ENTRY(cpa->entry));
	if(text && *text)
	{
		Conv		conv;

		file = conv_begin(&conv, text);

		/* Enforce difference between clone and copy: clone name is without path. */
		switch(cpa->action)
		{
			case CPA_COPYAS:
			case CPA_SYMLINKAS:
				if(strchr(text, G_DIR_SEPARATOR) == NULL)
					g_snprintf(dest, sizeof dest, "%s%s%s", dst->dir.path, G_DIR_SEPARATOR_S, file);
				else
					stu_strncpy(dest, file, sizeof dest);
				break;
			case CPA_CLONE:
			case CPA_SYMLINKCLONE:
				if(strchr(text, G_DIR_SEPARATOR) != NULL)
				{
					conv_end(&conv);
					return 0;
				}
				g_snprintf(dest, sizeof dest, "%s%s%s", src->dir.path, G_DIR_SEPARATOR_S, file);
				break;
		}
		conv_end(&conv);
		ores = ovw_overwrite_file(min, dest, dp_full_name(src, DP_ROW_INDEX(src, row)));
		if(ores == OVW_SKIP)
			return 1;
		else if(ores == OVW_CANCEL)
		{
			if(errno)
				err_set(min, errno, op_name(cpa->action), dp_full_name(src, DP_ROW_INDEX(src, row)));
			return 0;
		}
		else if(ores == OVW_PROCEED_FILE)
		{
			if(!del_delete_file(min, dest))
				return 0;
		}
		else if(ores == OVW_PROCEED_DIR)
		{
			if(!del_delete_dir(min, dest, FALSE))
				return 0;
		}

		if((cpa->action == CPA_COPYAS) || (cpa->action == CPA_CLONE))	/* Copy? */
		{
			if(S_ISDIR(DP_ROW_LSTAT(row).st_mode))
				copy_dir(min, DP_ROW_NAME(row), dest);
			else
				copy_file(min, DP_ROW_NAME(row), dest, &DP_ROW_LSTAT(row));
		}
		else								/* Link. */
			symlink(dp_full_name(src, DP_ROW_INDEX(src, row)), dest);
	}
	if(errno == 0)
		dp_unselect(src, DP_ROW_INDEX(src, row));
	return errno == 0;
}

static void cpa_free(gpointer user)
{
	CpaInfo	*cpa = user;

	if(cpa->ovw_open)
	{
		ovw_overwrite_end(cpa->min);
		pgs_progress_end(cpa->min);
	}
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-09-12 -	Fire up either a COPYAS, CLONE, or SYMLINK command, depending on the <action> arg. */
static gint copyas_or_clone(MainInfo *min, DirPane *src, DirPane *dst, CpaAction action)
{
	static CpaInfo	cpa;

	cpa.action = action;
	cpa.min	   = min;
	cpa.ovw_open = FALSE;

	cpa.vbox = gtk_vbox_new(FALSE, 0);
	cpa.label = gtk_label_new(_("Copy As"));
	cpa.entry = gtk_entry_new_with_max_length(FILENAME_MAX - 1);
	gtk_box_pack_start(GTK_BOX(cpa.vbox), cpa.label, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(cpa.vbox), cpa.entry, FALSE, FALSE, 0);

	if(cpa.action == CPA_COPYAS)
		return cmd_generic(min, _("Copy As"), CGF_NOALL, cpa_copy_body, cpa_action, cpa_free, &cpa);
	else if(cpa.action == CPA_CLONE)
		return cmd_generic(min, _("Clone"), CGF_NOALL | CGF_NODST | CGF_SRC, cpa_copy_body, cpa_action, cpa_free, &cpa);
	else if(cpa.action == CPA_SYMLINKAS)
		return cmd_generic(min, _("Symbolic Link As"), CGF_NOALL, cpa_link_body, cpa_action, cpa_free, &cpa);
	else if(cpa.action == CPA_SYMLINKCLONE)
		return cmd_generic(min, _("Symbolic Link Clone"), CGF_NOALL | CGF_NODST | CGF_SRC, cpa_link_body, cpa_action, cpa_free, &cpa);
	return 0;
}

/* ----------------------------------------------------------------------------------------- */

gint cmd_copyas(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	return copyas_or_clone(min, src, dst, CPA_COPYAS);
}

gint cmd_clone(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	return copyas_or_clone(min, src, dst, CPA_CLONE);
}

gint cmd_symlinkas(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	return copyas_or_clone(min, src, dst, CPA_SYMLINKAS);
}

gint cmd_symlinkclone(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	return copyas_or_clone(min, src, dst, CPA_SYMLINKCLONE);
}
