#! /usr/bin/env python

# Genetic
# Copyright (C) 2001 Jean-Baptiste LAMY
#
# This program is free software. See README or LICENSE for the license terms.

import random, string, math, operator, Tkinter
from genetic import organism, lifecycle, mutablelist

# This will solve the traveller saleman problem (=TSP) with a different, MUCH
# efficient approach than demo_saleman.py.
#
# The saleman must travel from city to city, and pass in each city; but he
# wants to walk as less as possible... Are genetic algos his best chance ?


# Define a City class. Cities are defined by 2 coordinates x and y on a map.
class City:
  next_id = 0
  
  def __init__(self, x, y):
    self.x, self.y = x, y
    
    self.id = City.next_id
    self.name = "city" + str(self.id)
    City.next_id = City.next_id + 1
    
  def __repr__(self): return self.name
  

# Define a list of cities.

cities = []


# Define the function to minimize.
# 
# Here, we'll use a MutableOrderList, i.e. a list whose order can mutate
# (= item are swapped in the list).
# 
# MutableOrderList extends the Chromosom class. The gene that contains the list
# is called "data", and so the arg of journey_distance is called "data" too.

def journey_distance(data):
  # data is : [first_city, second_city, ...]
  return reduce(operator.add, map(lambda cityA, cityB: math.sqrt((cityA.x - cityB.x) ** 2 + (cityA.y - cityB.y) ** 2), data, data[1:] + [data[0]]))


# Create our class of organism, with a "journey_distance" characteristic that
# we want to minimize.

class TSPSolver(organism.Organism):
  characteristics = [
    organism.Characteristic("journey_distance",
                            journey_distance,
                            organism.RECESSIVE_PHENOTYPE)
    ]

  # An organism is better (= smaller) than another if the journey is shorter.
  def __cmp__(self, other):
    return cmp(self.journey_distance, other.journey_distance)


SIZE = 250

class Graph(Tkinter.Canvas):
  def __init__(self, master):
    Tkinter.Canvas.__init__(self, master, bg = "white", width = SIZE, height = SIZE)
    self.bind("<Button-1>", self.leftClick )
    self.line = None
    
    #self.reset()
    
  def reset(self):
    listA = cities[:]
    random.shuffle(listA)
    organismA = TSPSolver([
      mutablelist.MutableOrderList(data = listA),
      ])
    
    listB = cities[:]
    random.shuffle(listB)
    organismB = TSPSolver([
      mutablelist.MutableOrderList(data = listB),
      ])
    
    self.organisms = [organismA, organismB]
    
  def leftClick (self, event): self.addcity(City(event.x, event.y))
  
  def addcity(self, city):
    cities.append(city)
    self.create_text(city.x - 0, city.y - 0, text = "+")
    # Phenotypes have changed, because there is one more city !
    self.reset()
    
  def generation(self, nb = 1):
    self.organisms = lifecycle.run(self.organisms, 1, nb, 50, 7)
    
  def update(self):
    if self.line: self.delete(self.line)
    best   = self.organisms[0]
    cities = best.journey_distance_args[0]
    
    xy = map(lambda city: (city.x, city.y), cities)
    xy.append(xy[0]) # close the figure
    self.line = self.create_line(*xy)

class GraphFrame(Tkinter.Tk):
  def __init__(self):
    Tkinter.Tk.__init__(self, className = "genetic saleman traveler")
    Tkinter.Label(self, text = "Click some points below, then run 1 or 10 generations !\nI will genetically compute the shortest ways between all your points !", wraplength = SIZE + 30).pack()
    self.graph = Graph(self)
    self.graph.pack()
    Tkinter.Button(self, text = "1 generation"  , command = self.generation1  ).pack()
    Tkinter.Button(self, text = "10 generations", command = self.generation10 ).pack()
        
  def generation1(self, event = None):
    if len(cities) < 2: print "No enough cities ! Click in the white rectangle !"
    else:
      self.graph.generation(1)
      self.graph.update()
        
  def generation10(self, event = None):
    if len(cities) < 2: print "No enough cities ! Click in the white rectangle !"
    else:
      self.graph.generation(10)
      self.graph.update()
    

graphframe = GraphFrame()
Tkinter.mainloop()
