static const char* op_c_source =
"/* This file is an image processing operation for GEGL                        \n"
" * GEGL is free software; you can redistribute it and/or                      \n"
" * modify it under the terms of the GNU Lesser General Public                 \n"
" * License as published by the Free Software Foundation; either               \n"
" * version 3 of the License, or (at your option) any later version.           \n"
" *                                                                            \n"
" * GEGL is distributed in the hope that it will be useful,                    \n"
" * but WITHOUT ANY WARRANTY; without even the implied warranty of             \n"
" * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          \n"
" * Lesser General Public License for more details.                            \n"
" *                                                                            \n"
" * You should have received a copy of the GNU Lesser General Public           \n"
" * License along with GEGL; if not, see <http://www.gnu.org/licenses/>.       \n"
" *                                                                            \n"
" *                                                                            \n"
" * Neon filter for GIMP for BIPS                                              \n"
" * Original by Spencer Kimball                                                \n"
" * Port to GEGL Copyright 2017 Peter O'Regan <peteroregan@gmail.com>          \n"
" *                                                                            \n"
" * This filter works in a manner similar to the \"edge\"                      \n"
" *  plug-in, but uses the first derivative of the gaussian                    \n"
" *  operator to achieve resolution independence. The IIR                      \n"
" *  method of calculating the effect is utilized to keep                      \n"
" *  the processing time constant between large and small                      \n"
" *  standard deviations.                                                      \n"
" *                                                                            \n"
" *                                                                            \n"
" *  A description of the algorithm may be found in Deriche (1993).            \n"
" *  In short, it is a fourth order recursive filter with causal and           \n"
" *  anti-causal parts which are computed separately and summed. The           \n"
" *  Gaussian is approximated using a closely-matched exponential              \n"
" *  whose coefficients in IIR form have been optimized to minimize            \n"
" *  the error against the Gaussian over a large range of std devs.            \n"
" *                                                                            \n"
" *  Deriche, R. \"Recursively Implementing the Gaussian and its Derivatives,\" in\n"
" *    Proc. Rapports de Recherche, No. 1893. April 1993.                      \n"
" */                                                                           \n"
"                                                                              \n"
"                                                                              \n"
"#include \"config.h\"                                                         \n"
"#include <glib/gi18n-lib.h>                                                   \n"
"#include <math.h>                                                             \n"
"#include <stdio.h>                                                            \n"
"                                                                              \n"
"#ifdef GEGL_PROPERTIES                                                        \n"
"                                                                              \n"
"property_double (radius, _(\"Radius\"), 5.0)                                  \n"
"   description (_(\"Radius of effect (in pixels)\"))                          \n"
"   value_range (1, 1500.0)                                                    \n"
"   ui_gamma    (2.0)                                                          \n"
"   ui_range    (1, 50.0)                                                      \n"
"   ui_meta     (\"unit\", \"pixel-distance\")                                 \n"
"                                                                              \n"
"                                                                              \n"
"property_double (amount, _(\"Intensity\"), 0.0)                               \n"
"   description (_(\"Strength of Effect\"))                                    \n"
"   value_range (0.0, 100.0)                                                   \n"
"   ui_range    (0.0, 3.0)                                                     \n"
"                                                                              \n"
"#else                                                                         \n"
"                                                                              \n"
"#define GEGL_OP_FILTER                                                        \n"
"#define GEGL_OP_NAME edge_neon                                                \n"
"#define GEGL_OP_C_SOURCE edge-neon.c                                          \n"
"                                                                              \n"
"#include \"gegl-op.h\"                                                        \n"
"                                                                              \n"
"#define GEGL_SCALE_NONE 1.0                                                   \n"
"#define NEON_MAXRGB 1.0                                                       \n"
"#define NEON_MINRGB 0.0                                                       \n"
"                                                                              \n"
"                                                                              \n"
"// Forward declarations                                                       \n"
"static void      find_constants      (gdouble           n_p[],                \n"
"                                      gdouble           n_m[],                \n"
"                                      gdouble           d_p[],                \n"
"                                      gdouble           d_m[],                \n"
"                                      gdouble           bd_p[],               \n"
"                                      gdouble           bd_m[],               \n"
"                                      gdouble           std_dev);             \n"
"                                                                              \n"
"static void      transfer_pixels     (gfloat           *src1,                 \n"
"                                      gfloat           *src2,                 \n"
"                                      gfloat           *dest,                 \n"
"                                      gint              bytes,                \n"
"                                      gint              width);               \n"
"                                                                              \n"
"static void      combine_to_gradient (gfloat           *dest,                 \n"
"                                      gfloat           *src2,                 \n"
"                                      gint              bytes,                \n"
"                                      gint              width,                \n"
"                                      gdouble           amount);              \n"
"                                                                              \n"
"                                                                              \n"
"                                                                              \n"
"static void neon_prepare (GeglOperation *operation)                           \n"
"{                                                                             \n"
"  //const Babl *src_format = gegl_operation_get_source_format (operation, \"input\");\n"
"  gegl_operation_set_format (operation, \"input\", babl_format (\"R'G'B'A float\"));\n"
"  gegl_operation_set_format (operation, \"output\", babl_format(\"R'G'B'A float\"));\n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static gboolean                                                               \n"
"neon_process (GeglOperation     *op,                                          \n"
"         GeglBuffer             *in_buf,                                      \n"
"         GeglBuffer             *out_buf,                                     \n"
"         const GeglRectangle    *roi,                                         \n"
"         gint                   level)                                        \n"
"{                                                                             \n"
"    // Used later for traversal                                               \n"
"    GeglRectangle cur_roi;                                                    \n"
"    gint          x0, y0;                                                     \n"
"    gint          width, height;                                              \n"
"    gint          chpp, bpp, aidx;     //Number of channels & bytes per pixel \n"
"                                                                              \n"
"    //Pointers to buffer locations & iterators                                \n"
"    gfloat       *val_p, *val_m, *vp, *vm;                                    \n"
"    gint          i, j, b;                                                    \n"
"    gint          row, col, bufsz;                                            \n"
"    gint          terms; // # active terms during filter (active along edges) \n"
"    gdouble       vanish_pt;                                                  \n"
"                                                                              \n"
"    /* Retrieve a pointer to GeglProperties via Chant */                      \n"
"    const Babl *rgba32 = babl_format(\"R'G'B'A float\");                      \n"
"    GeglProperties *o = GEGL_PROPERTIES (op);                                 \n"
"    gdouble      radius, amount;                                              \n"
"    gdouble      std_dev;                                                     \n"
"                                                                              \n"
"    //Linear buffers                                                          \n"
"    gfloat       *dest;                                                       \n"
"    gfloat       *src, *vert, *sp_p, *sp_m;                                   \n"
"                                                                              \n"
"    //Coefficient holders                                                     \n"
"    gdouble       n_p[5], n_m[5];                                             \n"
"    gdouble       d_p[5], d_m[5];                                             \n"
"    gdouble       bd_p[5], bd_m[5];                                           \n"
"    gdouble       initial_p[4];                                               \n"
"    gdouble       initial_m[4];                                               \n"
"                                                                              \n"
"                                                                              \n"
"    radius = o->radius;                                                       \n"
"    amount = o->amount;                                                       \n"
"                                                                              \n"
"    width = roi->width;                                                       \n"
"    height = roi->height;                                                     \n"
"    x0 = roi->x;                                                              \n"
"    y0 = roi->y;                                                              \n"
"    chpp = 4; //RGBA                                                          \n"
"    aidx = 3; //Alpha index                                                   \n"
"    bpp = babl_format_get_bytes_per_pixel(rgba32);                            \n"
"                                                                              \n"
"    radius = radius + 1.0;                                                    \n"
"    vanish_pt = 1.0/255.0; //8-bit holdover, for now. (TODO)                  \n"
"    std_dev = sqrt(-(radius * radius) / (2 * log(vanish_pt)));                \n"
"                                                                              \n"
"    find_constants (n_p, n_m, d_p, d_m, bd_p, bd_m, std_dev);                 \n"
"                                                                              \n"
"    bufsz = bpp* MAX (width,height); //RGBA                                   \n"
"                                                                              \n"
"    //Buffers for causal and anti-causal parts                                \n"
"    val_p = g_new (gfloat, bufsz);                                            \n"
"    val_m = g_new (gfloat, bufsz);                                            \n"
"                                                                              \n"
"    src  = g_new (gfloat, bufsz);                                             \n"
"    vert = g_new (gfloat, bufsz);                                             \n"
"    dest = g_new (gfloat, bufsz);                                             \n"
"                                                                              \n"
"    //Vertical pass of the 1D IIR                                             \n"
"    for (col=0; col<width; col++)                                             \n"
"    {                                                                         \n"
"      memset (val_p, 0.0, height * bpp);                                      \n"
"      memset (val_m, 0.0, height * bpp);                                      \n"
"                                                                              \n"
"      //Acquire an entire column                                              \n"
"      gegl_rectangle_set (&cur_roi,x0+col,y0,1,height);                       \n"
"      gegl_buffer_get (in_buf, &cur_roi, GEGL_SCALE_NONE,rgba32,src,GEGL_AUTO_ROWSTRIDE,GEGL_ABYSS_NONE);\n"
"                                                                              \n"
"      sp_p = src;                                                             \n"
"      sp_m = src + (height - 1) * chpp;                                       \n"
"      vp = val_p;                                                             \n"
"      vm = val_m + (height - 1) * chpp;                                       \n"
"                                                                              \n"
"      /*  Set up the first vals  */                                           \n"
"      for (i = 0; i < chpp; i++)                                              \n"
"      {                                                                       \n"
"          initial_p[i] = sp_p[i];                                             \n"
"          initial_m[i] = sp_m[i];                                             \n"
"      }                                                                       \n"
"      for (row = 0; row < height; row++)                                      \n"
"      {                                                                       \n"
"        gfloat *vpptr, *vmptr;                                                \n"
"        terms = (row < 4) ? row : 4; //Magic constant 4 due to 4th order filter\n"
"        for (b = 0; b < 3; b++)      //RGB channels. Skip Alpha.              \n"
"        {                                                                     \n"
"          vpptr = vp + b; vmptr = vm + b;                                     \n"
"                                                                              \n"
"          for (i = 0; i <= terms; i++)                                        \n"
"          {                                                                   \n"
"            *vpptr += n_p[i] * sp_p[(-i * chpp) + b] -                        \n"
"                      d_p[i] * vp[(-i * chpp) + b];                           \n"
"            *vmptr += n_m[i] * sp_m[(i * chpp) + b] -                         \n"
"                      d_m[i] * vm[(i * chpp) + b];                            \n"
"          }                                                                   \n"
"          for (j = i; j <= 4; j++)                                            \n"
"              {                                                               \n"
"                *vpptr += (n_p[j] - bd_p[j]) * initial_p[b];                  \n"
"                *vmptr += (n_m[j] - bd_m[j]) * initial_m[b];                  \n"
"              }                                                               \n"
"        }                                                                     \n"
"        //Copy Alpha Straight                                                 \n"
"        vp[aidx] = sp_p[aidx];                                                \n"
"        vm[aidx] = sp_m[aidx];                                                \n"
"                                                                              \n"
"                                                                              \n"
"        sp_p += chpp;                                                         \n"
"        sp_m -= chpp;                                                         \n"
"        vp   += chpp;                                                         \n"
"        vm   -= chpp;                                                         \n"
"        }                                                                     \n"
"                                                                              \n"
"      transfer_pixels (val_p, val_m, dest, chpp, height);                     \n"
"      gegl_buffer_set (out_buf, &cur_roi, 0,rgba32,dest,GEGL_AUTO_ROWSTRIDE); \n"
"                                                                              \n"
"    }                                                                         \n"
"                                                                              \n"
"    for (row=0; row<height; row++)                                            \n"
"    {                                                                         \n"
"        memset (val_p, 0.0, width * bpp);                                     \n"
"        memset (val_m, 0.0, width * bpp);                                     \n"
"        //Acquire an entire row                                               \n"
"        gegl_rectangle_set (&cur_roi, x0, y0+row, width, 1);                  \n"
"        gegl_buffer_get (in_buf, &cur_roi, GEGL_SCALE_NONE, rgba32, src, GEGL_AUTO_ROWSTRIDE, GEGL_ABYSS_NONE);\n"
"        gegl_buffer_get (out_buf, &cur_roi, GEGL_SCALE_NONE, rgba32, vert, GEGL_AUTO_ROWSTRIDE, GEGL_ABYSS_NONE);\n"
"                                                                              \n"
"        sp_p = src;                                                           \n"
"        sp_m = src + (width - 1) * chpp;                                      \n"
"        vp = val_p;                                                           \n"
"        vm = val_m + (width - 1) * chpp;                                      \n"
"                                                                              \n"
"        /*  Set up the first vals  */                                         \n"
"        for (i = 0; i < chpp; i++)                                            \n"
"        {                                                                     \n"
"          initial_p[i] = sp_p[i];                                             \n"
"          initial_m[i] = sp_m[i];                                             \n"
"        }                                                                     \n"
"                                                                              \n"
"        for (col = 0; col < width; col++)                                     \n"
"        {                                                                     \n"
"          gfloat *vpptr, *vmptr;                                              \n"
"                                                                              \n"
"          terms = (col < 4) ? col : 4;                                        \n"
"                                                                              \n"
"          for (b = 0; b < 3; b++) //RGB|A                                     \n"
"          {                                                                   \n"
"            vpptr = vp + b; vmptr = vm + b;                                   \n"
"                                                                              \n"
"            for (i = 0; i <= terms; i++)                                      \n"
"            {                                                                 \n"
"              *vpptr += n_p[i] * sp_p[(-i * chpp) + b] -                      \n"
"                        d_p[i] * vp[(-i * chpp) + b];                         \n"
"              *vmptr += n_m[i] * sp_m[(i * chpp) + b] -                       \n"
"                        d_m[i] * vm[(i * chpp) + b];                          \n"
"            }                                                                 \n"
"                                                                              \n"
"            for (j = i; j <= 4; j++)                                          \n"
"            {                                                                 \n"
"              *vpptr += (n_p[j] - bd_p[j]) * initial_p[b];                    \n"
"              *vmptr += (n_m[j] - bd_m[j]) * initial_m[b];                    \n"
"            }                                                                 \n"
"          }                                                                   \n"
"          vp[aidx] = sp_p[aidx];                                              \n"
"          vm[aidx] = sp_m[aidx];                                              \n"
"                                                                              \n"
"          sp_p += chpp;                                                       \n"
"          sp_m -= chpp;                                                       \n"
"          vp   += chpp;                                                       \n"
"          vm   -= chpp;                                                       \n"
"        }                                                                     \n"
"                                                                              \n"
"        transfer_pixels (val_p, val_m, dest, chpp, width);                    \n"
"        combine_to_gradient (dest, vert, chpp, width, amount);                \n"
"        gegl_buffer_set (out_buf, &cur_roi, 0, rgba32, dest, GEGL_AUTO_ROWSTRIDE);\n"
"    }                                                                         \n"
"                                                                              \n"
"  /*  free up buffers  */                                                     \n"
"  g_free (val_p);                                                             \n"
"  g_free (val_m);                                                             \n"
"  g_free (src);                                                               \n"
"  g_free (vert);                                                              \n"
"  g_free (dest);                                                              \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/*  This function combines the causal and anticausal segments                 \n"
"    It also prevents each pixel from clippping.                               \n"
"    TODO: Originally, the Alpha channel is included in all scaling operatings, and doubled on the transfer operation.\n"
"    It appears that transparency is never preserved.                          \n"
"    I am unsure if this is intended operation, and it should likely be fixed in the speed update.\n"
"    */                                                                        \n"
"static void                                                                   \n"
"transfer_pixels (gfloat *src1,                                                \n"
"                 gfloat *src2,                                                \n"
"                 gfloat *dest,                                                \n"
"                 gint    chpp,                                                \n"
"                 gint    width)                                               \n"
"{                                                                             \n"
"  gint    b;                                                                  \n"
"  gint    bend = chpp * width;                                                \n"
"  gdouble sum;                                                                \n"
"                                                                              \n"
"  for (b = 0; b < bend; b++)                                                  \n"
"    {                                                                         \n"
"      sum = *src1++ + *src2++;                                                \n"
"                                                                              \n"
"      if (sum > NEON_MAXRGB)                                                  \n"
"        sum = NEON_MAXRGB;                                                    \n"
"      else if (sum < NEON_MINRGB)                                             \n"
"        sum = NEON_MINRGB;                                                    \n"
"                                                                              \n"
"      *dest++ = (gdouble) sum;                                                \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"combine_to_gradient (gfloat *dest,                                            \n"
"                     gfloat *src2,                                            \n"
"                     gint    bytes,                                           \n"
"                     gint    width,                                           \n"
"                     gdouble amount)                                          \n"
"{                                                                             \n"
"  gint    b;                                                                  \n"
"  gint    bend = bytes * width;                                               \n"
"  gdouble h, v;                                                               \n"
"  gdouble sum;                                                                \n"
"  //9.0 is a magic constant. I am not sure why it is 9.0.                     \n"
"  gdouble scale = (1.0 + 9.0 * amount);                                       \n"
"                                                                              \n"
"  for (b = 0; b < bend; b++)                                                  \n"
"    {                                                                         \n"
"                                                                              \n"
"      /* scale result */                                                      \n"
"      h = *src2++;                                                            \n"
"      v = *dest;                                                              \n"
"                                                                              \n"
"      sum = sqrt (h*h + v*v) * scale;                                         \n"
"                                                                              \n"
"      if (sum > NEON_MAXRGB)                                                  \n"
"        sum = NEON_MAXRGB;                                                    \n"
"      else if (sum < NEON_MINRGB)                                             \n"
"        sum = NEON_MINRGB;                                                    \n"
"                                                                              \n"
"      *dest++ = (gdouble) sum;                                                \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"find_constants (gdouble n_p[],                                                \n"
"                gdouble n_m[],                                                \n"
"                gdouble d_p[],                                                \n"
"                gdouble d_m[],                                                \n"
"                gdouble bd_p[],                                               \n"
"                gdouble bd_m[],                                               \n"
"                gdouble std_dev)                                              \n"
"{                                                                             \n"
"  gdouble a0, a1, b0, b1, c0, c1, w0, w1;                                     \n"
"  gdouble w0n, w1n, cos0, cos1, sin0, sin1, b0n, b1n;                         \n"
"                                                                              \n"
"  /* Coefficients which minimize difference between the                       \n"
"     approximating function and the Gaussian derivative.                      \n"
"     Calculated via an offline optimization process.                          \n"
"     See Deriche (1993).                                                      \n"
"  */                                                                          \n"
"                                                                              \n"
"  a0 =  0.6472;                                                               \n"
"  a1 =  4.531;                                                                \n"
"  b0 =  1.527;                                                                \n"
"  b1 =  1.516;                                                                \n"
"  c0 = -0.6494;                                                               \n"
"  c1 = -0.9557;                                                               \n"
"  w0 =  0.6719;                                                               \n"
"  w1 =  2.072;                                                                \n"
"                                                                              \n"
"  w0n  = w0 / std_dev;                                                        \n"
"  w1n  = w1 / std_dev;                                                        \n"
"  cos0 = cos (w0n);                                                           \n"
"  cos1 = cos (w1n);                                                           \n"
"  sin0 = sin (w0n);                                                           \n"
"  sin1 = sin (w1n);                                                           \n"
"  b0n  = b0 / std_dev;                                                        \n"
"  b1n  = b1 / std_dev;                                                        \n"
"                                                                              \n"
"  n_p[4] = 0.0;                                                               \n"
"  n_p[3] = exp (-b1n - 2 * b0n) * (c1 * sin1 - cos1 * c0) + exp (-b0n - 2 * b1n) * (a1 * sin0 - cos0 * a0);\n"
"  n_p[2] = 2 * exp (-b0n - b1n) * ((a0 + c0) * cos1 * cos0 - cos1 * a1 * sin0 - cos0 * c1 * sin1) + c0 * exp (-2 * b0n) + a0 * exp (-2 * b1n);\n"
"  n_p[1] = exp (-b1n) * (c1 * sin1 - (c0 + 2 * a0) * cos1) + exp (-b0n) * (a1 * sin0 - (2 * c0 + a0) * cos0);\n"
"  n_p[0] = a0 + c0;                                                           \n"
"                                                                              \n"
"  d_p[4] = exp (-2 * b0n - 2 * b1n);                                          \n"
"  d_p[3] = -2 * cos0 * exp (-b0n - 2 * b1n) - 2 * cos1 * exp (-b1n - 2 * b0n);\n"
"  d_p[2] = 4 * cos1 * cos0 * exp (-b0n - b1n) + exp (-2 * b1n) + exp (-2 * b0n);\n"
"  d_p[1] = -2 * exp (-b1n) * cos1 - 2 * exp (-b0n) * cos0;                    \n"
"  d_p[0] = 0.0;                                                               \n"
"                                                                              \n"
"  n_m[4] = d_p[4] * n_p[0] - n_p[4];                                          \n"
"  n_m[3] = d_p[3] * n_p[0] - n_p[3];                                          \n"
"  n_m[2] = d_p[2] * n_p[0] - n_p[2];                                          \n"
"  n_m[1] = d_p[1] * n_p[0] - n_p[1];                                          \n"
"  n_m[0] = 0.0;                                                               \n"
"                                                                              \n"
"  d_m[4] = d_p[4];                                                            \n"
"  d_m[3] = d_p[3];                                                            \n"
"  d_m[2] = d_p[2];                                                            \n"
"  d_m[1] = d_p[1];                                                            \n"
"  d_m[0] = d_p[0];                                                            \n"
"                                                                              \n"
"  {                                                                           \n"
"  // Back-calculate values for edges                                          \n"
"    gint    i;                                                                \n"
"    gdouble sum_n_p, sum_n_m, sum_d;                                          \n"
"    gdouble a, b;                                                             \n"
"                                                                              \n"
"    sum_n_p = 0.0;                                                            \n"
"    sum_n_m = 0.0;                                                            \n"
"    sum_d   = 0.0;                                                            \n"
"                                                                              \n"
"                                                                              \n"
"    for (i = 0; i <= 4; i++)                                                  \n"
"      {                                                                       \n"
"        sum_n_p += n_p[i];                                                    \n"
"        sum_n_m += n_m[i];                                                    \n"
"        sum_d   += d_p[i];                                                    \n"
"      }                                                                       \n"
"                                                                              \n"
"    a = sum_n_p / (1 + sum_d);                                                \n"
"    b = sum_n_m / (1 + sum_d);                                                \n"
"                                                                              \n"
"    for (i = 0; i <= 4; i++)                                                  \n"
"      {                                                                       \n"
"        bd_p[i] = d_p[i] * a;                                                 \n"
"        bd_m[i] = d_m[i] * b;                                                 \n"
"      }                                                                       \n"
"  }                                                                           \n"
"}                                                                             \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"neon_get_bounding_box (GeglOperation *self)                                   \n"
"{                                                                             \n"
"  GeglRectangle  result = { 0, 0, 0, 0 };                                     \n"
"  GeglRectangle *in_rect;                                                     \n"
"                                                                              \n"
"  in_rect = gegl_operation_source_get_bounding_box (self, \"input\");         \n"
"  if (in_rect)                                                                \n"
"    {                                                                         \n"
"      result = *in_rect;                                                      \n"
"    }                                                                         \n"
"  return result;                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"neon_get_required_for_output (GeglOperation        *operation,                \n"
"                         const gchar         *input_pad,                      \n"
"                         const GeglRectangle *roi)                            \n"
"{                                                                             \n"
"                                                                              \n"
"  GeglRectangle result = *gegl_operation_source_get_bounding_box (operation, \"input\");\n"
"                                                                              \n"
"  /* Don't request an infinite plane */                                       \n"
"  if (gegl_rectangle_is_infinite_plane (&result))                             \n"
"    return *roi;                                                              \n"
"                                                                              \n"
"  return result;                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"neon_get_cached_region (GeglOperation        *operation,                      \n"
"                                  const GeglRectangle *roi)                   \n"
"{                                                                             \n"
"  GeglRectangle result = *gegl_operation_source_get_bounding_box (operation, \"input\");\n"
"                                                                              \n"
"  if (gegl_rectangle_is_infinite_plane (&result))                             \n"
"    return *roi;                                                              \n"
"                                                                              \n"
"  return result;                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* Pass-through when trying to perform on an infinite plane                   \n"
" */                                                                           \n"
"static gboolean                                                               \n"
"operation_process (GeglOperation        *operation,                           \n"
"                   GeglOperationContext *context,                             \n"
"                   const gchar          *output_prop,                         \n"
"                   const GeglRectangle  *result,                              \n"
"                   gint                  level)                               \n"
"{                                                                             \n"
"  GeglOperationClass  *operation_class;                                       \n"
"                                                                              \n"
"  const GeglRectangle *in_rect =                                              \n"
"    gegl_operation_source_get_bounding_box (operation, \"input\");            \n"
"                                                                              \n"
"  operation_class = GEGL_OPERATION_CLASS (gegl_op_parent_class);              \n"
"                                                                              \n"
"  if (in_rect && gegl_rectangle_is_infinite_plane (in_rect))                  \n"
"    {                                                                         \n"
"      gpointer in = gegl_operation_context_get_object (context, \"input\");   \n"
"      gegl_operation_context_take_object (context, \"output\",                \n"
"                                          g_object_ref (G_OBJECT (in)));      \n"
"      return TRUE;                                                            \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* chain up, which will create the needed buffers for our actual            \n"
"   * process function                                                         \n"
"   */                                                                         \n"
"  return operation_class->process (operation, context, output_prop, result,   \n"
"                                   gegl_operation_context_get_level (context));\n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"gegl_op_class_init (GeglOpClass *klass)                                       \n"
"{                                                                             \n"
"                                                                              \n"
"  GeglOperationClass       *operation_class;                                  \n"
"  GeglOperationFilterClass *filter_class;                                     \n"
"                                                                              \n"
"  operation_class = GEGL_OPERATION_CLASS (klass);                             \n"
"  filter_class    = GEGL_OPERATION_FILTER_CLASS (klass);                      \n"
"                                                                              \n"
"  /* Override methods of inherited class */                                   \n"
"                                                                              \n"
"  filter_class->process                    = neon_process;                    \n"
"                                                                              \n"
"                                                                              \n"
"  operation_class->prepare                 = neon_prepare;                    \n"
"  operation_class->process                 = operation_process;               \n"
"  operation_class->get_bounding_box        = neon_get_bounding_box;           \n"
"  operation_class->get_required_for_output = neon_get_required_for_output;    \n"
"  operation_class->get_cached_region       = neon_get_cached_region;          \n"
"  operation_class->opencl_support          = 0;                               \n"
"  operation_class->threaded                = 0; //Due to IIR Implementation (smear-carry), we require single-process, linear operation.\n"
"  operation_class->want_in_place           = 0; //IIR Causes buffer build-up. \n"
"  operation_class->no_cache                = 1; //IIR Causes buffer build-up. \n"
"                                                                              \n"
"  gegl_operation_class_set_keys (operation_class,                             \n"
"    \"name\",       \"gegl:edge-neon\",                                       \n"
"    \"title\",      _(\"Neon Edge Detection\"),                               \n"
"    \"categories\", \"edge-detect\",                                          \n"
"    \"description\",                                                          \n"
"        _(\"Performs edge detection using a Gaussian derivative method\"),    \n"
"        NULL);                                                                \n"
"                                                                              \n"
"                                                                              \n"
"                                                                              \n"
"  /*                                                                          \n"
"  gchar                    *composition = \"<?xml version='1.0' encoding='UTF-8'?>\"\n"
"    \"<gegl>\"                                                                \n"
"    \"<node operation='gegl:edge-neon'>\"                                     \n"
"    \"  <params>\"                                                            \n"
"    \"    <param name='radius'>2</param>\"                                    \n"
"    \"    <param name='amount'>1</param>\"                                    \n"
"    \"  </params>\"                                                           \n"
"    \"</node>\"                                                               \n"
"    \"<node operation='gegl:load'>\"                                          \n"
"    \"  <params>\"                                                            \n"
"    \"    <param name='path'>standard-input.png</param>\"                     \n"
"    \"  </params>\"                                                           \n"
"    \"</node>\"                                                               \n"
"    \"</gegl>\";                                                              \n"
"                                                                              \n"
"  */                                                                          \n"
"}                                                                             \n"
"                                                                              \n"
"#endif                                                                        \n"
;
