/*
 *  Main authors:
 *     Christian Schulte <schulte@gecode.org>
 *
 *  Copyright:
 *     Christian Schulte, 2004
 *
 *  Last modified:
 *     $Date: 2005-08-28 14:06:57 +0200 (Sun, 28 Aug 2005) $ by $Author: schulte $
 *     $Revision: 2243 $
 *
 *  This file is part of Gecode, the generic constraint
 *  development environment:
 *     http://www.gecode.org
 *
 *  See the file "LICENSE" for information on usage and
 *  redistribution of this file, and for a
 *     DISCLAIMER OF ALL WARRANTIES.
 *
 */



#include "search/bab-reco.hh"

namespace Gecode { namespace Search {

  /*
   * The invarinat maintained by the engine is:
   *   For all nodes stored at a depth less than mark, there
   *   is no guarantee of betterness. For those above the mark,
   *   betterness is guaranteed.
   *
   * The engine maintains the path on the stack for the current
   * node to be explored.
   *
   */

  BabReCoEngine::~BabReCoEngine(void) {
    ds.reset();
    delete best;
    delete cur;
  }

  size_t
  BabReCoEngine::stacksize(void) const {
    return ds.size();
  }

  bool
  BabReCoEngine::explore(Space*& s1, Space*& s2) {
    /*
     * Upon entry, cur can be either NULL (after a solution
     * has been returned) or set to a space that has been
     * requested to be constrained.
     * When a solution is found, the solution is returned in s1
     * and true is returned.
     * When a space is requested to be constrained, the space
     * to be constrained is returned in s1 and s2 refers to the
     * currently best solution. In this case false is returned.
     *
     */
    while (true) {
      if (cur == NULL) {
	if (!ds.next(*this)) {
	  s1 = NULL;
	  return true;
	}
	cur = ds.recompute(d,*this);
	FullStatistics::current(cur);
	if (ds.entries() <= mark) {
	  // Next space needs to be constrained
	  mark = ds.entries();
	  d  = 0; // Force copy!
	  s1 = cur;
	  s2 = best;
	  return false;
	}
      }
      unsigned int alt;
      switch (cur->status(alt,propagate)) {
      case SS_FAILED:
	fail++;
	delete cur;
	cur = NULL;
	FullStatistics::current(NULL);
	break;
      case SS_SOLVED:
	delete best;
	best = cur;
	mark = ds.entries();
	s1 = best->clone();
	clone++;
	cur = NULL;
	FullStatistics::current(NULL);
	return true;
      case SS_BRANCH:
	{
	  Space* c;
	  if ((d == 0) || (d >= c_d)) {
	    c = cur->clone();
	    clone++;
	    d = 1;
	  } else {
	    c = NULL;
	    if (alt > 1)
	      d++;
	  }
	  BranchingDesc* desc = ds.push(cur,c,alt);
	  FullStatistics::push(c,desc);
	  cur->commit(0,desc,propagate);
	  commit++;
	  break;
	}
      }
    }
    return true;
  }

}}
	
// STATISTICS: search-any
