/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "gdis.h"
#include "coords.h"
#include "matrix.h"
#include "numeric.h"
#include "opengl.h"
#include <GL/gl.h>

/* externals */
extern struct sysenv_pak sysenv;

/***************************/
/* free an array of points */
/***************************/
void gl_free_points(struct point_pak *p)
{
g_return_if_fail(p->num_points != 0);

if (p->num_points)
  {
  g_free(p->x[0]);
  g_free(p->x[1]);
  g_free(p->x[2]);
  p->num_points = 0;
  }
}

/***********************************/
/* more efficient point generation */
/***********************************/
#define DEBUG_GLO_INIT_SPHERE 0
void gl_init_sphere(struct point_pak *p)
{
gint i, j, n;
gdouble layers, divisions, alpha, beta, phi, theta, sp, cp, st, ct;

/* number of layers - azimuthal sweep */
layers = sysenv.render.sphere_quality+1.0;
alpha = 0.5*PI/layers;

/* alloc */
n=0;
for (i=0 ; i<(gint) 1.0+layers ; i++)
  n += i;
n *= 6;
n++;

p->x[0] = g_malloc(n * sizeof(gdouble));
p->x[1] = g_malloc(n * sizeof(gdouble));
p->x[2] = g_malloc(n * sizeof(gdouble));
p->num_points = n;
p->type = sysenv.render.sphere_quality;

*(p->x[0]+0) = 0.0;
*(p->x[1]+0) = 0.0;
*(p->x[2]+0) = -1.0;

n=1;

for (i=1 ; i<(gint) 1.0+layers ; i++)
  {
#if DEBUG_GLO_INIT_SPHERE
printf("[%d/%d]\n", i, (gint) layers);
#endif

  phi = i*alpha;
  sp = tbl_sin(phi);
  cp = tbl_cos(phi);

/* divisions at this level - hexagonal sweep */
  divisions = i*6.0;
  beta = 2.0*PI/divisions;
  for (j=0 ; j<(gint) divisions ; j++)
    {
    theta = j*beta;
    st = tbl_sin(theta);
    ct = tbl_cos(theta);
 
    *(p->x[0]+n) = ct*sp;
    *(p->x[1]+n) = st*sp;
    *(p->x[2]+n) = -cp;
    n++;

#if DEBUG_GLO_INIT_SPHERE
printf("%7.4f %7.4f %7.4f\n", ct*sp, st*sp, -cp);
#endif
    }
  }
g_assert(p->num_points == n);
}

/*********************************/
/* more efficient sphere drawing */
/*********************************/
#define DEBUG_GLO_DRAW_SPHERE 0
void gl_draw_sphere(struct point_pak *p, gdouble *x, gdouble rad)
{
gint i, j, div, m, n, mstart, nstart;
gdouble vec[3];

g_return_if_fail(p->num_points != 0);

/* layer loop (phi) */
n=1;
mstart = 0;
for (i=1 ; i<p->type+2 ; i++)
  {
  nstart = n;
  div = i*6.0;
  m = mstart;

#if DEBUG_GLO_DRAW_SPHERE
printf("%d : [m,n start = %d,%d]\n", i, mstart, nstart);
#endif

/* strip loop (theta) */
  glBegin(GL_TRIANGLE_STRIP);
  for (j=0 ; j<div ; j++)
    {
#if DEBUG_GLO_DRAW_SPHERE
printf("> : m,n = %d,%d\n", m, n);
#endif

/* lower */
    vec[0] = *(p->x[0]+n);
    vec[1] = *(p->x[1]+n);
    vec[2] = *(p->x[2]+n);
    glNormal3dv(vec);
    VEC3MUL(vec, rad);
    ARR3ADD(vec, x);
    glVertex3dv(vec);
    n++;

/* upper */
    vec[0] = *(p->x[0]+m);
    vec[1] = *(p->x[1]+m);
    vec[2] = *(p->x[2]+m);
    glNormal3dv(vec);
    VEC3MUL(vec, rad);
    ARR3ADD(vec, x);
    glVertex3dv(vec);

/* staggered doubling */
    m++;
    if ((j+1) % i == 0)
      m--;
/* cycle around */
    if (m >= nstart)
      {
      m -= nstart;
      m += mstart;
      }
    }

/* finish off the triangle layer */
  vec[0] = *(p->x[0]+nstart);
  vec[1] = *(p->x[1]+nstart);
  vec[2] = *(p->x[2]+nstart);
  glNormal3dv(vec);
  VEC3MUL(vec, rad);
  ARR3ADD(vec, x);
  glVertex3dv(vec);

  vec[0] = *(p->x[0]+mstart);
  vec[1] = *(p->x[1]+mstart);
  vec[2] = *(p->x[2]+mstart);
  glNormal3dv(vec);
  VEC3MUL(vec, rad);
  ARR3ADD(vec, x);
  glVertex3dv(vec);

/* finished current layer */
  glEnd();

  mstart = nstart;
  }
#if DEBUG_GLO_DRAW_SPHERE
printf("n : %d\n", n);
#endif
}

/***************************/
/* create digitized circle */
/***************************/
void gl_init_circle(struct point_pak *circle, gint segments)
{
gint i;
gdouble da, theta;
gdouble vec[3];

g_return_if_fail(segments != 0);

/* allocate for digitized circle */
circle->x[0] = g_malloc((segments+1) * sizeof(gdouble));
circle->x[1] = g_malloc((segments+1) * sizeof(gdouble));
circle->x[2] = g_malloc((segments+1) * sizeof(gdouble));
circle->num_points = segments + 1;

/* first point (theta=0) */
*(circle->x[0]) = 1.0;
*(circle->x[1]) = 0.0;
*(circle->x[2]) = 0.0;
/* sweep */
vec[2] = 0.0;
da = 2.0 * PI / (gdouble) segments;
for (i=1 ; i<segments ; i++) 
  {
  theta = (gdouble) i * da;
/* store point on rim (doubles as the normal) */
  vec[0] = tbl_cos(theta);
  vec[1] = tbl_sin(theta);
  normalize(vec, 2);
  *(circle->x[0]+i) = vec[0];
  *(circle->x[1]+i) = vec[1];
  *(circle->x[2]+i) = vec[2];
  }
/* duplicate first point, so the circle is closed */
*(circle->x[0]+segments) = 1.0;
*(circle->x[1]+segments) = 0.0;
*(circle->x[2]+segments) = 0.0;
}

/*******************************/
/* draw a pre-digitized circle */
/*******************************/
void gl_draw_circle(struct point_pak *circle, gdouble *x, gdouble rad)
{
gint i;
gdouble vec[3];

g_return_if_fail(circle->num_points != 0);

glBegin(GL_POLYGON);
for (i=0 ; i<circle->num_points ; i++) 
  {
/* get points */
  vec[0] = *(circle->x[0]+i);
  vec[1] = *(circle->x[1]+i);
  vec[2] = *(circle->x[2]+i);
  VEC3MUL(vec, rad);
  ARR3ADD(vec, x);
  glVertex3dv(vec);
  }
glEnd();
}

/*****************************/
/* draw a pre-digitized ring */
/*****************************/
void gl_draw_ring(struct point_pak *circle, gdouble *x, gdouble rad1, gdouble rad2)
{
gint i;
gdouble vec[3];

g_return_if_fail(circle->num_points != 0);

glBegin(GL_QUAD_STRIP);
for (i=0 ; i<circle->num_points ; i++) 
  {
/* inner point */
  vec[0] = *(circle->x[0]+i);
  vec[1] = *(circle->x[1]+i);
  vec[2] = *(circle->x[2]+i);
  VEC3MUL(vec, rad2);
  ARR3ADD(vec, x);
  glVertex3dv(vec);

/* outer point */
  vec[0] = *(circle->x[0]+i);
  vec[1] = *(circle->x[1]+i);
  vec[2] = *(circle->x[2]+i);
  VEC3MUL(vec, rad1);
  ARR3ADD(vec, x);
  glVertex3dv(vec);
  }
glEnd();
}

/*********************************************/
/* draw cylinder from a pre-digitized circle */
/*********************************************/
#define DEBUG_GL_DRAW_CYLINDER 0
void gl_draw_cylinder(struct point_pak *circle, gdouble *pa, gdouble *pb, gdouble rad)
{
gint i, flag=0;
gdouble len1, len2;
gdouble p1[3], p2[3], p[3], vec[3], mat[9], mop[9];

g_return_if_fail(circle->num_points != 0);

/* force ordering (for correct quad normals) */
if (pa[2] > pb[2])
  {
  ARR3SET(p1, pb);
  ARR3SET(p2, pa);
  }
else
  {
  ARR3SET(p1, pa);
  ARR3SET(p2, pb);
  }

/* get vector from point 1 to 2 */
ARR3SET(vec, p2);
ARR3SUB(vec, p1);

/* normalize x */
len1 = VEC3MAG(vec);
vec[0] /= len1;
/* normalize y,z (projection) */
len2 = sqrt(vec[1]*vec[1] + vec[2]*vec[2]);
if (len2 > FRACTION_TOLERANCE)
  {
  vec[1] /= len2;
  vec[2] /= len2;
  len2 /= len1;
  }
else
  flag++;

#if DEBUG_GL_DRAW_CYLINDER
printf("--------------------\n");
P3VEC("input: ", vec);
#endif

/* z axis coincidence */
VEC3SET(&mat[0], len2, 0.0, vec[0]);
VEC3SET(&mat[3], 0.0, 1.0, 0.0);
VEC3SET(&mat[6], -vec[0], 0.0, len2);
/* yz plane coincidence (if not already satisfied) */
if (!flag)
  {
  VEC3SET(&mop[0], 1.0, 0.0, 0.0);
  VEC3SET(&mop[3], 0.0, vec[2], vec[1]);
  VEC3SET(&mop[6], 0.0, -vec[1], vec[2]);
/* construct transformation matrix */
  matmat(mop, mat);
  }

/* check */
#if DEBUG_GL_DRAW_CYLINDER
P3MAT("matrix: ", mat);
VEC3SET(vec, 0.0, 0.0, 1.0);
P3VEC("before: ", vec);
vecmat(mat, vec);
P3VEC("after: ", vec);
printf("--------------------\n");
#endif

glBegin(GL_QUAD_STRIP);
for (i=0 ; i<circle->num_points ; i++) 
  {
/* get point on rim (doubles as the normal) */
  vec[0] = *(circle->x[0]+i);
  vec[1] = *(circle->x[1]+i);
  vec[2] = *(circle->x[2]+i);
/* rotate the digitized circle */
  vecmat(mat, vec);
/* submit the normal */
  glNormal3dv(vec);
/* get points at required radius */
  VEC3MUL(vec, rad);
/* point on disk 2 */
  ARR3SET(p, p2);
  ARR3ADD(p, vec);
  glVertex3dv(p);
/* point on disk 1 */
  ARR3SET(p, p1);
  ARR3ADD(p, vec);
  glVertex3dv(p);
  }
glEnd();
}

/******************************************/
/* draw a cylinder between points v1 & v2 */
/******************************************/
/* NB: used for drawing vectors */
void draw_cylinder(gdouble *va, gdouble *vb, gdouble rad, guint segments)
{
gint i;
gdouble theta, st, ct;
gdouble v1[3], v2[3], v[3], v12[3], p[3], q[3];

g_return_if_fail(segments != 0);

/* force ordering of v1 & v2 to get correct quad normals */
if (va[2] > vb[2])
  {
  ARR3SET(v1, vb);
  ARR3SET(v2, va);
  }
else
  {
  ARR3SET(v1, va);
  ARR3SET(v2, vb);
  }

/* vector from v1 to v2 */
ARR3SET(v12, v2);
ARR3SUB(v12, v1);

/* create vectors p and q, co-planar with the cylinder's cross-sectional disk */
ARR3SET(p, v12);
if (p[0] == 0.0 && p[1] == 0.0)
  p[0] += 1.0;
else
  p[2] += 1.0;
crossprod(q, p, v12);
crossprod(p, v12, q);
/* do the normalization outside the segment loop */
normalize(p, 3);
normalize(q, 3);

/* build the cylinder from rectangular segments */
/* TODO - see if vertex arrays give any speedup here */
glBegin(GL_QUAD_STRIP);
for (i=0 ; i<=segments ; i++) 
  {
/* sweep out a circle */
  theta = (gdouble) i * 2.0 * PI / (gdouble) segments;
  st = tbl_sin(theta);
  ct = tbl_cos(theta);
/* construct normal */
  v12[0] = ct * p[0] + st * q[0];
  v12[1] = ct * p[1] + st * q[1];
  v12[2] = ct * p[2] + st * q[2];
/* set the normal for the two subseqent points */
  glNormal3dv(v12);
/* get the vector from centre to rim */
  VEC3MUL(v12, rad);
/* point on disk 1 */
  ARR3SET(v, v2);
  ARR3ADD(v, v12);
  glVertex3dv(v);
/* point on disk 2 */
  ARR3SET(v, v1);
  ARR3ADD(v, v12);
  glVertex3dv(v);
  }
glEnd();
}

/*************************************/
/* draw a cone defined by two points */
/*************************************/
void draw_cone(gdouble *va, gdouble *vb, gdouble base, gint segments)
{
gint i;
gdouble theta, st, ct;
gdouble v1[3], v2[3], v[3];
gdouble p[3], q[3], v12[3];

g_return_if_fail(segments != 0);

/* force ordering of v1 & v2 to get correct quad normals */
if (va[2] > vb[2])
  {
  ARR3SET(v1, vb);
  ARR3SET(v2, va);
  }
else
  {
  ARR3SET(v1, va);
  ARR3SET(v2, vb);
  }

/* vector from v1 to v2 */
ARR3SET(v12, v2);
ARR3SUB(v12, v1);

/* create vectors p and q, co-planar with the cylinder's cross-sectional disk */
ARR3SET(p, v12);
if (p[0] == 0.0 && p[1] == 0.0)
  p[0] += 1.0;
else
  p[2] += 1.0;
crossprod(q, p, v12);
crossprod(p, v12, q);

normalize(p, 3);
normalize(q, 3);

/* build the cone from triangular segments */
glBegin(GL_TRIANGLE_FAN);

/* tip */
glVertex3dv(vb);

/* base */
for (i=0 ; i<=segments ; i++) 
  {
/* sweep out a circle */
  theta = (gdouble) i * 2.0 * PI / (gdouble) segments;
  st = tbl_sin(theta);
  ct = tbl_cos(theta);
/* construct normal */
  v12[0] = ct * p[0] + st * q[0];
  v12[1] = ct * p[1] + st * q[1];
  v12[2] = ct * p[2] + st * q[2];
  glNormal3dv(v12);
/* point 1 on disk 1 */
  ARR3SET(v, va);
  v[0] += base*v12[0];
  v[1] += base*v12[1];
  v[2] += base*v12[2];
  glVertex3dv(v);
  }
glEnd();
}

/*********************************************/
/* draw a vector cylinder between two points */
/*********************************************/
void draw_vector(gdouble *va, gdouble *vb, gdouble size)
{
gdouble vec[3];

draw_cylinder(va, vb, size, 5);

ARR3SET(vec, vb);
ARR3SUB(vec, va);
normalize(vec, 3);
VEC3MUL(vec, 4.0*size);
ARR3ADD(vec, vb);

draw_cone(vb, vec, 3*size, 9);
}

/*********************************************************/
/* draw a circular arc from p1 to p2, with p0 the centre */
/*********************************************************/
void draw_arc(gdouble *p0, gdouble *p1, gdouble *p2)
{
gint i, num_points;
gdouble angle, theta, st, ct;
gdouble v[3], v1[3], v2[3], len1, len2;

/* get vector to arc start */
ARR3SET(v1, p1);
ARR3SUB(v1, p0);

/* get vector to arc end */
ARR3SET(v2, p2);
ARR3SUB(v2, p0);

/* scale back a bit */
VEC3MUL(v1, 0.5);
VEC3MUL(v2, 0.5);

/* get lengths */
len1 = VEC3MAG(v1);
len2 = VEC3MAG(v2);

/* make them both equal to the smaller */
if (len2 > len1)
  {
  VEC3MUL(v2, len1/len2);
  }
else
  {
  VEC3MUL(v1, len2/len1);
  }

angle = via(v1, v2, 3);

num_points = 32;

glBegin(GL_LINE_STRIP);
/* sweep from start to end */
for (i=0 ; i<=num_points ; i++)
  {
/* sweep out a circle */
  theta = (gdouble) i * 0.5 * PI / (gdouble) num_points;
  st = tbl_sin(theta);
  ct = tbl_cos(theta);
/* construct normal */
  ARR3SET(v, p0);
  v[0] += ct * v1[0] + st * v2[0];
  v[1] += ct * v1[1] + st * v2[1];
  v[2] += ct * v1[2] + st * v2[2];
  glVertex3dv(v);
  }
glEnd();
}

