/*
Copyright (C) 2000 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

/* irix */
#define _BSD_SIGNALS 1

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <signal.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>


#include "gdis.h"
#include "task.h"
#include "file.h"
#include "parse.h"
#include "gtkshorts.h"
#include "interface.h"
#include "dialog.h"

/* top level data structure */
extern struct sysenv_pak sysenv;

#ifndef __WIN32
#include <sys/wait.h>
#endif

/* task manager globals */
GtkWidget *task_list=NULL;
GtkWidget *task_label1, *task_label2, *task_label3;
gint selected_task = -1;
gint task_info_pid = 0;
gint task_thread_pid = -1;
GArray *psarray = NULL;
GtkListStore *task_list_ls=NULL;
GtkWidget *task_list_tv=NULL;
char *strptime(const char *, const char *, struct tm *);

/***************************************************/
/* submit a job to be processed by the task thread */
/***************************************************/
#define DEBUG_SUBMIT_TASK 0
void 
submit_task(gchar *label,
            gpointer func1, gpointer arg1,
            gpointer func2, gpointer arg2,
            gpointer model)
{
struct task_pak *task;

g_assert(func1 != NULL);

/* duplicate the task data */
task = g_malloc(sizeof(struct task_pak));

/* system data */
task->pid = task_thread_pid;
task->status = QUEUED;
task->time = NULL;
task->message = NULL;
task->pcpu = 0.0;
task->pmem = 0.0;
task->progress = 0.0;
task->locked_model = model;
if (model)
  ((struct model_pak *) model)->locked = TRUE;

/* user data */
task->label = g_strdup(label);
task->primary = func1;
task->cleanup = func2;
task->ptr1 = arg1;
task->ptr2 = arg2;

#if DEBUG_SUBMIT_TASK
printf("locking...\n");
#endif
/* we're modifying the task list, so protect it */
g_mutex_lock(sysenv.task_mutex);
#if DEBUG_SUBMIT_TASK
printf("queueing [%p] ...\n", task);
#endif
sysenv.task_list = g_slist_append(sysenv.task_list, task);
g_mutex_unlock(sysenv.task_mutex);
#if DEBUG_SUBMIT_TASK
printf("unlocking...\n");
#endif

/* wake the task thread */
g_cond_signal(sysenv.task_cond);
}

/********************************/
/* remove and reorder task list */
/********************************/
#define DEBUG_REMOVE_TASK 0
gint remove_task(struct task_pak *task)
{
g_assert(task != NULL);
g_assert(sysenv.task_list != NULL);

/* free task data */
if (task->label)
  g_free(task->label);
if (task->message)
  g_free(task->message);

#if DEBUG_REMOVE_TASK
printf("removing task: %p\n", task);
#endif

g_mutex_lock(sysenv.task_mutex);
sysenv.task_list = g_slist_remove(sysenv.task_list, task);
if (task->locked_model)
  ((struct model_pak *) task->locked_model)->locked = FALSE;
g_free(task);
g_mutex_unlock(sysenv.task_mutex);

#if DEBUG_REMOVE_TASK
printf("remaining: %d (%d)\n", g_slist_length(sysenv.tasks), sysenv.num_tasks);
#endif

return(0);
}

/****************************************/
/* process available jobs (as a thread) */
/****************************************/
#define DEBUG_TASK_THREAD 0
gpointer task_thread(gpointer ptr)
{
struct task_pak *task;

task_thread_pid = getpid();

#if DEBUG_TASK_THREAD
printf("Task thread pid = %d\n", task_thread_pid);
#endif

/* inf. loop */
for (;;)
  {
  g_mutex_lock(sysenv.task_mutex);

#if DEBUG_TASK_THREAD
printf("Blocking... \n");
#endif

/* block while there are no pending tasks */
/* NB: g_cond_wait() unlocks the mutex while sleeping */
/* and then locks it again when it wakes */ 
  while (!sysenv.task_list)
    g_cond_wait(sysenv.task_cond, sysenv.task_mutex);

/* get the task info & then allow the list to be altered */
  sysenv.running_tasks++;
  task = sysenv.task_list->data;
  g_mutex_unlock(sysenv.task_mutex);

#if DEBUG_TASK_THREAD
printf("Executing task: %p\n", task);
#endif

/* perform the task */
  task->status = RUNNING;
  task->primary(task->ptr1, task);

/* make thread safe for widget updating */
  gdk_threads_enter();

/* NEW - display any error/status message */
  if (task->message)
    show_text(ERROR, task->message);

/* post-primary task */
  if (task->cleanup)
    task->cleanup(task->ptr2);

/* outside main event loop - flush before releasing lock */
  gdk_flush();
  gdk_threads_leave();

/* update the task list */
  sysenv.running_tasks--;
  remove_task(task);

/* job completion notification */
  gdk_beep();
  }
}

/**************************************/
/* start task thread & init variables */
/**************************************/
#define DEBUG_INIT_TASK_THREAD 0
gint init_task_thread(void)
{
#ifdef G_THREADS_ENABLED
/* init GTK thread system */
g_thread_init(NULL);
gdk_threads_init();

/* conditional for waking up the task_thread() */
sysenv.task_cond = g_cond_new();
/* mutex protects simultaneous access of sysenv.task_list */
sysenv.task_mutex = g_mutex_new();
/* list of tasks to be run in task_thread() */
sysenv.task_list = NULL;

if (!g_thread_supported())
  {
  printf("Thread init failed.\n");
  return(1);
  }
else
  sysenv.task_thread = g_thread_create(task_thread, NULL, FALSE, NULL);

#if DEBUG_INIT_TASK_THREAD
printf("created task thread: %p\n", sysenv.task_thread);
#endif

#else
printf("gthreads not supported...\n");
#endif

return(0);
}

/**************************/
/* a simple sleep routine */
/**************************/
void delay(gint usecs)
{
#ifndef __WIN32
struct timespec req;

req.tv_sec = usecs / 1000000;
req.tv_nsec = 1000*(usecs - 1000000*req.tv_sec);

/* TODO - if terminated early - call again with remainder */
nanosleep(&req, NULL);
#endif
}

/*******************/
/* setup child i/o */
/*******************/
void task_io(gchar **argv)
{
gint i;

i=0;
while (argv[i])
  {
  printf("[%s] ", argv[i]);
  i++;
  }
}

/**************************************/
/* platform independant task spawning */
/**************************************/
#define DEBUG_TASK_SYNC 0
gint task_sync(const gchar *command) 
{
gint argc, pid, status;
gint si, so;
guint flags;
gchar **argv;
struct task_pak *task;
GError *error=NULL;

/* checks */
if (!command)
  return(1);

/* parse the command */
/* TODO - free argv??? */
if (!g_shell_parse_argv(command, &argc, &argv, &error))
  {
  printf("task_sync(): shell parse error.\n");
  printf(" > %s\n", error->message);
  return(2);
  }

#if DEBUG_TASK_SYNC
{
gint i;
for (i=0 ; i<=argc ; i++)
  printf("[%s]", argv[i]);
printf("\n");
}
#endif

/* don't reap - allows us to wait for the child's exit */
flags = G_SPAWN_DO_NOT_REAP_CHILD;

/* special case to cope with GULP ugliness */
if (find_char(command, '>', LAST))
  {
  gsize len;
  gchar *inp, *out, *buff;
  GIOChannel *gsi, *gso, *gfi, *gfo;

/* spawn flags */
  flags |= G_SPAWN_FILE_AND_ARGV_ZERO;
  flags |= G_SPAWN_LEAVE_DESCRIPTORS_OPEN;
/* run background task with file descriptors */
/* si - standard input, so - standard output */
  if (!g_spawn_async_with_pipes(sysenv.cwd, argv, NULL, flags, NULL, NULL, &pid,
                                &si, &so, NULL, &error))
    {
    printf("task_sync(): spawn failed.\n");
    printf(" > %s\n", error->message);
    return(4);
    }
/* give the current task the child's pid */
  g_mutex_lock(sysenv.task_mutex);
  task = g_slist_nth_data(sysenv.task_list, 0);
/* NB: some jobs (eg MS electrostatic calcs) are not formally a task */
  if (task)
    task->pid = pid;
  g_mutex_unlock(sysenv.task_mutex);
/* setup output chanels */
  gso = g_io_channel_unix_new(so);
  out = g_build_filename(sysenv.cwd, argv[4], NULL);
  gfo = g_io_channel_new_file(out, "w", &error);
  g_free(out);
/* setup input chanels */
  gsi = g_io_channel_unix_new(si);
  inp = g_build_filename(sysenv.cwd, argv[2], NULL);
  gfi = g_io_channel_new_file(inp, "r", &error);
  g_free(inp);
/* pipe the input file to the child's standard input chanel */
  g_io_channel_read_to_end(gfi, &buff, &len, &error);
  g_io_channel_write_chars(gsi, buff, len, NULL, &error);
  g_io_channel_shutdown(gsi, TRUE, &error);
  g_io_channel_shutdown(gfi, TRUE, &error);
  g_free(buff);
  close(si);
/* pipe the child's standard output chanel to the output file */
  g_io_channel_read_to_end(gso, &buff, &len, &error);
  g_io_channel_write_chars(gfo, buff, len, NULL, &error);
  g_io_channel_shutdown(gso, TRUE, &error);
  g_io_channel_shutdown(gfo, TRUE, &error);
  g_free(buff);
  close(so);
  }
else
  {
/* spawn flags */
  flags |= G_SPAWN_LEAVE_DESCRIPTORS_OPEN;
/* run normal shell task in the background */
  if (!g_spawn_async(sysenv.cwd, argv, NULL, flags, NULL, NULL, &pid, &error))
    {
    printf("task_sync(): spawn failed.\n");
    printf(" > %s\n", error->message);
    return(3);
    }
/* give the current task the child's pid */
  g_mutex_lock(sysenv.task_mutex);
  task = g_slist_nth_data(sysenv.task_list, 0);
/* NB: some jobs (eg MS electrostatic calcs) are not formally a task */
  if (task)
    task->pid = pid;
  g_mutex_unlock(sysenv.task_mutex);
/* cleanup */
  g_strfreev(argv);
/* wait for child to exit */
#ifdef __WIN32
/* TODO - WIN32 use WaitFor*() functions to do the same job */
/*
WaitForSingleObject(pid);
CloseHandle(pid);
*/
#else
  do 
    {
    if (waitpid(pid, &status, 0) == -1) 
      return(-1);
    else
      return(status);
    }
  while(1);
#endif
  }

/* cleanup */
g_strfreev(argv);
return(0);
}

#define EXIST(idx) ((idx) != -1)

gint find_pid(struct task_pak *curr_process, gint *pid)
{
if (curr_process->ppid == GPOINTER_TO_INT(pid))
  return (0);
else
  return(-1);
}

gint find_pid_index(gint pid)
{
gint i;

for (i = psarray->len - 1; i >= 0 && g_array_index(psarray, struct task_pak, i).pid != pid; i--);
  return(i);
}

#define DEBUG_ADD_FROM_TREE 0
void add_from_tree(struct task_pak *tdata, gint idx)
{
struct task_pak *process_ptr;
gint child;
div_t h_sec, sec, min;
gchar *h_sec_pos, *sec_pos, *min_pos, *hour_pos;

/* add in current process */  
process_ptr = &g_array_index(psarray, struct task_pak, idx);
tdata->pcpu += process_ptr->pcpu;
tdata->pmem += process_ptr->pmem;
/* parse the cpu time */
#if defined(__APPLE__) && defined(__MACH__) 
h_sec_pos = g_strrstr(process_ptr->time, ".") + 1;
sec_pos = h_sec_pos - 3;
min_pos = process_ptr->time;
hour_pos = NULL;
#else
hour_pos = process_ptr->time;
min_pos = hour_pos + 3;
sec_pos = min_pos + 3;
h_sec_pos = NULL;
#endif
tdata->h_sec += (gint) str_to_float(h_sec_pos);
h_sec = div(tdata->h_sec, 100);
tdata->h_sec = h_sec.rem;
tdata->sec =tdata->sec + h_sec.quot + (gint) str_to_float(sec_pos);
sec = div(tdata->sec, 60);
tdata->sec = sec.rem;
tdata->min = tdata->min + sec.quot + (gint) str_to_float(min_pos);
min = div(tdata->min, 60);
tdata->min = min.rem;
tdata->hour = tdata->hour + min.quot + (gint) str_to_float(hour_pos);
#if DEBUG_ADD_FROM_TREE
printf("%s hour=%d min=%d, sec=%d, hsec=%d\n", process_ptr->time, tdata->hour, tdata->min, tdata->sec, tdata->h_sec);
#endif

/* process children */
for (child = process_ptr->child; EXIST(child); child = g_array_index(psarray, struct task_pak, child).sister)
  add_from_tree(tdata, child);
}

/*****************************************************/
/* get the results of a ps on the requested process */
/*****************************************************/
#define DEBUG_CALC_TASK_INFO 0
void calc_task_info(struct task_pak *tdata)
{
gint num_tokens;
gint me, parent, sister;
gint found;
gchar line[LINELEN], *line_no_time, **buff, *cmd;
struct task_pak curr_process, *process_ptr;
struct tm start_tm;
time_t oldest_time;
FILE *fp;

/* dispose of task list if it exists */
if (psarray != NULL)
  g_array_free(psarray, TRUE);

/* initialise process record */
curr_process.parent = curr_process.child = curr_process.sister = -1;

/* setup ps command */

/*
 * SG's - 'ps -Ao "pid ppid pcpu vsz etime comm"
 * 			(unfortunately, no %mem - and vsz is absolute :-(
 */
#if DEBUG_CALC_TASK_INFO
printf("making ps command string\n");
#ifdef __sgi
cmd = g_strdup_printf("ps -Ao \"pid ppid pcpu vsz etime comm\"");
#else
cmd = g_strdup_printf("ps -o \"lstart pid ppid %%cpu %%mem time ucomm\"");
#endif
#else
#ifdef __sgi
cmd = g_strdup_printf("ps -Ao \"pid ppid pcpu vsz etime comm\"");
#else
cmd = g_strdup_printf("ps -axo \"lstart pid ppid %%cpu %%mem time ucomm\"");
#endif
#endif

/* run ps command */
fp = popen(cmd, "r");
g_free(cmd);
if (!fp)
  {
  show_text(ERROR, "unable to launch ps command\n");
  return;
  }

/* skip title line */
if (fgetline(fp, line))
  {
  show_text(ERROR, "unable to read first line of output from ps command\n");
  return;
  }
    
/* load data into array */
psarray = g_array_new(FALSE, FALSE, sizeof(struct task_pak));
while (!fgetline(fp, line))
  {
#if DEBUG_CALC_TASK_INFO
printf("%s", line);
#endif
/* extract what we want */
#ifdef __WIN32
/* TODO - win32 replacement? */
  line_no_time = NULL;
#else
/* first get start time */
  line_no_time = strptime(line, "%c", &start_tm);
#endif
  if (line_no_time == NULL)
    {
    show_text(ERROR, "file produced by ps not understood\n");
    return;
    }
  curr_process.start_time = mktime(&start_tm);
  buff = tokenize(line_no_time, &num_tokens);
  if (num_tokens >= 5)
    {
    curr_process.pid = (int) str_to_float(*(buff+0));
    curr_process.ppid = (int) str_to_float(*(buff+1));
    curr_process.pcpu = str_to_float(*(buff+2));
    curr_process.pmem = str_to_float(*(buff+3));
    curr_process.time = g_strdup(*(buff+4));
    g_array_append_val(psarray, curr_process);
    }
  g_strfreev(buff);
  }
pclose(fp);

/* Build the process hierarchy. Every process marks itself as first child */
/* of it's parent or as sister of first child of it's parent */
/* algorithm taken from pstree.c by Fred Hucht */

for (me = 0; me < psarray->len; me++)
  {
  parent = find_pid_index(g_array_index(psarray, struct task_pak, me).ppid);
  if (parent != me && parent != -1)
    { /* valid process, not me */
    g_array_index(psarray, struct task_pak, me).parent = parent;
    if (g_array_index(psarray, struct task_pak, parent).child == -1) /* first child */
      g_array_index(psarray, struct task_pak, parent).child = me;
    else
      {
      for (sister = g_array_index(psarray, struct task_pak, parent).child; EXIST(g_array_index(psarray, struct task_pak, sister).sister); sister = g_array_index(psarray, struct task_pak, sister).sister);
      g_array_index(psarray, struct task_pak, sister).sister = me;
      }
    }
  }

#if DEBUG_CALC_TASK_INFO
printf("process list\n");
for (me = 0; me < psarray->len; me++)
  {
  process_ptr = &g_array_index(psarray, struct task_pak, me);
  printf("pid: %d ppid: %d pcpu: %.1f pmem: %.1f date: %s parent: %d child: %d sister: %d time: %s", process_ptr->pid,  process_ptr->ppid, process_ptr->pcpu, process_ptr->pmem, process_ptr->time, process_ptr->parent, process_ptr->child, process_ptr->sister, ctime(&process_ptr->start_time));
  }
#endif

/* set tdata's pid to 0 so if this routine fails, it won't kill gdis! */
tdata->pcpu =  tdata->pmem = 0;
tdata->h_sec = tdata->sec = tdata->min = tdata->hour = 0;

/* scan through the running processes and find the task thread */
oldest_time = time(NULL);
found = FALSE;
for (me = 0; me < psarray->len; me++)
  {
  process_ptr = &g_array_index(psarray, struct task_pak, me);
  if (process_ptr->pid == tdata->pid)
    {
    found = TRUE;

      #if DEBUG_CALC_TASK_INFO
      printf("pid: %d time:%s\n", child_pid, ctime(&oldest_time));
      #endif
    break;
    }
  }
if (found)
  {
    /* find child */
/*    process_ptr = &g_array_index(psarray, struct task_pak, child_idx);
    child_idx = process_ptr->child;
    process_ptr = &g_array_index(psarray, struct task_pak, child_idx);
    child_pid = tdata->pid = process_ptr->pid;
    #if DEBUG_CALC_TASK_INFO
    printf("pid: %d time:%s\n", child_pid, ctime(&oldest_time));
    #endif*/

  add_from_tree(tdata, me);
  }

/* FIXME - core dumps here in analysis (esp. if >1 jobs queued) */
if (tdata->time)
  g_free(tdata->time);

/*
sprintf(line, "%02d:%02d:%02d", tdata->hour, tdata->min, tdata->sec);
tdata->time = g_strdup(line);
*/
tdata->time = g_strdup_printf("%02d:%02d:%02d", tdata->hour, tdata->min, tdata->sec);
 

/* search through any children of the the oldest child */
/*
pid = child_pid;
while ((list = g_slist_find_custom(palist, GINT_TO_POINTER(pid), (gpointer) find_pid)) != NULL)
  {
  curr_process = (struct task_pak *) list->data;
  pid = tdata->pid = curr_process->pid;
  tdata->pcpu += curr_process->pcpu;
  tdata->pmem += curr_process->pmem;
  if (tdata->time)
      g_free(tdata->time);
  tdata->time = g_strdup(curr_process->time);
  }
#if DEBUG_CALC_TASK_INFO
printf("FINAL TOTALS\n");
printf("pid: %d pcpu: %.1f pmem: %.1f time: %s\n", tdata->pid, tdata->pcpu, tdata->pmem, tdata->time);
#endif

free_slist(palist);
*/

/* hack for round off possibly giving >100% */
if (tdata->pcpu > 100.0)
  tdata->pcpu = 100.0;
if (tdata->pmem > 100.0)
  tdata->pmem = 100.0;
}

/**********************************/
/* get the currently selected row */
/**********************************/
gint task_list_selected(void)
{
gint row=0;
GtkTreeModel *treemodel;
GtkTreeSelection *selection;
GtkTreeIter iter;

treemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(task_list_tv));
selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(task_list_tv));

if (gtk_tree_model_get_iter_first(treemodel, &iter))
  {
  do
    {
    if (gtk_tree_selection_iter_is_selected(selection, &iter))
      return(row);
    row++;
    }
  while (gtk_tree_model_iter_next(treemodel, &iter));
  }
return(-1);
}

/*********************************/
/* rewrite the current task list */
/*********************************/
gint update_task_info(void)
{
gint i, row, num_rows=0;
gchar *txt, *info[6];
GSList *tlist=NULL;
GtkTreeIter iter;
GtkTreeModel *treemodel;
GtkTreeSelection *selection;
struct task_pak *task=NULL;

/* checks */
if (!dialog_active(TASKMAN))
  return(TRUE);
if (!task_list_ls)
  return(TRUE);

/* store */
row = task_list_selected();

/* update task summary info */
txt = g_strdup_printf("%d", g_slist_length(sysenv.task_list));
gtk_label_set_text(GTK_LABEL(task_label1), txt);
g_free(txt);

txt = g_strdup_printf("%d", sysenv.running_tasks);
gtk_label_set_text(GTK_LABEL(task_label2), txt);
g_free(txt);

txt = g_strdup_printf("%d", sysenv.max_running_tasks);
gtk_label_set_text(GTK_LABEL(task_label3), txt);
g_free(txt);

/* re-populate */
gtk_list_store_clear(task_list_ls);
for (tlist=sysenv.task_list ; tlist ; tlist=g_slist_next(tlist)) 
  {
  task = (struct task_pak *) tlist->data;
  gtk_list_store_append(task_list_ls, &iter);

/* column 0 */
  if (task->pid && (task->status == RUNNING))
    info[0] = g_strdup_printf("%d", task->pid);
  else
    info[0] = g_strdup(" ");

/* column 1 */
  if (task->label)
    info[1] = g_strdup(task->label);
  else
    info[1] = g_strdup(" Unknown ");

/* column 2-5 */
  switch(task->status)
    {
    case QUEUED:
      info[2] = g_strdup("Queued");
      info[3] = g_strdup_printf(" ");
      info[4] = g_strdup_printf(" ");
      info[5] = g_strdup_printf(" ");
      break;

    case RUNNING:
      calc_task_info(task);
      if (task->progress > 0.0)
        info[2] = g_strdup_printf("%5.1f%%", task->progress);
      else
        info[2] = g_strdup("Running");
      info[3] = g_strdup_printf("%5.1f", task->pcpu);
      info[4] = g_strdup_printf("%5.1f", task->pmem);
      info[5] = g_strdup_printf(" %s", task->time);
      break;

    default:
      info[2] = g_strdup("Unknown");
      info[3] = g_strdup_printf(" ");
      info[4] = g_strdup_printf(" ");
      info[5] = g_strdup_printf(" ");
    }

/* fill out the row */
  for (i=0 ; i<6 ; i++)
    gtk_list_store_set(task_list_ls, &iter, i, info[i], -1);

  num_rows++;
  } 

/* restore selected row, or the previous (if possible) if deleted  */
if (row >= num_rows && row)
  row--;
if (row >= 0)
  {
  treemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(task_list_tv));
  if (gtk_tree_model_iter_nth_child(treemodel, &iter, NULL, row))
    {
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(task_list_tv)); 
    if (selection)
      gtk_tree_selection_select_iter(selection, &iter);
    }
  }

return(TRUE);
}

/******************/
/* kill a process */
/******************/
#define DEBUG_KILL_FROM_TREE 0
void kill_from_tree(gint idx)
{
struct task_pak *process_ptr;
gint child;

#if DEBUG_KILL_FROM_TREE
printf("Killing task with index %d\n", idx);
#endif
process_ptr = &g_array_index(psarray, struct task_pak, idx);
/* process children */
for (child = process_ptr->child; EXIST(child); child = g_array_index(psarray, struct task_pak, child).sister)
  kill_from_tree(child);
/* don't kill gdis thread itself (eg analysis task) */
/* FIXME - win32 kill() equivalent? */
if (process_ptr->pid > 0 && process_ptr->pid != task_thread_pid)
  {
#ifdef __WIN32
/*
CloseHandle(process_ptr->pid);
*/
#else
  kill((pid_t) process_ptr->pid, SIGTERM);
#endif
  }
}

/************************************************/
/* kill a process and all it's spawned children */
/************************************************/
#define DEBUG_KILL_TASKS 0
void kill_tasks(struct task_pak *task)
{
struct task_pak *process_ptr;
gint me, found;

g_assert(task != NULL);

#if DEBUG_KILL_TASKS
printf("Killing task [%p], pid %d and its children\n", task, task->pid);
#endif

/* find task */
found = FALSE;
for (me = 0; me < psarray->len; me++)
  {
  process_ptr = &g_array_index(psarray, struct task_pak, me);
  if (process_ptr->pid == task->pid)
    {
      found = TRUE;
      break;
    }
  }
  if (found)
    kill_from_tree(me);
}

/***********************************/
/* kill/remove tasks from the list */
/***********************************/
#define DEBUG_TASK_LIST_DELETE 0
gint task_list_delete(GtkWidget *w, gint mode)
{
GSList *tlist=NULL;
struct task_pak *task=NULL;

/* NB: signals will trigger the task update stuff */
switch(mode)
  {
  case SINGLE:
    task = g_slist_nth_data(sysenv.task_list, task_list_selected());
    if (task)
      {
/* if running kill, else delete from list */
      if (task->status == RUNNING)
        kill_tasks(task);
      else
        {
#if DEBUG_TASK_LIST_DELETE
printf("removing task %d\n", selected_task);
#endif
        remove_task(task);
        }
      }
#if DEBUG_TASK_LIST_DELETE
    else
      printf("Invalid task number: %d\n", selected_task);
#endif
    break;

  case ALL:
/* remove all queued tasks first, so they won't be run when we kill the running ones */
    tlist = sysenv.task_list;
    while (tlist != NULL)
      {
      task = (struct task_pak *) tlist->data;
      tlist = g_slist_next(tlist);
/* remove only AFTER we've gone to the next element */
      if (task->status != RUNNING)
        remove_task(task);
      }

/* remove the running tasks */
    tlist = sysenv.task_list;
    while (tlist != NULL)
      {
      task = (struct task_pak *) tlist->data;
      tlist = g_slist_next(tlist);
/* FIXME - cores dumps often here for multiple md_analysis jobs */
      if (task->status == RUNNING)
        kill_tasks(task);
      }
    break;

  default:
    printf("task_list_delete(): bad argument.\n");
  }

/* updates */
update_task_info();

return(FALSE);
}

/***********************/
/* task dialog cleanup */
/***********************/
void task_cleanup(void)
{
gtk_list_store_clear(task_list_ls);
task_list_ls = NULL;
task_list_tv = NULL;
}

/****************************************************/
/* a task dialog for viewing/killing existing tasks */
/****************************************************/
void task_dialog()
{
gint i, id;
gchar *titles[6] = {"  PID  ", "       Job       ", "   Status   ", 
                    "  \% CPU  ", "  \% Mem  ", "  Time  "};
GtkWidget *frame, *vbox, *hbox, *label;
GtkWidget *swin;
GtkCellRenderer *renderer;
GtkTreeViewColumn *column;

gchar *txt;
struct dialog_pak *task_dialog;

/* request a new dialog */
if ((id = request_dialog(sysenv.active, TASKMAN)) < 0)
  return;
task_dialog = &sysenv.dialog[id];

/* create new dialog */
task_dialog->win = gtk_dialog_new();
gtk_window_set_title(GTK_WINDOW (task_dialog->win), "Task Manager");
gtk_window_set_default_size(GTK_WINDOW(task_dialog->win), 440, 340);
g_signal_connect(GTK_OBJECT(task_dialog->win), "destroy",
                 GTK_SIGNAL_FUNC(event_close_dialog), (gpointer) id);

/* Frame */
frame = gtk_frame_new (NULL);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(task_dialog->win)->vbox),frame,FALSE,FALSE,0); 
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);

/* create a hbox in the frame */
hbox = gtk_hbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(frame), hbox);
gtk_container_set_border_width(GTK_CONTAINER(hbox), PANEL_SPACING);

/* create vbox */
vbox = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
/* labels - running tasks, total tasks, allowed running tasks */
label = gtk_label_new("Number of tasks ");
gtk_misc_set_alignment(GTK_MISC(label), 0.0f, 0.0f);
gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, FALSE, 0);
label = gtk_label_new("Number of running tasks ");
gtk_misc_set_alignment(GTK_MISC(label), 0.0f, 0.0f);
gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, FALSE, 0);
label = gtk_label_new("Allowed number of running tasks ");
gtk_misc_set_alignment(GTK_MISC(label), 0.0f, 0.0f);
gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, FALSE, 0);

/* create vbox */
vbox = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);
/* values - running tasks, total tasks, allowed running tasks */
txt = g_strdup_printf("%d", sysenv.num_tasks);
task_label1 = gtk_label_new(txt);
gtk_misc_set_alignment(GTK_MISC(task_label1), 0.0f, 0.0f);
gtk_box_pack_start(GTK_BOX(vbox), task_label1, TRUE, FALSE, 0);
g_free(txt);

txt = g_strdup_printf("%d", sysenv.running_tasks);
task_label2 = gtk_label_new(txt);
gtk_misc_set_alignment(GTK_MISC(task_label2), 0.0f, 0.0f);
gtk_box_pack_start(GTK_BOX(vbox), task_label2, TRUE, FALSE, 0);
g_free(txt);

txt = g_strdup_printf("%d", sysenv.max_running_tasks);
task_label3 = gtk_label_new(txt);
gtk_misc_set_alignment(GTK_MISC(task_label3), 0.0f, 0.0f);
gtk_box_pack_start(GTK_BOX(vbox), task_label3, TRUE, FALSE, 0);
g_free(txt);

/* Frame */
frame = gtk_frame_new (NULL);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(task_dialog->win)->vbox),frame,TRUE,TRUE,0); 
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
/* create a vbox in the frame */
vbox = gtk_vbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);

/* scrolled model pane */
swin = gtk_scrolled_window_new(NULL, NULL);
gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW (swin),
                               GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
gtk_box_pack_start(GTK_BOX(vbox), swin, TRUE, TRUE, 0);


task_list_ls = gtk_list_store_new(6, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING,
                                     G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING);
task_list_tv = gtk_tree_view_new_with_model(GTK_TREE_MODEL(task_list_ls));
gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(swin), task_list_tv);
for (i=0 ; i<6 ; i++)
  {
  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(titles[i], renderer, "text", i, NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(task_list_tv), column);
  }

/* control buttons */
gtksh_icon_button(GTK_STOCK_REMOVE, "Remove",
                  task_list_delete, (gpointer) SINGLE,
                  GTK_DIALOG(task_dialog->win)->action_area);

gtksh_icon_button(GTK_STOCK_CLEAR, "Remove all",
                  task_list_delete, (gpointer) ALL,
                  GTK_DIALOG(task_dialog->win)->action_area);

gtksh_stock_button(GTK_STOCK_CLOSE,
                   event_close_dialog, GINT_TO_POINTER(id),
                   GTK_DIALOG(task_dialog->win)->action_area);

/* done */
gtk_widget_show_all(task_dialog->win);
/* refresh labels */
update_task_info();
}

