/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <ctype.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>

#include "gdis.h"
#include "coords.h"
#include "matrix.h"
#include "sginfo.h"
#include "space.h"
#include "gtkshorts.h"
#include "interface.h"
#include "dialog.h"
#include "opengl.h"

/* main structure */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/* values for directions */
/*
GtkWidget *imagespin[MAX_DIALOGS][6];
*/

/**********/
/* sginfo */
/**********/
static int str_ibegin(const char *s1, const char *s2) /* string ignore-case */
{                                                     /* begin              */
char u1, u2;

while (*s1 && *s2)
  {
  u1 = toupper(*s1++);
  u2 = toupper(*s2++);
  if (u1 < u2)
    return -1;
  else if (u1 > u2)
    return  1;
  }
if (*s2)
  return -1;
return 0;
}

/**********/
/* sginfo */
/**********/
/* TODO - we know what the input will be - trim */
gint BuildSgInfo(T_SgInfo *SgInfo, const gchar *SgName)
{
gint                VolLetter;
const T_TabSgName  *tsgn;


/* look for "VolA", "VolI", or "Hall" */

while (*SgName && isspace((int) *SgName)) SgName++;

VolLetter = -1;

if      (isdigit((int) *SgName))
  VolLetter = 'A';
else if (str_ibegin(SgName, "VolA") == 0)
  {
  VolLetter = 'A';
  SgName += 4;
  }
else if (str_ibegin(SgName, "VolI") == 0 || str_ibegin(SgName, "Vol1") == 0)
  {
  VolLetter = 'I';
  SgName += 4;
  }
else if (str_ibegin(SgName, "Hall") == 0)
  {
  VolLetter = 0;
  SgName += 4;
  }

while (*SgName && isspace((int) *SgName)) 
  SgName++;

/* default is "VolA" */

if (VolLetter == -1)
  VolLetter = 'A';

/* if we don't have a Hall symbol do a table look-up */

tsgn = NULL;

if (VolLetter)
  {
  tsgn = FindTabSgNameEntry(SgName, VolLetter);
  if (tsgn == NULL) return -1; /* no matching table entry */
  SgName = tsgn->HallSymbol;
  }

/* Allocate memory for the list of Seitz matrices and
   a supporting list which holds the characteristics of
   the rotation parts of the Seitz matrices
*/

SgInfo->MaxList = 192; /* absolute maximum number of symops */

SgInfo->ListSeitzMx
    = g_malloc(SgInfo->MaxList * sizeof (*SgInfo->ListSeitzMx));

if (SgInfo->ListSeitzMx == NULL) 
  {
  SetSgError("Not enough core");
  return -1;
  }

SgInfo->ListRotMxInfo
    = g_malloc(SgInfo->MaxList * sizeof (*SgInfo->ListRotMxInfo));

if (SgInfo->ListRotMxInfo == NULL) 
  {
  SetSgError("Not enough core");
  return -1;
  }

/* Initialize the SgInfo structure */

InitSgInfo(SgInfo);
SgInfo->TabSgName = tsgn; /* in case we know the table entry */

/* Translate the Hall symbol and generate the whole group */

ParseHallSymbol(SgName, SgInfo);
if (SgError != NULL)
  return -1;

  /* Do some book-keeping and derive crystal system, point group,
     and - if not already set - find the entry in the internal
     table of space group symbols
   */

return CompleteSgInfo(SgInfo);
}

/*****************************************/
/* generate positions of a complete cell */
/*****************************************/
void fn_fill_cell(struct model_pak *model)
{
gint j, s, smin;
GSList *cores, *list;
struct core_pak *core, *sr_core;
struct shel_pak *sr_shell;

/* include an inversion (ie mult by -1) operation? */
smin = 0;
if (model->sginfo.centric == -1)
  smin = -2;

/* prepending shells - this'll preserve order */
model->shels = g_slist_reverse(model->shels);

/* loop over (asymetric only - list order!!!) atoms */
cores = NULL;
for (list=model->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;

/* operate only on primary (asymmetric unit) atoms */
/* TODO - delete all non primary prior to this */
  if (!core->primary)
    continue;

/* s = +1 or -1 for inversion of matrix elements */
  for (s=1 ; s>smin ; s-=2)
    {
/* loop over group operations */
/* NB: include j=0 in case of inversion centre ie (-x,-y,-z) */
    for (j=(s+1)/2 ; j<model->sginfo.order ; j++)
      {
/* new, symmetry related core */
      sr_core = dup_core(core);
      cores = g_slist_prepend(cores, sr_core);

      sr_core->primary = FALSE;

/* get current (ith) coords */
      VEC3MUL(sr_core->x, s);
      vecmat(*(model->sginfo.matrix+j), sr_core->x);
      sr_core->x[0] += *(*(model->sginfo.offset+j)+0);
      sr_core->x[1] += *(*(model->sginfo.offset+j)+1);
      sr_core->x[2] += *(*(model->sginfo.offset+j)+2);

      if (sr_core->shell)
        {
        sr_shell = sr_core->shell;
        model->shels = g_slist_prepend(model->shels, sr_shell);

        sr_shell->primary = FALSE;

        VEC3MUL(sr_shell->x, s);
        vecmat(*(model->sginfo.matrix+j), sr_shell->x);
        sr_shell->x[0] += *(*(model->sginfo.offset+j)+0);
        sr_shell->x[1] += *(*(model->sginfo.offset+j)+1);
        sr_shell->x[2] += *(*(model->sginfo.offset+j)+2);
        }
      }         /* loop - operations in the group */
    }
  }             /* loop - primary atoms */

/* update lists */
model->cores = g_slist_concat(model->cores, cores);

/* constrain & enforce no duplicates */
/* NB: do this before pbc constrain, as it updates core-shell linkages */
remove_duplicates(model);
/* NB: core-shell linkages must be correct before we call this */
pbc_constrain_atoms(model);
}

/**************************/
/* do space group loopkup */
/**************************/
#define DEBUG_GEN_POS 0
gint genpos(struct model_pak *data)
{
gint f, i, j, nt;
gint m;
gint iList, nTrV, iTrV, nLoopInv, iLoopInv, iMatrix;
gchar *label;
gdouble dab, dac, dbc;
GString *name;
const T_RTMx  *lsmx;
const gint *r, *t, *TrV;
T_RTMx SMx;
T_SgInfo SgInfo;

g_return_val_if_fail(data != NULL, 1);

/* request info via number or name */
/* cell choice (TODO - neaten this ugly code) */

name = g_string_new(NULL);

if (data->sginfo.spacenum > 0)
  g_string_sprintf(name, "%d", data->sginfo.spacenum);
else
  g_string_sprintf(name, "%s", g_strstrip(data->sginfo.spacename));

/* cell choice */
if (data->sginfo.cellchoice)
  g_string_sprintfa(name, ":%d", data->sginfo.cellchoice);

/* if trigonal lattice, determine if cell is in hex or rhombo format */
if (g_strrstr(data->sginfo.spacename, "R") || data->sginfo.spacenum == 146 || data->sginfo.spacenum == 148 || data->sginfo.spacenum == 155 || data->sginfo.spacenum == 160 || data->sginfo.spacenum == 161 || data->sginfo.spacenum == 166 || data->sginfo.spacenum == 167)
  {
  dab = (data->pbc[0] - data->pbc[1]);
  dac = (data->pbc[0] - data->pbc[2]);
  dbc = (data->pbc[1] - data->pbc[2]);
/* better tolerances? */
  if (dab*dab < 0.001)
    if (dac*dac < 0.001)
      if (dbc*dbc < 0.001)
        g_string_sprintfa(name, ":R");   /* primitive (rhombohedral) */
  }

#if DEBUG_GEN_POS
printf("retrieving as [%s]\n", name->str);
#endif

/* main call */
if (BuildSgInfo(&SgInfo, name->str) != 0)
  {
/* CURRENT - if order > 0 -> fill the cell */
  if (data->sginfo.order)
    {
    show_text(WARNING, "No spacegroup: using explicit matrices.\n");
    fn_fill_cell(data);
/* NB: we must constrain here if the periodic bond calculation is */
/* to be done correctly. Since we can have auto unfragment on */
/* this shouldn't be a problem for models with parts sticking out */
/*
    pbc_constrain_atoms(data);
*/
    return(0);
    }
  return(2);
  }
g_string_free(name, TRUE);

/* process returned space group name */
label = strdup(SgInfo.TabSgName->SgLabels);
for (i=0 ; i<strlen(label) ; i++)
  if (*(label+i) == '_')
    *(label+i) = ' ';

/* copy label to avoid any funny GULP characters */
i=strlen(label);
while(i>0)
  {
  if (*(label+i) == '=')
    {
    i++;
    break;
    }
  i--;
  }

/* fill in space group name */
g_free(data->sginfo.spacename);
data->sginfo.spacename = g_strdup_printf("%s",label+i);

/* fill in number (if an invalid value was supplied) */
if (data->sginfo.spacenum < 1)
  data->sginfo.spacenum = SgInfo.TabSgName->SgNumber;

/* crystal lattice type */
data->sginfo.lattice = SgInfo.XtalSystem;
g_free(data->sginfo.latticename);
switch(SgInfo.XtalSystem)
  {
  case XS_CUBIC:
    data->sginfo.latticename = g_strdup("Cubic");
    break;
  case XS_TETRAGONAL:
    data->sginfo.latticename = g_strdup("Tetragonal");
    break;
  case XS_ORTHOROMBIC:
    data->sginfo.latticename = g_strdup("Orthorhombic");
    break;
  case XS_HEXAGONAL:
    data->sginfo.latticename = g_strdup("Hexagonal");
    break;
  case XS_TRIGONAL:
    data->sginfo.latticename = g_strdup("Trigonal");
    break;
  case XS_MONOCLINIC:
    data->sginfo.latticename = g_strdup("Monoclinic");
    break;
  case XS_TRICLINIC:
    data->sginfo.latticename = g_strdup("Triclinic");
    break;
  default:
    data->sginfo.latticename = g_strdup("Unknown");
  }

/* point group */
data->sginfo.pointgroup = SgInfo.PointGroup;
/* centred */
data->sginfo.centric = SgInfo.Centric;

#if DEBUG_GEN_POS
printf("   Space group name: %s\n",data->sginfo.spacename);
printf(" Space group number: %d\n",data->sginfo.spacenum);
printf("       Lattice type: %c\n",SgInfo.LatticeInfo->Code);
printf("              Order: %d\n",SgInfo.OrderL);
printf("   Inversion center: ");
if (SgInfo.Centric == -1)
  printf("yes\n");
else
  printf("no\n");
#endif

/* CURRENT */
if (!data->sginfo.order)
  {

/* OrderP/OrderL - which one??? */
data->sginfo.order = SgInfo.OrderL;
/* allocate for order number of pointers (to matrices) */
data->sginfo.matrix = (gdouble **) g_malloc(SgInfo.OrderL*sizeof(gdouble *));
data->sginfo.offset = (gdouble **) g_malloc(SgInfo.OrderL*sizeof(gdouble *));

iMatrix = 0;

nLoopInv = Sg_nLoopInv(&SgInfo);

nTrV = SgInfo.LatticeInfo->nTrVector;
 TrV = SgInfo.LatticeInfo->TrVector;

/* matrix counter */
m=0;
for (iTrV = 0; iTrV < nTrV; iTrV++, TrV += 3)
  {
  for (iLoopInv = 0; iLoopInv < nLoopInv; iLoopInv++)
    {
    if (iLoopInv == 0)
      f =  1;
    else
      f = -1;

    lsmx = SgInfo.ListSeitzMx;

/* loop over all matrices (order of the group) */
    for (iList = 0; iList < SgInfo.nList; iList++, lsmx++)
      {
      for (i = 0; i < 9; i++)
        SMx.s.R[i] = f * lsmx->s.R[i];
      for (i = 0; i < 3; i++)
        SMx.s.T[i] = iModPositive(f * lsmx->s.T[i] + TrV[i], STBF);

      r = SMx.s.R;
      t = SMx.s.T;

      *(data->sginfo.matrix+m) = (gdouble *) g_malloc(9*sizeof(gdouble));
      *(data->sginfo.offset+m) = (gdouble *) g_malloc(3*sizeof(gdouble));

      for (i = 0; i < 3; i++, t++)
        {
        for (j = 0; j < 3; j++, r++)
          {
/* mth general position */
          *(*(data->sginfo.matrix+m)+3*i+j) = (gdouble) *r;
          }
        nt = iModPositive(*t, STBF);
        if (nt >  STBF / 2)
          nt -= STBF;
/* offset matrix (3x1) */
        *(*(data->sginfo.offset+m)+i) = (gdouble) nt / (gdouble) STBF;
        }
      m++;
      }
    }
  }

/* number of matrices matches the order? */
  if (m != data->sginfo.order)
    printf("Serious error in genpos()\n");
  }
else
  {
#if DEBUG_GEN_POS
printf("Skipping symmetry matrix generation.\n");
#endif
  m = data->sginfo.order;
  }

#if DEBUG_GEN_POS
printf("Symmetry matrices: %d\n", m);
for (i=0 ; i<m ; i++)
  {
  printf("matrix %d :",i);
  for (j=0 ; j<9 ; j++)
    printf(" %4.1f",*(*(data->sginfo.matrix+i)+j));
  printf("\n");

  printf("offset %d :",i);
  for (j=0 ; j<3 ; j++)
    printf(" %4.1f",*(*(data->sginfo.offset+i)+j));
  printf("\n");
  }
#endif

/* NEW - store the raw SgInfo structure for later calls (eg hkl absences) */
if (data->sginfo.raw)
  g_free(data->sginfo.raw);
data->sginfo.raw = g_malloc(sizeof(SgInfo));
memcpy(data->sginfo.raw, &SgInfo, sizeof(SgInfo));

/* generate symmetry related cores */
fn_fill_cell(data);

/* NB: we must constrain here if the periodic bond calculation is */
/* to be done correctly. Since we can have auto unfragment on */
/* this shouldn't be a problem for models with parts sticking out */
/*
pbc_constrain_atoms(data);
*/

#if DEBUG_GEN_POS
printf("[*] num atoms: %d\n", g_slist_length(data->cores));
printf("[*] num shels: %d\n", g_slist_length(data->shels));
#endif

return(0);
}

/***************************************************************/
/* ensure a model's coordinates are stored in cartesian format */
/***************************************************************/
void make_cartesian(struct model_pak *model)
{
GSList *list;
struct core_pak *core;
struct shel_pak *shell;

g_assert(model != NULL);

for (list=model->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  vecmat(model->latmat, core->x);
  core->primary = TRUE;
  core->orig = TRUE;
  }
for (list=model->shels ; list ; list=g_slist_next(list))
  {
  shell = (struct shel_pak *) list->data;
  vecmat(model->latmat, shell->x);
  shell->primary = TRUE;
  shell->orig = TRUE;
  }
model->fractional = FALSE;
}

/***********************************************/
/* periodicity dialog hook to make a supercell */
/***********************************************/
void make_supercell(GtkWidget *w, struct model_pak *model)
{
gint i;
gdouble v[3], m[3];
GSList *list, *ilist;
struct core_pak *core, *core2;
struct shel_pak *shell;
struct image_pak *image;

/* get model ptr assoc. with this dialog */
g_assert(model != NULL);

/* all (non-deleted) atoms are now primary & non fractional */
delete_commit(model);
for (list=model->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  core->orig = TRUE;
  core->primary = TRUE;

/* duplicate images */
  for (ilist=model->images ; ilist ; ilist=g_slist_next(ilist))
    {
    image = (struct image_pak *) ilist->data;

/* core */
    core2 = dup_core(core);
    ARR3ADD(core2->x, image->pic);
    model->cores = g_slist_prepend(model->cores, core2);

/* if shell was also duplicated, update & add to list */
    if (core2->shell)
      {
      ARR3ADD((core2->shell)->x, image->pic);
      model->shels = g_slist_prepend(model->shels, core2->shell);
      }
    }
  }
free_slist(model->images);
model->images = NULL;

/* and the shells */
for (list=model->shels ; list ; list=g_slist_next(list))
  {
  shell = (struct shel_pak *) list->data;
  shell->orig = TRUE;
  shell->primary = TRUE;
  }

/* get multiple of lattice vectors desired */
m[0] = fabs(model->image_limit[1] + model->image_limit[0]);
m[1] = fabs(model->image_limit[3] + model->image_limit[2]);
m[2] = fabs(model->image_limit[5] + model->image_limit[4]);

/* NEW - scale up the surface bulk energy */
model->gulp.sbulkenergy *= m[0] * m[1] * m[2];

/* scale the fractional coordinates down */
for (list=model->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  core->x[0] /= m[0];
  core->x[1] /= m[1];
  core->x[2] /= m[2];
  }
for (list=model->shels ; list ; list=g_slist_next(list))
  {
  shell = (struct shel_pak *) list->data;
  shell->x[0] /= m[0];
  shell->x[1] /= m[1];
  shell->x[2] /= m[2];
  }

/* get current lattice vectors and scale up */
for (i=0 ; i<3 ; i++)
  {
  v[0] = model->latmat[i];
  v[1] = model->latmat[i+3];
  v[2] = model->latmat[i+6];
  VEC3MUL(v, m[i]);
  model->latmat[i] = v[0];
  model->latmat[i+3] = v[1];
  model->latmat[i+6] = v[2];
  }

/* update cell param related data */
model->construct_pbc = TRUE;

/* NB: this closes the periodicity box (since symmetry is changed) */
make_p1(model);

/* TODO - currently, tell gdis to do pbond calc again */
/* but a more efficient way would be to recalc */
model->done_pbonds = FALSE;

prep_model(model);

model->image_limit[0] = 0.0;
model->image_limit[1] = 1.0;
model->image_limit[2] = 0.0;
model->image_limit[3] = 1.0;
model->image_limit[4] = 0.0;
model->image_limit[5] = 1.0;

redraw_canvas(SINGLE);

/* sometimes this causes strange redraw hicups (doesn't crash AFAIK tho') */
/*
space_info(model);
*/
}

/***********************************/
/* periodic image creation routine */
/***********************************/
#define DEBUG_UPDATE_IMAGES 0
void update_images(gint mode, struct model_pak *model)
{
gint a, b, c, i;
gint num_cells, num_images;
struct image_pak *image;

/* checks */
g_assert(model != NULL);

#if DEBUG_UPDATE_IMAGES
printf("---------------------------\n");
printf("model : %d\n", model->number);
printf("a : %f - %f\n", model->image_limit[0], model->image_limit[1]);
printf("b : %f - %f\n", model->image_limit[2], model->image_limit[3]);
printf("c : %f - %f\n", model->image_limit[4], model->image_limit[5]);
printf("---------------------------\n");
#endif

num_images = 0;
/* check mode */
switch(mode)
  {
  case CREATE:
/* update image list */
    free_slist(model->images);
    model->images = NULL;
    num_cells = (model->image_limit[0]+model->image_limit[1])
              * (model->image_limit[2]+model->image_limit[3])
              * (model->image_limit[4]+model->image_limit[5]);
    num_images = num_cells-1;
/* setup for pic iteration */
    a = -model->image_limit[0];
    b = -model->image_limit[2];
    c = -model->image_limit[4];
    for (i=num_cells ; i-- ; )
      {
/* image increment */
      if (a == model->image_limit[1])
        {
        a = -model->image_limit[0];
        b++;
        if (b == model->image_limit[3])
          {
          b = -model->image_limit[2];
          c++;
          if (c == model->image_limit[5])
            break;
          }
        }
/* don't duplicate the primary cell */
      if (a || b || c)
        {
        image = g_malloc(sizeof(struct image_pak));

        VEC3SET(image->pic, a, b, c);
        model->images = g_slist_prepend(model->images, image);
        }
      a++;
      }
    break;

  case INITIAL:
/* reset values to default */
    for (i=0 ; i<model->periodic ; i++)
      {
      model->image_limit[2*i] = 0;
      model->image_limit[2*i+1] = 1;
      }
/* delete all images */
    free_slist(model->images);
    model->images = NULL;
    break;
  }

gtksh_relation_update(model);
}

/*********************/
/* spin button hooks */
/*********************/
void gtk_refresh_images(GtkWidget *w, gint id)
{
gint model;
struct model_pak *data;

model = sysenv.dialog[id].model;
data = model_ptr(model, RECALL);
g_return_if_fail(data != NULL);

/* atom colour/display updates */
update_images(CREATE, data);
init_objs(CENT_COORDS, data);
pick_model(model);
}

void gtk_reset_images(GtkWidget *w, gint id)
{
gint model;
struct model_pak *data;

model = sysenv.dialog[id].model;
data = model_ptr(model, RECALL);
g_return_if_fail(data != NULL);

/* atom colour/display updates */
update_images(INITIAL, data);
init_objs(CENT_COORDS, data);
pick_model(model);
}

/****************************/
/* display property updates */
/****************************/
void asym_refresh(GtkWidget *w, struct model_pak *model)
{
GSList *list;
struct core_pak *core;
struct shel_pak *shel;

g_assert(model != NULL);

if (model->periodic)
  {
  for (list=model->cores ; list ; list=g_slist_next(list))
    {
    core = (struct core_pak *) list->data;
    if (core->primary)
      continue;

    if (model->asym_on)
      core->status |= HIDDEN;
    else
      core->status &= ~HIDDEN;
    }
  for (list=model->shels ; list ; list=g_slist_next(list))
    {
    shel = (struct shel_pak *) list->data;
    if (shel->primary)
      continue;
    if (model->asym_on)
      shel->status |= HIDDEN;
    else
      shel->status &= ~HIDDEN;
    }
  }

/* update */
init_objs(REDO_COORDS, model);
redraw_canvas(SINGLE);
}

/***************************/
/* Space Group info dialog */
/***************************/
void space_info(struct model_pak *data)
{
gint i, j, k, l, m, id, rows, cols, need_sign;
gchar *dir;
gdouble *mat, value;
GString *info, *item;
GtkWidget *table, *frame;
GtkWidget *vbox, *hbox, *vbox1, *vbox2;
GtkWidget *label, *spin;
struct dialog_pak *sgid;

item = g_string_new(NULL);
info = g_string_new(NULL);

/* model checks */
g_return_if_fail(data != NULL);
g_return_if_fail(data->id != NODATA);
g_return_if_fail(data->periodic > 0);

/* retrieve a suitable dialog */
if ((id = request_dialog(data->number, SGINFO)) < 0)
  return;
sgid = &sysenv.dialog[id];

/* setup the dialog */
sgid->win = gtk_dialog_new();
gtk_window_set_title (GTK_WINDOW (sgid->win), "Space Group Info");
g_signal_connect(GTK_OBJECT(sgid->win), "destroy",
                 GTK_SIGNAL_FUNC(event_close_dialog), (gpointer) id);

/* dual pane layout */
hbox = gtk_hbox_new(FALSE,0);
gtk_container_add(GTK_CONTAINER(GTK_BOX(GTK_DIALOG(sgid->win)->vbox)),hbox);

/* left & right vboxes */
vbox1 = gtk_vbox_new (FALSE, 0);
vbox2 = gtk_vbox_new (FALSE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox1, TRUE, TRUE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox2, FALSE, FALSE, 0);

/* LEFT FRAME - MAIN INFO */
g_string_sprintf(info,"%s",data->basename);

frame = gtk_frame_new (info->str);
gtk_box_pack_start(GTK_BOX(vbox1), frame, TRUE, TRUE, 0);
gtk_container_set_border_width(GTK_CONTAINER (frame), PANEL_SPACING);
vbox = gtk_vbox_new(FALSE, 5);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame),vbox);

/* table for nice spacing */
table = gtk_table_new(2, 6, FALSE);
gtk_container_add(GTK_CONTAINER(GTK_BOX(vbox)),table);

label = gtk_label_new ("Space Group name:       ");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,0,1);
gtk_misc_set_alignment(GTK_MISC(label), 1.0f, 0.0f);
g_string_sprintf(info,"%s",data->sginfo.spacename);
label = gtk_label_new (g_strstrip(info->str));
gtk_table_attach_defaults(GTK_TABLE(table),label,1,2,0,1);
gtk_misc_set_alignment(GTK_MISC(label), 0.5f, 0.0f);

label = gtk_label_new ("Space Group number:       ");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,1,2);
gtk_misc_set_alignment(GTK_MISC(label), 1.0f, 0.0f);

/* include cell choice */
if (data->sginfo.cellchoice)
  g_string_sprintf(info,"%d : %d",data->sginfo.spacenum
                                 ,data->sginfo.cellchoice);
else
  {
  if (data->sginfo.spacenum > 0)
    g_string_sprintf(info,"%d",data->sginfo.spacenum);
  else
    g_string_sprintf(info,"?");
  }
label = gtk_label_new (g_strstrip(info->str));
gtk_table_attach_defaults(GTK_TABLE(table),label,1,2,1,2);
gtk_misc_set_alignment(GTK_MISC(label), 0.5f, 0.0f);

label = gtk_label_new ("Crystal system:       ");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,2,3);
gtk_misc_set_alignment(GTK_MISC(label), 1.0f, 0.0f);
g_string_sprintf(info,"%s",g_strstrip(data->sginfo.latticename));
label = gtk_label_new (g_strstrip(info->str));
gtk_table_attach_defaults(GTK_TABLE(table),label,1,2,2,3);
gtk_misc_set_alignment(GTK_MISC(label), 0.5f, 0.0f);

label = gtk_label_new ("Point group:       ");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,3,4);
gtk_misc_set_alignment(GTK_MISC(label), 1.0f, 0.0f);
i = PG_Index(data->sginfo.pointgroup);
g_string_sprintf(info,"%s",PG_Names[i]);
label = gtk_label_new (g_strstrip(info->str));
gtk_table_attach_defaults(GTK_TABLE(table),label,1,2,3,4);
gtk_misc_set_alignment(GTK_MISC(label), 0.5f, 0.0f);

label = gtk_label_new ("Atoms in asymmetric cell:       ");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,4,5);
gtk_misc_set_alignment(GTK_MISC(label), 1.0f, 0.0f);
g_string_sprintf(info,"%d",data->num_asym);
label = gtk_label_new (g_strstrip(info->str));
gtk_table_attach_defaults(GTK_TABLE(table),label,1,2,4,5);
gtk_misc_set_alignment(GTK_MISC(label), 0.5f, 0.0f);

label = gtk_label_new ("Atoms in full cell:       ");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,5,6);
gtk_misc_set_alignment(GTK_MISC(label), 1.0f, 0.0f);
g_string_sprintf(info,"%d",data->num_atoms);
label = gtk_label_new (g_strstrip(info->str));
gtk_table_attach_defaults(GTK_TABLE(table),label,1,2,5,6);
gtk_misc_set_alignment(GTK_MISC(label), 0.5f, 0.0f);

/* section */
frame = gtk_frame_new("Cell parameters");
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);

vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

hbox = gtk_hbox_new(TRUE, 5);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
g_string_sprintf(info, "lengths: %6.2f  %6.2f  %6.2f", data->pbc[0],
                              data->pbc[1], data->pbc[2]);
label = gtk_label_new(info->str);
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

hbox = gtk_hbox_new(TRUE, 5);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
g_string_sprintf(info, " angles: %6.2f  %6.2f  %6.2f", 180*data->pbc[3]/PI,
                       180*data->pbc[4]/PI, 180*data->pbc[5]/PI);
label = gtk_label_new(info->str);
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

/* volume calc, TODO - area calc for surface? */
if (data->periodic == 3)
  {
/* display */
  hbox = gtk_hbox_new(TRUE, 5);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  g_string_sprintf(info, " volume: %12.6f             ",
                             calc_volume(data->latmat));
  label = gtk_label_new(info->str);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  }

/* FRAME */
frame = gtk_frame_new ("General positions");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(sgid->win)->vbox),frame,TRUE,TRUE,0);
gtk_container_set_border_width (GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new (FALSE, 5);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);
gtk_container_add (GTK_CONTAINER(frame),vbox);
/* table for nice spacing */
/* more intelligent dimension choice? */
cols = 4;
rows = data->sginfo.order / cols;
table = gtk_table_new(cols, rows, FALSE);
gtk_container_add(GTK_CONTAINER(vbox), table);

/* order of the group (ie mth matrix) */
m=0;
/* loop over rows */
while (m<data->sginfo.order)
  {
/* get row(i) and column(j) */
  i = m / cols;
  j = m % cols;
/* construct the three generator elements */
  mat = *(data->sginfo.matrix+m);
  g_string_sprintf(info," ");
  for (k=0 ; k<3 ; k++)
    {
/* offset elements */
    value = *(*(data->sginfo.offset+m)+k);
    need_sign = 0;
/* separate 2nd and 3rd elements with a comma */
    if (k)
      g_string_append(info, ", ");
/* clear the string for the new element to append */
    g_string_assign(item, "");
/* append the offset */
    if (fabs(2.0*value-1.0) < FRACTION_TOLERANCE)
      {
      g_string_sprintfa(item, "1/2");
      need_sign++;
      }
    if (fabs(2.0*value+1.0) < FRACTION_TOLERANCE)
      {
      g_string_sprintfa(item, "-1/2");
      need_sign++;
      }
    if (fabs(3.0*value-1.0) < FRACTION_TOLERANCE)
      {
      g_string_sprintfa(item, "1/3");
      need_sign++;
      }
    if (fabs(3.0*value+1.0) < FRACTION_TOLERANCE)
      {
      g_string_sprintfa(item, "-1/3");
      need_sign++;
      }
    if (fabs(4.0*value-1.0) < FRACTION_TOLERANCE)
      {
      g_string_sprintfa(item, "1/4");
      need_sign++;
      }
    if (fabs(4.0*value+1.0) < FRACTION_TOLERANCE)
      {
      g_string_sprintfa(item, "-1/4");
      need_sign++;
      }
/* append the appropriate matrix element(s) */
    for (l='x' ; l<='z' ; l++)
      {
      if (*mat > FRACTION_TOLERANCE)
        {
        if (need_sign)
          g_string_sprintfa(item, "+%c", l);
        else
          {
          g_string_sprintfa(item, "%c", l);
          need_sign++;
          }
        }
      if (*mat < -FRACTION_TOLERANCE)
        {
        g_string_sprintfa(item, "-%c",l);
        need_sign++;
        }
      mat++;
      }
    g_string_append(info, item->str);
    }
/* write the generator to the table */
  label = gtk_label_new(info->str);
  gtk_table_attach_defaults(GTK_TABLE(table),label,j,j+1,i,i+1);
  m++;
  }

/* RIGHT FRAME - PERIODIC IMAGE CONSTRUCTION */
frame = gtk_frame_new ("Periodic image construction");
gtk_box_pack_start(GTK_BOX(vbox2), frame, TRUE, FALSE, 0);

gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

/* use table for nice spacing */
table = gtk_table_new(3, 4, FALSE);
gtk_container_add(GTK_CONTAINER(vbox), table);

/* image directions */
label = gtk_label_new ("-");
gtk_table_attach_defaults(GTK_TABLE(table),label,1,2,0,1);
label = gtk_label_new ("+");
gtk_table_attach_defaults(GTK_TABLE(table),label,2,3,0,1);

/* labels and spinners */
dir = g_strdup("a");
for (i=0 ; i<data->periodic ; i++)
  {
  label = gtk_label_new(dir);
  gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,i+1,i+2);

/* negative direction */
  spin = gtksh_direct_spin(NULL, &data->image_limit[2*i], 0, 10, 1,
                           gtk_refresh_images, (gpointer) id, NULL);
  gtk_table_attach_defaults(GTK_TABLE(table), spin, 1, 2, i+1, i+2);

/* positive direction */
  spin = gtksh_direct_spin(NULL, &data->image_limit[2*i+1], 1, 10, 1,
                           gtk_refresh_images, (gpointer) id, NULL);

  gtk_table_attach_defaults(GTK_TABLE(table), spin, 2, 3, i+1, i+2);

  dir[0]++;
  }
g_free(dir);

/* reset images */
hbox = gtk_hbox_new(TRUE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, PANEL_SPACING);
gtksh_button("  Reset  ", gtk_reset_images, GINT_TO_POINTER(id), hbox, TF);

/* FRAME */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX (vbox2), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);

gtksh_direct_check("Asymmetric cell", &data->asym_on, asym_refresh, data, vbox);

gtksh_button_x("Make supercell", make_supercell, data, vbox);

/* exit button */
gtksh_stock_button(GTK_STOCK_CLOSE, event_close_dialog, GINT_TO_POINTER(id),
                   GTK_DIALOG(sgid->win)->action_area);

/* done, draw, exit */
gtk_widget_show_all(sgid->win);
/* active this model */
pick_model(data->number);

g_string_free(info, TRUE);
g_string_free(item, TRUE);
}

