/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>

#include "gdis.h"
#include "file.h"
#include "parse.h"
#include "coords.h"
#include "matrix.h"
#include "molcon.h"
#include "spatial.h"
#include "surface.h"
#include "sginfo.h"
#include "task.h"
#include "gtkshorts.h"
#include "interface.h"
#include "dialog.h"
#include "opengl.h"

/* re-entrant surface search step (Ang) */
#define LIFT_STEP 0.01
/* re-entrant surface search sweep (radians) */
#define SWEEP_STEP 0.05

/* main pak structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/* molsurf globals */
/* TODO - get rid of this (or get it updated), so we can switch between models */
struct model_pak *ms_data=NULL;

/******************************************/
/* build exterior/neighbourhood atom list */
/******************************************/
/* NB: should speed up calc - as well as eliminating problems */
/* due to contributions from cavities */

/************************************************/
/* gaussian based distance to molecular surface */
/************************************************/
/* r is position, a is probe radius, list is core list */
/* TODO - in principle this should handle surfaces, but the trouble is there */
/* is no confinement to the surface cell */
gdouble ms_dist(gdouble *r, GSList *list)
{
gint i, j, m, ni;
gdouble a, sum, x[3];
GSList *item;
struct core_pak *core;

ni = pow(3, ms_data->periodic);

sum = 0.0;
for (m=ni ; m-- ; )
  {
/* periodic images (NB: only works for periodicity = 2) */
  i = (m % 3) - 0.5*ms_data->periodic;
  j = (m / 3) - 0.5*ms_data->periodic;

  for (item=list ; item ; item=g_slist_next(item))
    {
    core = (struct core_pak *) item->data;

    a = ms_data->csurf.prad + elements[core->atom_code].vdw;

    ARR3SET(x, core->x);
    VEC3ADD(x, i, j, 0);
    vecmat(ms_data->latmat, x);
    ARR3SUB(x, r);

    sum += exp(-(VEC3MAG(x) - a)/ms_data->csurf.dsize);
    }
  }
return(-ms_data->csurf.dsize*log(sum));
}

/*********************************************/
/* gaussian based molecular surface gradient */
/*********************************************/
struct core_pak *ms_grad(struct smv_pak *point, GSList *list)
{
gint i, j, m, ni, touch;
gdouble a, e, dx, sum1[3], sum2[3], x[3], tmp[3];
GSList *item;
struct core_pak *core, *touch_core;
/*
gint flag;
*/

VEC3SET(sum1, 0.0, 0.0, 0.0);
VEC3SET(sum2, 0.0, 0.0, 0.0);

#define TOUCH_TOL 0.8

/* periodic images */
ni = pow(3, ms_data->periodic);
touch = 0;
touch_core = NULL;

/*
ni = 1;
P3VEC("cent: ", ms_data->centroid);
a = r[0]*r[0] + r[1]*r[1];
if (a < FRACTION_TOLERANCE)
  {
  flag = 1;
  P3VEC("ms_grad: ", r);
  }
else
  flag = 0;
*/

for (item=list ; item ; item=g_slist_next(item))
  {
  core = (struct core_pak *) item->data;

  a = ms_data->csurf.prad + elements[core->atom_code].vdw;

  for (m=ni ; m-- ; )
    {
/* periodic images (NB: only works for periodicity = 2) */
    i = (m % 3) - 0.5*ms_data->periodic;
    j = (m / 3) - 0.5*ms_data->periodic;

/* get separation vector */
      ARR3SET(x, core->x);

/*
i = j = 0;
*/

    VEC3ADD(x, i, j, 0);
    vecmat(ms_data->latmat, x);

/*
      ARR3SUB(x, r);
      dx = VEC3MAG(x);
*/

    ARR3SET(tmp, x);
    ARR3SUB(tmp, point->rx);
    dx = VEC3MAG(tmp);

/* vector components of the gradient */
    e = exp(-(dx - a)/ms_data->csurf.dsize);

/*
if (flag)
  {
  if (e > 0.1)
    printf("core: %p  (i,j = %d,%d)  e = %f\n", core, i, j, e);
  }
*/
  
    VEC3ADD(sum1, e, e, e);

    e /= (dx * ms_data->csurf.dsize);

    sum2[0] += e * (x[0] - point->rx[0]);
    sum2[1] += e * (x[1] - point->rx[1]);
    sum2[2] += e * (x[2] - point->rx[2]);

    if (e > TOUCH_TOL)
      {
/* high contribution -> touching */
      touch_core = core;
      touch++;
      }
    }
  }

/* compute overall normal */
for (i=3 ; i-- ; )
  point->nx[i] = sum2[i]/sum1[i];

normalize(point->nx, 3);

/* only return a core if it was the only one touching the supplied point */
if (touch == 1)
  return(touch_core);
return(NULL);
}

/**********************************************/
/* nail vec by a VERTICAL drop to the surface */
/**********************************************/
gdouble ms_drop_vec(gdouble *vec, struct model_pak *model)
{
gint i, j, m, flag;
gdouble z, zmax, dist, dz, x[3];
GSList *list;
struct core_pak *core;

g_assert(ms_data->periodic == 2);

zmax = -999999999.9;
flag = 0;
for (list=model->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;

  dist = ms_data->csurf.prad + elements[core->atom_code].vdw;

/* periodic images */
  for (m=0 ; m<9 ; m++)
    {
    i = (m % 3) - 1;
    j = (m / 3) - 1;
/* periodic image */
    ARR3SET(x, core->x);
    VEC3ADD(x, i, j, 0);

    vecmat(ms_data->latmat, x);

/* save current core z */
    z = x[2];

/* check horizontal separation */
    ARR3SUB(x, vec);
    x[2] = 0.0;
    if (VEC3MAGSQ(x) < dist*dist)
      {
/* get z coord of probe if touching current core */
      z += sqrt(dist*dist - x[0]*x[0] - x[1]*x[1]);
      if (z > zmax)
        zmax = z;
      flag++;
      }
    }
  }

if (flag)
  {
  dz = vec[2] - zmax;
  vec[2] = zmax;
  }
else
  {
  printf("Error: failed to locate surface during drop.\n");
  dz = 0.0;
  }
 
return(dz);
}

/************************************************/
/* nail vec to the CLOSEST point on the surface */
/************************************************/
#define DEBUG_MS_NAIL_VEC 0
gdouble ms_nail_point(struct smv_pak *point, struct model_pak *data)
{
gdouble dist, disp[3];

#if DEBUG_MS_NAIL_VEC
P3VEC(" vec: ", point->rx);
#endif

/* get displacement vector */
dist = ms_dist(point->rx, data->cores);
ms_grad(point, data->cores);
ARR3SET(disp, point->nx);

VEC3MUL(disp, dist);

#define LIMIT_CROSSOVER 0
#if LIMIT_CROSSOVER
  {
gdouble ratio, sep, dp, x[3];
GSList *list;
struct smv_pak *point2;

/* NEW - check adjacencies */
for (list=point->adj ; list ; list=g_slist_next(list))
  {
  point2 = (struct smv_pak *) list->data;

/* get relative vector */
  ARR3SET(x, point2->rx);
  ARR3SUB(x, point->rx);
  sep = VEC3MAG(x);
  normalize(x, 3);

/* project displacement vector onto this */
  ARR3MUL(x, disp);
  dp = x[0] + x[1] + x[2];
  
/* prevent a displacement of more than half the separation */
  if (dp > 0.5*sep)
    {
printf("scaling back...\n");
    ratio = 0.5*sep*dist/dp;
    normalize(disp, 3);
    VEC3MUL(disp, ratio);
    }
  }
  }
#endif


/* relocate the point */
ARR3ADD(point->rx, disp);

#if DEBUG_MS_NAIL_VEC
printf("dist: %f\n", dist);
P3VEC(" vec: ", point->rx);
#endif

return(dist);
}

/***************************************/
/* create a triangular surface segment */
/***************************************/
struct smt_pak *new_triangle(struct smv_pak *p1, struct smv_pak *p2, struct smv_pak *p3)
{
struct smt_pak *triangle;

triangle = g_malloc(sizeof(struct smt_pak));

/* point references */
triangle->point[0] = p1;
triangle->point[1] = p2;
triangle->point[2] = p3;

return(triangle);
}

/*******************************/
/* allocate for a new midpoint */
/*******************************/
struct smv_pak *new_midpoint(struct smv_pak *p1, struct smv_pak *p2)
{
struct smv_pak *mp;

mp = g_malloc(sizeof(struct smv_pak));

/* init */
ARR3SET(mp->rx, p1->rx);
ARR3ADD(mp->rx, p2->rx);
VEC3MUL(mp->rx, 0.5);

mp->adj = NULL;
mp->adj = g_slist_prepend(mp->adj, p1);
mp->adj = g_slist_prepend(mp->adj, p2);

return(mp);
}

/*************************************************/
/* compute the real area of a triangular spatial */
/*************************************************/
gdouble triangle_area(struct smt_pak *triangle)
{
gdouble a[3], b[3], c[3];

/* form side vectors */
ARR3SET(a, triangle->point[1]->rx);
ARR3SUB(a, triangle->point[0]->rx);
ARR3SET(b, triangle->point[2]->rx);
ARR3SUB(b, triangle->point[0]->rx);

/* compute area via crossproduct */
crossprod(c, a, b);

return(0.5*VEC3MAG(c));
}

/****************************************/
/* calculate new molecular surface area */
/****************************************/
gdouble molsurf_area(GSList *tlist)
{
gdouble area=0.0;
GSList *item;
struct smt_pak *triangle;

for (item=tlist ; item ; item=g_slist_next(item))
  {
  triangle = (struct smt_pak *) item->data;

  area += triangle_area(triangle);
  }
return(area);
}

/**********************************/
/* molecular surface scaling type */
/**********************************/
GtkWidget *epot_vbox, *surf_epot_min, *surf_epot_max, *surf_epot_div;
GtkWidget *epot_grid_pts;

void epot_scale_sensitivity(GtkWidget *w, struct model_pak *model)
{
if (model->gulp.epot_autoscale)
  gtk_widget_set_sensitive(GTK_WIDGET(epot_vbox), FALSE);
else
  gtk_widget_set_sensitive(GTK_WIDGET(epot_vbox), TRUE);
}

/**************************/
/* common colour routines */
/**************************/
void ms_afm_colour(gdouble *colour, gdouble value, struct model_pak *data)
{
/* TODO - need AFM min/max stored in data */
}

void ms_epot_colour(gdouble *colour, gdouble value, struct model_pak *data)
{
gdouble r, g, b;

r = g = b = 1.0;

if (value < 0.0)
  {
  b *= 1.0 - value/data->gulp.epot_min;
  g *= 1.0 - value/data->gulp.epot_min;
  }
else
  {
  r *= 1.0 - value/data->gulp.epot_max;
  g *= 1.0 - value/data->gulp.epot_max;
  }

VEC3SET(colour, r, g, b);
}

/****************************/
/* min/max point calculator */
/****************************/
void ms_get_zlimits(gdouble *min, gdouble *max, GSList *list)
{
GSList *item;
struct smv_pak *point;

g_assert(list != NULL);

/* init */
point = list->data;
*min = *max = point->rx[2];
/* loop */
for (item=list ; item ; item=g_slist_next(item))
  {
  point = (struct smv_pak *) item->data;

  if (point->rx[2] < *min)
    *min = point->rx[2];
  if (point->rx[2] > *max)
    *max = point->rx[2];
  }
}

/*****************************************/
/* uses GULP's new epot list calc method */
/*****************************************/
gint ms_get_epot(GSList *list, struct model_pak *model)
{
gint run;
GSList *item;
struct vec_pak *vec;
struct smv_pak *point;

/* checks */
g_assert(model != NULL);

/* setup the site coordinate list */
if (model->gulp.epot_vecs)
  free_slist(model->gulp.epot_vecs);
model->gulp.epot_vecs = NULL;

/* fill out the list */
for (item=list ; item ; item=g_slist_next(item))
  {
  point = (struct smv_pak *) item->data;

  vec = g_malloc(sizeof(struct vec_pak));

  ARR3SET(vec->rx, point->rx);

  model->gulp.epot_vecs = g_slist_prepend(model->gulp.epot_vecs, vec);
  }
model->gulp.epot_vecs = g_slist_reverse(model->gulp.epot_vecs);

/* create a single point GULP input file */
run = model->gulp.run;
model->gulp.run = E_SINGLE;
write_gulp("gok.gin", model);
model->gulp.run = run;

/* run the GULP calc & get the results */
if (exec_gulp("gok.gin", "gok.got"))
  return(1);
if (read_gulp_output("gok.got", model))
  return(1);

/* failsafes (min & max are frequently on the denominator) */
if (model->gulp.epot_autoscale)
  {
  if (fabs(model->gulp.epot_min) < FRACTION_TOLERANCE)
    model->gulp.epot_min = -0.01;
  if (fabs(model->gulp.epot_max) < FRACTION_TOLERANCE)
    model->gulp.epot_max = 0.01;
  if (model->gulp.epot_divisions < 1)
    model->gulp.epot_divisions = 10;

/* update widget values */
  if (GTK_IS_ENTRY(surf_epot_min))
    gtk_entry_set_text(GTK_ENTRY(surf_epot_min), 
                       g_strdup_printf("%f", model->gulp.epot_min));

  if (GTK_IS_ENTRY(surf_epot_max))
    gtk_entry_set_text(GTK_ENTRY(surf_epot_max), 
                       g_strdup_printf("%f", model->gulp.epot_max));
  }
else
  {
/* get values from widgets */
  if (GTK_IS_ENTRY(surf_epot_div))
    model->gulp.epot_divisions = (gint) str_to_float
                                (gtk_entry_get_text(GTK_ENTRY(surf_epot_div)));

  if (GTK_IS_ENTRY(surf_epot_min))
    model->gulp.epot_min = str_to_float
                          (gtk_entry_get_text(GTK_ENTRY(surf_epot_min)));
  if (GTK_IS_ENTRY(surf_epot_max))
      model->gulp.epot_max = str_to_float(gtk_entry_get_text
                            (GTK_ENTRY(surf_epot_max)));
  }

return(0);
}

/*************************/
/* colour the point list */
/*************************/
/* NEW - replacement for colour_molsurf() */
void ms_colour_points(GSList *list, struct model_pak *model)
{
gdouble z, zmin=0.0, zmax=0.0;
gdouble *q;
GSList *item, *item2=NULL;
struct smv_pak *point;
struct core_pak *core;

/* setup */
switch (model->csurf.colour_method)
  {
  case MS_EPOT:
    if (ms_get_epot(list, model))
      {
      show_text(ERROR, "Electrostatic calculation failed.\n");
/* destroy spatials? */
/*
      del_molsurf(model);
      return;
*/
/* or, colour them a default colour (grey) */
      break;
      }
    item2 = model->gulp.epot_vals;
    break;

  case MS_AFM:
    ms_get_zlimits(&zmin, &zmax, list);
    break;
  }

/* assign colour */
for (item=list ; item ; item=g_slist_next(item))
  {
  point = (struct smv_pak *) item->data;

/* compute point normal */
/* NB: this is required for 2D surfaces, since the vertex normals */
/* have not been calculated before this point */
  core = ms_grad(point, model->cores);

  switch (model->csurf.colour_method)
    {
    case MS_AFM:
      z = point->rx[2] - zmin;
      z *= 1.0/(zmax - zmin);
      VEC3SET(point->colour, 0.7*z+0.2, 0.8*z, 0.5*z*z*z*z);
      break;

    case MS_EPOT:
      if (item2)
        {
        q = (gdouble *) item2->data;
        ms_epot_colour(point->colour, *q, model);
        item2 = g_slist_next(item2);
        }
      else
        {
/* default colour (ie electrostatic calc has failed) */
        VEC3SET(point->colour, 0.5, 0.5, 0.5);
        }
      break;

    case MS_TOUCH:
/* assign colour if one (and only one) atom was "touched" */
      if (core)
        {
        ARR3SET(point->colour, core->colour);
        VEC3MUL(point->colour, 1.0/65535.0);
        }
      else
        {
/* default colour */
        ARR3SET(point->colour, sysenv.render.rsurf_colour);
        }
      break;
    }
  }
}

/*******************************************/
/* test if a point "punctures" the molsurf */
/*******************************************/
/* return a clipped dx if molsurf would be punctured by the drop */
void ms_test_point(struct smv_pak *point, struct model_pak *model)
{
GSList *list;
gdouble u, v, norm[3], res[3], x[3];
struct core_pak *core;

/* get point surface normal */
ms_data->csurf.prad += 0.1;
ms_grad(point, model->cores);
ARR3SET(norm, point->nx);
ms_data->csurf.prad -= 0.1;

/* loop over all atoms */
for (list=model->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;

/* get raw cartesian coords */
  ARR3SET(x, core->x);
  vecmat(model->latmat, x);

/* project separation vector onto surface */
  ARR3SUB(x, point->rx);
  proj_vop(res, x, norm);
  u = VEC3MAG(res);

/* project separation vector onto surface normal */
  ARR3MUL(x, norm);
  v = VEC3MAG(x);

printf("u,v = %f,%f\n", u, v);

  }

/* if close enough - test if within point adjacency "sphere" */

/* constrain so no atoms "puncture" the molsurf */

}

/**********************************************/
/* test if a triangle "punctures" the molsurf */
/**********************************************/
/* return's NULL if ok, else a new midpoint */
struct smv_pak *ms_test_triangle(struct smt_pak *triangle, struct model_pak *model)
{
return(NULL);
}

/***********************************************************/
/* nail/drop (ie refine) all points comprising the surface */
/***********************************************************/
void ms_mold_points(GSList *points, struct model_pak *model)
{
gdouble drop=0.0;
GSList *list;
struct smv_pak *point;

for (list=points ; list ; list=g_slist_next(list))
  {
  point = (struct smv_pak *) list->data;

  if (model->periodic == 2)
    drop += ms_drop_vec(point->rx, model);
  else
    drop += ms_nail_point(point, model);
  }
}

/**************************************************/
/* attempt to locate & remove surface "punctures" */
/**************************************************/
gint ms_heal_edges(GSList *points, struct model_pak *model)
{
gint n, p=0;
gdouble cut2, x[3], dx[3];
GSList *list1, *list2, *list3;
struct smv_pak *point1, *point2;
struct core_pak *core;

/* even spacing */
/* for all points in the current mesh */
for (list1=points ; list1 ; list1=g_slist_next(list1))
  {
  point1 = (struct smv_pak *) list1->data;

/* ignore anything with too few adjacent points */
  n = g_slist_length(point1->adj);
/* NB: edge points shouldn't have 6 adjacencies, so 2D should be ok */
  if (n < 6)
    continue;

/* get centroid */
  for (list2=point1->adj ; list2 ; list2=g_slist_next(list2))
    {
    point2 = (struct smv_pak *) list2->data;

/* unique edges only */
    if (point2 < point1)
      continue;

    ARR3SET(x, point1->rx);
    ARR3ADD(x, point2->rx);
    VEC3MUL(x, 0.5);

    for (list3=model->cores ; list3 ; list3=g_slist_next(list3))
      {
      core = (struct core_pak *) list3->data;

/*
    cut2 = model->csurf.prad + elements[core->atom_code].vdw;
*/
/* don't want cutoff to be too stringent */
      cut2 = elements[core->atom_code].vdw;
      cut2 *= cut2;
  
      ARR3SET(dx, core->x);
      vecmat(model->latmat, dx);
      ARR3SUB(dx, x);

      if (VEC3MAGSQ(dx) < cut2)
        {
        p++;

/* lift the points above the surface */
        ARR3SUB(point1->rx, point1->nx); 
        ARR3SUB(point1->rx, point2->nx); 
        ARR3SUB(point2->rx, point1->nx); 
        ARR3SUB(point2->rx, point2->nx); 
        ARR3SUB(x, point2->nx); 
        ARR3SUB(x, point1->nx); 

/* nail the new point location */
/*
        if (model->periodic == 2)
          {
          ms_drop_vec(point1->rx, model);
          ms_drop_vec(point2->rx, model);
          }
        else
          {
          ms_nail_point(point1, model);
          ms_nail_point(point2, model);
          }
*/
        }
      }
    }
  }
return(p);
}

/**********************************************/
/* attempt to maximally space the mesh points */
/**********************************************/
/* method 1 */
void ms_space_points(GSList *points, struct model_pak *model)
{
gint n;
GSList *list1, *list2;
struct smv_pak *point1, *point2;

/* even spacing */
/* for all points in the current mesh */
for (list1=points ; list1 ; list1=g_slist_next(list1))
  {
  point1 = (struct smv_pak *) list1->data;

/* ignore anything with too few adjacent points */
  n = g_slist_length(point1->adj);
/* NB: edge points shouldn't have 6 adjacencies, so 2D should be ok */
  if (n < 6)
    continue;

/* get centroid */
  for (list2=point1->adj ; list2 ; list2=g_slist_next(list2))
    {
    point2 = (struct smv_pak *) list2->data;

    ARR3ADD(point1->rx, point2->rx);
    }
  n++;
  VEC3MUL(point1->rx, 1.0/(gdouble) n);

/* nail the new point location */
  if (model->periodic == 2)
    ms_drop_vec(point1->rx, model);
  else
    ms_nail_point(point1, model);
  }
}

/**********************************************/
/* attempt to maximally space the mesh points */
/**********************************************/
/* method 2 */
void ms_space_points2(GSList *points, struct model_pak *model)
{
gint n;
gdouble avg, len, x[3], dx[3];
GSList *list1, *list2, *list3;
struct smv_pak *point1, *point2;

/* even spacing */
/* for all points in the current mesh */
for (list1=points ; list1 ; list1=g_slist_next(list1))
  {
  point1 = (struct smv_pak *) list1->data;

/* ignore anything with too few adjacent points */
  n = g_slist_length(point1->adj);
/* NB: edge points shouldn't have 6 adjacencies, so 2D should be ok */
  if (n < 6)
    {

  if (model->periodic == 2)
    ms_drop_vec(point1->rx, model);
  else
    ms_nail_point(point1, model);

    continue;
    }

/* compute average distance (squared) to adjacent points */
  avg = 0.0;
  for (list2=point1->adj ; list2 ; list2=g_slist_next(list2))
    {
    point2 = (struct smv_pak *) list2->data;

    ARR3SET(x, point1->rx);
    ARR3SUB(x, point2->rx);

    avg += VEC3MAG(x);
    }
  avg *= 1.0/(gdouble) n;

/* sum difference vectors between actual and average length */
  VEC3SET(dx, 0.0, 0.0, 0.0);
  for (list3=point1->adj ; list3 ; list3=g_slist_next(list3))
    {
    point2 = (struct smv_pak *) list3->data;

    ARR3SET(x, point1->rx);
    ARR3SUB(x, point2->rx);

    len = VEC3MAG(x);

    VEC3MUL(x, 1.0/len);

    VEC3MUL(x, (avg-len));

    ARR3ADD(dx, x);
    }

  normalize(dx, 3);
  VEC3MUL(dx, 0.1*avg);

/* apply sum of difference vectors to the central point */
  ARR3ADD(point1->rx, dx);

  if (model->periodic == 2)
    ms_drop_vec(point1->rx, model);
  else
    ms_nail_point(point1, model);
  }
}

/******************************/
/* test for vertex puncturing */
/******************************/
gint ms_test_vertex_punctures(GSList *points, struct model_pak *model)
{
gint p=0;
gdouble cut2, dx[3];
GSList *list1, *list2;
struct smv_pak *point1;
struct core_pak *core;

for (list1=points ; list1 ; list1=g_slist_next(list1))
  {
  point1 = (struct smv_pak *) list1->data;

  for (list2=model->cores ; list2 ; list2=g_slist_next(list2))
    {
    core = (struct core_pak *) list2->data;

/*
    cut2 = model->csurf.prad + elements[core->atom_code].vdw;
*/
/* don't want cutoff to be too stringent */
    cut2 = elements[core->atom_code].vdw;
    cut2 *= cut2;

    ARR3SET(dx, core->x);
    vecmat(model->latmat, dx);
    ARR3SUB(dx, point1->rx);

    if (VEC3MAGSQ(dx) < cut2)
      {
      p++;
      }
    }
  }
return(p);
}

/***************************************************/
/* attempt to locate & remove triangle "punctures" */
/***************************************************/
gint ms_heal_vertex_punctures(GSList *points, struct model_pak *model)
{
gint n, p=0;
gdouble cut2, cent[3], dx[3];
GSList *list1, *list2, *list3;
struct core_pak *core;
struct smv_pak *point, *point2;

/* loop over triangle midpoints */
for (list1=points ; list1 ; list1=g_slist_next(list1))
  {
  point = (struct smv_pak *) list1->data;

  for (list2=model->cores ; list2 ; list2=g_slist_next(list2))
    {
    core = (struct core_pak *) list2->data;

/*
    cut2 = model->csurf.prad + elements[core->atom_code].vdw;
*/
/* don't want cutoff to be too stringent */
    cut2 = elements[core->atom_code].vdw;
    cut2 *= cut2;

    ARR3SET(dx, core->x);
    vecmat(model->latmat, dx);
    ARR3SUB(dx, point->rx);

    if (VEC3MAGSQ(dx) < cut2)
      {
      p++;

/* leave edges alone - we may not even need to correct them... */
  if (g_slist_length(point->adj) < 6)
    continue;

/* get centroid of adjacencies */
  n = 0;
  VEC3SET(cent, 0.0, 0.0, 0.0);
  for (list3=point->adj ; list3 ; list3=g_slist_next(list3))
    {
    point2 = (struct smv_pak *) list3->data;

    ARR3ADD(cent, point2->rx);
    n++;
    }
  VEC3MUL(cent, 1.0/(gdouble) n);

/* check vector to centroid is outward facing */
/* NB: -nx is vertex normal */
   if (via(cent, point->nx, 3) > PI)
     {
/* outward */
     ARR3SET(point->rx, cent);
     }
   else
     {
/* inward */
     ARR3SUB(cent, point->rx);
     VEC3MUL(cent, -1.0);
     ARR3ADD(point->rx, cent);
     }


      }
    }
  }
return(p);
}

/***************************************************/
/* attempt to locate & remove triangle "punctures" */
/***************************************************/
gint ms_heal_punctures(GSList *triangles, struct model_pak *model)
{
gint n, p;
gdouble cut2, x[3], dx[3], nx[3];
GSList *list1, *list2;
struct core_pak *core;
struct smt_pak *triangle;

/* loop over triangle midpoints */
p=0;
for (list1=triangles ; list1 ; list1=g_slist_next(list1))
  {
  triangle = (struct smt_pak *) list1->data;

  VEC3SET(x, 0.0, 0.0, 0.0);
  for (n=3 ; n-- ; )
    {
    ARR3ADD(x, triangle->point[n]->rx);
    }
  VEC3MUL(x, 0.333333333);

  for (list2=model->cores ; list2 ; list2=g_slist_next(list2))
    {
    core = (struct core_pak *) list2->data;

/*
    cut2 = model->csurf.prad + elements[core->atom_code].vdw;
*/
/* don't want cutoff to be too stringent */
    cut2 = elements[core->atom_code].vdw;
    cut2 *= cut2;

    ARR3SET(dx, core->x);
    vecmat(model->latmat, dx);
    ARR3SUB(dx, x);

    if (VEC3MAGSQ(dx) < cut2)
      {
      p++;

/* compute average normal */
      ARR3SET(nx, triangle->point[0]->nx);
      ARR3ADD(nx, triangle->point[1]->nx);
      ARR3ADD(nx, triangle->point[2]->nx);

      VEC3MUL(nx, -0.333333333 * elements[core->atom_code].vdw);

/* lift the triangle above the surface */
      ARR3ADD(triangle->point[0]->rx, nx); 
      ARR3ADD(triangle->point[1]->rx, nx); 
      ARR3ADD(triangle->point[2]->rx, nx); 

      ARR3ADD(x, nx);
      }
    }
  }
return(p);
}

/*****************************************/
/* initialization for isolated molecules */
/*****************************************/
void fn_surf_init_iso(GSList **plist, GSList **tlist, struct model_pak *model)
{
struct smv_pak *p1, *p2, *p3, *p4, *p5;
struct smt_pak *triangle;

/* point init */
p1 = g_malloc(sizeof(struct smv_pak));
p2 = g_malloc(sizeof(struct smv_pak));
p3 = g_malloc(sizeof(struct smv_pak));
p4 = g_malloc(sizeof(struct smv_pak));
p5 = g_malloc(sizeof(struct smv_pak));

/* NEW - with the ms_test_point() scheme, molsurf must start completely outside mol */
VEC3SET(p1->rx,  0.00,  2.00,  0.00);
VEC3SET(p2->rx,  1.73, -1.00,  0.00);
VEC3SET(p3->rx, -1.73, -1.00,  0.00);
VEC3SET(p4->rx,  0.00,  0.00,  1.73);
VEC3SET(p5->rx,  0.00,  0.00, -1.73);

VEC3MUL(p1->rx, model->rmax);
VEC3MUL(p2->rx, model->rmax);
VEC3MUL(p3->rx, model->rmax);
VEC3MUL(p4->rx, model->rmax);
VEC3MUL(p5->rx, model->rmax);

ARR3ADD(p1->rx, model->centroid);
ARR3ADD(p2->rx, model->centroid);
ARR3ADD(p3->rx, model->centroid);
ARR3ADD(p4->rx, model->centroid);
ARR3ADD(p5->rx, model->centroid);

/* construct adj list */
p1->adj = NULL;
p2->adj = NULL;
p3->adj = NULL;
p4->adj = NULL;
p5->adj = NULL;
p1->adj = g_slist_prepend(p1->adj, p2);
p1->adj = g_slist_prepend(p1->adj, p3);
p1->adj = g_slist_prepend(p1->adj, p4);
p1->adj = g_slist_prepend(p1->adj, p5);

p2->adj = g_slist_prepend(p2->adj, p1);
p2->adj = g_slist_prepend(p2->adj, p3);
p2->adj = g_slist_prepend(p2->adj, p4);
p2->adj = g_slist_prepend(p2->adj, p5);

p3->adj = g_slist_prepend(p3->adj, p1);
p3->adj = g_slist_prepend(p3->adj, p2);
p3->adj = g_slist_prepend(p3->adj, p4);
p3->adj = g_slist_prepend(p3->adj, p5);

p4->adj = g_slist_prepend(p4->adj, p1);
p4->adj = g_slist_prepend(p4->adj, p2);
p4->adj = g_slist_prepend(p4->adj, p3);

p5->adj = g_slist_prepend(p5->adj, p1);
p5->adj = g_slist_prepend(p5->adj, p2);
p5->adj = g_slist_prepend(p5->adj, p3);

/* construct point list */
*plist = g_slist_prepend(*plist, p1);
*plist = g_slist_prepend(*plist, p2);
*plist = g_slist_prepend(*plist, p3);
*plist = g_slist_prepend(*plist, p4);
*plist = g_slist_prepend(*plist, p5);

/* NB: not doing this right away -> more triangles when the 1st nail */
/* is done which gives a better change of avoiding a bad start */
/*
ms_nail_point(p1, model);
ms_nail_point(p2, model);
ms_nail_point(p3, model);
ms_nail_point(p4, model);
ms_nail_point(p5, model);
*/

/* construct triangle list */

/* facet */
triangle = new_triangle(p2, p1, p4);
*tlist = g_slist_prepend(*tlist, triangle);

/* facet */
triangle = new_triangle(p3, p2, p4);
*tlist = g_slist_prepend(*tlist, triangle);

/* facet */
triangle = new_triangle(p1, p3, p4);
*tlist = g_slist_prepend(*tlist, triangle);

/* facet */
triangle = new_triangle(p1, p2, p5);
*tlist = g_slist_prepend(*tlist, triangle);

/* facet */
triangle = new_triangle(p2, p3, p5);
*tlist = g_slist_prepend(*tlist, triangle);

/* facet */
triangle = new_triangle(p3, p1, p5);
*tlist = g_slist_prepend(*tlist, triangle);
}

/**********************************/
/* initialization for 2D surfaces */
/**********************************/
void fn_surf_init_2d(GSList **plist, GSList **tlist, struct model_pak *model)
{
struct smv_pak *p1, *p2, *p3, *p4, *p5;
struct smt_pak *triangle;

p1 = g_malloc(sizeof(struct smv_pak));
p2 = g_malloc(sizeof(struct smv_pak));
p3 = g_malloc(sizeof(struct smv_pak));
p4 = g_malloc(sizeof(struct smv_pak));
p5 = g_malloc(sizeof(struct smv_pak));

/* TODO - might need to scan for highest z */
VEC3SET(p1->rx,  0.00,  0.00,  0.0);
VEC3SET(p2->rx,  1.00,  0.00,  0.0);
VEC3SET(p3->rx,  0.00,  1.00,  0.0);
VEC3SET(p4->rx,  1.00,  1.00,  0.0);
VEC3SET(p5->rx,  0.50,  0.50,  0.0);

vecmat(model->latmat, p1->rx);
vecmat(model->latmat, p2->rx);
vecmat(model->latmat, p3->rx);
vecmat(model->latmat, p4->rx);
vecmat(model->latmat, p5->rx);

/* construct adj list */
p1->adj = NULL;
p2->adj = NULL;
p3->adj = NULL;
p4->adj = NULL;
p5->adj = NULL;

p1->adj = g_slist_prepend(p1->adj, p2);
p1->adj = g_slist_prepend(p1->adj, p3);
p1->adj = g_slist_prepend(p1->adj, p5);

p2->adj = g_slist_prepend(p2->adj, p1);
p2->adj = g_slist_prepend(p2->adj, p4);
p2->adj = g_slist_prepend(p2->adj, p5);

p3->adj = g_slist_prepend(p3->adj, p1);
p3->adj = g_slist_prepend(p3->adj, p4);
p3->adj = g_slist_prepend(p3->adj, p5);

p4->adj = g_slist_prepend(p4->adj, p2);
p4->adj = g_slist_prepend(p4->adj, p3);
p4->adj = g_slist_prepend(p4->adj, p5);

p5->adj = g_slist_prepend(p5->adj, p1);
p5->adj = g_slist_prepend(p5->adj, p2);
p5->adj = g_slist_prepend(p5->adj, p3);
p5->adj = g_slist_prepend(p5->adj, p4);

/* construct point list */
*plist = g_slist_prepend(*plist, p1);
*plist = g_slist_prepend(*plist, p2);
*plist = g_slist_prepend(*plist, p3);
*plist = g_slist_prepend(*plist, p4);
*plist = g_slist_prepend(*plist, p5);

/* construct triangle list */
/* facet */
triangle = new_triangle(p2, p5, p1);
*tlist = g_slist_prepend(*tlist, triangle);

/* facet */
triangle = new_triangle(p4, p5, p2);
*tlist = g_slist_prepend(*tlist, triangle);

/* facet */
triangle = new_triangle(p3, p5, p4);
*tlist = g_slist_prepend(*tlist, triangle);

/* facet */
triangle = new_triangle(p5, p3, p1);
*tlist = g_slist_prepend(*tlist, triangle);
}

/*********************************/
/* compute the molecular surface */
/*********************************/
#define DEBUG_CALC_MOLSURF 0
void calc_molsurf(GtkWidget *w, struct model_pak *data)
{
gint i, invert, depth, mask, t_num;
gchar *text;
gdouble area1, radius, t_rms, dx[3];
GSList *item1, *item2, *plist1, *plist2, *tlist1;
struct smv_pak *point, *point1, *point2, *point3, *midpt12, *midpt23, *midpt13;
struct smt_pak *triangle, *triangle1, *triangle2;
struct vec_pak *vec;
struct spatial_pak *spatial;

ms_data = data;

/* remove old molsurf (if any) */
delete_spatial_type(SPATIAL_MOLSURF, data);
/* rescale, so mesh init gets the correct dimensions */
init_objs(CENT_COORDS, data);

/* init the triangular mesh */
plist1=tlist1=NULL;
switch (data->periodic)
  {
  case 0:
    fn_surf_init_iso(&plist1, &tlist1, data);
    break;

  case 2:
    fn_surf_init_2d(&plist1, &tlist1, data);
    break;

  default:
    show_text(WARNING, "Unsupported model periodicity.\n");
  }

#if DEBUG_CALC_MOLSURF
printf("req depth = %d\n", (gint) data->csurf.depth);
printf("# points = %d\n", g_slist_length(plist1));
printf("# triangles = %d\n", g_slist_length(tlist1));
#endif

/* store target probe radius */
radius = data->csurf.prad;

/* main sub-division loop */
i = 0;
for (depth=0 ; depth < (gint) data->csurf.depth ; depth++)
  {

t_rms = 0.0;
t_num = 0;

/* compute new point list (midpoints) */
  plist2 = NULL;
  for (item1=plist1 ; item1 ; item1=g_slist_next(item1))
    {
    point1 = (struct smv_pak *) item1->data;
    item2 = point1->adj;
    while (item2)
      {
      point2 = (struct smv_pak *) item2->data;
      item2 = g_slist_next(item2);

/* avoid double counting */
      if (point1 < point2)
        {
        point3 = new_midpoint(point1, point2);

/* compute approx. triangle size */
        ARR3SET(dx, point3->rx);
        ARR3SUB(dx, point1->rx);
        t_rms += VEC3MAGSQ(dx);
        t_num++;

        plist2 = g_slist_prepend(plist2, point3);
        }
      }
    }

/* compute approx. triangle size */
t_rms /= (9.0*t_num);
t_rms = sqrt(t_rms);

/* adjust the probe radius (wrt triangle size) to avoid surface punctures */
if (t_rms > radius)
  data->csurf.prad = t_rms;
else
  data->csurf.prad = radius;

#if DEBUG_CALC_MOLSURF
printf("setting probe size: %f\n", data->csurf.prad);
#endif

/* for all midpoints, update source point adjacencies */
  for (item1=plist2 ; item1 ; item1=g_slist_next(item1))
    {
    midpt12 = (struct smv_pak *) item1->data;

    point1 = g_slist_nth_data(midpt12->adj, 0);
    point2 = g_slist_nth_data(midpt12->adj, 1);

    point1->adj = g_slist_prepend(point1->adj, midpt12);
    point2->adj = g_slist_prepend(point2->adj, midpt12);
    point1->adj = g_slist_remove(point1->adj, point2);
    point2->adj = g_slist_remove(point2->adj, point1);
    }

/* compute new triangle list */
  for (item1=tlist1 ; item1 ; item1=g_slist_next(item1))
    {
    triangle1 = (struct smt_pak *) item1->data;
/*
printf("tri: %p\n", triangle1);
printf("nmp: %d\n", g_slist_length(plist2));
*/

    point1 = triangle1->point[0];
    point2 = triangle1->point[1];
    point3 = triangle1->point[2];

/* find the midpoints by searching through the newly created midpoint list */
    midpt12 = midpt13 = midpt23 = NULL;
    for (item2=plist2 ; item2 ; item2=g_slist_next(item2))
      {
      point = (struct smv_pak *) item2->data;

      mask = 0;
      if (g_slist_find(point->adj, point1))
        mask |= 1;
      if (g_slist_find(point->adj, point2))
        mask |= 2;
      if (g_slist_find(point->adj, point3))
        mask |= 4;

      switch(mask)
        {
        case 3:
          midpt12 = point;
          break;
        case 5:
          midpt13 = point;
          break;
        case 6:
          midpt23 = point;
          break;
        }
      }

/* TODO - midpoints should always be found, but if not */
/* is there a better way to handle these cases? */
    g_assert(midpt12 != NULL);
    g_assert(midpt13 != NULL);
    g_assert(midpt23 != NULL);

/* triangle 1 */
    triangle2 = new_triangle(midpt12, point2, midpt23);
    tlist1 = g_slist_prepend(tlist1, triangle2);

/* triangle 2 */
    triangle2 = new_triangle(midpt13, midpt23, point3);
    tlist1 = g_slist_prepend(tlist1, triangle2);

/* triangle 3 */
    triangle2 = new_triangle(midpt13, midpt12, midpt23);
    tlist1 = g_slist_prepend(tlist1, triangle2);

/* triangle 4 - alter the old one */
    triangle1->point[1] = midpt12;
    triangle1->point[2] = midpt13;

/* new adjacencies */
    midpt12->adj = g_slist_prepend(midpt12->adj, midpt23);
    midpt12->adj = g_slist_prepend(midpt12->adj, midpt13);
    midpt13->adj = g_slist_prepend(midpt13->adj, midpt12);
    midpt13->adj = g_slist_prepend(midpt13->adj, midpt23);
    midpt23->adj = g_slist_prepend(midpt23->adj, midpt12);
    midpt23->adj = g_slist_prepend(midpt23->adj, midpt13);
    }

/* join the two point lists */
  plist1 = g_slist_concat(plist1, plist2);

/* space & nail the points */
/* TODO - combine for speedup? */
  ms_mold_points(plist1, data);

  ms_space_points2(plist1, data);

/*
  ms_space_points(plist1, data);
  ms_space_points2(plist1, data);
  i = ms_heal_punctures(tlist1, data);
  i = ms_heal_edges(plist1, data);
  i = ms_heal_vertex_punctures(plist1, data);
  printf("Punctures: %d\n", i);
*/

  }

/*NB: this computes surface normals as well - important for 2D cases */
/* as normals HAVE NOT been calculated before this due to drop mechanism */
ms_colour_points(plist1, data);

/* lower the surface by the probe radius -> give mol. surf rather than acc. surf */
if (data->csurf.type == MS_MOLECULAR)
  {
  for (item1=plist1 ; item1 ; item1=g_slist_next(item1))
    {
    point1 = (struct smv_pak *) item1->data;

/* below is a means to conserve the surface cell, but it gets the edge */
/* normals wrong - so don't worry - the surface is still correct, even */
/* if it does have wavy boundaries */
/*
  if (data->periodic == 2)
    {
    gint flag = 0;
    ARR3SET(dx, point1->rx);
    vecmat(data->ilatmat, dx);

    if (dx[0] < FRACTION_TOLERANCE)
      flag++;
    if (dx[1] < FRACTION_TOLERANCE)
      flag++;
    if (dx[0] > 1.0-FRACTION_TOLERANCE)
      flag++;
    if (dx[1] > 1.0-FRACTION_TOLERANCE)
      flag++;
    }

  if (flag)
    point1->rx[2] -= data->csurf.prad;
  else
    {
    ARR3SET(dx, point1->nx);
    VEC3MUL(dx, data->csurf.prad);
    ARR3ADD(point1->rx, dx);
    }
*/
    ARR3SET(dx, point1->nx);
    VEC3MUL(dx, data->csurf.prad);
    ARR3ADD(point1->rx, dx);
    }
  }

/* get the area */
/* TODO - volume etc. */
area1 = molsurf_area(tlist1);

/* output summary */
text = g_strdup_printf("Surface area = %f A^2\n", area1);
show_text(STANDARD, text);

#if DEBUG_CALC_MOLSURF
printf("SA = %f\n", area1);
printf("# points = %d\n", g_slist_length(plist1));
printf("# triangles = %d\n", g_slist_length(tlist1));
#endif

/* reverse vertex ordering for inverted lattice matrices */
if (det(data->latmat) < 0.0)
  invert = 1;
else
  invert = 0;

/* NB: best to calc the normals here - so it's indep. of the colouring method */
for (item1=tlist1 ; item1 ; item1=g_slist_next(item1))
  {
  triangle = (struct smt_pak *) item1->data;

/* init a new spatial */
  spatial = g_malloc(sizeof(struct spatial_pak));
  spatial->type = SPATIAL_MOLSURF;

/* fill out the vertex list */
  spatial->data = NULL;
  for (i=3 ; i-- ; )
    {
    vec = g_malloc(sizeof(struct vec_pak));

/* copy vertex colour, coords & normal */
    ARR3SET(vec->colour, triangle->point[i]->colour);
    ARR3SET(vec->x, triangle->point[i]->rx);
    ARR3SET(vec->n, triangle->point[i]->nx);
    VEC3MUL(vec->n, -1.0);

    vecmat(data->ilatmat, vec->n);
    vecmat(data->ilatmat, vec->x);

/* display the normal */
/*
fn_make_spatial(SPATIAL_VECTOR, vec->x, vec->n, data);
*/

    spatial->data = g_slist_prepend(spatial->data, vec);
    }
/* vertex ordering */
  if (invert)
    spatial->data = g_slist_reverse(spatial->data);

  data->spatial = g_slist_prepend(data->spatial, spatial);
  }


/* free the point & triangle lists */
free_slist(plist1);
free_slist(tlist1);

/* display */
init_objs(CENT_COORDS, data);
/*
colour_molsurf(data);
*/
data->csurf_on = TRUE;
redraw_canvas(SINGLE);
}

/************************/
/* simple deletion hook */
/************************/
void del_molsurf(GtkWidget *w, struct model_pak *data)
{
/* remove any previous surfaces */
delete_spatial_type(SPATIAL_MOLSURF, data);
data->csurf_on = FALSE;
redraw_canvas(SINGLE);
}

/*******************************************/
/* Molecular surface colour mode selection */
/*******************************************/
void ms_colour_mode(GtkWidget *entry)
{
const gchar *tmp;
struct model_pak *data;

/* checks */
data = model_ptr(sysenv.active, RECALL);
if (!data)
  return;

g_assert(entry != NULL);

tmp = gtk_entry_get_text(GTK_ENTRY(entry));

if (g_ascii_strncasecmp(tmp,"Touch",5) == 0)
  data->csurf.colour_method = MS_TOUCH;
if (g_ascii_strncasecmp(tmp,"AFM",3) == 0)
  data->csurf.colour_method = MS_AFM;
if (g_ascii_strncasecmp(tmp,"Electrostatic",13) == 0)
  data->csurf.colour_method = MS_EPOT;

redraw_canvas(ALL);
}

/*************/
/* callbacks */
/*************/
void cb_ms_accessible(struct model_pak *model)
{
model->csurf.type = MS_ACCESSIBLE;
}

void cb_ms_molecular(struct model_pak *model)
{
model->csurf.type = MS_MOLECULAR;
}

/***************************/
/* Molecular surface setup */
/***************************/
void surf_dialog()
{
gint id;
GtkWidget *frame, *vbox, *hbox, *button, *combo, *label;
GList *list;
struct dialog_pak *surf_dialog;
struct model_pak *data;

/* checks */
data = model_ptr(sysenv.active, RECALL);
if (!data)
  return;
if (data->id == MORPH)
  return;
if (data->periodic == 3)
  return;

/* create dialog */
if ((id = request_dialog(sysenv.active, SURF)) < 0)
  return;
surf_dialog = &sysenv.dialog[id];

surf_dialog->win = gtk_dialog_new();
g_signal_connect(GTK_OBJECT(surf_dialog->win), "destroy",
                 GTK_SIGNAL_FUNC(event_close_dialog), (gpointer) id);
gtk_window_set_title(GTK_WINDOW(surf_dialog->win), "Molecular surfaces");


/* frame for spinner setup */
frame = gtk_frame_new (NULL);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(surf_dialog->win)->vbox),frame,FALSE,TRUE,0); 
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);

gtksh_direct_spin("D size", &data->csurf.dsize, 0.01, 1.0, 0.05, NULL, NULL, vbox);
gtksh_direct_spin("Triangulation depth", &data->csurf.depth, 0, 10, 1, NULL, NULL, vbox);
gtksh_direct_spin("Probe radius", &data->csurf.prad, 0.1, 10.0, 0.01, NULL, NULL, vbox);

/* surface contour type */
frame = gtk_frame_new ("Surface type");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(surf_dialog->win)->vbox),frame,FALSE,TRUE,0); 
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);

new_radio_group(0, vbox, TT);

button = add_radio_button("Accessible surface", (gpointer) cb_ms_accessible, data);
if (data->csurf.type == MS_ACCESSIBLE)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

button = add_radio_button("Molecular surface", (gpointer) cb_ms_molecular, data);
if (data->csurf.type == MS_MOLECULAR)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* surface colour scheme */
frame = gtk_frame_new ("Colour method");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(surf_dialog->win)->vbox),frame,FALSE,TRUE,0); 
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, PANEL_SPACING);

/* combo box for colour mode */
list = NULL;
list = g_list_prepend(list, "Touch");
list = g_list_prepend(list, "AFM");
list = g_list_prepend(list, "Electrostatic");
list = g_list_reverse(list);

combo = gtk_combo_new();
gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), FALSE);
gtk_combo_set_popdown_strings(GTK_COMBO(combo), list);
gtk_box_pack_start(GTK_BOX(hbox), combo, FALSE, FALSE, PANEL_SPACING);
g_signal_connect(GTK_OBJECT(GTK_COMBO(combo)->entry), "changed", 
                 GTK_SIGNAL_FUNC(ms_colour_mode), GTK_COMBO(combo)->entry);

/* electrostatic potential scale setup */
/* NEW - scale setup */
/*
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE,0);

new_check_button("Electrostatic autoscaling", epot_scale_toggle, data,
                 data->gulp.epot_autoscale, hbox);
*/
gtksh_direct_check("Electrostatic autoscaling", &data->gulp.epot_autoscale,
                   epot_scale_sensitivity, data, vbox);

/* hide/show control */
epot_vbox = gtk_vbox_new(TRUE, 0);
gtk_box_pack_start(GTK_BOX(vbox), epot_vbox, FALSE, TRUE,0);

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(epot_vbox), hbox, FALSE, TRUE,0);
label = gtk_label_new("maximum ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
surf_epot_max = gtk_entry_new();
gtk_box_pack_end(GTK_BOX(hbox), surf_epot_max, FALSE, FALSE, 0);
gtk_entry_set_text(GTK_ENTRY(surf_epot_max), 
                   g_strdup_printf("%f", data->gulp.epot_max));

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(epot_vbox), hbox, FALSE, TRUE,0);
label = gtk_label_new("minimum ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
surf_epot_min = gtk_entry_new();
gtk_box_pack_end(GTK_BOX(hbox), surf_epot_min, FALSE, FALSE, 0);
gtk_entry_set_text(GTK_ENTRY(surf_epot_min), 
                   g_strdup_printf("%f", data->gulp.epot_min));

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(epot_vbox), hbox, FALSE, TRUE,0);
label = gtk_label_new("divisions ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
surf_epot_div = gtk_entry_new();
gtk_box_pack_end(GTK_BOX(hbox), surf_epot_div, FALSE, FALSE, 0);
gtk_entry_set_text(GTK_ENTRY(surf_epot_div), 
                   g_strdup_printf("%d", data->gulp.epot_divisions));

/* make, hide, close - terminating buttons */
gtksh_stock_button(GTK_STOCK_EXECUTE, calc_molsurf, data,
                   GTK_DIALOG(surf_dialog->win)->action_area);

gtksh_stock_button(GTK_STOCK_REMOVE, del_molsurf, data,
                   GTK_DIALOG(surf_dialog->win)->action_area);

gtksh_stock_button(GTK_STOCK_CLOSE, event_close_dialog, GINT_TO_POINTER(id),
                   GTK_DIALOG(surf_dialog->win)->action_area);

/* done */
gtk_widget_show_all(surf_dialog->win);

/* initialize widget states */
epot_scale_sensitivity(NULL, data);
}

