/*
Copyright (C) 2003 by Andrew Lloyd Rohl and Sean David Fleming

andrew@power.curtin.edu.au
sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>

#include "gdis.h"
#include "coords.h"
#include "file.h"
#include "task.h"
#include "matrix.h"
#include "spatial.h"
#include "gtkshorts.h"
#include "interface.h"
#include "opengl.h"

extern struct sysenv_pak sysenv;

#define CONTROL_TAB_LABEL " Control "
#define BASIS_TAB_LABEL " Basis Set "
#define OPTIMISATION_TAB_LABEL " Optimisation "

enum {
  TIMLIM, MWORDS, WIDE_OUTPUT
};

struct callback_pak run_control[] = {
{"Time limit (minutes) ", TIMLIM},
{"Memory limit (megawords) ", MWORDS},
{NULL, 0}
};

enum {
  TOTAL_Q, MULTIPLICITY
};

struct callback_pak econfig[] = {
{"Total charge ", TOTAL_Q},
{"Multiplicity ", MULTIPLICITY},
{NULL, 0}
};

enum {
  NUM_P, NUM_D, NUM_F
};

struct callback_pak pol_funcs[] = {
{"p ", NUM_P},
{"d ", NUM_D},
{"f ", NUM_F},
{NULL, 0}
};

enum {
  HEAVY_DIFFUSE, HYDROGEN_DIFFUSE, NSTEP, GAMESS_TITLE, GAMESS_ENERGY, INPUTFILE, MODELNAME
};

struct basis_pak
{
gchar *label;
GMSBasisType basis;
gint ngauss;
};

struct basis_pak basis_sets[] = {
{"User Defined", GMS_USER, 0},
{"MNDO", GMS_MNDO, 0},
{"AM1", GMS_AM1, 0},
{"PM3", GMS_PM3, 0},
{"MINI", GMS_MINI, 0},
{"MIDI", GMS_MIDI, 0},
{"STO-2G", GMS_STO, 2},
{"STO-3G", GMS_STO, 3},
{"STO-4G", GMS_STO, 4},
{"STO-5G", GMS_STO, 5},
{"STO-6G", GMS_STO, 6},
{"3-21G", GMS_N21, 3},
{"6-21G", GMS_N21, 6},
{"4-31G", GMS_N31, 4},
{"5-31G", GMS_N31, 5},
{"6-31G", GMS_N31, 6},
{"6-311G", GMS_N311, 6},
{"DZV", GMS_DZV, 0},
{"DH", GMS_DH, 0},
{"TZV", GMS_TZV, 0},
{"MC", GMS_MC, 0},
{NULL, 0, 0}
};

/*********************/
/* run a gamess file */
/*********************/
#define DEBUG_EXEC_GAMESS 0
gint exec_gamess(gchar *input)
{
gchar *cmd;

/* checks */
if (!sysenv.gamess_path)
  return(-1);

cmd = g_strdup_printf("%s %s", sysenv.gamess_path, input);

#if DEBUG_EXEC_GAMESS
printf("executing: [%s]\n",cmd);
#endif

task_sync(cmd);

g_free(cmd);
return(0);
}

/*****************************/
/* execute a gamess run (task) */
/*****************************/
#define DEBUG_EXEC_GAMESS_TASK 0
void exec_gamess_task(struct model_pak *model)
{
/* construct output filename */
g_free(model->gamess.out_file);
model->gamess.out_file = g_strdup_printf("%s.gmot", model->basename);

/* save input file and execute */
file_save_as(model->gamess.temp_file, model);
exec_gamess(model->gamess.temp_file);
}

/*****************************/
/* process a gamess run (task) */
/*****************************/
#define DEBUG_PROC_GAMESS 0
void proc_gamess_task(gpointer *ptr)
{
GString *line;
struct model_pak *data;

/* TODO - model locking (moldel_ptr RO/RW etc) to prevent screw ups */
g_return_if_fail(ptr != NULL);
data = (struct model_pak *) ptr;

if (data->gamess.run_type < GMS_OPTIMIZE && !data->animation)
  {
/* same model (ie with current energetics dialog) so update */
    file_load(data->gamess.out_file, data);
/* update energy (TODO - only if successful) */
    line = g_string_new(NULL);
    if (data->gamess.have_energy)
      g_string_sprintf(line,"%f",data->gamess.energy);
    else
      g_string_sprintf(line, "not yet calculated");
/* is there a dialog entry to be updated? */
    if (GTK_IS_ENTRY(data->gamess.energy_entry))
      gtk_entry_set_text(GTK_ENTRY(data->gamess.energy_entry), line->str);
    g_string_free(line, TRUE);
  }
else
  {
/* TODO - make it possile to get dialog data by request */
/* so that we can check if a dialog exsits to be updated */
/* get new coords */
/* create new model for the minimized result */
  file_load(data->gamess.out_file, NULL);
  }

redraw_canvas(ALL);

return;
}

/*************************************************/
/* controls the use of GAMESS to optimise coords */
/*************************************************/
#define DEBUG_RUN_GAMESS 0
void run_gamess(GtkWidget *w, struct model_pak *data)
{
/* checks */
g_return_if_fail(data != NULL);
if (!sysenv.gamess_path)
  {
  show_text(ERROR, "GAMESS executable was not found.\n");
  return;
  }

#if DEBUG_RUN_GAMESS
printf("output file: %s\n", data->gamess.out_file);
#endif

submit_task("Gamess", &exec_gamess_task, data, &proc_gamess_task, data, data);
}

/*********************************/
/* GAMESS execution type toggles */
/*********************************/
void gamess_exe_run(struct model_pak *mdata)
{
mdata->gamess.exe_type = GMS_RUN;
}
void gamess_exe_check(struct model_pak *mdata)
{
mdata->gamess.exe_type = GMS_CHECK;
}
void gamess_exe_debug(struct model_pak *mdata)
{
mdata->gamess.exe_type = GMS_DEBUG;
}

/***************************/
/* GAMESS run type toggles */
/***************************/
void gamess_run_single(struct model_pak *data)
{
data->gamess.run_type = GMS_ENERGY;
}
void gamess_run_gradient(struct model_pak *data)
{
data->gamess.run_type = GMS_GRADIENT;
}
void gamess_run_hessian(struct model_pak *data)
{
data->gamess.run_type = GMS_HESSIAN;
}
void gamess_run_optimize(struct model_pak *data)
{
data->gamess.run_type = GMS_OPTIMIZE;
}

/***************************/
/* GAMESS SCF type toggles */
/***************************/
void gamess_scf_rhf(struct model_pak *data)
{
data->gamess.scf_type = GMS_RHF;
}
void gamess_scf_uhf(struct model_pak *data)
{
data->gamess.scf_type = GMS_UHF;
}
void gamess_scf_rohf(struct model_pak *data)
{
data->gamess.scf_type = GMS_ROHF;
}
void gamess_units_angs(struct model_pak *data)
{
data->gamess.units = GMS_ANGS;
}
void gamess_units_bohr(struct model_pak *data)
{
data->gamess.units = GMS_BOHR;
}
void gamess_optimise_qa(struct model_pak *data)
{
data->gamess.opt_type = GMS_QA;
}
void gamess_optimise_nr(struct model_pak *data)
{
data->gamess.opt_type = GMS_NR;
}
void gamess_optimise_rfo(struct model_pak *data)
{
data->gamess.opt_type = GMS_RFO;
}
void gamess_optimise_schlegel(struct model_pak *data)
{
data->gamess.opt_type = GMS_SCHLEGEL;
}
void event_destroy_user_data_from_object(GtkObject *object, gpointer user_data)
{
g_free(user_data);
}

/*****************************/
/* basis set selection event */
/*****************************/
void basis_selected(GtkWidget *w, struct model_pak *model)
{
gint i=0;
const gchar *label;

label = gtk_entry_get_text(GTK_ENTRY(w));

while (basis_sets[i].label)
  {
  if (g_strcasecmp(label, basis_sets[i].label) == 0)
    {
    model->gamess.basis = basis_sets[i].basis;
    model->gamess.ngauss = basis_sets[i].ngauss;
    }
  i++;
  }
}

/***************************/
/* text handling callbacks */
/***************************/

gint event_pol_text_fields(GtkWidget *w, gpointer *obj)
{
gint id;
struct model_pak *data;

/* checks */
g_return_val_if_fail(w != NULL, FALSE);

/* ascertain type of modification required */
id = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(w), "id"));

data = model_ptr(sysenv.active, RECALL);
sysenv.moving = FALSE;

switch(id)
  {
  case GAMESS_TITLE:
    g_free(data->gamess.title);
    data->gamess.title = g_strdup(gtk_entry_get_text(GTK_ENTRY(w)));
    break;
  case INPUTFILE:
    g_free(data->gamess.temp_file);
    data->gamess.temp_file = g_strdup(gtk_entry_get_text(GTK_ENTRY(w)));
    break; 
  }

return(FALSE);
}

/*********************************/
/* GAMESS job setup/results page */
/*********************************/
void gamess_widget(GtkWidget *box, struct model_pak *data)
{
gint i, j;
GString *line;
GList *list;
GtkWidget *hbox, *vbox, *vbox1, *vbox2, *vbox3, *page;
GtkWidget *frame, *button, *label, *entry, *notebook;
GtkWidget *combo;

/* checks */
g_assert(box != NULL);
g_assert(data != NULL);

/* string manipulation scratchpad */
line = g_string_new(NULL);

/* print method blurb */
vbox = gtk_vbox_new(TRUE, 0);
gtk_box_pack_start(GTK_BOX(box), vbox, FALSE, FALSE, 0);
g_string_sprintf(line,"\nQuantum mechanics method\nGAMESS GUI\n");
label = gtk_label_new(line->str);
gtk_container_add(GTK_CONTAINER(vbox), label);

/* create notebook */
notebook = gtk_notebook_new();
gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
gtk_container_add(GTK_CONTAINER(box), notebook);
gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), FALSE);

/* Control page */
page = gtk_vbox_new(FALSE, 0);
label = gtk_label_new (CONTROL_TAB_LABEL);
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
hbox = gtk_hbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(page), hbox);

/* split panel */
vbox1 = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox1, FALSE, FALSE, 0);
vbox2 = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox2, FALSE, FALSE, 0);
vbox3 = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox3, FALSE, FALSE, 0);

/* frame for execution type */
frame = gtk_frame_new("Execution type");
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);

/* radio buttons for execution type */
new_radio_group(0, vbox, TT);
button = add_radio_button("run", (gpointer) gamess_exe_run, data);
if (data->gamess.exe_type == GMS_RUN)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

button = add_radio_button("check", (gpointer) gamess_exe_check, data);
if (data->gamess.exe_type == GMS_CHECK)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

button = add_radio_button("debug", (gpointer) gamess_exe_debug, data);
if (data->gamess.exe_type == GMS_DEBUG)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* frame for run type */
frame = gtk_frame_new("Run type");
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);

/* radio buttons for run type */
new_radio_group(0, vbox, TT);
button = add_radio_button("single point", (gpointer) gamess_run_single, data);
if (data->gamess.run_type == GMS_ENERGY)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

button = add_radio_button("gradient", (gpointer) gamess_run_gradient, data);
if (data->gamess.run_type == GMS_GRADIENT)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

button = add_radio_button("hessian", (gpointer) gamess_run_hessian, data);
if (data->gamess.run_type == GMS_HESSIAN)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

button = add_radio_button("optimize", (gpointer) gamess_run_optimize, data);
if (data->gamess.run_type == GMS_OPTIMIZE)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* frame for SCF type */
frame = gtk_frame_new("SCF type");
gtk_box_pack_start(GTK_BOX(vbox2), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);

/* radio buttons for SCF type */
new_radio_group(0, vbox, TT);
button = add_radio_button("RHF", (gpointer) gamess_scf_rhf, data);
if (data->gamess.scf_type == GMS_RHF)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

button = add_radio_button("UHF", (gpointer) gamess_scf_uhf, data);
if (data->gamess.scf_type == GMS_UHF)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

button = add_radio_button("ROHF", (gpointer) gamess_scf_rohf, data);
if (data->gamess.scf_type == GMS_ROHF)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* frame for input units */
frame = gtk_frame_new("Input units");
gtk_box_pack_start(GTK_BOX(vbox2), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);

/* radio buttons for SCF type */
new_radio_group(0, vbox, TT);
button = add_radio_button("Angstrom", (gpointer) gamess_units_angs, data);
if (data->gamess.units == GMS_ANGS)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

button = add_radio_button("Bohr", (gpointer) gamess_units_bohr, data);
if (data->gamess.units == GMS_BOHR)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* frame for SCF options */
frame = gtk_frame_new("SCF options");
gtk_box_pack_start(GTK_BOX(vbox2), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtksh_direct_spin("Maximum iterations", &data->gamess.maxit, 1, 150, 1,
                  NULL, NULL, vbox);

/* frame for time, memory and output */
frame = gtk_frame_new("Run control");
gtk_box_pack_start(GTK_BOX(vbox3), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);

/* Add spinners */
i=0;
while (run_control[i].label)
  {
  switch (run_control[i].id)
    {
    case TIMLIM:
      gtksh_direct_spin(run_control[i].label,
                        &data->gamess.time_limit, 0, 30000, 10,
                        NULL, NULL, vbox);
      break;

    case MWORDS:
      gtksh_direct_spin(run_control[i].label,
                        &data->gamess.mwords, 1, 150, 1,
                        NULL, NULL, vbox);
      break;

    default:
      g_assert_not_reached();
    }
  i++;
  }

/* wide output button */
gtksh_direct_check("Wide output", &data->gamess.wide_output, NULL, NULL, vbox);

/* frame for charge and multiplicity */
frame = gtk_frame_new("Electronic configuration");
gtk_box_pack_start(GTK_BOX(vbox3), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);

/* Add spinners */
i=0;
while (econfig[i].label)
  {
  switch (econfig[i].id)
    {
    case TOTAL_Q:
      gtksh_direct_spin(econfig[i].label,
                        &data->gamess.total_charge, -5, 5, 1,
                        NULL, NULL, vbox);
      break;

    case MULTIPLICITY:
      gtksh_direct_spin(econfig[i].label,
                        &data->gamess.multiplicity, 1, 5, 1,
                        NULL, NULL, vbox);
      break;

    default:
      g_assert_not_reached();
    }
  i++;
  }
  
/* basis set page */
page = gtk_vbox_new(FALSE,0);
label = gtk_label_new (BASIS_TAB_LABEL);
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(page), hbox, FALSE, FALSE, 0);

/* split panel */
vbox1 = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox1, TRUE, TRUE, 0);
vbox2 = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox2, FALSE, FALSE, 0);

/* frame for basis set */
frame = gtk_frame_new("Basis set");
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
j = -1;
i = 0;
list = NULL;
while (basis_sets[i].label)
  {
  list = g_list_append(list, basis_sets[i].label);
  if ((basis_sets[i].basis == data->gamess.basis) && (basis_sets[i].ngauss == data->gamess.ngauss))
    j = i;
  i++;
  }

combo = gtk_combo_new();
gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), FALSE);
gtk_combo_set_popdown_strings(GTK_COMBO(combo), list);
gtk_box_pack_start(GTK_BOX(vbox), combo, FALSE, FALSE, 0);
/* NB: set text BEFORE the changed event is connected */
if (j >= 0)
  gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(combo)->entry), basis_sets[j].label);
g_signal_connect(GTK_OBJECT(GTK_COMBO(combo)->entry), "changed", 
                 GTK_SIGNAL_FUNC(basis_selected), data);

/* frame for polarization functions set */
frame = gtk_frame_new("Polarization functions");
gtk_box_pack_start(GTK_BOX(vbox2), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);

/* Add spinners */
i=0;
while (pol_funcs[i].label)
  {
  switch (pol_funcs[i].id)
    {
    case NUM_P:
      gtksh_direct_spin(pol_funcs[i].label, &data->gamess.num_p, 0, 3, 1,
                        NULL, NULL, vbox);
      break;
    case NUM_D:
      gtksh_direct_spin(pol_funcs[i].label, &data->gamess.num_d, 0, 3, 1,
                        NULL, NULL, vbox);
      break;
    case NUM_F:
      gtksh_direct_spin(pol_funcs[i].label, &data->gamess.num_f, 0, 1, 1,
                        NULL, NULL, vbox);
      break;
    default:
      g_assert_not_reached();
    }
  i++;
  }

/* frame for polarization functions set */
frame = gtk_frame_new("Diffuse functions");
gtk_box_pack_start(GTK_BOX(vbox2), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtksh_direct_check("Heavy atoms (s & p)", &data->gamess.have_heavy_diffuse,
                    NULL, NULL, vbox);
gtksh_direct_check("Hydrogen (s only)", &data->gamess.have_hydrogen_diffuse,
                    NULL, NULL, vbox);

/* frame for optimisation settings */
page = gtk_vbox_new(FALSE,0);
label = gtk_label_new (OPTIMISATION_TAB_LABEL);
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(page), hbox, FALSE, FALSE, 0);

/* split panel */
vbox1 = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox1, FALSE, FALSE, 0);
vbox2 = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox2, FALSE, FALSE, 0);

/* optimiser to use */
frame = gtk_frame_new("Optimiser");
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);

/* radio buttons for run type */
new_radio_group(0, vbox, TT);
button = add_radio_button("Quadratic approximation", (gpointer) gamess_optimise_qa, data);
if (data->gamess.opt_type == GMS_QA)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
button = add_radio_button("Newton-Raphson", (gpointer) gamess_optimise_nr, data);
if (data->gamess.opt_type == GMS_NR)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
button = add_radio_button("RFO", (gpointer) gamess_optimise_rfo, data);
if (data->gamess.opt_type == GMS_RFO)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
button = add_radio_button("Quasi-NR", (gpointer) gamess_optimise_schlegel, data);
if (data->gamess.opt_type == GMS_SCHLEGEL)
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

/* frame for optimization cycles */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);

/* optimization cycles */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
gtksh_direct_spin("Maximum cycles", &data->gamess.nstep,
                  0, 500, 10, NULL, NULL, hbox);


/* setup the filename entry */

/* setup some gulp file name defaults */
if (g_ascii_strncasecmp(data->gamess.temp_file, "none", 4) == 0)
  {
  g_free(data->gamess.temp_file);
  data->gamess.temp_file = g_strdup_printf("%s.inp", data->basename); 
  }

frame = gtk_frame_new("Files");
gtk_box_pack_start(GTK_BOX(box),frame,FALSE,FALSE,0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
hbox = gtk_hbox_new(FALSE, 0);
gtk_container_add (GTK_CONTAINER(frame),hbox);
gtk_container_set_border_width(GTK_CONTAINER(hbox), PANEL_SPACING);

/* left vbox */
vbox = gtk_vbox_new(TRUE, 2);
gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);

/* GAMESS input filename */
label = gtk_label_new(" Job input file ");
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);

/* right vbox */
vbox = gtk_vbox_new(TRUE, 2);
gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);

/* temporary input */
entry = gtk_entry_new_with_max_length(LINELEN);
gtk_entry_set_text(GTK_ENTRY(entry), data->gamess.temp_file);
gtk_box_pack_start(GTK_BOX(vbox), entry, TRUE, TRUE, 0);
g_signal_connect(GTK_OBJECT (entry), "changed",
                          GTK_SIGNAL_FUNC (event_pol_text_fields),
                         NULL);
g_object_set_data(G_OBJECT(entry), "id", (gpointer) INPUTFILE);

/* details */
frame = gtk_frame_new("Details");
gtk_box_pack_start(GTK_BOX(box),frame,FALSE,FALSE,0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
hbox = gtk_hbox_new(FALSE, 0);
gtk_container_add (GTK_CONTAINER(frame),hbox);

/* left vbox */
vbox = gtk_vbox_new(TRUE, 2);
gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);

if (g_ascii_strncasecmp(data->gamess.title, "none", 4) == 0)
  {
  g_free(data->gamess.title);
  data->gamess.title = g_strdup(data->basename); 
  }

/* temporary GAMESS input filename */
label = gtk_label_new("Structure name");
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
label = gtk_label_new("Title");
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
label = gtk_label_new("Total energy (Hartrees)");
gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);

/* right vbox */
vbox = gtk_vbox_new(TRUE, 2);
gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);

entry = gtk_entry_new_with_max_length(LINELEN);
gtk_entry_set_text(GTK_ENTRY(entry), data->basename);
gtk_box_pack_start (GTK_BOX (vbox), entry, TRUE, TRUE, 0);
g_signal_connect(GTK_OBJECT(entry), "changed",
                   GTK_SIGNAL_FUNC(name_entry_changed),
                   (gpointer) data);

entry = gtk_entry_new_with_max_length(LINELEN);
gtk_entry_set_text(GTK_ENTRY(entry), data->gamess.title);
gtk_box_pack_start (GTK_BOX (vbox), entry, TRUE, TRUE, 0);
g_signal_connect(GTK_OBJECT (entry), "changed",
                          GTK_SIGNAL_FUNC (event_pol_text_fields),
                         NULL);
g_object_set_data(G_OBJECT(entry), "id", (gpointer) GAMESS_TITLE);

data->gamess.energy_entry = gtk_entry_new_with_max_length(LINELEN);
if (data->gamess.have_energy)
  g_string_sprintf(line,"%f",data->gamess.energy);
else
  g_string_sprintf(line,"not yet calculated");

gtk_entry_set_text(GTK_ENTRY(data->gamess.energy_entry),line->str);
gtk_box_pack_start (GTK_BOX (vbox), data->gamess.energy_entry, TRUE, TRUE, 0);
gtk_entry_set_editable(GTK_ENTRY(data->gamess.energy_entry), FALSE);

/* done */
gtk_widget_show_all(box);

g_string_free(line, TRUE);
}

