/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#ifdef __sun
#include <sys/dirent.h>
#else
#ifdef __WIN32
#include <dirent.h>
#else
#include <sys/dir.h>
#include <sys/param.h>
#endif
#endif

#include "gdis.h"
#include "coords.h"
#include "edit.h"
#include "file.h"
#include "parse.h"
#include "matrix.h"
#include "interface.h"

#define DEBUG_MORE 0
#define MAX_KEYS 15

/* main structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/*************************************************/
/* routines to simplify writing GULP input files */
/*************************************************/
void fprint_core(FILE *fp, struct core_pak *core, struct model_pak *data)
{
gint i;
gdouble vec[3];

/* get cartesian coords */
ARR3SET(vec, core->x);
vecmat(data->latmat, vec); 

/* set fractional part */
for (i=0 ; i<data->periodic ; i++)
  if (data->fractional)
    vec[i] = core->x[i];

/* only print specific atom charges if we read them in */
/* in that fashion (ie lookup_charge is false) */
if (core->lookup_charge)
  {
/* FIXME - what about if we have a non unit SOF? */
/* this could be a problem as GULP's internal charges don't always */
/* match GDIS's - difficulty is in multiple valencies (eg Mn 2+/4+) */
  fprintf(fp,"%-4s core %11.6f %11.6f %11.6f\n",
          core->label, vec[0], vec[1], vec[2]);
  }
else
  {
/* NB: 7 dp for charges - 8 occasionally introduced small errors (cf GULP) */
  fprintf(fp,"%-4s core %11.6f %11.6f %11.6f  %11.7lf  %3.1f\n",
          core->label, vec[0], vec[1], vec[2], core->charge, core->sof);
  }
}

void fprint_shell(FILE *fp, struct shel_pak *shell, struct model_pak *data)
{
gint i;
gdouble vec[3];

/* get cartesian coordinates */
ARR3SET(vec, shell->x);
vecmat(data->latmat, vec); 

/* set fractional part */
for (i=0 ; i<data->periodic ; i++)
  if (data->fractional)
    vec[i] = shell->x[i];

/* only print specific atom charges if we read them in */
if (shell->lookup_charge)
  {
  fprintf(fp,"%-4s shel %11.6f %11.6f %11.6f\n",
          shell->label, vec[0], vec[1], vec[2]);
  }
else
  {
  fprintf(fp,"%-4s shel %11.6f %11.6f %11.6f  %11.7lf  %3.1f\n",
          shell->label, vec[0], vec[1], vec[2], shell->charge, 1.0);
  }
}

/************************************************/
/* debugging - dump the vibrational frequencies */
/************************************************/
void dump_phonons(struct model_pak *model)
{
GSList *list;

for (list=model->phonons ; list ; list=g_slist_next(list))
  {
  printf("[%s]\n", (gchar *) list->data);
  }
}

/**********************************/
/* print appropriate region label */
/**********************************/
void write_region_header(FILE *fp, gint region, struct model_pak *model)
{
/* check periodicity */
switch (model->periodic)
  {
  case 1:
    if (model->fractional)
      fprintf(fp,"pfractional region %d", region+1);
    else
      fprintf(fp,"cartesian region %d", region+1);
    break;

  case 2:
    if (model->fractional)
      fprintf(fp,"sfractional region %d", region+1);
    else
      fprintf(fp,"cartesian region %d", region+1);
    break;

/* 3 or none */
  default:
    if (model->fractional)
      fprintf(fp,"fractional");
    else
      fprintf(fp,"cartesian");
    break;
  }

/* TODO - is there a better way? (what if region 3 exists & is not meant to be moved?) */
if (model->gulp.dock)
  if (region == 2)
    fprintf(fp," rigid xyz");

fprintf(fp,"\n");
}

/*********************/
/* write a GULP file */
/*********************/
#define DEBUG_WRITE_GULP 0
gint write_gulp(gchar *filename, struct model_pak *data)
{
gint i, libflag, flag, region;
GString *line;
GSList *list;
struct core_pak *core;
struct shel_pak *shell;
struct vec_pak *vec;
FILE *fp;

g_return_val_if_fail(data != NULL, 1);

/* is this really necessary??? */
unlink(filename);

/* is a valid potential library specified? */
libflag = 0;
if (g_ascii_strncasecmp(data->gulp.libfile, "none", 4) != 0 &&
    strlen(data->gulp.libfile))
  {
/* test for library existance */
  if (!access(data->gulp.libfile, R_OK))
    libflag++;
  else
    printf("Warning: library %s not found.\n", data->gulp.libfile);
  }

/* off we go */
fp = fopen(filename, "wt");
if (!fp)
  {
  printf("Failed to open %s.\n", filename);
  return(2);
  }

/* NEW - docking */
/* FIXME? - this will irretrievably set the region to 3 for the docking calc */
if (data->gulp.dock)
  {
  for (list=data->selection ; list ; list=g_slist_next(list))
    {
    core = (struct core_pak *) list->data;

    core->region = 2;
    }
  }

/* run type */
line = g_string_new(NULL);
switch(data->gulp.run)
  {
  case E_SINGLE:
    g_string_assign(line,"single");
    break;
  case E_OPTIMIZE:
    g_string_assign(line,"opti");
    break;
  case MD:
    g_string_assign(line,"md");
    break;
/* no run -> single pt (valid default?) */
  default:
    g_string_assign(line,"single");
    break;
  }

switch(data->gulp.method)
  {
  case CONP:
    g_string_append(line," conp");
    break;
  case CONV:
    g_string_append(line," conv");
    break;
  default:
    break;
  }

switch(data->gulp.optimiser)
  {
  case BFGS_OPT:
    g_string_append(line," bfgs");
    break;
  case CONJ_OPT:
    g_string_append(line," conjugate");
    break;
  case RFO_OPT:
    g_string_append(line," rfo");
    break;
  }

if (data->gulp.unit_hessian)
    g_string_append(line," unit");

switch(data->gulp.coulomb)
  {
  case MOLE:
    g_string_append(line," mole");
    break;
  case MOLMEC:
    g_string_append(line," molmec");
    break;
  case MOLQ:
    g_string_append(line," molq");
    break;
  default:
    break;
  }

/* CURRENT */
if (data->gulp.phonon)
  {
  g_string_append(line," phonon");
  if (data->gulp.eigen)
    g_string_append(line," eigen");
  }

/* additional keywords */
if (data->gulp.qeq)
  g_string_append(line," qeq");
if (data->gulp.free)
  g_string_append(line," free");
if (data->gulp.zsisa)
  g_string_append(line," zsisa");
if (data->gulp.compare)
  g_string_append(line," comp");
if (data->gulp.fix)
  g_string_append(line," fix");
/* if nosym specified, force output of non-prim cell for consistency */
if (data->gulp.nosym)
  g_string_append(line," nosym full");
/* electrostatic potential calcs */
if (g_slist_length(data->gulp.epot_vecs))
  g_string_append(line," epot");

/* print out the keywords line */
fprintf(fp,"%s\n\n",line->str);
fprintf(fp, "# ");
gdis_blurb(fp);
fprintf(fp, "#\n\n");

if (data->periodic == 2)
  {
/* NB: this surface data is only meaningful if generate from within GDIS */
  if (VEC3MAGSQ(data->surface.depth_vec) > FRACTION_TOLERANCE)
    {
    fprintf(fp, "#       miller: %d %d %d\n",
                     (gint) data->surface.miller[0],
                     (gint) data->surface.miller[1],
                     (gint) data->surface.miller[2]);
    fprintf(fp, "#        shift: %lf\n", data->surface.shift);
    fprintf(fp, "# region sizes: %d %d\n\n",
                    (gint) data->surface.region[0],
                    (gint) data->surface.region[1]);
    }
  }

/* optional data */
if (data->gulp.maxcyc > 0)
  fprintf(fp,"maxcyc %d\n\n", (gint) data->gulp.maxcyc);

if (data->gulp.optimiser2 != SWITCH_OFF)
  {
  fprintf(fp,"switch ");
  switch(data->gulp.optimiser2)
    {
    case CONJ_OPT:
      fprintf(fp,"conj ");
      break;
    case RFO_OPT:
      fprintf(fp,"rfo ");
      break;
    case BFGS_OPT:
    default:
      fprintf(fp,"bfgs ");
      break;
    }

  switch(data->gulp.switch_type)
    {
    case CYCLE:
      fprintf(fp,"cycle %d\n\n", (gint) data->gulp.switch_value);
      break;
    case GNORM:
    default:
      fprintf(fp,"gnorm %lf\n\n", data->gulp.switch_value);
      break;
    }
  }

/* only write if we have a valid supplied library */
if (libflag)
  fprintf(fp,"library %s\n\n", data->gulp.libfile);

fprintf(fp,"name %s\n\n",data->basename);

if (g_ascii_strncasecmp(data->gulp.dump_file, "none", 4))
  fprintf(fp,"\ndump %s\n\n",data->gulp.dump_file);


/* md stuff */
if (data->gulp.temp > FRACTION_TOLERANCE)
  fprintf(fp,"temperature %lf\n\n",data->gulp.temp);

switch(data->gulp.ensemble)
  {
  case NVE:
    fprintf(fp,"ensemble nve\n\n");
    break;
  case NVT:
    fprintf(fp,"ensemble nvt\n\n");
    break;
  case NPT:
    fprintf(fp,"ensemble npt\n\n");
    break;
  }

/* output section */
if (data->animation)
  fprintf(fp, "output trajectory %s\n", data->gulp.trj_file);

if (data->gulp.mov_file)
  {
  fprintf(fp, "output movie arc %s\n", data->gulp.mov_file);
  if (data->gulp.run == MD)
    fprintf(fp, "mdarchive %s\n", data->gulp.mov_file);
  }

#if DEBUG_WRITE_GULP
printf("  periodic: %d\n", data->periodic);
printf("fractional: %d\n", data->fractional);
#endif

switch(data->periodic)
  {
  case 3:
    if (data->construct_pbc)
      {
/* NB: gulp matrices are transposed wrt gdis */
      fprintf(fp,"vectors\n");
      fprintf(fp,"%15.10f %15.10f %15.10f\n",
                  data->latmat[0], data->latmat[3], data->latmat[6]);
      fprintf(fp,"%15.10f %15.10f %15.10f\n",
                  data->latmat[1], data->latmat[4], data->latmat[7]);
      fprintf(fp,"%15.10f %15.10f %15.10f\n",
                  data->latmat[2], data->latmat[5], data->latmat[8]);
      }
    else
      {
      fprintf(fp,"cell\n");
      fprintf(fp,"%lf %lf %lf  %lf %lf %lf\n",data->pbc[0],data->pbc[1],data->pbc[2],
                data->pbc[3]*180/PI, data->pbc[4]*180/PI, data->pbc[5]*180/PI);
      }
  break;
  case 2:
    if (data->construct_pbc)
      {
      fprintf(fp,"svectors\n");
      fprintf(fp,"%lf  %lf\n%lf  %lf\n",data->latmat[0], data->latmat[3],
                                    data->latmat[1], data->latmat[4]);
      }
    else
      {
      fprintf(fp,"scell\n");
      fprintf(fp,"%lf %lf %lf\n",data->pbc[0], data->pbc[1],
                              180.0*data->pbc[5]/PI);
      }
  break;
  }

/* coord section write */
flag = region = 0;
for (;;)
  {
/* cores */
  for (list=data->cores ; list ; list=g_slist_next(list))
    {
    core = (struct core_pak *) list->data;

    if (core->status & DELETED)
      continue;
    if (!core->primary)
      continue;

    switch (data->periodic)
      {
      case 2:
        if (core->region != region)
          break;
      default:
/* write suitable header if it hasn't yet been written */
if (!flag)
  {
  write_region_header(fp, region, data);
  flag++;
  }
        fprint_core(fp, core, data);
      }
    }

/* shells */
  for (list=data->shels ; list ; list=g_slist_next(list))
    {
    shell = (struct shel_pak *) list->data;

    if (shell->status & DELETED)
      continue;
    if (!shell->primary)
      continue;

/* if the model is a surface, only write the current region */
    switch (data->periodic)
      {
      case 2:
        if (shell->region != region)
          break;
      default:
        fprint_shell(fp, shell, data);
      }
    }

/* exit after 1st loop as no other regions if not a surface */
/* FIXME - a bit inelegant */
  if (data->periodic != 2)
    break;
  else
    region++;
  if (region > 3)
    break;
  flag = 0;
  }


/* post coord data */
switch(data->periodic)
  {
  case 2:
/* write dhkl for Eatt calcs */
    if (data->surface.dspacing > 0.1 && !data->gulp.no_eatt)
      fprintf(fp, "\ndhkl %lf\n",data->surface.dspacing);

/* write surface bulk energy for Esurf calcs */
    if (!data->gulp.no_esurf)
      fprintf(fp, "\nsbulkenergy %lf\n",data->gulp.sbulkenergy);
    break;

/* SG info */
  case 3:
    fprintf(fp,"\nspace\n%s\n",g_strstrip(data->sginfo.spacename));
    break;
  }
fprintf(fp,"\n");



/* extra data */
if (data->gulp.num_elem)
  {
  fprintf(fp,"elements\n");
  for (i=0 ; i<data->gulp.num_elem ; i++)
    fprintf(fp,"%s\n",*(data->gulp.elem+i));
  fprintf(fp,"end\n\n");
  }
if (data->gulp.num_species)
  {
  fprintf(fp,"species\n");
  for (i=0 ; i<data->gulp.num_species ; i++)
    fprintf(fp,"%s\n",*(data->gulp.species+i));
  fprintf(fp,"end\n\n");
  }
for (i=0 ; i<data->gulp.num_potentials ; i++)
  fprintf(fp,"%s\n",*(data->gulp.pot+i));

/* deprec. - potgrid calc */
/*
if (data->gulp.potgrid)
  {
  fprintf(fp,"potgrid  ");
  fprintf(fp,"%f %f %f  %f %f %f %d %d %d\n\n",
    data->gulp.grid_dim[0], data->gulp.grid_dim[3],
    data->gulp.grid_dim[1], data->gulp.grid_dim[4],
    data->gulp.grid_dim[2], data->gulp.grid_dim[5],
    data->gulp.grid_pts[0], data->gulp.grid_pts[1], data->gulp.grid_pts[2]);
  }
*/

if (data->gulp.epot_vecs)
  {
  fprintf(fp, "potsites cart\n");

  for (list=data->gulp.epot_vecs ; list ; list=g_slist_next(list))
    {
    vec = (struct vec_pak *) list->data;

    fprintf(fp, "%f  %f  %f\n", vec->rx[0], vec->rx[1], vec->rx[2]);
    }

/* CURRENT - GULP bug - can't end on blank line */
  fprintf(fp, "\nprint 1\n");
  }


/* specified kpoints? */
if (data->periodic && data->gulp.num_kpoints)
  {
/* FIXME - allow for more kpoints in future */
  fprintf(fp, "kpoints %d\n", data->gulp.num_kpoints);
  for (i=0 ; i<data->periodic ; i++ )
    fprintf(fp, "%lf ", data->gulp.kpoints[i]);
  fprintf(fp, "\n\n");
  }

/* unrecognized stuff */
if (data->gulp.extra)
  {
  fprintf(fp, "#--- Additional options/data\n");
  fprintf(fp, "%s\n", data->gulp.extra);
  }

/* done */
fclose(fp);
g_string_free(line, TRUE);
return(0);
}

/*****************/
/* debugging aid */
/*****************/
void print_gulp_flags(gint *list, gint size)
{
gint i;

printf("Flags:");
for (i=0 ; i<size ; i++)
  {
/* active? */
  if (*(list+i))
    {
    switch(i)
      {
      case NAME:
        printf(" name");
        break;
      case SURFACE_CELL:
        printf(" scell");
        break;
      case CELL:
        printf(" cell");
        break;
      case FRAC:
        printf(" frac");
        break;
      case S_FRAC:
        printf(" sfrac");
        break;
      case CART:
        printf(" cart");
        break;
      case DUMP_FILE:
        printf(" dump");
        break;
      case SURFACE_VECTORS:
        printf(" svec");
        break;
      case LATTICE_VECTORS:
        printf(" lvec");
        break;
      }
    }
  }
printf("\n");
}

/****************************/
/* new gulp parsing routine */
/****************************/
#define DEBUG_LOAD_GULP 0
gint read_gulp(gchar *filename, struct model_pak *data)
{
gint i, j, nc=0, ns=0;
gint model, current, type, code, fix;
guint region;
gint gen_flag, new_flag, flag_count, old_count, *flag;
gint run, method, optimiser, optimiser2, unit, switch_type;
gint coulomb, maxcyc, qeq, free, zsisa, compare, nosym, phonon, eigen;
gint num_kpoints, num_potentials, num_species, num_elem;
gint num_keyword, num_tokens;
gint *keyword;
gchar **buff, *tmp, line[LINELEN];
/* TODO - use some of glib's auto grow strings to remove size limitations */
gchar *species[MAX_POTENTIALS], *elem[MAX_POTENTIALS];
gchar *pot[MAX_POTENTIALS];
gdouble vec1[3], vec2[3], vec3[3], kpoints[3], switch_value;
GString *pad, *extra;
GSList *list;
struct elem_pak elem_data;
struct core_pak *core;
struct shel_pak *shell;
FILE *fp;

/* checks */
g_return_val_if_fail(data != NULL, 1);
g_return_val_if_fail(filename != NULL, 1);

fp = fopen(filename, "rt");
if (!fp)
  return(1);

pad = g_string_new(NULL);
/* append lines we don't recognize to this string */
extra = g_string_new("\n");

/* make sure this is assigned b4 parse as we don't */
/* want to encounter data that goes into a model ptr b4 */
/* one has been assigned - which may not happen if keyword */
/* order is changed around */
model = data->number;
current = model-1;

/* setup */
num_keyword=num_keys();
flag = g_malloc(num_keyword * sizeof(gint));
for (i=0 ; i<num_keyword ; i++)
  *(flag+i) = 0;
flag_count = 0;
old_count=0;

/* no default gulp run type */
/* so we don't force anything that wasn't in the file */
run = method = optimiser = optimiser2 = switch_type = -1;
switch_value = -1.0;
coulomb = NOBUILD;
fix = FALSE;
maxcyc = 0;
num_potentials = num_species = num_elem = 0;
region = REGION1A;
qeq = free = zsisa = compare = nosym = phonon = eigen = unit = FALSE;
VEC3SET(kpoints, 0.0, 0.0, 0.0);
num_kpoints = 0;

/* main option search */
keyword = get_keyword(line, MAX_KEYS);
for(;;)
  {
/* search for keyword(s) */
  while (!*keyword)
    {
    if (fgetline(fp, line))
      goto done;
    g_free(keyword);
/* FIXME - use get_keywords() */
    keyword = get_keyword(line, MAX_KEYS);

/* save lines that have no keywords (for later printing) */
    if (!*keyword)
      {
/* ignore lines with only one char (eg '\n') */
      if (strlen(line) > 1)
        g_string_sprintfa(extra,"%s\n", g_strstrip(line));
      }
    }
/* search for starting keywords */
  if (*keyword)
    {
/* PROCESS FOR FLAG SETTING */
    for (i=1 ; i<=*keyword ; i++)
      {
      switch(*(keyword+i))
        {
/* these are to be applied to all loaded models */
        case E_SINGLE:
        case E_OPTIMIZE:
        case MD:
          run = *(keyword+i);
          break;
        case QEQ:
          qeq = TRUE;
          break;
        case FREE_ENERGY:
          free = TRUE;
          break;
        case ZSISA:
          zsisa = TRUE;
          break;
        case COMPARE:
          compare = TRUE;
          break;
        case NOSYM:
          nosym = TRUE;
          break;
        case FIX:
          fix = TRUE;
          break;
        case CONP:
        case CONV:
          method = *(keyword+i);
          break;
        case BFGS_OPT:
        case CONJ_OPT:
        case RFO_OPT:
/* cope with switch option triggering this */
          if (*(keyword+1) == SWITCH_ON)
            optimiser2 = *(keyword+i);             /* switch option */
          else
            optimiser = *(keyword+i);              /* keyword */
          break;
        case UNIT_HESSIAN:
          unit = TRUE;
          break;
        case MOLE:
        case MOLMEC:
        case MOLQ:
          coulomb = *(keyword+i);
          break;
        case PHONON:
          phonon = TRUE;
          break;
        case EIGEN:
          eigen = TRUE;
          break;
        case MAXCYC:
          sscanf(get_token_pos(line,1),"%d",&maxcyc);
          break;
        case SWITCH_ON:
/* scan for a number ANYWHERE ie the switch condition */
/* damn julian for making this so fiddly to parse */
          buff = tokenize(line, &num_tokens);
          for (j=0 ; j<num_tokens ; j++)
            {
/* check 1st character - this'll fail on things like .01 */
            if (g_ascii_isdigit(*(*(buff+j))))
              switch_value = str_to_float(*(buff+j));
            }
          g_strfreev(buff);
          break;
        case GNORM:
        case CYCLE:
/* should only be a switch condition type */
          if (*(keyword+1) == SWITCH_ON)
            switch_type = *(keyword+i);
          break;

        case S_FRAC:
        case FRAC:
          buff = get_tokens(line, 4);
          if (g_ascii_strncasecmp("region",*(buff+1),6) == 0)
            region = str_to_float(*(buff+2)) - 1;
          g_strfreev(buff);
          data->region_empty[region] = FALSE; 
/* only change flags if it's region 1 */
/* that way region 2 -> keep on reading as if nothing happened */
/* (apart from the fact that we now have a different region number) */
          if (region == REGION1A)
            flag[*(keyword+i)]++;
          break;
        case CART:
          buff = get_tokens(line, 4);
          if (g_ascii_strncasecmp("region",*(buff+1),6) == 0)
            region = str_to_float(*(buff+2)) - 1;
          g_strfreev(buff);
          data->region_empty[region] = FALSE; 
/* begin new model only if it's region 1 */
          if (region == REGION1A)
            flag[CART]++;
          break;

/* other keywords that can initiate a new model */
        case CELL:
        case NAME:
        case POLYMER_CELL:
        case POLYMER_VECTOR:
        case SURFACE_CELL:
        case SURFACE_VECTORS:
        case LATTICE_VECTORS:
          flag[*(keyword+i)]++;
          break;
        }
      old_count = flag_count;
      }

#if DEBUG_LOAD_GULP
print_gulp_flags(flag, num_keyword);
#endif

/* reset found flag */
    *keyword=0;
/* PROCESS FOR MODEL CREATION */
    new_flag=flag_count=0;
    for (i=0 ; i<num_keyword ; i++)
      {
/* repeated keyword -> next model in file */
      if (flag[i] == 2)
        {
/* check we have minimum data */
        if (!flag[CART])
          if (!flag[FRAC] && !flag[CELL])
            {
            printf("***ERROR: incomplete data.\n");
            return(1);
            }
        new_flag++;
        }
/* sum the number of starting keywords currently found */
      if (flag[i])
        flag_count++;
      }
/* NEW - new model only on 1st occurrence of any data start keyword */
    if (flag_count == 1 && old_count == 0)
      new_flag++;

/* are we required to make a new model? */
    if (new_flag)
      {
      current++;
#if DEBUG_LOAD_GULP
printf("Creating new model: %d\n",current);
#endif
/* special case for 1st time around (ie ptr is already assigned) */
      if (current > model)
        data = model_ptr(current, ASSIGN);
      if (data == NULL)
        return(2);
      data->id = GULP;
/* init indices for num cores & shells read into this model */
      nc=ns=0;
/* setup name */
      g_free(data->basename);
      g_string_sprintf(pad,"%s_%d",strdup_basename(filename),current-model+1);
      data->basename = g_strdup(pad->str);
      data->gulp.srcfile = strdup(filename);
/* reset flags unless first model (special case) */
      if (new_flag && current > model)
        {
/* new models beyond the first may be triggered by a repeat */
/* option (eg the second occurrance of name) in which case */
/* we only want to dec the flag (ie from 2 to 1 - NB: all other */
/* flags should be set to zero) so the parser won't make a new */
/* model if it bumps into another main data option (eg cell, cart...) */
        for (i=0 ; i<num_keyword ; i++)
          {
/* NEW - safer way of initializing flags */
          if (flag[i] >= 2)
            flag[i] = 1;
          else
            flag[i] = 0;
          }
        }
      }
    }

/* PROCESS FOR DATA - options only (ie 1st keyword) */
    switch(*(keyword+1))
      {
/* avoid keyword parsing the crap in a title/end block */
      case TITLE:
      case WEIGHT:
        g_string_sprintfa(extra, "%s\n", g_strstrip(line));
        while (*(keyword+1) != END)
          {
          if (fgetline(fp,line))
            goto done;
          g_string_sprintfa(extra, "%s\n", g_strstrip(line));
          keyword = get_keyword(line, MAX_KEYS);
          }
        break;

/* hack for getting an associated .trg file */
      case OUTPUT:
        buff = get_tokens(line, 5);
        if (g_ascii_strncasecmp(*(buff+1),"traj",4) == 0)
          {
          g_free(data->gulp.trj_file);
          data->gulp.trj_file = g_strdup(*(buff+2));
          data->animation = TRUE;
          }
        if (g_ascii_strncasecmp(*(buff+1),"mov",3) == 0)
          {
          if (data->gulp.mov_file);
            g_free(data->gulp.mov_file);
          data->gulp.mov_file = g_strdup(*(buff+3));
          }
        g_strfreev(buff);
        break;

/* avoid keyword parsing the crap in an observables/end block */
      case GULP_OBSERVABLES:
        g_string_sprintfa(extra, "%s\n", g_strstrip(line));
        while (*(keyword+1) != END)
          {
          if (fgetline(fp,line))
            goto done;
          g_string_sprintfa(extra, "%s\n", g_strstrip(line));
          keyword = get_keyword(line, MAX_KEYS);
          }
        break;
/* NEW stuff */
        case ENSEMBLE:
          buff = get_tokens(line, 3);
          if (g_ascii_strncasecmp(*(buff+1),"nve",3) == 0)
            data->gulp.ensemble = NVE;
          if (g_ascii_strncasecmp(*(buff+1),"nvt",3) == 0)
            data->gulp.ensemble = NVT;
          if (g_ascii_strncasecmp(*(buff+1),"npt",3) == 0)
            data->gulp.ensemble = NPT;
          g_strfreev(buff);
          break;
        case TEMPERATURE:
          buff = get_tokens(line, 3);
          sscanf(*(buff+1),"%lf",&(data->gulp.temp));
          g_strfreev(buff);
          break;
        case SUPER_CELL:
          buff = get_tokens(line, 5);
          sscanf(*(buff+1),"%d",&(data->gulp.super[0]));
          sscanf(*(buff+2),"%d",&(data->gulp.super[1]));
          sscanf(*(buff+3),"%d",&(data->gulp.super[2]));
          g_strfreev(buff);
          break;
/* model specific data - the *data ptr MUST be allocated */
        case NAME:
          g_free(data->basename);
          data->basename = g_strstrip(g_strdup(get_token_pos(line,1)));
          strcpy(data->filename, data->basename);
          break;
        case GULP_LIBRARY:
          g_free(data->gulp.libfile);
          data->gulp.libfile = g_strstrip(g_strdup(get_token_pos(line,1)));
          break;
        case SPACE:
/* anything on the rest of the line? */
          buff = get_tokens(line, 2);
/* position of actual space data */
          i = 1;
/* no, use the next line instead */
          if (!strlen(*(buff+1)))
            {
            fgetline(fp,line);
            i = 0;
            }
          tmp = g_strdup(g_strstrip(get_token_pos(line, i)));
          gen_flag=0;
          for (i=0 ; i<strlen(tmp) ; i++)
            {
/* any alphabetic chars => space *name* */
            if (g_ascii_isalpha(*(tmp+i)))
              {
              g_free(data->sginfo.spacename);
              data->sginfo.spacename = g_strdup_printf("%s",tmp);
/* indicate that name should used in lookup */
              data->sginfo.spacenum = -1;
              gen_flag++;
              break;
              }
            }
/* no alphabetic chars, assume space group number */
          if (!gen_flag)
            sscanf(tmp,"%3d",&data->sginfo.spacenum);

          g_strfreev(buff);
          g_free(tmp);
          break;
        case ORIGIN:
          buff = get_tokens(line, 5);
/* FIXME - could be coords & not just the cell choice */
          if (strlen(*(buff+3)))
            printf("Warning: unsupported origin specification.\n");
          else
            sscanf(*(buff+1),"%d",&(data->sginfo.cellchoice));
          g_strfreev(buff);
          break;

        case POLYMER_CELL:
          if (fgetline(fp,line))
            return(4);
          buff = get_tokens(line, 3);

          data->pbc[0] = str_to_float(*(buff+0));
          data->periodic = 1;
          break;

        case POLYMER_VECTOR:
          if (fgetline(fp,line))
            return(4);
          buff = get_tokens(line, 4);

          vec1[0] = str_to_float(*(buff+0));
          vec1[1] = str_to_float(*(buff+1));
          vec1[2] = str_to_float(*(buff+2));
          data->latmat[0] = vec1[0];
          data->latmat[3] = vec1[1];
          data->latmat[6] = vec1[2];
          data->construct_pbc = TRUE;
          data->periodic = 1;
          make_latmat(data);
          break;

/* read in gulp surface cell */
        case SURFACE_CELL:
          if (fgetline(fp,line))
            return(4);
          buff = get_tokens(line, 5);
          data->pbc[0] = str_to_float(*(buff+0));
          data->pbc[1] = str_to_float(*(buff+1));
          data->pbc[5] = str_to_float(*(buff+2));
          data->pbc[5] *= PI/180.0;
          data->periodic = 2;
          break;

/* read in gulp surface vectors */
/* NB: gdis wants transposed gulp matrix */
        case SURFACE_VECTORS:
          if (fgetline(fp,line))
            return(4);
          buff = get_tokens(line, 3);
/* is this always colinear to x? (FIXME - no) */
          vec1[0] = str_to_float(*(buff+0));
          vec2[0] = str_to_float(*(buff+1));
          g_strfreev(buff);
          if (fgetline(fp,line))
            return(5);
          buff = get_tokens(line, 3);
          vec1[1] = str_to_float(*(buff+0));
          vec2[1] = str_to_float(*(buff+1));
          g_strfreev(buff);
          VEC3SET(&(data->latmat[0]), vec1[0], vec1[1], 0.0);
          VEC3SET(&(data->latmat[3]), vec2[0], vec2[1], 0.0);
          VEC3SET(&(data->latmat[6]), 0.0, 0.0, 1.0);
          data->construct_pbc = TRUE;
          data->periodic = 2;
          make_latmat(data);
          break;

        case CELL:
/* check if anything is on the rest of the line */
          buff = get_tokens(line, 8);
          i = 1;
          if (!strlen(*(buff+6)))
            {
/* nope - get next line (if possible) */
            if (fgetline(fp,line))
              return(3);
            g_strfreev(buff);
            buff = get_tokens(line, 7);
/* exit with error if not enough tokens */
            if (!strlen(*(buff+5)))
              return(4);
            i = 0;
            }
/* parse for cell data */
          data->pbc[0] = str_to_float(*(buff+i+0));
          data->pbc[1] = str_to_float(*(buff+i+1));
          data->pbc[2] = str_to_float(*(buff+i+2));
          data->pbc[3] = str_to_float(*(buff+i+3));
          data->pbc[4] = str_to_float(*(buff+i+4));
          data->pbc[5] = str_to_float(*(buff+i+5));
          g_strfreev(buff);
/* hack to determine if 2D or 3D */
/* FIXME - possible for pbc[3] or pbc[4] to be 0 instead */
          if (fabs(data->pbc[2]) < FRACTION_TOLERANCE)
            data->periodic = 2;
          else
            data->periodic = 3;
/* degrees -> radians */
          data->pbc[3] *= PI/180.0;
          data->pbc[4] *= PI/180.0;
          data->pbc[5] *= PI/180.0;
          break;

/* read in gulp lattice vectors */
/* NB: gdis wants transposed gulp matrix */
        case LATTICE_VECTORS:
          if (fgetline(fp,line))
            return(4);
          buff = get_tokens(line, 4);
          sscanf(*(buff+0),"%lf",&vec1[0]);
          sscanf(*(buff+1),"%lf",&vec2[0]);
          sscanf(*(buff+2),"%lf",&vec3[0]);
          g_strfreev(buff);
          if (fgetline(fp,line))
            return(4);
          buff = get_tokens(line, 4);
          sscanf(*(buff+0),"%lf",&vec1[1]);
          sscanf(*(buff+1),"%lf",&vec2[1]);
          sscanf(*(buff+2),"%lf",&vec3[1]);
          g_strfreev(buff);
          if (fgetline(fp,line))
            return(4);
          buff = get_tokens(line, 4);
          sscanf(*(buff+0),"%lf",&vec1[2]);
          sscanf(*(buff+1),"%lf",&vec2[2]);
          sscanf(*(buff+2),"%lf",&vec3[2]);
          g_strfreev(buff);
/* NEW - use the supplied lattice matrix (it will be normalized, however) */
          ARR3SET(&(data->latmat[0]), vec1);
          ARR3SET(&(data->latmat[3]), vec2);
          ARR3SET(&(data->latmat[6]), vec3);
/* NB: data->pbc's are constructed in make_latmat() */
          data->construct_pbc = TRUE;
          data->periodic = 3;
          make_latmat(data);
          break;

/* main coord read in */
/* NB: errors -> break (not return) so we cope with empty regions */
        case S_FRAC:
        case FRAC:
          data->fractional = TRUE;
        case CART:
/* au check */
          buff = get_tokens(line,7);
          gen_flag = 0;
          if (g_ascii_strncasecmp("au",*(buff+1),2) == 0)
            gen_flag++;
/* get new line */
          if (fgetline(fp, line))
            break;
/* insufficient items => stupid gulp frac/cart \n number \n coords format */
          buff = tokenize(line, &num_tokens);
          if (num_tokens < 3)
            if (fgetline(fp, line))
              break;
          g_strfreev(buff);
          g_free(keyword);
          keyword = get_keyword(line, MAX_KEYS);
/* core,shell indices */
          i=j=0;
          while (!*keyword)
            {
            buff = get_tokens(line,7);
/* terminate on a blank/insufficient line */
            if (!strlen(*(buff+3)))
              goto last;
/* if an atom type (ie core/shell) is specified - adjust data positions */
            type = 0;
            if (g_ascii_strncasecmp(*(buff+1),"core",4) == 0)
              type=1;
            if (g_ascii_strncasecmp(*(buff+1),"shel",4) == 0)
              type=2;
/* read appropriately */
            switch(type)
              {
              case 0:
              case 1:

                core = new_core(*buff, data);
                data->cores = g_slist_prepend(data->cores, core);

                core->region = region;
                core->x[0] = str_to_float(*(buff+1+type));
                core->x[1] = str_to_float(*(buff+2+type));
                core->x[2] = str_to_float(*(buff+3+type));

                if (strlen(*(buff+4+type)))
                  {
                  core->lookup_charge = FALSE;
                  core->charge = str_to_float(*(buff+4+type));
                  }
                if (strlen(*(buff+5+type)))
                  {
                  data->has_sof = TRUE;
                  core->sof = str_to_float(*(buff+5+type));
                  }
                tmp = g_strdup(get_token_pos(line,4+type));
                strcpy(core->tail, g_strstrip(tmp));
                g_free(tmp);
                if (gen_flag)
                  {
                  VEC3MUL(core->x, AU2ANG);
                  }
                nc++;
                break;

              case 2:
                shell = new_shell(*buff, data);
                data->shels = g_slist_prepend(data->shels, shell);

                shell->region = region;
                shell->x[0] = str_to_float(*(buff+2));
                shell->x[1] = str_to_float(*(buff+3));
                shell->x[2] = str_to_float(*(buff+4));

                if (strlen(*(buff+5)))
                  {
                  shell->lookup_charge = FALSE;
                  shell->charge = str_to_float(*(buff+5));
                  }
                tmp = g_strdup(get_token_pos(line,5));
                strcpy(shell->tail, g_strstrip(tmp));
                g_free(tmp);
                if (gen_flag)
                  {
                  VEC3MUL(shell->x, AU2ANG);
                  }

                ns++;
                break;
              }
            g_strfreev(buff);
/* get new line & check for keywords */
            if (fgetline(fp, line))
              break;
            g_free(keyword);
            keyword = get_keyword(line, MAX_KEYS);
            }
last:;
/* finished coord read */
          break;

/* one liners... */
        case D_HKL:
          buff = get_tokens(line, 2);
          sscanf(*(buff+1),"%lf",&data->surface.dspacing);
          g_strfreev(buff);
        case SBULK_ENERGY:
          buff = get_tokens(line, 2);
          sscanf(*(buff+1),"%lf",&data->gulp.sbulkenergy);
          g_strfreev(buff);
          break;
        case TOTAL_ENERGY:
          buff = get_tokens(line, 2);
          sscanf(*(buff+1),"%lf",&data->gulp.energy);
          g_strfreev(buff);
          break;
        case SPECIES:
          while (!fgetline(fp,line))
            {
/* end on a keyword */
            g_free(keyword);
            keyword = get_keyword(line, MAX_KEYS);
            if (*keyword)
              break;
            buff = get_tokens(line, 4);
/* is the 1st item an element? (no - end) */
            i = elem_type(*buff);
            if (i && num_species < MAX_POTENTIALS)
              {
              *(species+num_species) = g_strdup(g_strstrip(line));
              num_species++;
              g_strfreev(buff);
              }
            else
              {
              g_strfreev(buff);
              break;
              }
            }
          break;

        case ELEMENT:
          while (!fgetline(fp,line))
            {
/* end on a non element data keyword */
            g_free(keyword);
            keyword = get_keyword(line, MAX_KEYS);
            if (*(keyword+1) != COVALENT && *(keyword+1) != IONIC)
                break;
            buff = get_tokens(line, 3);
/* end if insufficient data */
            if (!strlen(*(buff+2)))
              {
              g_strfreev(buff);
              break;
              }
/* is the 2nd item an element? (symbol check) */
            if (!elem_type(*(buff+1)))
              {
/* FIXME - a better way to cope with ugly fortran ** possibility? */
              if (*(*(buff+1)) == '*')
                {
                g_strfreev(buff);
/* ignore - skip to next line */
                continue;
                }
/* no - atomic number check */
              if (atoi(*(buff+1)) < 0 || atoi(*(buff+1)) > sysenv.num_elements)
                {
/* not a recognized element - exit element loop */
                g_strfreev(buff);
                break;
                }
              }
/* otherwise save if we have the room */
            if (num_elem < MAX_POTENTIALS)
              {
              *(elem+num_elem) = g_strdup(g_strstrip(line));
              num_elem++;
              }
            g_strfreev(buff);
            }
          break;

        case KPOINTS:
          if (fgetline(fp,line))
            return(5);
          buff = tokenize(line, &num_tokens);
          i = 0;
          while (i<3 && num_tokens)
            {
            kpoints[i] = str_to_float(*(buff+i));
            num_tokens--;
            i++;
            }
          num_kpoints++;
          break;

        case DUMP_FILE:
/* ignore, for the moment */
          break;
        case HARMONIC:
        case MORSE:
        case BUCKINGHAM:
        case LENNARD:
        case THREE:
        case UREYBRADLEY:
        case FOUR:
        case TORSION:
        case OUTOFPLANE:
        case SPRING:
          if (num_potentials < MAX_POTENTIALS)
            {
/* TODO - use only one gchar ptr for potential storage */
/* store potential type line */
            g_string_sprintf(pad,"%s\n", g_strstrip(line));
/* NEW - while not a keyword, keep tacking the lines on */
            while (!fgetline(fp,line))
              {
/* end on a keyword */
              g_free(keyword);
              keyword = get_keyword(line, MAX_KEYS);
              if (*keyword)
                break;
              g_string_sprintfa(pad,"%s\n", g_strstrip(line));
              }
            *(pot+num_potentials) = g_strdup(pad->str);
            num_potentials++;

/* if missing next line - don't include */
/*
            if (fgetline(fp,line))
              g_free(*(pot1+num_potentials));
            else
              {
              *(pot2+num_potentials) = g_strdup(g_strstrip(line));
              num_potentials++;
              }
*/
            }
          break;

        }
  }
/* finished with file reading */
done:;

/* summary */
#if DEBUG_LOAD_GULP
printf("-------------------------------------\n");
printf("        Structures found : %d [%d;%d]\n",current-model+1,model,current);
printf("                run type : %d\n",run);
printf("             method type : %d\n",method);
printf("          optimiser type : %d\n",optimiser);
printf("    secondary  optimiser : %d\n",optimiser2);
printf("             switch type : %d\n",switch_type);
printf("            switch start : %lf\n",switch_value);
printf("coulomb subtraction type : %d\n",coulomb);
printf("         iteration limit : %d\n",maxcyc);
printf("        potentials found : %d\n",num_potentials);
printf("      species data found : %d\n",num_species);
printf("      element data found : %d\n",num_elem);
printf("             atoms found : %d\n",nc);
printf("            shells found : %d\n",ns);
printf("-------------------------------------\n");
#endif

/* init models & exit */
for (i=model ; i<=current ; i++)
  {
  data = model_ptr(i, RECALL);
  data->mode = FREE;
  if (data == NULL)
    return(7);
  strcpy(data->filename, filename);

/* no depth info for loaded surfaces, so pretend they're MARVIN style */
  if (data->periodic == 2)
    data->surface.true_cell = FALSE;

/* transfer universal gulp data */
  data->gulp.run = run;
  data->gulp.method = method;
  data->gulp.optimiser = optimiser;
  if (optimiser2 > 0)
    {
    data->gulp.optimiser2 = optimiser2;
    data->gulp.switch_type = switch_type;
    data->gulp.switch_value = switch_value;
    }
  data->gulp.unit_hessian = unit;
  data->gulp.coulomb = coulomb;
  data->gulp.maxcyc = maxcyc;
  data->gulp.qeq = qeq;
  data->gulp.free = free;
  data->gulp.zsisa = zsisa;
  data->gulp.compare = compare;
  data->gulp.nosym = nosym;
  data->gulp.fix = fix;
  data->gulp.phonon = phonon;
  data->gulp.eigen = eigen;
  data->gulp.num_kpoints = num_kpoints;
  ARR3SET(data->gulp.kpoints, kpoints);

  if (num_potentials)
    {
    data->gulp.num_potentials = num_potentials;
    data->gulp.pot = g_malloc(num_potentials*sizeof(gchar *));
    for (j=0 ; j<num_potentials ; j++)
      *(data->gulp.pot+j) = g_strdup(*(pot+j));
    }

  if (num_species)
    {
    data->gulp.num_species = num_species;
    data->gulp.species = g_malloc(num_species*sizeof(gchar *));
    for (j=0 ; j<num_species ; j++)
      {
/* save */
      *(data->gulp.species+j) = g_strdup(*(species+j));
/* TODO - overwrite any coord line charges with species values */
      buff = tokenize(*(species+j), &num_keyword);
/* TODO - core/shell */
      type = elem_type(*buff);

      get_elem_data(type, &elem_data, data);

      if (g_ascii_strncasecmp(*(buff+1), "shel", 4) == 0)
        {
        elem_data.shell_charge = str_to_float(*(buff+2));

        for (list=data->shels ; list ; list=g_slist_next(list))
          {
          shell = (struct shel_pak *) list->data;
          if (shel_match(*buff, shell))
            {
            shell->charge = elem_data.shell_charge;
            shell->lookup_charge = FALSE;
            }
          }
        }
      else
        {
/* core can be omitted */
        if (g_ascii_strncasecmp(*(buff+1), "core", 4) == 0)
          elem_data.charge = str_to_float(*(buff+2));
        else
          elem_data.charge = str_to_float(*(buff+1));

        for (list=data->cores ; list ; list=g_slist_next(list))
          {
          core = (struct core_pak *) list->data;

          if (core_match(*buff, core))
            {
            core->charge = elem_data.charge;
            core->lookup_charge = FALSE;
            }
          }

        }
      g_strfreev(buff);
      }
    }

  if (num_elem)
    {
    data->gulp.num_elem = num_elem;
    data->gulp.elem = g_malloc(num_elem*sizeof(gchar *));
    for (j=0 ; j<num_elem ; j++)
      {
      *(data->gulp.elem+j) = g_strdup(*(elem+j));
/* process any cova data */
      if (g_ascii_strncasecmp(*(elem+j), "cova", 4) == 0)
        {
        buff = get_tokens(*(elem+j), 4);
        code = elem_type(*(buff+1));
/* no match - perhaps atomic number was specified */
        if (!code)
          code = atoi(*(buff+1));
        if (!get_elem_data(code, &elem_data, data))
          {
          elem_data.cova = str_to_float(*(buff+2));
          put_elem_data(&elem_data, data);
          }

        g_strfreev(buff);
        }
      }
    }
/* extra (unrecognized lines) */
  if (strlen(extra->str))
    data->gulp.extra = g_strdup(extra->str);

/* display init */
  prep_model(data);
  }

/* clean up & exit */
/* attached data to all model, now we can free it */
for (i=0 ; i<num_potentials ; i++)
  {
/*
printf("%s\n",*(pot+i));
*/
  g_free(*(pot+i));
  }
for (i=0 ; i<num_elem ; i++)
  {
/*
printf("%s\n",*(elem+i));
*/
  g_free(*(elem+i));
  }
for (i=0 ; i<num_species ; i++)
  {
/*
printf("%s\n",*(species+i));
*/
  g_free(*(species+i));
  }

/*
printf("super = %d %d %d\n",
  data->gulp.super[0],data->gulp.super[1],data->gulp.super[2]);
printf("ensemble = %d\n",data->gulp.ensemble);
printf("temp = %lf\n",data->gulp.temp);
*/

g_string_free(pad, TRUE);
g_string_free(extra, TRUE);
g_free(keyword);

fclose(fp);
return(0);
}

/*****************************/
/* generic gulp output parse */
/*****************************/
/* NB: new model is not made, as only energy etc. values */
/* are of interest if we're parsing the results of a gulp run */
#define DEBUG_READ_GULP_OUTPUT 0
gint read_gulp_output(gchar *filename, struct model_pak *data)
{
gint i, j, m, n, flag, model, region, primitive=FALSE;
gint count=0, dflag=0, fflag=0, pflag=0, read_pass=0, num_tokens;
gchar line[LINELEN], **buff, coord;
gdouble *value;
GSList *flist=NULL;
GString *title;
struct core_pak *core;
struct shel_pak *shell;
FILE *fp;

/* checks */
g_return_val_if_fail(data != NULL, 1);
g_return_val_if_fail(filename != NULL, 2);

fp = fopen(filename, "rt");
if (!fp)
  return(3);

/* init */
model = data->number;
data->region_empty[REGION1A] = FALSE;
data->gulp.esurf[0] = 0.0;
data->gulp.esurf[1] = 0.0;
data->gulp.eatt[0] = 0.0;
data->gulp.eatt[1] = 0.0;
if (data->gulp.epot_autoscale)
  {
  data->gulp.epot_max = -999999999.0;
  data->gulp.epot_min = 999999999.0;
  }

#if DEBUG_READ_GULP_OUTPUT
printf("Grafted: %d\n", data->grafted);
#endif

/* init for new model (ie not on the tree) */
if (!data->grafted)
  {
  data->id = GULPOUT;
  g_free(data->basename);
  data->basename = strdup_basename(filename);
  strcpy(data->filename, data->basename);
  data->region_empty[REGION1A] = FALSE;
  data->gulp.esurf[0] = 0.0;
  data->gulp.esurf[1] = 0.0;
  data->gulp.eatt[0] = 0.0;
  data->gulp.eatt[1] = 0.0;
  }

/* look for interesting output */
flag=0;
while(!fgetline(fp,line))
  {
/* GULP trapping */
  if (g_ascii_strncasecmp("!! ERROR",line,8) == 0)
    {
    show_text(ERROR, line);
    return(3);
    }
  if (g_ascii_strncasecmp("  **** Unit cell is not charge neutral",line,38) == 0)
    {
/* print the charge error plus the subsequent line, which gives more info */
    show_text(ERROR, line);
    if (!fgetline(fp,line))
      show_text(ERROR, line);
    return(4);
    }
  if (g_ascii_strncasecmp("  **** Warning", line, 14) == 0)
    show_text(WARNING, line);

/* periodicity */
  if (g_ascii_strncasecmp("  Dimensionality", line, 16) == 0)
    {
    buff = tokenize(line, &num_tokens); 
    if (num_tokens > 2)
      data->periodic = str_to_float(*(buff+2));
    g_strfreev(buff);
    }

/* space group */
  if (g_ascii_strncasecmp("  Space group ", line, 14) == 0)
    {
    if (!data->grafted)
      {
      for (i=strlen(line) ; i-- ; )
        {
        if (line[i] == ':')
          {
          data->sginfo.spacenum = -1;
          g_free(data->sginfo.spacename);
          data->sginfo.spacename = g_strdup(g_strstrip(&line[i+1]));

          if (data->sginfo.spacename[0] == 'R')
            primitive = TRUE;

#if DEBUG_READ_GULP_OUTPUT
printf("space group: %s\n", data->sginfo.spacename);
#endif
          }
        }
      }
    }

/* minimized energy search */
  if (g_ascii_strncasecmp("  Final energy =",line,16) == 0)
    {
    buff = tokenize(line, &num_tokens);
    if (num_tokens > 3)
      {
      data->gulp.energy = str_to_float(*(buff+3));
      flag++;
      }
    g_strfreev(buff);
    }

/* surface energy search */
  if (g_ascii_strncasecmp("  Surface energy ",line,17) == 0)
    {
    buff = tokenize(line, &num_tokens);
    if (num_tokens > 6)
      {
      if (data->gulp.esurf[0] == 0.0)
        data->gulp.esurf[0] = str_to_float(*(buff+5));
      else
        data->gulp.esurf[1] = str_to_float(*(buff+5));

      g_free(data->gulp.esurf_units);
      data->gulp.esurf_units = g_strdup(*(buff+6));
      }
    g_strfreev(buff);
    }

/* attachment energy search 1 */
  if (g_ascii_strncasecmp("  Attachment energy ",line,20) == 0)
    {
    buff = tokenize(line, &num_tokens);
    if (num_tokens > 4)
      {
      if (data->gulp.eatt[0] == 0.0)
        data->gulp.eatt[0] = str_to_float(*(buff+3));
      else
        data->gulp.eatt[1] = str_to_float(*(buff+3));
      g_free(data->gulp.eatt_units);
      data->gulp.eatt_units = g_strdup(*(buff+4));
      }
    g_strfreev(buff);
    }

/* attachment energy search 2 */
  if (g_ascii_strncasecmp("  Attachment energy/",line,20) == 0)
    {
    buff = tokenize(line, &num_tokens);
    if (num_tokens > 3)
      {
/* NB: assumes this match occurs just after the previous match */
      if (data->gulp.eatt[1] == 0.0)
        data->gulp.eatt[0] = str_to_float(*(buff+3));
      else
        data->gulp.eatt[1] = str_to_float(*(buff+3));
      g_free(data->gulp.eatt_units);
      data->gulp.eatt_units = g_strdup("eV/unit");
      }

    g_strfreev(buff);
    }

/* search for gnorm */
  if (g_ascii_strncasecmp("  Final Gnorm  =",line,16) == 0)
    {
    buff = tokenize(line, &num_tokens);
    if (num_tokens > 3)
      {
      data->gulp.gnorm = str_to_float(*(buff+3));
      flag++;
      }
    g_strfreev(buff);
    }

/* single point energy search */
  if (g_ascii_strncasecmp("  Total lattice energy",line,22) == 0)
    {
    buff = tokenize(line, &num_tokens);
/* if nothing on the rest of this line, then should be on the next */
    switch (num_tokens)
      {
      case 4:
      g_strfreev(buff);
      while(!fgetline(fp,line))
        {
        if (primitive)
          {
          if (g_ascii_strncasecmp("    Primitive unit cell",line,23) == 0)
            {
            buff = tokenize(line, &num_tokens);
            if (num_tokens > 5)
              data->gulp.energy = str_to_float(*(buff+4));
            g_strfreev(buff);
            break;
            }
          }
        else
          {
          if (g_ascii_strncasecmp("    Non-primitive unit cell",line,27) == 0)
            {
            buff = tokenize(line, &num_tokens);
            if (num_tokens > 5)
              data->gulp.energy = str_to_float(*(buff+4));
            g_strfreev(buff);
            break;
            }
          }
        }
      break;

      case 6:
      if (g_ascii_strncasecmp("eV",*(buff+5),2) == 0)
        {
        data->gulp.energy = str_to_float(*(buff+4));
        g_strfreev(buff);
        }
      break;

      }
    }

/* single point free energy search */
  if (g_ascii_strncasecmp("  Initial surface dipole ",line,25) == 0)
    {
    buff = tokenize(line, &num_tokens);
    if (num_tokens > 4)
      data->gulp.sdipole = str_to_float(*(buff+4));
    g_strfreev(buff);
    }

/* single point free energy search */
  if (g_ascii_strncasecmp("  Total free energy",line,19) == 0)
    {
    buff = tokenize(line, &num_tokens);
    if (num_tokens > 5)
      if (g_ascii_strncasecmp("eV",*(buff+5),2) == 0)
        data->gulp.energy = str_to_float(*(buff+4));
    g_strfreev(buff);
    }

/* NEW - vibrational info search */
  if (g_ascii_strncasecmp(" Frequencies (cm-1) and Eigenvectors",line,36) == 0)
    fflag++;
  if (g_ascii_strncasecmp("  Zero point energy",line,19) == 0)
    fflag=0;

  if (fflag && g_ascii_strncasecmp(" Frequency",line,10) == 0)
    {
/* space group processing here, since we want to attach */
/* the vibration lists to ALL atoms in the full cell, */
/* not just the asymmetric ones. */
    if (!data->grafted)
      {
      prep_model(data);
      pflag++;
      }

    buff = tokenize(line, &n);
    for (i=1 ; i<n ; i++)
      {
      flist = g_slist_prepend(flist, *(buff+i));
      }
/* NB: don't free the actual tokens as we've appended them to the list */
    g_free(buff);

/* skip to data */
    while (!fgetline(fp,line))
      {
      buff = tokenize(line, &n);
      if (n > 1)
        coord = **(buff+1);
      else
        coord = '?';
      g_strfreev(buff);
      if (coord == 'x')
        break;
      }

/* read the eigenvectors in */
    m=0;
    while (m<3*g_slist_length(data->cores))
      {
      buff = tokenize(line, &n);
/* blank line termination */
      if (!n)
        break;
/* get reference atom number (NB: GULP starts at 1, list starts at 0) */
      i = (gint) str_to_float(*(buff+0));
      i--;

      core = g_slist_nth_data(data->cores, i);
      g_assert(core != NULL);

/*
      if (i<0 || i>data->num_atoms-1)
        {
        printf("Bad line: %s\n", line);
        printf("ref atom: %d, num atoms: %d\n", i, data->num_atoms);
        g_assert_not_reached();
        }
*/
      for (j=2 ; j<n ; j++)
        {
/* get the value */
        value = g_malloc(sizeof(gdouble));
        *value = str_to_float(*(buff+j));
/* get reference coordinate */
        coord = **(buff+1);
        switch(coord)
          {
          case 'x':
            core->vibx_list = g_slist_append(core->vibx_list,
                                         (gpointer *) value);
            break;
          case 'y':
            core->viby_list = g_slist_append(core->viby_list,
                                         (gpointer *) value);
            break;
          case 'z':
            core->vibz_list = g_slist_append(core->vibz_list,
                                         (gpointer *) value);
            break;
          default:
            g_assert_not_reached();
          }
        }

      g_strfreev(buff);
/* next line (if available) */
      if (fgetline(fp,line))
        break;
      m++;
      }
    }

/* minimized coords search */
/* TODO - unminimized coords? */
/* added 'of surface' to avoid reading in growth slice */
  dflag=0;
  if (g_ascii_strncasecmp("  Final asymmetric unit coordinates",line,35) == 0
   || g_ascii_strncasecmp("  Final fractional coordinates",line,30) == 0
   || g_ascii_strncasecmp("  Fractional coordinates of asymmetric",line,38) == 0
   || g_ascii_strncasecmp("  Final fractional/Cartesian coordinates", line, 40) == 0
   || g_ascii_strncasecmp("  Mixed fractional/Cartesian coordinates of surface", line, 51) == 0)
    {
    data->fractional = TRUE;
    dflag++;
    }
/* don't read in coords if already on the tree (eg single pt) */
  if (dflag && !data->grafted)
    {
/* enforce empty core/shell lists */
    free_core_list(data);
/* skip the two (minimum number) ----- dividers */
    region = REGION1A;
    count=0;
    while (count < 2)
      {
      if (fgetline(fp,line))
        return(4);
      if (g_ascii_strncasecmp("-------", line, 7) == 0)
        count++;
      }

/* loop until we run out of atomic data lines */
    i=j=n=0;
    flag=0;
/* read as many atoms as we can */
    while(!fgetline(fp,line))
      {
/* TODO - dont exit if ----- (could be region divider) */
      if (g_ascii_strncasecmp("-------", line, 7) == 0)
        continue;
/* determine region */
      if (g_ascii_strncasecmp("  Region ", line, 9) == 0)
        {
        buff = get_tokens(line, 3);
        switch(*(*(buff+1)))
          {
          case '1':
#if DEBUG_READ_GULP_OUTPUT
printf("Labelling region 1...\n");
#endif
            region = REGION1A;
            break;
          case '2':
#if DEBUG_READ_GULP_OUTPUT
printf("Labelling region 2...\n");
#endif
            region = REGION2A;
            break;
          default:
            printf("WARNING: unknown region specification/\n");
            region = REGION1A;
          }
        g_strfreev(buff);
        data->region_empty[region] = FALSE;
        continue;
        }

/* NEW - sometimes GULP doesn't number sequentially eg core/shell transition */
      buff = get_tokens(line,11);
/* NEW - GULP sometimes puts *'s after coords (region 1 only?) */
      if (g_ascii_strncasecmp("*", *(buff+4), 1) == 0)
        {
/* shuffle to get coords in expected positions */
        g_free(*(buff+4)); 
        *(buff+4) = g_strdup(*(buff+5));
        g_free(*(buff+5)); 
        *(buff+5) = g_strdup(*(buff+7));
        }

/* exit only if insufficient data items (NB: changes b/w sections, min=6) */
      if (!strlen(*(buff+6)))
        break;
      else
        {
/* read the data in - assume the same order */
        if (g_ascii_strncasecmp("c", *(buff+2), 1) == 0)
          {
          core = new_core(*(buff+1), data);
          data->cores = g_slist_prepend(data->cores, core);

          if (!read_pass)
            core->region = region;
          core->x[0] = str_to_float(*(buff+3));
          core->x[1] = str_to_float(*(buff+4));
          core->x[2] = str_to_float(*(buff+5));
          core->charge = str_to_float(*(buff+6));
          core->lookup_charge = FALSE;
/*
          core->sof = str_to_float(*(buff+7));
*/
          i++;

#if DEBUG_READ_GULP_OUTPUT
printf("coords: %lf %lf %lf\n", core->x[0], core->x[1], core->x[2]);
#endif
          }

        if (g_ascii_strncasecmp("s", *(buff+2), 1) == 0)
          {
          shell = new_shell(*(buff+1), data);
          data->shels = g_slist_prepend(data->shels, shell);

          if (!read_pass)
            shell->region = region;
          shell->x[0] = str_to_float(*(buff+3));
          shell->x[1] = str_to_float(*(buff+4));
          shell->x[2] = str_to_float(*(buff+5));
          shell->charge = str_to_float(*(buff+6));
          shell->lookup_charge = FALSE;

#if DEBUG_READ_GULP_OUTPUT
printf("coords: %lf %lf %lf\n", shell->x[0], shell->x[1], shell->x[2]);
#endif

/* elem setup (colour, code etc.) */
          j++;
          }
        }
      g_strfreev(buff);
      }

/* done one pass => read in unrelaxed coords if reading optimization */
    read_pass++;

#if DEBUG_READ_GULP_OUTPUT
printf("Retrieved %d atoms & %d shells (periodic).\n",i,j);
#endif
    }

/* cell parameters - search 1 */
  if (g_ascii_strncasecmp("  Final cell parameters",line,22) == 0 && !data->grafted)
    {
/* skip to data */
    fgetline(fp,line);
    fgetline(fp,line);
/* get cell lengths */
    for (i=0 ; i<6 ; i++)
      {
      if (fgetline(fp,line))
        break;
      buff = get_tokens(line,3);
      sscanf(*(buff+1),"%lf",&data->pbc[i]);
      g_strfreev(buff);
      }
/* convert to radians */
    data->pbc[3] *= PI/180.0;
    data->pbc[4] *= PI/180.0;
    data->pbc[5] *= PI/180.0;
    }

/* cell parameters - search 2 */
  if (g_ascii_strncasecmp("  Non-primitive lattice parameters",line,34) == 0 && !data->grafted)
    {
/* skip to data */
    if (fgetline(fp,line))
      break;
    if (fgetline(fp,line))
      break;
/* get cell lengths */
    buff = get_tokens(line,9);
    sscanf(*(buff+2),"%lf",&data->pbc[0]);
    sscanf(*(buff+5),"%lf",&data->pbc[1]);
    sscanf(*(buff+8),"%lf",&data->pbc[2]);
    if (fgetline(fp,line))
      break;
/* get cell angles */
    g_strfreev(buff);
    buff = get_tokens(line,6);
    sscanf(*(buff+1),"%lf",&data->pbc[3]);
    sscanf(*(buff+3),"%lf",&data->pbc[4]);
    sscanf(*(buff+5),"%lf",&data->pbc[5]);
/* convert to radians */
    data->pbc[3] *= PI/180.0;
    data->pbc[4] *= PI/180.0;
    data->pbc[5] *= PI/180.0;
    g_strfreev(buff);
    }

/* cell parameters - search 3 */
  if (g_ascii_strncasecmp("  Final surface cell parameters ",line,32) == 0 && !data->grafted)
    {
    data->periodic = 2;
    data->fractional = TRUE;
    data->pbc[2] = 0.0;
    data->pbc[3] = PI/2.0;
    data->pbc[4] = PI/2.0;
/* skip to 1st line of data */
    fgetline(fp,line);
    fgetline(fp,line);
    if (fgetline(fp,line))
      break;
    buff = get_tokens(line,4);
    sscanf(*(buff+1),"%lf",&data->pbc[0]);
    g_strfreev(buff);

    if (fgetline(fp,line))
      break;
    buff = get_tokens(line,4);
    sscanf(*(buff+1),"%lf",&data->pbc[1]);
    g_strfreev(buff);

    if (fgetline(fp,line))
      break;
    buff = get_tokens(line,4);
    sscanf(*(buff+1),"%lf",&data->pbc[5]);
/* convert to radians */
    data->pbc[5] *= PI/180.0;
    g_strfreev(buff);
    }

/* cell parameters - search 4 (eg a CONV surface calc) */
  if (g_ascii_strncasecmp("  Surface cell parameters ",line, 26) == 0 && !data->grafted)
    {
    data->periodic = 2;
    data->fractional = TRUE;
    data->pbc[2] = 0.0;
    data->pbc[3] = PI/2.0;
    data->pbc[4] = PI/2.0;
/* skip to 1st line of data */
    fgetline(fp,line);
    if (fgetline(fp,line))
      break;
    buff = get_tokens(line,6);
    sscanf(*(buff+2),"%lf",&data->pbc[0]);
    sscanf(*(buff+5),"%lf",&data->pbc[5]);
/* convert to radians */
    data->pbc[5] *= PI/180.0;
    g_strfreev(buff);
    if (fgetline(fp,line))
      break;
    buff = get_tokens(line,10);
    sscanf(*(buff+2),"%lf",&data->pbc[1]);
    g_strfreev(buff);
    }

/* cell parameters - search 5 (unrelaxed) */
  if (g_ascii_strncasecmp("  Cell parameters (Angstroms/Degrees):",line,38) == 0 && !data->grafted)
    {
/* skip blank line */
    fgetline(fp,line);
/* get cell lengths */
    for (i=0 ; i<3 ; i++)
      {
      if (fgetline(fp,line))
        break;
      buff = get_tokens(line,7);
/* length */
      sscanf(*(buff+2),"%lf",&data->pbc[i]);
/* angle */
      sscanf(*(buff+5),"%lf",&data->pbc[i+3]);
      g_strfreev(buff);
      }
/* convert to radians */
    data->pbc[3] *= PI/180.0;
    data->pbc[4] *= PI/180.0;
    data->pbc[5] *= PI/180.0;
    }

/* cell parameters - search 6 */
  if (g_ascii_strncasecmp("  Surface Cartesian vectors (Angstroms) :",line,41) == 0 && !data->grafted)
    {
    data->periodic = 2;
    data->construct_pbc = TRUE;
/* skip blank line */
    fgetline(fp,line);
/* get vec 1 */
    fgetline(fp,line);
    buff = get_tokens(line,4);
    data->latmat[0] = str_to_float(*buff);
    data->latmat[3] = str_to_float(*(buff+1));
    data->latmat[6] = 0.0;
    g_strfreev(buff);
/* get vec 2 */
    fgetline(fp,line);
    buff = get_tokens(line,4);
    data->latmat[1] = str_to_float(*buff);
    data->latmat[4] = str_to_float(*(buff+1));
    data->latmat[7] = 0.0;
/* set vec 3 */
    data->latmat[2] = 0.0;
    data->latmat[5] = 0.0;
    data->latmat[8] = 1.0;
    g_strfreev(buff);
    }

/* cell parameters - search 8 */
  if (g_ascii_strncasecmp("  Primitive cell parameters :",line,29) == 0 && !data->grafted)
    {
    data->periodic = 3;
/* skip blank line */
    fgetline(fp,line);
/* get line 1 */
    fgetline(fp,line);
    buff = get_tokens(line,12);
    data->pbc[0] = str_to_float(*(buff+8));
    data->pbc[3] = str_to_float(*(buff+11));
    g_strfreev(buff);
/* get line 2 */
    fgetline(fp,line);
    buff = get_tokens(line,12);
    data->pbc[1] = str_to_float(*(buff+8));
    data->pbc[4] = str_to_float(*(buff+11));
    g_strfreev(buff);
/* get line 2 */
    fgetline(fp,line);
    buff = get_tokens(line,12);
    data->pbc[2] = str_to_float(*(buff+8));
    data->pbc[5] = str_to_float(*(buff+11));
    g_strfreev(buff);
/* convert to radians */
    data->pbc[3] *= PI/180.0;
    data->pbc[4] *= PI/180.0;
    data->pbc[5] *= PI/180.0;
    }

/* cell parameters - search 9 */
  if (g_ascii_strncasecmp("  Polymer cell parameter",line,24) == 0 && !data->grafted)
    {
/* init */
    data->periodic = 1;
    data->construct_pbc = FALSE;
/* blank line skip */
    fgetline(fp,line);
    fgetline(fp,line);
    buff = tokenize(line, &num_tokens);
    if (num_tokens > 2)
      data->pbc[0] = str_to_float(*(buff+2));
    }

/* cell parameters - search 10 */
  if (g_ascii_strncasecmp("  Final polymer cell parameter",line,30) == 0 && !data->grafted)
    {
/* init */
    data->periodic = 1;
    data->construct_pbc = FALSE;
/* blank line skip */
    fgetline(fp,line);
    fgetline(fp,line);
    fgetline(fp,line);
    buff = tokenize(line, &num_tokens);
    if (num_tokens > 2)
      data->pbc[0] = str_to_float(*(buff+1));
    }

/* isolated coords search */
  if ((g_ascii_strncasecmp("  Final cartesian coordinates",line,29) == 0 
   ||  g_ascii_strncasecmp("  Cartesian coordinates of cluster",line,34) == 0) 
   && !data->grafted)
    {
    data->periodic = 0;
    data->fractional = FALSE;

/* enforce empty core/shell lists */
    free_core_list(data);

    for (i=0 ; i<5 ; i++)
      fgetline(fp,line);
/* loop until we run out of atomic data lines */
    i=j=n=0;
    flag=0;
/* don't read any more than num_atoms -> memory allocation problems! */
    while(!fgetline(fp,line) && !flag)
      {
      buff = get_tokens(line,8);
/* exit if atom number is not consecutive */ 
      sscanf(*buff,"%d",&m);
      if (m != ++n)
        flag=1;
      else
        {
/* read the data in - assume the same order */
        if (g_ascii_strncasecmp("c",*(buff+2),1) == 0)
          {
          core = new_core(*(buff+1), data);
          data->cores = g_slist_prepend(data->cores, core);

          core->x[0] = str_to_float(*(buff+3));
          core->x[1] = str_to_float(*(buff+4));
          core->x[2] = str_to_float(*(buff+5));
          core->charge = str_to_float(*(buff+6));
          core->lookup_charge = FALSE;
          }

        if (g_ascii_strncasecmp("s",*(buff+2),1) == 0)
          {
          shell = new_shell(*(buff+1), data);
          data->shels = g_slist_prepend(data->shels, shell);

          shell->x[0] = str_to_float(*(buff+3));
          shell->x[1] = str_to_float(*(buff+4));
          shell->x[2] = str_to_float(*(buff+5));
          shell->charge = str_to_float(*(buff+6));
          shell->lookup_charge = FALSE;
          }
        }
      g_strfreev(buff);
      }

#if DEBUG_READ_GULP_OUTPUT
printf("Retrieved %d atoms & %d shells (cluster).\n",i,j);
#endif
    }

/* NEW */
  if (g_ascii_strncasecmp("  Electrostatic potential at input sites", line, 40) == 0)
    {
/* skip to data */
    for (i=5 ; i-- ; )
      fgetline(fp,line);

/* clean the list */
    if (data->gulp.epot_vals)
      free_slist(data->gulp.epot_vals);
    data->gulp.epot_vals = NULL;

/* acquire lines */
    while (!fgetline(fp, line))
      {
      if (g_ascii_strncasecmp("----------",line,10) == 0)
        break;

      buff = tokenize(line, &num_tokens);
      if (num_tokens > 4)
        {
/* get potential value */
        value = g_malloc(sizeof(gdouble));
        *value = str_to_float(*(buff+4));
        data->gulp.epot_vals = g_slist_prepend(data->gulp.epot_vals, value);

/* record min & max (if scale is not forced by the user) */
        if (data->gulp.epot_autoscale)
          {
          if (*value > data->gulp.epot_max)
            data->gulp.epot_max = *value;
          if (*value < data->gulp.epot_min)
            data->gulp.epot_min = *value;
          }
        }
      g_strfreev(buff);
      }

/* NB: we use list prepend -> reverse to get matching direction with epot_vecs */
    data->gulp.epot_vals = g_slist_reverse(data->gulp.epot_vals); 

/* confirm that we got sufficient points */
#if DEBUG_READ_GULP_OUTPUT
    printf("got: %d, expected: %d\n",
           g_slist_length(data->gulp.epot_vecs),
           g_slist_length(data->gulp.epot_vals));
    if (data->gulp.epot_autoscale)
      printf("min: %f, max: %f\n", data->gulp.epot_min, data->gulp.epot_max);
#endif
    }

/* deprec. - electrostatic potential grid */
  if (g_ascii_strncasecmp("  Electrostatic potential on a grid", line, 35) == 0)
    {
    printf("Epot data found!\n");
    fgetline(fp,line);
    fgetline(fp,line);
    buff = get_tokens(line,12);

    data->gulp.grid_pts[0] = (gint) str_to_float(*(buff+3));
    data->gulp.grid_pts[1] = (gint) str_to_float(*(buff+5));
    data->gulp.grid_pts[2] = (gint) str_to_float(*(buff+7));
    g_strfreev(buff);

/* alloc */
    n = data->gulp.grid_pts[0] * data->gulp.grid_pts[1] * data->gulp.grid_pts[2];
    data->gulp.epot_grid = g_malloc0(n * sizeof(gdouble));

    for (i=5 ; i-- ; )
      fgetline(fp,line);
/* acquire points */
    m = 0;
    while (!fgetline(fp,line))
      {
      if (g_ascii_strncasecmp("----------",line,10) == 0)
        break;

      buff = get_tokens(line,9);

/* get potential value */
      *(data->gulp.epot_grid+m) = str_to_float(*(buff+3));

      g_strfreev(buff);
      m++;
      }

    if (m != n)
      show_text(ERROR, "ERROR: Insufficient potential grid points.\n");
    }
  }

#if DEBUG_READ_GULP_OUTPUT
P3VEC("cell lengths ",&data->pbc[0]);
P3VEC("cell angles ",&data->pbc[3]);
#endif

/* surfaces are always const vol */
if (data->periodic == 2)
  data->gulp.method = CONV;

/* init & save the list */
if (flist)
  data->phonons = g_slist_reverse(flist);
else
  data->phonons = NULL;
data->num_phonons = g_slist_length(data->phonons);

#if DEBUG_READ_GULP_OUTPUT
printf("vibrational eigenvectors: %d\n", data->num_phonons);
if (data->num_phonons)
  dump_phonons(data);
#endif

g_free(data->title);
title = g_string_new("");
g_string_append_printf(title, "E");
g_string_append_printf(title, " = %.5f eV", data->gulp.energy);
if (data->gulp.gnorm >= 0.0)
  g_string_append_printf(title, ", gnorm = %.5f", data->gulp.gnorm);
data->title = g_strdup(title->str);
g_string_free(title, TRUE);


/* init for display (if not already displayed) */
if (!data->grafted && !pflag)
  prep_model(data);

fclose(fp);
return(0);
}

/**************************************/
/* show the current file stream error */
/**************************************/
void print_file_error(FILE *fp)
{

printf(">>> [offet = %ld] ", ftell(fp));

if (feof(fp))
  printf("error = EOF\n");
else
  printf("error = %d\n", ferror(fp));

clearerr(fp);
}

/***********************************/
/* GULP dynamics trajectory header */
/***********************************/
/* macro for fortran start/end record padding */
#define RECORD fread(&pad, sizeof(int), 1, fp)

/* NB: I experimented with fseek AND fsetpos, but these seem to fail */
/* with EOF errors even tho ftell reports a position < file size */

#define DEBUG_TRJ_HEADER 0
gint read_trj_header(FILE *fp, struct model_pak *model)
{
int pad, num_atoms, periodic;
double version;

/* record 1 - version */
RECORD;
fread(&version, sizeof(version), 1, fp);
if (fabs(version) > 100.0)
  {
  swap_bytes(&version, sizeof(version));
  if (fabs(version) > 100.0)
    {
    printf("Error: file seems garbled.\n");
    return(1);
    }
  model->trj_swap = TRUE;
  }
RECORD;
/* record 2 */
RECORD;
/* # of atoms + shells */
fread(&num_atoms, sizeof(num_atoms), 1, fp);
if (model->trj_swap)
  swap_bytes(&num_atoms, sizeof(num_atoms));
/* dimension */
fread(&periodic, sizeof(periodic), 1, fp);
if (model->trj_swap)
  swap_bytes(&periodic, sizeof(periodic));
RECORD;

#if DEBUG_TRJ_HEADER
printf("trj version: %lf\n", version);
printf(" Swap bytes: %d\n", model->trj_swap);
printf("total atoms: %d\n", num_atoms);
#endif

return(0);
}

/**********************************/
/* GULP dynamics trajectory frame */
/**********************************/
#define DEBUG_TRJ_FRAME 0
gint read_trj_frame(FILE *fp, struct model_pak *model, gint replace_coords)
{
gint i, j;
int pad, num_atoms, periodic;
double time, ke, pe, temp, *x[6];
double cell[9], velc[9];
GSList *list;
struct core_pak *core;
struct shel_pak *shell;

/* standard frame read */
num_atoms = model->expected_cores + model->expected_shells;
periodic = model->periodic;

/* alloc for x,y,z & vx, vy, vz */
for (j=0 ; j<6 ; j++)
  x[j] = g_malloc(num_atoms * sizeof(double));

/* read one frame */
RECORD;

if (!fread(&time, sizeof(time), 1, fp))
  print_file_error(fp);
if (model->trj_swap)
  swap_bytes(&time, sizeof(time));

if (!fread(&ke, sizeof(ke), 1, fp))
  print_file_error(fp);
if (model->trj_swap)
  swap_bytes(&ke, sizeof(ke));

if (!fread(&pe, sizeof(pe), 1, fp))
  print_file_error(fp);
if (model->trj_swap)
  swap_bytes(&pe, sizeof(pe));

if (!fread(&temp, sizeof(temp), 1, fp))
  print_file_error(fp);
if (model->trj_swap)
  swap_bytes(&temp, sizeof(temp));

RECORD;

#if DEBUG_TRJ_FRAME
printf("[ %1dD ][ %d atoms]", periodic, num_atoms);
printf("[ time : %.2lf ]",time);
printf("[ ke : %.2lf ]",ke);
printf("[ pe : %.2lf ]",pe);
printf("[ temp : %.2lf ]\n",temp);
#endif

/* loop over x,y,z & assoc velocity components */
for (j=0 ; j<6 ; j++)
  {
/* NB: cores first, then shells */
  RECORD;
  for (i=0 ; i<num_atoms ; i++)
    if (!fread(x[j]+i, sizeof(double), 1, fp))
      {
      print_file_error(fp);
      break;
      }
  RECORD;
  }

/* write core data */
list = model->cores;
if (replace_coords)
  for (i=0 ; i<model->expected_cores ; i++)
    {
    if (list)
      core = (struct core_pak *) list->data;
    else
      {
      core = new_core("X", model);
      model->cores = g_slist_append(model->cores, core);
      list = g_slist_last(model->cores);
      }

    core->x[0] = *(x[0]+i);
    core->x[1] = *(x[1]+i);
    core->x[2] = *(x[2]+i);

    core->v[0] = *(x[3]+i);
    core->v[1] = *(x[4]+i);
    core->v[2] = *(x[5]+i);
  
    list = g_slist_next(list);
    }

/* write shells data */
list = model->shels;
if (replace_coords)
  for (i=0 ; i<model->expected_shells ; i++)
    {
    if (list)
      shell = (struct shel_pak *) list->data;
    else
      {
      shell = new_shell("X", model);
      model->shels = g_slist_append(model->shels, shell);
      list = g_slist_last(model->shels);
      }

    shell->x[0] = *(x[0]+i);
    shell->x[1] = *(x[1]+i);
    shell->x[2] = *(x[2]+i);
  
    shell->v[0] = *(x[3]+i);
    shell->v[1] = *(x[4]+i);
    shell->v[2] = *(x[5]+i);
  
    list = g_slist_next(list);
    }

/* read cell info */
if (model->gulp.ensemble == NPT)
  {
/* get cell vectors */
  RECORD;
  fread(cell, sizeof(double), 9, fp);
  RECORD;
/* in help.txt the 9 should be nstrains, but wtf is nstrains??? */
  RECORD;
  fread(velc, sizeof(double), 9, fp);
  RECORD;
/* compute latmat/ilatmat */
  memcpy(model->latmat, cell, 9*sizeof(gdouble));
  model->construct_pbc = TRUE;
  make_latmat(model); 
  }

/* convert input cartesian to fractional */
if (replace_coords)
  if (model->fractional)
    latmat_fudge(model);

/* clean up */
for (j=0 ; j<6 ; j++)
  g_free(x[j]);

return(0);
}

