/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#ifndef __WIN32
#include <sys/param.h>
#endif

#include "gdis.h"
#include "coords.h"
#include "file.h"
#include "parse.h"
#include "matrix.h"
#include "interface.h"

/* main structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/* a SIESTA special... */
struct species_pak
{
gint number;   /* can be -ve -> ghost */
gchar *label;  /* same as core->label */
};

/****************/
/* file writing */
/****************/
gint write_fdf(gchar *filename, struct model_pak *data)
{
gint i, code;
gdouble x[3], depth;
GSList *list, *clist, *species_list, *normal_list, *ghost_list;
struct core_pak *core;
struct species_pak *species_data;
FILE *fp;

/* checks */
g_return_val_if_fail(data != NULL, 1);
g_return_val_if_fail(filename != NULL, 2);

/* open the file */
fp = fopen(filename,"wt");
if (!fp)
  return(3);

/* print header */
fprintf(fp, "# ");
gdis_blurb(fp);
fprintf(fp, "#\n\n");
fprintf(fp, "SystemLabel      %s\n\n", data->basename);
fprintf(fp, "NumberOfAtoms    %d\n\n", g_slist_length(data->cores));

/* init the normal, ghost, and overall species lists */
normal_list = find_unique(LABEL_NORMAL, data);
ghost_list = find_unique(LABEL_GHOST, data);
species_list = NULL;

i = g_slist_length(normal_list) + g_slist_length(ghost_list);
fprintf(fp, "NumberOfSpecies  %d\n", i);

/* write the species (unique atom types) block */
fprintf(fp, "%%block ChemicalSpeciesLabel\n");

i=1;
for (list=normal_list ; list ; list=g_slist_next(list))
  {
  code = elem_type(list->data);
  fprintf(fp, "  %3d  %3d  %s\n", i, code, (gchar *) list->data);
  i++;

/* update the species identifier list */
  species_data = g_malloc(sizeof(struct species_pak));
  species_data->label = list->data;
  species_data->number = code;
  species_list = g_slist_append(species_list, species_data);
  }

for (list=ghost_list ; list ; list=g_slist_next(list))
  {
  code = elem_type(list->data);
  fprintf(fp, "  %3d  %3d  %s\n", i, -code, (gchar *) list->data);
  i++;

/* update the species identifier list */
  species_data = g_malloc(sizeof(struct species_pak));
  species_data->label = list->data;
  species_data->number = -code;
  species_list = g_slist_append(species_list, species_data);
  }
fprintf(fp, "%%endblock ChemicalSpeciesLabel\n\n");


/* write the lattice data */
if (data->periodic)
  {
  fprintf(fp, "LatticeConstant 1.0 Ang\n");
  if (data->construct_pbc)
    {
/* NB: siesta matrices are transposed wrt gdis */
    fprintf(fp, "%%block LatticeVectors\n");
    fprintf(fp,"%15.10f %15.10f %15.10f\n",
                data->latmat[0], data->latmat[3], data->latmat[6]);
    fprintf(fp,"%15.10f %15.10f %15.10f\n",
                data->latmat[1], data->latmat[4], data->latmat[7]);
    fprintf(fp,"%15.10f %15.10f %15.10f\n",
                data->latmat[2], data->latmat[5], data->latmat[8]);
    fprintf(fp, "%%endblock LatticeVectors\n\n");
    }
  else
    {
    fprintf(fp, "%%block LatticeParameters\n");
/* if it's a surface - print Dhkl (mult by region sizes) */
    if (data->periodic == 2)
      {
/* get depth info */
      depth = (data->surface.region[0]+data->surface.region[1])
                                          *data->surface.depth;
/* no depth info - make it large enough to fit everything */
      if (depth < POSITION_TOLERANCE)
        depth = 2.0*data->rmax;

      fprintf(fp, "  %f  %f  %f", data->pbc[0], data->pbc[1], depth);
      }
    else
      fprintf(fp, "  %f  %f  %f", data->pbc[0], data->pbc[1], data->pbc[2]);

    fprintf(fp, "  %f  %f  %f\n", R2D*data->pbc[3], R2D*data->pbc[4], R2D*data->pbc[5]);
    fprintf(fp, "%%endblock LatticeParameters\n\n");
    }
  }

/* write the atoms - for surfaces (and isolated molecules), coords must be cartesian */
if (data->fractional)
  fprintf(fp, "AtomicCoordinatesFormat Fractional\n");
else
  fprintf(fp, "AtomicCoordinatesFormat NotScaledCartesianAng\n");

fprintf(fp, "%%block AtomicCoordinatesAndAtomicSpecies\n");

for (clist=data->cores ; clist ; clist=g_slist_next(clist))
  {
  core = (struct core_pak *) clist->data;
  if (core->status & DELETED)
    continue;

  ARR3SET(x, core->x);

/* NB: want fractional if 3D periodic, otherwise cartesian */
  if (!data->fractional)
    vecmat(data->latmat, x);

/* find corresponding number of this element in the species block */
  i=1;
  for (list=species_list ; list ; list=g_slist_next(list))
    {
    species_data = (struct species_pak *) list->data;

    if (g_ascii_strcasecmp(core->label, species_data->label) == 0)
      {
      if (core->ghost)
        {
        if (species_data->number == -core->atom_code)
          break;
        }
      else
        {
        if (species_data->number == core->atom_code)
          break;
        }
      }
    i++;
    }

/* found? */
  if (list)
    fprintf(fp,"  %14.9f  %14.9f  %14.9f    %d\n", x[0], x[1], x[2], i);
  else
    printf("write_fdf() error: bad species type.\n");
  }
fprintf(fp, "%%endblock AtomicCoordinatesAndAtomicSpecies\n\n");

free_slist(species_list);

g_slist_free(normal_list);
g_slist_free(ghost_list);

fclose(fp);
return(0);
}

/**************************/
/* fdf input file reading */
/**************************/
#define DEBUG_READ_FDF 0
gint read_fdf(gchar *filename, struct model_pak *data)
{
gint i, num_tokens, num_atoms=0;
gchar **buff, *text;
GSList *species_list;
struct core_pak *core;
struct species_pak *species_data;
FILE *fp;

/* checks */
g_return_val_if_fail(data != NULL, 1);
g_return_val_if_fail(filename != NULL, 2);

fp = fopen(filename, "rt");
if (!fp)
  return(3);

species_list = NULL;
for (;;)
  {
  buff = get_tokenized_line(fp, &num_tokens);
  if (!buff)
    break;

  if (g_ascii_strncasecmp("NumberOfAtoms", *buff, 13) == 0)
    {
    if (num_tokens > 1)
      num_atoms = (gint) str_to_float(*(buff+1));
    }

  if (g_ascii_strncasecmp("SystemLabel", *buff, 11) == 0)
    {
    if (num_tokens > 1)
      {
      g_free(data->basename);
      data->basename = g_strdup(*(buff+1));
      }
    }

  if (g_ascii_strncasecmp("\%block", *buff, 6) == 0 && num_tokens > 1)
    {
    if  (g_ascii_strncasecmp("ChemicalSpecies", *(buff+1), 15) == 0)
      {
      for (;;)
        {
        g_strfreev(buff);
        buff = get_tokenized_line(fp, &num_tokens);
        if (!buff)
           break;
        if (g_ascii_strncasecmp("\%endblock", *buff, 9) == 0)
          break;

        if (num_tokens > 2)
          {
          species_data = g_malloc(sizeof(struct species_pak));

          species_data->number = str_to_float(*(buff+1));
          species_data->label = g_strdup(*(buff+2));
          species_list = g_slist_append(species_list, species_data);
          }
        }
      }
    }

  if (g_ascii_strncasecmp("\%block", *buff, 6) == 0 && num_tokens > 1)
    {
    if  (g_ascii_strncasecmp("LatticeParameters", *(buff+1), 17) == 0)
      {
      g_strfreev(buff);
      buff = get_tokenized_line(fp, &num_tokens);
      if (num_tokens > 5)
        {
        data->pbc[0] = str_to_float(*(buff+0));
        data->pbc[1] = str_to_float(*(buff+1));
        data->pbc[2] = str_to_float(*(buff+2));
        data->pbc[3] = D2R*str_to_float(*(buff+3));
        data->pbc[4] = D2R*str_to_float(*(buff+4));
        data->pbc[5] = D2R*str_to_float(*(buff+5));
        }
/* hack for determining if it's a (GDIS created) surface */
      if (fabs(data->pbc[2]) < FRACTION_TOLERANCE)
        data->periodic = 2;
      else
        data->periodic = 3;
      }

    if  (g_ascii_strncasecmp("LatticeVectors", *(buff+1), 14) == 0)
      {
      for (i=0 ; i<3 ; i++)
        {
        g_strfreev(buff);
        buff = get_tokenized_line(fp, &num_tokens);

        data->latmat[0+i] = str_to_float(*(buff+0));
        data->latmat[3+i] = str_to_float(*(buff+1));
        data->latmat[6+i] = str_to_float(*(buff+2));
        }
      data->construct_pbc = TRUE;
      data->periodic = 3;
      }
    }

  if (g_ascii_strncasecmp("AtomicCoordinatesFormat", *buff, 23) == 0)
    {
    if (num_tokens > 1)
      {
      if (g_ascii_strncasecmp("Fractional", *(buff+1), 10) == 0)
        data->fractional = TRUE;
      else
        data->fractional = FALSE;
      }
    else
      printf("WARNING: unexpected fdf format reading atomic coordinates type.\n");
    }

  if (g_ascii_strncasecmp("\%block", *buff, 6) == 0 && num_tokens > 1)
    {
    if (g_ascii_strncasecmp("AtomicCoordinates", *(buff+1), 17) == 0)
      {
      for (;;)
        {
        g_strfreev(buff);
        buff = get_tokenized_line(fp, &num_tokens);
        if (!buff)
          break;
        if (g_ascii_strncasecmp("\%endblock", *buff, 9) == 0)
          break;

        if (num_tokens > 3)
          {
/* find corresponding number of this element in the species block */
/* NB: list counts from 0, fdf counts from 1 (hence the minus 1) */
          if (species_list)
            {
            i = abs((gint) str_to_float(*(buff+3)) - 1);
            species_data = g_slist_nth_data(species_list, i);

            core = new_core(species_data->label, data);

/* translucent ghost atom */
            if (species_data->number < 0)
              {
              core->ghost = TRUE;
              core->colour[3] = 0.5;
              }
            }
          else
            core = new_core("X", data);

          data->cores = g_slist_prepend(data->cores, core);

          core->x[0] = str_to_float(*(buff+0));
          core->x[1] = str_to_float(*(buff+1));
          core->x[2] = str_to_float(*(buff+2));
          }
        }
      }
    }

/* loop cleanup */
  g_strfreev(buff);
  }

/* free the species list */
free_slist(species_list);

/* check cores */
data->cores = g_slist_reverse(data->cores);
i = g_slist_length(data->cores);
if (num_atoms != i)
  {
  text = g_strdup_printf("Inconsistancy reading %s: expected %d cores, but found %d.\n",
                          filename, num_atoms, i);
  show_text(ERROR, text);
  g_free(text);
  }

/* model setup */
strcpy(data->filename, filename);

prep_model(data);

return(0);
}

/*******************************************/
/* read single SIESTA output configuration */
/*******************************************/
#define DEBUG_READ_SOUT 0
gint read_sout_block(FILE *fp, struct model_pak *data)
{
gint num_tokens;
gchar **temp, **buff, line[LINELEN];
GString *title;
GSList *clist;
struct core_pak *core;

clist = data->cores;

/* get 1st line of coords */
if (fgetline(fp, line))
  return(1);
buff = tokenize(line, &num_tokens);

while (num_tokens > 4)
  {
  if (clist)
    {
    core = (struct core_pak *) clist->data;
    clist = g_slist_next(clist);
    }
  else
    {
    core = new_core(*(buff+4), data);
    data->cores = g_slist_append(data->cores, core);
    }

  core->x[0] = str_to_float(*(buff+0));
  core->x[1] = str_to_float(*(buff+1));
  core->x[2] = str_to_float(*(buff+2));

/* get next line */
  g_strfreev(buff);
  if (fgetline(fp, line))
    return(2);
  buff = tokenize(line, &num_tokens);
  }
g_strfreev(buff);

/* attempt to get the lattice matrix */
/* if not found, use the initial values */
data->construct_pbc = FALSE;
while (!fgetline(fp, line))
  {
/* if next frame encoutered - assume there is no outcell data & exit */
  if (g_ascii_strncasecmp(line, "siesta:        Begin CG move", 28) == 0)
    break;
  /* new version of siests doesn't have the siesta: */
  if (g_ascii_strncasecmp(line, "                            Begin CG move", 41) == 0)
    break;

/* acquire cell lengths */
  if (g_ascii_strncasecmp(line, "outcell: Cell vector modules", 28) == 0)
    {
    temp = g_strsplit(line, ":", 3);
    buff = tokenize(*(temp+2), &num_tokens);
    g_strfreev(temp);

    if (num_tokens > 2)
      {
      data->pbc[0] = str_to_float(*(buff+0));
      data->pbc[1] = str_to_float(*(buff+1));
      data->pbc[2] = str_to_float(*(buff+2));
      }
    else
      printf("Unexpected data format reading cell lengths.\n");
    g_strfreev(buff);
    }

/* acquire cell angles */
  if (g_ascii_strncasecmp(line, "outcell: Cell angles", 20) == 0)
    {
    temp = g_strsplit(line, ":", 3);
    buff = tokenize(*(temp+2), &num_tokens);
    g_strfreev(temp);

    if (num_tokens > 2)
      {
/* get the angles (in radians) */
      data->pbc[3] = D2R*str_to_float(*(buff+0));
      data->pbc[4] = D2R*str_to_float(*(buff+1));
      data->pbc[5] = D2R*str_to_float(*(buff+2));
      }
    else
      printf("Unexpected data format reading cell angles.\n");
    g_strfreev(buff);
    
/* no more data - break out */
/*
    break;
*/
    }
    
  if (g_ascii_strncasecmp(line, "siesta: E_KS(eV)", 16) == 0)
    {
    buff = tokenize(line, &num_tokens);
    data->siesta.energy = str_to_float(*(buff+3));
    data->siesta.have_energy = TRUE;
    g_strfreev(buff);
    }
    
  if (g_strrstr(line, "constrained") != NULL)
    {
    buff = tokenize(line, &num_tokens);
    data->siesta.max_grad = str_to_float(*(buff+1));
    data->siesta.have_max_grad = TRUE;
    g_strfreev(buff);
    }

  }

g_free(data->title);
title = g_string_new("");
if (data->siesta.have_energy)
  {
  g_string_append_printf(title, "E");
  g_string_append_printf(title, " = %.4f eV, ", data->siesta.energy);
  }
if (data->siesta.have_max_grad)
  g_string_append_printf(title, "max grad = %.5f", data->siesta.max_grad);
data->title = g_strdup(title->str);
g_string_free(title, TRUE);

return(0);
}

/*******************************/
/* SIESTA output frame reading */
/*******************************/
gint read_sout_frame(FILE *fp, struct model_pak *data)
{
    return(read_sout_block(fp, data));
}

/********************************/
/* Read in a SIESTA output file */
/********************************/
/* could almost use the read_fdf() code, as SIESTA spits outa copy */
/* however, we do need to know the number of frames for animation... */
gint read_sout(gchar *filename, struct model_pak *data)
{
gint flag, frame, num_tokens;
gchar **buff, line[LINELEN];
FILE *fp;

fp = fopen(filename, "rt");
if (!fp)
  return(1);

flag=frame=0;
while (!fgetline(fp, line))
  {
/* default cell dimensions */
  if (g_ascii_strncasecmp("\%block Lattice", line, 14) == 0)
    {
    if (fgetline(fp, line))
      return(2);
    buff = tokenize(line, &num_tokens);
    if (num_tokens > 5)
      {
      data->pbc[0] = str_to_float(*(buff+0));
      data->pbc[1] = str_to_float(*(buff+1));
      data->pbc[2] = str_to_float(*(buff+2));
      data->pbc[3] = D2R*str_to_float(*(buff+3));
      data->pbc[4] = D2R*str_to_float(*(buff+4));
      data->pbc[5] = D2R*str_to_float(*(buff+5));
      data->construct_pbc = FALSE;
      data->periodic = 3;
      }
    g_strfreev(buff);
    }

/* coordinates */
  if (g_ascii_strncasecmp(line, "outcoor: Atomic coordinates (fractional)", 40) == 0)
    {
    data->fractional = TRUE;
    add_frame_offset(fp, data);
/* only load the required frame */
    if (frame == data->cur_frame)
      {
      read_sout_block(fp, data);
      flag++;
      }
    frame++;
    }

/* coordinates */
  if (g_ascii_strncasecmp(line, "outcoor: Atomic coordinates (Ang", 32) == 0)
    {
    data->fractional = FALSE;
    add_frame_offset(fp, data);
/* only load the required frame */
    if (frame == data->cur_frame)
      {
      read_sout_block(fp, data);
      flag++;
      }
    frame++;
    }

/* seems to increment the number of frames but doesn't set the animation pointer! */
/* line only occurs at end of a run that runs out of cycles so commenting out! */
/*  if (g_ascii_strncasecmp(line, "outcoor: Final", 14) == 0)
    frame++;*/
  }

/* done */
if (flag)
  {
  strcpy(data->filename, filename);
  g_free(data->basename);
  data->basename = strdup_basename(filename);

  data->num_frames = frame;

  prep_model(data);
  }
else
  return(2);

return(0);
}

