package GCExport::GCExportTarGz;

###################################################
#
#  Copyright 2005-2006 Tian
#
#  This file is part of GCstar.
#
#  GCstar is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  GCstar is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with GCstar; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
###################################################

use strict;

use GCExport::GCExportBase;

{
    package GCExport::GCExporterTarGz;

    use File::Copy;
    use File::Basename;
    use Cwd;
    use XML::Simple;
    use GCUtils 'glob';
    use base qw(GCExport::GCExportBaseClass);

    sub new
    {
        my ($proto, $parent) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new($parent);
        bless ($self, $class);

        $self->checkModule('Compress::Zlib');
        $self->checkModule('Archive::Tar');

        return $self;
    }
    
    sub process
    {
        my ($self, $options) = @_;

        $self->{options} = $options;
        $self->{options}->{withPictures} = 1;
        $self->{fileName} = $options->{file};
        $self->{fileName} .= '.tar.gz' if ($self->{fileName} !~ m/\.tar\.gz$/);
        
        chdir dirname($self->{fileName});

        $self->{original} = $options->{collection};
        #$self->{original} =~ s/\\/\//g if ($^O =~ /win32/i);
        $self->{origDir} = dirname($self->{original});
        
        my $listFile = 'collection.gcs';
        my $baseDir = 'tmp_items_tar_gz';
        my $imagesSubDir = 'images';
        my $imageDir = $baseDir.'/'.$imagesSubDir;

        mkdir $baseDir;
        mkdir $imageDir;
        my $currentDir = getcwd;

        my @tmpArray = @{$options->{items}};
        my $item;
        my $line;
        open LIST, '>'.$baseDir.'/'.$listFile;
        binmode(LIST, ':utf8' );
        
        my $xmlModel = '';
        my $xmlPreferences = '';
        my $collectionType;
        if ($self->{model}->isPersonal)
        {
            $xmlModel = $self->{model}->toString($GCData::inlineCollectionTag);
            $xmlPreferences = $self->{model}->{preferences}->toXmlString;
            $collectionType = 'inline';
        }
        else
        {
            $collectionType = $self->{model}->getName;
        }
        my $information = ' <information>
';
        $information .= "  <$_>".$options->{originalList}->{information}->{$_}."</$_>\n"
            foreach (keys %{$options->{originalList}->{information}});
        $information .= ' </information>';

        print LIST '<?xml version="1.0" encoding="UTF-8"?>
<collection type="',$collectionType,'" items="',scalar @tmpArray,'">
',$xmlModel,'
',$xmlPreferences,'
',$information,'
';

        
        foreach $item(@tmpArray)
        {
            chdir $self->{origDir};
            my $imageName;

            #$line = " <item>\n";
            #Perform the transformation for each image value
            #my $pic;
            foreach my $pic(@{$self->{model}->{fieldsImage}})
            {
                $item->{$pic} = $self->duplicatePicture($item, $pic, $currentDir.'/'.$imageDir,
                                                        $item->{$self->{model}->{commonFields}->{title}});
            }

            my $xs = XML::Simple->new;
            $line = $xs->XMLout($item,
                                NoAttr => 1,
                                RootName => 'item');
            print LIST "$line\n";        

            $self->restoreInfo($item);
        }
        print LIST "</collection>\n";
        close LIST;

        chdir $currentDir;

        my $tar = Archive::Tar->new();
        chdir $baseDir;

        $tar->add_files($listFile, $imagesSubDir);
        my @images = glob $imagesSubDir.'/*';
        $tar->add_files($_) foreach (@images);
        (my $tarfile = $self->{fileName}) =~ s/\.gz$//;
        $tar->write($tarfile);

        my $gz = Compress::Zlib::gzopen($self->{fileName}, "wb");
        open(TAR, $tarfile) or die "Cannot open $tarfile ";
        binmode(TAR);
        my $buff;
        while (read(TAR, $buff, 8 * 2**10))
        {
            $gz->gzwrite($buff);
        }
        $gz->gzclose;
        close TAR;
        unlink foreach (@images);

        unlink $listFile;
        rmdir $imagesSubDir;
        chdir '..';
        rmdir $baseDir;
        $tarfile =~ s/\\/\//g if ($^O =~ /win32/i);
        unlink $tarfile;

        return $self->getEndInfo;
    }

    sub getOptions
    {
        my $self = shift;
        my @options;
        return \@options;
    }

    sub getName
    {
        my $self = shift;
        
        return ".tar.gz";
    }
    
    sub hasErrors
    {
        my $self = shift;

        return '' if !$self->{errors};
        return "Following modules should be present to use this export :\n\n".$self->{errors};
    }

    sub getEndInfo
    {
        my $self = shift;
        my $message = $self->getLang->{Info}.$self->{fileName};
        return $message;
    }
}
