# Copyright (C) 2004,2005 by SICEm S.L.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
import gtk
import gobject

from gazpacho.propertyclass import UNICHAR_PROPERTIES
from gazpacho.loader import tags

from xml.sax.saxutils import escape, unescape
import xml.dom.minidom

from os.path import basename, splitext

HAS_NODES = "glade_util_has_nodes"
SELECTION_NODE_SIZE = 7

def hide_window(window):
    x, y = window.get_position()
    window.hide()
    window.move(x, y)

def get_parent(gtk_widget):
    from gazpacho.widget import get_widget_from_gtk_widget
    parent_widget = gtk_widget
    gwidget = None
    while True:
        parent_widget = parent_widget.get_parent()
        if parent_widget is None:
            return None
        gwidget = get_widget_from_gtk_widget(parent_widget)
        if gwidget is not None:
            return gwidget

    return None

def _get_window_positioned_in(widget):
    """This returns the window that the given widget's position is relative to.
    Usually this is the widget's parent's window. But if the widget is a
    toplevel, we use its own window, as it doesn't have a parent.
    Some widgets also lay out widgets in different ways """
    parent = widget.get_parent()
    if parent:
        return parent.window
    return widget.window

def _draw_nodes(window, gc, x, y, width, height):
    if width > SELECTION_NODE_SIZE and height > SELECTION_NODE_SIZE:
        window.draw_rectangle(gc, True, x, y,
                              SELECTION_NODE_SIZE, SELECTION_NODE_SIZE)
        window.draw_rectangle(gc, True, x, y + height -SELECTION_NODE_SIZE,
                              SELECTION_NODE_SIZE, SELECTION_NODE_SIZE)
        window.draw_rectangle(gc, True, x + width - SELECTION_NODE_SIZE, y,
                              SELECTION_NODE_SIZE, SELECTION_NODE_SIZE)
        window.draw_rectangle(gc, True, x + width - SELECTION_NODE_SIZE,
                              y + height - SELECTION_NODE_SIZE,
                              SELECTION_NODE_SIZE, SELECTION_NODE_SIZE)
    window.draw_rectangle(gc, False, x, y, width - 1, height - 1)


def _calculate_window_offset(gdkwindow):
    """ This calculates the offset of the given window within its toplevel.
    It also returns the toplevel """
    x = y = 0
    window = gdkwindow
    while True:
        if window.get_window_type() != gtk.gdk.WINDOW_CHILD:
            break
        tmp_x, tmp_y = window.get_position()
        x += tmp_x
        y += tmp_y
        window = window.get_parent()

    return (window, x, y)

def _can_draw_nodes(sel_widget):
    """ This returns TRUE if it is OK to draw the selection nodes for the given
    selected widget inside the given window that has received an expose event.
    For most widgets it returns TRUE, but if a selected widget is inside a
    widget like a viewport, that uses its own coordinate system, then it only
    returns TRUE if the expose window is inside the viewport as well """
    viewport = None
    # Check if the selected widget is inside a viewport
    widget = sel_widget.get_parent()
    while widget:
        if isinstance(widget, gtk.Viewport):
            viewport = widget
        widget = widget.get_parent()

    if viewport is None:
        return True

    # XXX TODO do more stuff if it is a viewport
    return False

def draw_nodes_idle(expose_win):
    """ This is called to redraw any selection nodes that intersect the given
    exposed window. It steps through all the selected widgets, converts the
    coordinates so they are relative to the exposed window, then calls
    glade_util_draw_nodes() to draw the selection nodes if appropriate """
    from gazpacho.widget import get_widget_from_gtk_widget
    if not expose_win.is_viewable():
        return False

    # Find the corresponding GtkWidget and GladeWidget
    expose_widget = expose_win.get_user_data()
    gc = expose_widget.style.black_gc
    expose_gwidget = get_widget_from_gtk_widget(expose_widget)
    if not expose_gwidget:
        expose_gwidget = get_parent(expose_widget)
    if not expose_gwidget:
        return False

    # Calculate the offset of the expose window within its toplevel
    expose_toplevel, expose_win_x, expose_win_y = _calculate_window_offset(expose_win)

    expose_win_w, expose_win_h = expose_win.get_size()

    # Step through all the selected widgets in the project
    for sel_widget in expose_gwidget.project.selection:
        sel_win = _get_window_positioned_in(sel_widget)

        if sel_win is None:
            continue
        
        # Calculate the offset of the selected widget's window within
        # its toplevel
        sel_toplevel, sel_x, sel_y = _calculate_window_offset(sel_win)

        # We only draw the nodes if the window that got the expose event is
        # in the same toplevel as the selected widget
        if expose_toplevel == sel_toplevel and _can_draw_nodes(sel_widget):
            x = sel_x + sel_widget.allocation.x - expose_win_x
            y = sel_y + sel_widget.allocation.y - expose_win_y
            w = sel_widget.allocation.width
            h = sel_widget.allocation.height

            # Draw the selection nodes if they intersect the
            # expose window bounds
            if x < expose_win_w and x + w >= 0 and \
               y < expose_win_h and y + h >= 0:
                _draw_nodes(expose_win, gc, x, y, w, h)
        
def queue_draw_nodes(window):
    """ This should be called whenever a widget in the interface receives an
    expose event. It sets up an idle function which will redraw any selection
    nodes that intersect the exposed window """
    gobject.idle_add(draw_nodes_idle, window)
    
def has_nodes(widget):
    nodes = widget.get_data(HAS_NODES)
    if nodes and nodes == 1: return True
    return False

def add_nodes(widget):
    widget.set_data(HAS_NODES, 1)
    widget.queue_draw()
    
def remove_nodes(widget):
    widget.set_data(HAS_NODES, 0)

    # We redraw the parent, since the selection rectangle may not be
    # cleared if we just redraw the widget itself
    if widget.parent:
        widget.parent.queue_draw()
    else:
        widget.queue_draw()

# xml utility functions
def xml_create_string_prop_node(xml_doc, prop_name, prop_value):
    node = xml_doc.createElement(tags.XML_TAG_PROPERTY)
    node.setAttribute(tags.XML_TAG_NAME, prop_name)
    text_node = xml_doc.createTextNode(escape(prop_value))
    node.appendChild(text_node)
    return node

def xml_get_text_from_node(xmlnode):
    text = ''
    for node in xmlnode.childNodes:
        if node.nodeType == node.TEXT_NODE:
            text += node.data
    return unescape(text)

def get_bool_from_string_with_default(the_string, default):
    if the_string in ['True', 'TRUE', 'true', 'yes', '1']:
        return True
    elif the_string in ['False', 'FALSE', 'false', 'no', '0']:
        return False
    else:
        return default
    
def xml_get_property_boolean(xmlnode, tag, default):
    value = xmlnode.getAttribute(tag)
    return get_bool_from_string_with_default(value, default)

def xml_filter_nodes(nodes, node_type):
    return [node for node in nodes if node.nodeType == node_type]

def write_xml(file, xml_node, indent=0):
    if xml_node.nodeType == xml_node.TEXT_NODE:
        file.write(xml_node.data)
        return

    file.write(' '*indent)
    
    file.write('<%s' % xml_node.tagName)
    if len(xml_node.attributes) > 0:
        attr_string = ' '.join(['%s="%s"' % (n, v) for n, v in \
                                xml_node.attributes.items()])
        file.write(' '+attr_string)

    children = [a for a in xml_node.childNodes \
                if a.nodeType != a.ATTRIBUTE_NODE]
    if children:
        has_text_child = False
        for child in children:
            if child.nodeType == child.TEXT_NODE:
                has_text_child = True
                break

        if has_text_child:
            file.write('>')
        else:
            file.write('>\n')
        for child in children:
            write_xml(file, child, indent+4)

        if not has_text_child:
            file.write(' '*indent)
        file.write('</%s>\n' % xml_node.tagName)
    else:
        file.write('/>\n')

def xml_read_raw_property(xml_node):
    assert xml_node.tagName == tags.XML_TAG_PROPERTY

    name = xml_node.getAttribute(tags.XML_TAG_NAME)
    name = name.replace('_', '-')
    value = xml_get_text_from_node(xml_node)
    return (name, value)

def xml_clean_node(node, node_types_to_remove):
    """Remove all the nodes of 'node' that are of type
    'node_types_to_remove'
    """
    for child_node in node.childNodes:
        if child_node.nodeType in node_types_to_remove:
            node.removeChild(child_node)
        xml_clean_node(child_node, node_types_to_remove)
        

def xml_find_node(xml_node, tag_name, **attributes):
    if xml_node.nodeType == xml_node.ELEMENT_NODE:
        if xml_node.tagName == tag_name:
            found = True
            for attr, value in attributes.items():
                if xml_node.getAttribute(attr) != value:
                    found = False
                    break
            if found:
                return xml_node
    # not found, look in the children
    for child_node in xml_node.childNodes:
        node = xml_find_node(child_node, tag_name, **attributes)
        if node is not None:
            return node
    

def xml_child_widgets(xml_node):
    """Return a list <child> nodes. Actually it return the children of those
    nodes
    """
    list1 = xml_filter_nodes(xml_node.childNodes, xml_node.ELEMENT_NODE)
    list2 = [child for child in list1 if child.tagName == tags.XML_TAG_CHILD]
    list3 = []
    for child_node in list2:
        for item_node in xml_filter_nodes(child_node.childNodes,
                                          child_node.ELEMENT_NODE):
            list3.append(item_node)
    return list3
    
def ask_for_number(title, name, min, max, default):
    dialog = gtk.Dialog(title, None,
                        gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT \
                        | gtk.DIALOG_NO_SEPARATOR,
                        (gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
    dialog.set_position(gtk.WIN_POS_MOUSE)
    dialog.set_border_width(12)
    dialog.set_default_response(gtk.RESPONSE_ACCEPT)
    
    label = gtk.Label(name)
    label.set_alignment(0.0, 0.5)

    spin_button = gtk.SpinButton()
    spin_button.set_increments(1, 5)
    spin_button.set_range(min, max)
    spin_button.set_numeric(False)
    spin_button.set_value(default)
    spin_button.set_property('activates-default', True)

    hbox = gtk.HBox(spacing=4)
    hbox.pack_end(spin_button)
    hbox.pack_end(label)

    dialog.vbox.pack_start(hbox)
    hbox.show_all()

    # even if the use destroys the dialog box, we retrieve the number and we
    # accept it. I.e., this function never fails
    dialog.run()
    number = spin_button.get_value_as_int()
    dialog.destroy()
    return number

METADATA_I18N_IS_TRANSLATABLE = 'i18n_is_translatable'
METADATA_I18N_HAS_CONTEXT = 'i18n_has_context'
METADATA_I18N_COMMENT = 'i18n_comment'

def gtk_widget_set_metadata(widget, prefix, key, value):
    full_key = '%s_%s' % (prefix, key)
    widget.set_data(full_key, value)

def gtk_widget_get_metadata(widget, prefix, key):
    full_key = '%s_%s' % (prefix, key)
    return widget.get_data(full_key)


def get_property_value_from_string(property_class, str_value):
    if property_class.type == gobject.TYPE_BOOLEAN:
        value = get_bool_from_string_with_default (str_value, True)
    elif property_class.type in (gobject.TYPE_FLOAT, gobject.TYPE_DOUBLE):
        value = float(str_value or 0.0)
    elif property_class.type == gobject.TYPE_INT:
        value = int(str_value or 0)
    elif property_class.type == gobject.TYPE_STRING:
        value = str_value or ""
    elif property_class.type == gobject.TYPE_UINT:
        if property_class.id in UNICHAR_PROPERTIES:
            value = unicode(str_value and str_value[0] or "")
        else:
            value = int(str_value or 0)
    elif property_class.type == gobject.TYPE_ENUM:
        value = int(str_value or 0)
    elif property_class.type == gobject.TYPE_FLAGS:
        value = int(str_value or 0)
    elif property_class.type == gobject.TYPE_OBJECT:
        value = str_value
    elif property_class.type is None:
        print _("Error: the property %s should not be of type None") % \
              property_classlass.name
        return None

    return value

# useful treeview functions
def unselect_when_clicked_on_empty_space(treeview, event):
    result = treeview.get_path_at_pos(int(event.x), int(event.y))
    if not result:
        selection = treeview.get_selection()
        selection.unselect_all()

def unselect_when_press_escape(treeview, event):
    if event.keyval == gtk.keysyms.Escape:
        selection = treeview.get_selection()
        selection.unselect_all()

def select_iter(treeview, item_iter):
    model = treeview.get_model()
    path = model.get_path(item_iter)
    treeview.expand_to_path(path)
    treeview.scroll_to_cell(path)
    treeview.get_selection().select_path(path)
