# Copyright (C) 2004,2005 by SICEm S.L. and Imendio
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
from xml.dom.minidom import parseString

import gtk
import gobject
import os

from gazpacho import util
from gazpacho.placeholder import Placeholder
from gazpacho.widget import Widget, get_widget_from_gtk_widget
from gazpacho.uieditor import UIEditor
from gazpacho.widgetregistry import widget_registry
from gazpacho.l10n import _

app = None

_ui_editor = None

# The root library is the name of the library Gazpacho will import to create
# the widgets. e.g import gtk
root_library = 'gtk'

# When loading the widget class from the catalog files we strip off this prefix
# for every widget
widget_prefix = 'Gtk'

def init_gladegtk(application):
    # Please note that this is ugly and nasty but it's not a big problem right
    # now since eventually gladegtk.py is gonna disappear
    global app
    global _ui_editor
    app = application
    _ui_editor = UIEditor()
    
def glade_gtk_widget_condition(widget_class):
    widget = widget_class.type()
    if isinstance(widget,(gtk.Button, gtk.MenuItem)):
        return True
    else:
        return False # XXX TODO check for no window widgets

def glade_gtk_widget_get_tooltip(widget):
    data = gtk.tooltips_data_get(widget)
    if data is not None:
        return data[2]

    return None

def glade_gtk_widget_set_tooltip(widget, tooltip):
    gwidget = get_widget_from_gtk_widget(widget)
    tooltips = gwidget.project.tooltips
    if tooltip:
        tooltips.set_tip(widget, tooltip, None)
    else:
        tooltips.set_tip(widget, None, None)

def glade_gtk_box_get_size(box):
    return len(box.get_children())

def glade_gtk_box_set_size(box, size):
    global app
    old_size = len(box.get_children())
    new_size = size

    if new_size == old_size:
        return
    elif new_size > old_size:
        # The box has grown. Add placeholders
        while new_size > old_size:
            box.add(Placeholder(app))
            old_size += 1
    else:
        # The box has shrunk. Remove the widgets that are not on those slots
        child = box.get_children()[-1]
        while old_size > new_size and child:
            gwidget = get_widget_from_gtk_widget(child)
            if gwidget: # It may be None, e.g a placeholder
                gwidget.project.remove_widget(child)

            box.remove(child)
            child = box.get_children()[-1]
            old_size -= 1
            
    box.set_data('glade_nb_placeholders', new_size)

def glade_gtk_text_view_get_text(text_view):
    buffer = text_view.get_buffer()
    (start, end) = buffer.get_bounds()
    return buffer.get_text(start, end, False)

def glade_gtk_text_view_set_text(text_view, text):
    text_view.get_buffer().set_text(text)
                 
def glade_gtk_notebook_get_n_pages(notebook):
    return notebook.get_n_pages()

def glade_gtk_notebook_set_n_pages(notebook, pages):
    global app
    gwidget = get_widget_from_gtk_widget(notebook)
    old_size = notebook.get_n_pages()
    new_size = pages

    if new_size == old_size:
        return

    if new_size > old_size:
        # The notebook has grown. Add pages
        while new_size > old_size:
            notebook.append_page(Placeholder(app), None)
            old_size += 1
            
    else:
        # The notebook has shrunk. Remove pages
        
        # Thing to remember is that GtkNotebook starts the
        # page numbers from 0, not 1 (C-style). So we need to do
        # old_size-1, where we're referring to "nth" widget.
        while old_size > new_size:
            child_widget = notebook.get_nth_page(old_size - 1)
            child_gwidget = get_widget_from_gtk_widget(child_widget)

            # If we got it, and it's not a placeholder, remove it from project
            if child_gwidget:
                child_gwidget.project.remove_widget(child_widget)

            notebook.remove_page(old_size - 1)
            old_size -= 1

def glade_gtk_table_set_n_common(table, value, for_rows):
    global app
    new_size = value
    p = for_rows and 'n-rows' or 'n-columns'
    old_size = table.get_property(p)
    if new_size == old_size:
        return

    if new_size < 1:
        return

    gwidget = get_widget_from_gtk_widget(table)
    if new_size > old_size:
        if for_rows:
            table.resize(new_size, table.get_property('n-columns'))
            for i in range(table.get_property('n-columns')):
                for j in range(old_size, table.get_property('n-rows')):
                    table.attach(Placeholder(app), i, i+1, j, j+1)
        else:
            table.resize(table.get_property('n-rows'), new_size)
            for i in range(old_size, table.get_property('n-columns')):
                for j in range(table.get_property('n-rows')):
                    table.attach(Placeholder(app), i, i+1, j, j+1)
    else:
        # Remove from the bottom up
        l = table.get_children()
        l.reverse()
        for child in l:
            p = for_rows and 'top-attach' or 'left-attach'
            start = table.child_get_property(child, p)
            p = for_rows and 'bottom-attach' or 'right-attach'
            end = table.child_get_property(child, p)

            # We need to completely remove it
            if start >= new_size:
                table.remove(child)
                continue
            # If the widget spans beyond the new border, we should resize it to
            # fit on the new table
            if end > new_size:
                p = for_rows and 'bottom-attach' or 'right-attach'
                table.child_set_property(child, p, new_size)
        table.resize(for_rows and new_size or table.get_property('n-rows'),
                     for_rows and table.get_property('n-columns') or new_size)

        p = for_rows and 'n-columns' or 'n-rows'
        table.set_data('glade_nb_placeholders',
                       new_size * table.get_property(p))
                       
    
def glade_gtk_table_set_n_rows(table, rows):
    glade_gtk_table_set_n_common(table, rows, True)

def glade_gtk_table_set_n_columns(table, columns):
    glade_gtk_table_set_n_common(table, columns, False)

def glade_gtk_button_set_stock():
    print 'gtk button set stock'

def glade_gtk_statusbar_get_has_resize_grip(statusbar):
    return statusbar.get_has_resize_grip()

def glade_gtk_statusbar_set_has_resize_grip(statusbar, value):
    statusbar.set_has_resize_grip(value)

def ignore(*args):
    pass

# Pre Create functions
def glade_gtk_table_pre_create(table, interactive=True):
    """ a GtkTable starts with a default size of 1x1, and setter/getter of
    rows/columns expect the GtkTable to hold this number of placeholders, so we
    should add it. """
    global app
    table.attach(Placeholder(app), 0, 1, 0, 1)

def glade_gtk_tree_view_pre_create(tree_view, interactive=True):
    model = gtk.ListStore(gobject.TYPE_STRING) # dummy model
    tree_view.set_model(model)

    renderer = gtk.CellRendererText()
    column = gtk.TreeViewColumn('Column 1', renderer, text=0)
    tree_view.append_column(column)

    column = gtk.TreeViewColumn('Column 2', renderer, text=0)
    tree_view.append_column(column)

def glade_gtk_combo_box_pre_create(combo, interactive=True):
    model = gtk.ListStore(gobject.TYPE_STRING)
    combo.set_model(model)

def glade_gtk_table_post_create(table, interactive=True):
    if not interactive:
        return
    gwidget = get_widget_from_gtk_widget(table)
    property_rows = gwidget.get_glade_property('n-rows')
    property_cols = gwidget.get_glade_property('n-columns')
    dialog = gtk.Dialog(_('Create a table'), None,
                        gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT | \
                        gtk.DIALOG_NO_SEPARATOR,
                        (gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
    dialog.set_position(gtk.WIN_POS_MOUSE)
    dialog.set_default_response(gtk.RESPONSE_ACCEPT)
    
    label_rows = gtk.Label(_('Number of rows')+':')
    label_rows.set_alignment(0.0, 0.5)
    label_cols = gtk.Label(_('Number of columns')+':')
    label_cols.set_alignment(0.0, 0.5)

    spin_button_rows = gtk.SpinButton()
    spin_button_rows.set_increments(1, 5)
    spin_button_rows.set_range(1.0, 10000.0)
    spin_button_rows.set_numeric(False)
    spin_button_rows.set_value(3)
    spin_button_rows.set_property('activates-default', True)
    
    spin_button_cols = gtk.SpinButton()
    spin_button_cols.set_increments(1, 5)
    spin_button_cols.set_range(1.0, 10000.0)
    spin_button_cols.set_numeric(False)
    spin_button_cols.set_value(3)
    spin_button_cols.set_property('activates-default', True)

    table = gtk.Table(2, 2, True)
    table.set_col_spacings(4)
    table.set_border_width(12)
    
    table.attach(label_rows, 0, 1, 0, 1)
    table.attach(spin_button_rows, 1, 2, 0, 1)

    table.attach(label_cols, 0, 1, 1, 2)
    table.attach(spin_button_cols, 1, 2, 1, 2)

    dialog.vbox.pack_start(table)
    table.show_all()

    # even if the user destroys the dialog box, we retrieve the number and we
    # accept it.  I.e., this function never fails
    dialog.run()

    property_rows.set(spin_button_rows.get_value_as_int())
    property_cols.set(spin_button_cols.get_value_as_int())

    dialog.destroy()
    
def glade_gtk_window_post_create(window, interactive=True):
    window.set_default_size(440, 250)

### Managing the custom editors for GtkImage
def image_icon_editor_add_widget(box, name, widget, expand):
    box.pack_start(widget, expand=expand)
    box.set_data(name, widget)

def glade_gtk_image_icon_editor_create():

    store = gtk.ListStore(str, str)
    completion = gtk.EntryCompletion()
    completion.set_model(store)

    cell = gtk.CellRendererPixbuf()
    completion.pack_start(cell, True)
    completion.add_attribute(cell, 'stock-id', 0)

    cell = gtk.CellRendererText()
    completion.pack_start(cell, True)
    completion.add_attribute(cell, 'text', 1)

    completion.set_text_column(0)
    completion.set_minimum_key_length(1)
    
    image_icon_create_stock_popup(completion)
    
    entry = gtk.Entry()
    entry.set_completion(completion)
    
    fc_button = gtk.Button(_('...'))

    hbox = gtk.HBox()
    image_icon_editor_add_widget(hbox, 'image-icon-entry', entry, True)
    image_icon_editor_add_widget(hbox, 'image-icon-file-chooser', fc_button, False)

    return hbox

def image_icon_create_stock_popup(completion):
    model = completion.get_model()
    
    for stock_id in gtk.stock_list_ids():
        if stock_id == 'gtk-missing-image': continue
        stock_item = gtk.stock_lookup(stock_id)
        if not stock_item:
            continue
        model.append([stock_id, stock_item[1]])

def image_icon_entry_changed(entry, proxy):
    if entry.get_text() in gtk.stock_list_ids():
        proxy.set_property('stock', entry.get_text())

def image_icon_file_clicked(button, (proxy, image, entry)):
    dialog = gtk.FileChooserDialog('Chooser', None, gtk.FILE_CHOOSER_ACTION_OPEN,
                                   buttons=(gtk.STOCK_CANCEL,gtk.RESPONSE_CANCEL,gtk.STOCK_OPEN,gtk.RESPONSE_OK))

    filter = gtk.FileFilter()
    filter.set_name("Images")
    filter.add_mime_type("image/png")
    filter.add_mime_type("image/jpeg")
    filter.add_mime_type("image/gif")
    filter.add_pattern("*.png")
    filter.add_pattern("*.jpg")
    filter.add_pattern("*.gif")

    dialog.add_filter(filter)
    
    response = dialog.run()
    if response == gtk.RESPONSE_OK and dialog.get_filename():
        entry.handler_block(entry.get_data('handler-id-changed'))
        entry.set_text(os.path.basename(dialog.get_filename()))
        entry.handler_unblock(entry.get_data('handler-id-changed'))
        proxy.set_property('file', dialog.get_filename())
        image.set_data('image-file-name', entry.get_text())

    dialog.destroy()
    return
 
def image_icon_reconnect(widget, signal, callback, userdata):
    handler_id = widget.get_data('handler-id-' + signal)
    if handler_id:
        widget.disconnect(handler_id)

    handler_id = widget.connect(signal, callback, userdata)
    widget.set_data('handler-id-' + signal, handler_id)

def glade_gtk_image_icon_editor_load(box, image, proxy):
    entry = box.get_data('image-icon-entry')
    toggle = box.get_data('image-icon-stock-chooser')
    fc_button = box.get_data('image-icon-file-chooser')

    stock_id = image.get_property('stock')
    if stock_id and stock_id != 'gtk-missing-image':
        text = stock_id
    elif image.get_data('image-file-name'):
        text = image.get_data('image-file-name')
    else:
        text = ''

    entry.set_text(text)

    image_icon_reconnect(entry, 'changed', image_icon_entry_changed, proxy)
    image_icon_reconnect(fc_button, 'clicked', image_icon_file_clicked, (proxy, image, entry))

def glade_gtk_image_icon_size_editor_create():
    store = gtk.ListStore(str, int)

    combo = gtk.ComboBox (store)
    
    cell = gtk.CellRendererText()
    combo.pack_start(cell, True)
    combo.add_attribute(cell, 'text', 0)

    return combo

def image_icon_size_setup_combo(editor, image):
    model = editor.get_model()
    model.clear()
    
    stock_id = image.get_property('stock')
    if not stock_id or stock_id == 'gtk-missing-image':
        editor.set_sensitive(False)
        return

    icon_set = gtk.icon_factory_lookup_default(stock_id)
    editor.set_sensitive(True)
    
    icon_size = image.get_property('icon-size')
    for size in icon_set.get_sizes():
        iter = model.append([gtk.icon_size_get_name(size), size])

        if size == icon_size:
            editor.handler_block(editor.get_data('handler-id-changed'))
            editor.set_active_iter(iter)
            editor.handler_unblock(editor.get_data('handler-id-changed'))

def image_icon_size_notify(image, param_spec, (editor, proxy)):
    image_icon_size_setup_combo(editor, image)

def image_icon_size_changed(editor, proxy):
    iter = editor.get_active_iter()
    if iter:
        model = editor.get_model()
        size = model.get_value(iter, 1)

        proxy.set_value(size)

def glade_gtk_image_icon_size_editor_load(editor, image, proxy):
    handler_id = image.get_data('image-notify-id')
    if handler_id:
        image.disconnect(handler_id)

    image_icon_reconnect(editor, 'changed', image_icon_size_changed, proxy)

    image_icon_size_setup_combo(editor, image)
    
    handler_id = image.connect('notify', image_icon_size_notify, (editor, proxy))
    image.set_data('image-notify-id', handler_id)
    return

### GtkDialog
def glade_gtk_dialog_post_create(dialog, interactive=True):
    global app
    gwidget = get_widget_from_gtk_widget(dialog)
    if not gwidget: return

    # create the GladeWidgets for internal children
    child_class = widget_registry.get_by_name('GtkVBox')
    if not child_class: return

    project = gwidget.project
    vbox_widget = Widget(child_class, project)
    vbox_widget.setup_internal_widget(dialog.vbox, 'vbox', gwidget.name)

    child_class = widget_registry.get_by_name('GtkHButtonBox')
    if not child_class: return

    dialog.action_area.pack_start(Placeholder(app))
    dialog.action_area.pack_start(Placeholder(app))

    action_area_widget = Widget(child_class, project)
    action_area_widget.setup_internal_widget(dialog.action_area,
                                             'action_area',
                                             gwidget.name)

    # set a reasonable default size for a dialog
    dialog.set_default_size(320, 260)

def glade_gtk_message_dialog_post_create(dialog, interactive=True):
    dialog.set_default_size(400, 115)

def glade_gtk_menu_bar_post_create(menu_bar, interactive=True):
    pass

def glade_gtk_tool_bar_post_create(tool_bar, interactive=True):
    pass

# Replace child functions
def glade_gtk_container_replace_child(current, new, container):
    if current is None:
        container.add(new)
        return
    
    if current.parent != container:
        return

    props = {}
    for pspec in gtk.container_class_list_child_properties(type(container)):
        props[pspec.name] = container.child_get_property(current, pspec.name)
        
    container.remove(current)
    container.add(new)

    for name, value in props.items():
        container.child_set_property(new, name, value)
        
def glade_gtk_notebook_replace_child(current, new, container):
    page_num = container.page_num(current)
    if page_num == -1:
        return

    page = container.get_nth_page(page_num)
    label = container.get_tab_label(current)

    container.remove_page(page_num)
    container.insert_page(new, label, page_num)

    new.show()
    container.set_current_page(page_num)

# Fill Empty functions
def glade_gtk_container_fill_empty(widget):
    global app
    widget.add(Placeholder(app))

def glade_gtk_dialog_fill_empty(dialog):
    global app
    dialog.vbox.pack_start(Placeholder(app))

def _glade_gtk_paned_set_position(paned):
    """This function get called until the paned is realized.
    Then it puts the bar in the middle.
    """
    if paned.flags() & gtk.REALIZED:
        alloc = paned.get_allocation()
        if isinstance(paned, gtk.VPaned):
            pos = alloc.height / 2
        else:
            pos = alloc.width / 2
        paned.set_position(pos)
        return False
    else:
        return True
    
def glade_gtk_paned_fill_empty(paned):
    global app
    paned.add1(Placeholder(app))
    paned.add2(Placeholder(app))
    # we want to set the position in the middle but
    # as the paned has not yet a parent it is not
    # realized we can't know its size
    gobject.idle_add(_glade_gtk_paned_set_position, paned)

# Get internal child functions
def glade_gtk_dialog_get_internal_child(dialog, name):
    return getattr(dialog, name, None)

def glade_gtk_dialog_child_property_applies(ancestor, widget, property_id):
    if property_id == 'response-id' and \
       isinstance(widget.parent, gtk.HButtonBox) and \
       isinstance(widget.parent.parent, gtk.VBox) and \
       widget.parent.parent.parent == ancestor:
        return True
    elif widget.parent == ancestor:
        return True

    return False

def glade_gtk_false(widget_class):
    return False

# menubar and toolbar functions
def _create_default_actions(gaction_group, project):
    default_actions = (
        # name, label, tooltip, stock_id, callback, accelerator
        (_('FileMenu'), _('_File'), '', None, '', ''),
        (_('New'), _('_New'), _('Create a new file'), gtk.STOCK_NEW, '', '<control>N'),
        (_('Open'), _('_Open'), _('Open a file'), gtk.STOCK_OPEN, '', '<control>O'),
        (_('Save'), _('_Save'), _('Save a file'), gtk.STOCK_SAVE, '', '<control>S'),
        (_('SaveAs'), _('Save _as'), _('Save with a differente name'), gtk.STOCK_SAVE_AS, '', ''),
        (_('Quit'), _('_Quit'), _('Quit the program'), gtk.STOCK_QUIT, '', '<control>Q'),
        (_('EditMenu'), _('_Edit'), '', None, '', ''),
        (_('Copy'), _('_Copy'), _('Copy selected object into the clipboard'), gtk.STOCK_COPY, '', '<control>C'),
        (_('Cut'), _('C_ut'), _('Cut selected object into the clipboard'), gtk.STOCK_CUT, '', '<control>X'),
        (_('Paste'), _('_Paste'), _('Paste object from the Clipboard'), gtk.STOCK_PASTE, '', '<control>V'),
        )
    for action in default_actions:
        pairs = zip(('name', 'label', 'tooltip',
                     'stock_id', 'callback', 'accelerator'),
                    action)
        values = dict(pairs)
        if gaction_group.get_action(values['name']) is None:
            app._command_manager.add_action(values, gaction_group, project)

def glade_gtk_menu_bar_post_create(menubar, interactive=True):
    gwidget = get_widget_from_gtk_widget(menubar)
    # A None in this list means a separator
    names = [gwidget.name, _('FileMenu'), _('New'), _('Open'), _('Save'),
             _('SaveAs'), None, _('Quit'), _('EditMenu'), _('Copy'), _('Cut'),
             _('Paste')]
    tmp = []
    for name in names:
        if name is None:
            tmp.append("sep%d" % gwidget.project.separator_counter)
        else:
            tmp.extend([name, name])
        
    ui_string = """
<menubar action="%s" name="%s">
  <menu action="%s" name="%s">
    <menuitem action="%s" name="%s"/>
    <menuitem action="%s" name="%s"/>
    <menuitem action="%s" name="%s"/>
    <menuitem action="%s" name="%s"/>
    <separator name="%s"/>
    <menuitem action="%s" name="%s"/>
  </menu>
  <menu action="%s" name="%s">
    <menuitem action="%s" name="%s"/>
    <menuitem action="%s" name="%s"/>
    <menuitem action="%s" name="%s"/>
  </menu>
</menubar>
""" % tuple(tmp)
    
    _internal_menu_or_tool_bar_post_create(menubar, ui_string)


def glade_gtk_tool_bar_post_create(toolbar, interactive=True):
    gwidget = get_widget_from_gtk_widget(toolbar)

    # None in this tuple represent a separator
    names = [gwidget.name, _('New'), _('Open'), _('Save'), None,
             _('Copy'), _('Cut'), _('Paste')]
    tmp = []
    for name in names:
        if name is None:
            tmp.append('sep%d' % gwidget.project.separator_counter)
        else:
            tmp.extend([name, name])
    
    ui_string = """
<toolbar action="%s" name="%s">
  <toolitem action="%s" name="%s"/>
  <toolitem action="%s" name="%s"/>
  <toolitem action="%s" name="%s"/>
  <separator name="%s"/>
  <toolitem action="%s" name="%s"/>
  <toolitem action="%s" name="%s"/>
  <toolitem action="%s" name="%s"/>
</toolbar>
""" % tuple(tmp)

    _internal_menu_or_tool_bar_post_create(toolbar, ui_string)
    
def _internal_menu_or_tool_bar_post_create(widget, ui_string):
    global app
    # create some default actions
    gwidget = get_widget_from_gtk_widget(widget)
    project = gwidget.project
    # look for the 'DefaultActions' action group
    gaction_group = None
    for gag in project.action_groups:
        if gag.name == 'DefaultActions':
            gaction_group = gag
            break
    # if not found, create it
    if gaction_group is None:
        app._command_manager.add_action_group('DefaultActions', project)
        gaction_group = project.action_groups[-1]
    
    # create some default actions
    _create_default_actions(gaction_group, project)
    
    new_widget = _get_new_widget_after_uimanager_change(widget, ui_string)
    gwidget.setup_widget(new_widget)
    gwidget.apply_properties()
    
def glade_gtk_menu_or_tool_bar_ui_editor_create():
    button = gtk.Button(_('Edit...'))
    button.set_data('connection-id', -1)
    return button

def _ui_editor_edit(widget, menubar, proxy):
    global _ui_editor
    if app.window:
        _ui_editor.set_transient_for(app.window)
    
    _ui_editor.set_widget(menubar, proxy)
    _ui_editor.present()

def glade_gtk_menu_or_tool_bar_ui_editor_load(button, widget, proxy):
    connection_id = button.get_data('connection-id')
    if connection_id != -1:
        button.disconnect(connection_id)

    connection_id = button.connect('clicked', _ui_editor_edit, widget, proxy)
    button.set_data('connection-id', connection_id)

def _get_new_widget_after_uimanager_change(widget, ui_string):
    gwidget = get_widget_from_gtk_widget(widget)
    ui_prop = gwidget.get_glade_property('ui')
    ui_prop.value = ui_string
    project = gwidget.project
    if project.uimanager is None:
        project.uimanager = gtk.UIManager()

    old_merge_id = widget.get_data('merge_id')
    if old_merge_id is not None:
        project.uimanager.remove_ui(old_merge_id)
        
    merge_id = project.uimanager.add_ui_from_string(ui_string)
    project.uimanager.ensure_update()
    new_widget = project.uimanager.get_widget('/'+gwidget.name)
    new_widget.set_data('merge_id', merge_id)
    return new_widget

def glade_gtk_menu_or_tool_bar_ui_set(widget, ui_string):
    if not ui_string:
        return

    gwidget = get_widget_from_gtk_widget(widget)
    project = gwidget.project
    new_widget = _get_new_widget_after_uimanager_change(widget, ui_string)
    
    if new_widget != widget:
        # we need to replace widget with new_widget
        container = widget.get_parent()

        props = {}
        for pspec in gtk.container_class_list_child_properties(type(container)):
            props[pspec.name] = container.child_get_property(widget, pspec.name)
        
        container.remove(widget)
        container.add(new_widget)

        for name, value in props.items():
            container.child_set_property(new_widget, name, value)

        gwidget.setup_widget(new_widget)
        
        project.widgets.remove(widget)
        project.widgets.append(new_widget)
