""" Python module for flow properties of perfect gases with heat transfer

In Rayleigh flow, heat addition increases the entropy and takes the process
towards the choking conditions, (M = M* = 1). This behaviour imposes a limit
to heat addition, therefore for a given initial Mach number there is a fixed
value of the maximum possible heat transfer (Qmax).

Assumptions:
1) One dimensional flow with Heat transfer
2) Constant area duct
3) Perfect gas with constant specific heats and molecular weights

"""
"""
 * Copyright (C) 2006 Varun Hiremath.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * Authors: Varun Hiremath, Venkattraman A
 * Version 0.1
"""

from scipy import optimize
from math import *

class RayleighFlow :
    def __init__(self, g= 1.4):
        """ g is the value of gamma (ratio of specific heats), default = 1.4
        """
        self.g = g
    
    def get_P_by_Pstar_from_M(self, M):
        """Returns the value of p/p* at Mach number M"""
        return (1+self.g)/(1+self.g*M**2)
    
    def get_Po_by_Postar_from_M(self, M):
        """Returns the value of po/po* at Mach number M"""
        return (1+self.g)/(1+self.g*M**2)*(2*(1+(self.g-1)/2*M**2)/(self.g+1))**(self.g/(self.g-1))
    
    def get_T_by_Tstar_from_M(self, M):
        """Returns the value of T/T* at Mach number M"""
        return ((1+self.g)/(1+self.g*M**2))**2*M**2
    
    def get_To_by_Tostar_from_M(self, M):
        """Returns the value of To/To* at Mach number M"""
        return 2*(1+self.g)*M**2*(1+(self.g-1)/2*M**2)/(1+self.g*M**2)**2
    
    def get_rho_by_rhostar_from_M(self, M):
        """Returns the value of rho/rho* at Mach number M"""
        return (1+self.g*M**2)/(1+self.g)/M**2
    
    def get_Qmax_by_CpT_from_M(self, M):
        """Returns the value of Qmax/CpT* at Mach number M"""
        return (M**2-1)**2/2/(self.g+1)/M**2
    
    def get_M_from_P_by_Pstar(self, P_by_Pstar):
        """Returns  Mach number M at p/p* equal to P_by_Pstar"""
        return (((1+self.g)/P_by_Pstar-1)/self.g)**0.5
    
    def get_Po_by_Postar_from_P_by_Pstar(self, P_by_Pstar):
        """Returns po/po* at p/p* equal to P_by_Pstar"""
        M = self.get_M_from_P_by_Pstar(P_by_Pstar)
        return self.get_Po_by_Postar_from_M(M)
    
    def get_T_by_Tstar_from_P_by_Pstar(self, P_by_Pstar):
        """Returns T/T* at p/p* equal to P_by_Pstar"""
        M = self.get_M_from_P_by_Pstar(P_by_Pstar)
        return self.get_T_by_Tstar_from_M(M)

    def get_To_by_Tostar_from_P_by_Pstar(self, P_by_Pstar):
        """Returns  To/To* at p/p* equal to P_by_Pstar"""
        M = self.get_M_from_P_by_Pstar(P_by_Pstar)
        return self.get_To_by_Tostar_from_M(M)

    def get_rho_by_rhostar_from_P_by_Pstar(self, P_by_Pstar):
        """Returns  rho/rho* at p/p* equal to P_by_Pstar"""
        M = self.get_M_from_P_by_Pstar(P_by_Pstar)
        return self.get_rho_by_rhostar_from_M(M)

    def get_Qmax_by_CpT_from_P_by_Pstar(self, P_by_Pstar):
        """Returns  Qmax/CpT at p/p* equal to P_by_Pstar"""
        M = self.get_M_from_P_by_Pstar(P_by_Pstar)
        return self.get_Qmax_by_CpT_from_M(M)

    def get_M_from_Po_by_Postar(self, Po_by_Postar):
        """Returns  Mach number M at po/po* equal to Po_by_Postar"""
        return (optimize.bisect(lambda M : self.get_Po_by_Postar_from_M(M) - Po_by_Postar, 0.001, 1.0),optimize.bisect(lambda M : self.get_Po_by_Postar_from_M(M) - Po_by_Postar, 1.0, 100.0))

    def get_P_by_Pstar_from_Po_by_Postar(self, Po_by_Postar =1.4):
        """Returns  p/p* from po/po* equal to Po_by_Postar"""
        M = self.get_M_from_Po_by_Postar(Po_by_Postar)
        return (self.get_P_by_Pstar_from_M(M[0]),self.get_P_by_Pstar_from_M(M[1]))

    def get_T_by_Tstar_from_Po_by_Postar(self, Po_by_Postar):
        """Returns T/T* at po/po* equal to Po_by_Postar"""
        M = self.get_M_from_Po_by_Postar(Po_by_Postar)
        return (self.get_T_by_Tstar_from_M(M[0]),self.get_T_by_Tstar_from_M(M[1]))

    def get_To_by_Tostar_from_Po_by_Postar(self, Po_by_Postar):
        """Returns To/To* at po/po* equal to Po_by_Postar"""
        M = self.get_M_from_Po_by_Postar(Po_by_Postar)
        return self.get_To_by_Tostar_from_M(M)

    def get_rho_by_rhostar_from_Po_by_Postar(self, Po_by_Postar):
        """Returns rho/rho* at po/po* equal to Po_by_Postar"""
        M = self.get_M_from_Po_by_Postar(Po_by_Postar)
        return (self.get_rho_by_rhostar_from_M(M[0]),self.get_rho_by_rhostar_from_M(M[1]))

    def get_Qmax_by_CpT_from_Po_by_Postar(self, Po_by_Postar):
        """Returns Qmax/CpT at po/po* equal to Po_by_Postar"""
        M = self.get_M_from_Po_by_Postar(Po_by_Postar)
        return (self.get_Qmax_by_CpT_from_M(M[0]),self.get_Qmax_by_CpT_from_M(M[1]))

    def get_M_from_T_by_Tstar(self, T_by_Tstar):
        """Returns M at T/T* equal to T_by_Tstar"""
        return (optimize.bisect(lambda x : self.get_T_by_Tstar_from_M(x) - T_by_Tstar, 0.001, 1.0/sqrt(self.g)),optimize.bisect(lambda x : self.get_T_by_Tstar_from_M(x) - T_by_Tstar, 1.0/sqrt(self.g), 100.0))

    def get_P_by_Pstar_from_T_by_Tstar(self, T_by_Tstar):
        """Returns p/p* at T/T* equal to T_by_Tstar"""
        M = self.get_M_from_T_by_Tstar(T_by_Tstar)
        return (self.get_P_by_Pstar_from_M(M[0]),self.get_P_by_Pstar_from_M(M[1]))

    def get_Po_by_Postar_from_T_by_Tstar(self, T_by_Tstar):
        """Returns po/po* at T/T* equal to T_by_Tstar"""
        M = self.get_M_from_T_by_Tstar(T_by_Tstar)
        return (self.get_Po_by_Postar_from_M(M[0]),self.get_Po_by_Postar_from_M(M[1]))

    def get_To_by_Tostar_from_T_by_Tstar(self, T_by_Tstar):
        """Returns To/To* at T/T* equal to T_by_Tstar"""
        M = self.get_M_from_T_by_Tstar(T_by_Tstar)
        return (self.get_To_by_Tostar_from_M(M[0]),self.get_To_by_Tostar_from_M(M[1]))

    def get_rho_by_rhostar_from_T_by_Tstar(self, T_by_Tstar):
        """Returns rho/rho* at T/T* equal to T_by_Tstar"""
        M = self.get_M_from_T_by_Tstar(T_by_Tstar)
        return (self.get_rho_by_rhostar_from_M(M[0]),self.get_rho_by_rhostar_from_M(M[1]))

    def get_Qmax_by_CpT_from_T_by_Tstar(self, T_by_Tstar):
        """Returns Qmax/CpT at T/T* equal to T_by_Tstar"""
        M = self.get_M_from_T_by_Tstar(T_by_Tstar)
        return (self.get_Qmax_by_CpT_from_M(M[0]),self.get_Qmax_by_CpT_from_M(M[1]))

    def get_M_from_To_by_Tostar(self, To_by_Tostar):
        """Returns M at To/To* equal to To_by_Tostar"""
        return (optimize.bisect(lambda x : self.get_To_by_Tostar_from_M(x) - To_by_Tostar, 0.001, 1.0/sqrt(self.g)),optimize.bisect(lambda x : self.get_To_by_Tostar_from_M(x) - To_by_Tostar, 1.0/sqrt(self.g), 100.0))

    def get_P_by_Pstar_from_To_by_Tostar(self, To_by_Tostar):
        """Returns p/p* at To/To* equal to To_by_Tostar"""
        M = self.get_M_from_To_by_Tostar(To_by_Tostar)
        return (self.get_P_by_Pstar_from_M(M[0]),self.get_P_by_Pstar_from_M(M[1]))

    def get_Po_by_Postar_from_To_by_Tostar(self, To_by_Tostar):
        """Returns po/po* at To/To* equal to To_by_Tostar"""
        M = self.get_M_from_To_by_Tostar(To_by_Tostar)
        return (self.get_Po_by_Postar_from_M(M[0]),self.get_Po_by_Postar_from_M(M[1]))

    def get_T_by_Tstar_from_To_by_Tostar(self, To_by_Tostar):
        """Returns T/T* at To/To* equal to To_by_Tostar"""
        M = self.get_M_from_To_by_Tostar(To_by_Tostar)
        return (self.get_T_by_Tstar_from_M(M[0]),self.get_T_by_Tstar_from_M(M[1]))

    def get_rho_by_rhostar_from_To_by_Tostar(self, To_by_Tostar):
        """Returns rho/rho* at To/To* equal to To_by_Tostar"""
        M = self.get_M_from_To_by_Tostar(To_by_Tostar)
        return (self.get_rho_by_rhostar_from_M(M[0]),self.get_rho_by_rhostar_from_M(M[1]))

    def get_Qmax_by_CpT_from_To_by_Tostar(self, To_by_Tostar):
        """Returns Qmax/CpT at To/To* equal to To_by_Tostar"""
        M = self.get_M_from_To_by_Tostar(To_by_Tostar)
        return (self.get_Qmax_by_CpT_from_M(M[0]),self.get_Qmax_by_CpT_from_M(M[1]))

    def get_M_from_rho_by_rhostar(self, rho_by_rhostar):
        """Returns  Mach number M at rho/rho* equal to rho_by_rhostar"""
        return (1.0/((1+self.g)*rho_by_rhostar - self.g))**0.5
    
    def get_P_by_Pstar_from_rho_by_rhostar(self, rho_by_rhostar):
        """Returns p/p* at rho/rho* equal to rho_by_rhostar"""
        M = self.get_M_from_rho_by_rhostar(rho_by_rhostar)
        return self.get_P_by_Pstar_from_M(M)

    def get_Po_by_Postar_from_rho_by_rhostar(self, rho_by_rhostar):
        """Returns po/po* at rho/rho* equal to rho_by_rhostar"""
        M = self.get_M_from_rho_by_rhostar(rho_by_rhostar)
        return self.get_Po_by_Postar_from_M(M)
    
    def get_T_by_Tstar_from_rho_by_rhostar(self, rho_by_rhostar):
        """Returns T/T* at rho/rho* equal to rho_by_rhostar"""
        M = self.get_M_from_rho_by_rhostar(rho_by_rhostar)
        return self.get_T_by_Tstar_from_M(M)

    def get_To_by_Tostar_from_rho_by_rhostar(self, rho_by_rhostar):
        """Returns  To/To* at rho/rho* equal to rho_by_rhostar"""
        M = self.get_M_from_rho_by_rhostar(rho_by_rhostar)
        return self.get_To_by_Tostar_from_M(M)

    def get_Qmax_by_CpT_from_rho_by_rhostar(self, rho_by_rhostar):
        """Returns  Qmax/CpT at rho/rho* equal to rho_by_rhostar"""
        M = self.get_M_from_rho_by_rhostar(rho_by_rhostar)
        return self.get_Qmax_by_CpT_from_M(M)
