#!/usr/bin/env python
# vim:sw=4:et
#
# Create a UML 2.0 datamodel from the Gaphor 0.2.0 model file.
#
# To do this we do the following:
# 1. read the model file with the gaphor parser
# 2. Create a object herarcy by ordering elements based on generalizations

# Recreate the model using some very dynamic class, so we can set all
# attributes and traverse them to generate the data model.

from gaphor.parser import parse, element, canvas, canvasitem
import sys, string, operator
import override

# The kind of attribute we're dealing with:
ATTRIBUTE = 0
ENUMERATION = 1

header = """# This file is generated by genUML2.py. DO NOT EDIT!

from properties import association, attribute, enumeration, derivedunion, redefine
"""

# redefine 'bool' for Python version < 2.3
if map(int, sys.version[:3].split('.')) < [2, 3]:
    header = header + "bool = int\n"

class Dynamic(object):

    def __init__(self):
        self.v = []

    def __getattr__(self, key):
        attr = self.__dict__.get(key)
        if attr is None:
            attr = self.__dict__[key] = Dynamic()
        return attr.v

    def __setattr__(self, key, value):
        attr = getattr(self, key)
        attr.v.append(value)

class DevNull:
    def write(self, s):
        pass

def msg(s):
    sys.stderr.write('  ')
    sys.stderr.write(s)
    sys.stderr.write('\n')
    sys.stderr.flush()

class Writer:

    def __init__(self, filename, overridesfile=None):
        self.overrides = overridesfile and override.Overrides(overridesfile) or None
        if filename:
            self.out = open(filename, 'w')
        else:
            self.out = sys.stdout

    def write(self, data):
        self.out.write(data)

    def close(self):
        self.out.close()

    def write_classdef(self, clazz):
        """Write a class definition (class xx(x): pass).
        First the parent classes are examined. After that its own definition
        is written. It is ensured that class definitions are only written
        once.
        
        For Diagram an exception is made: Diagram is imported from diagram.py"""
        if not clazz.written:
            s = ''
            for g in clazz.generalization:
                self.write_classdef(g)
                if s: s += ', '
                s = s + g['name']
            if not s: s = 'object'
            if not self.overrides.write_override(self, clazz['name']):
                self.write('class %s(%s): pass\n' % (clazz['name'], s))
        clazz.written = True

    def write_property(self, full_name, value):
        """Write a property to the file. If the property is overridden, use the
        overridden value. full_name should be like Class.attribute. value is
        free format text."""
        if not self.overrides.write_override(self, full_name):
            self.write('%s = %s\n' % (full_name, value))

    def write_attribute(self, a, enumerations={}):
        """Write a definition for attribute a. Enumerations may be a dict
        of enumerations, indexed by ID. These are used to identify enums.
        """
        kind, derived, name, type, default, lower, upper = parse_attribute(a)
        full_name = "%s.%s" % (a.class_name, name)
        if self.overrides.has_override(full_name):
            self.overrides.write_override(self, full_name)
        elif derived:
            msg('Ignoring derived attribute %s.%s: no definition' % (a.class_name, name))
        elif kind == ATTRIBUTE:
            self.write_property("%s.%s" % (a.class_name, name),
                                "attribute('%s', %s, %s, %s, %s)" % (name, type, default, lower, upper))
        elif kind == ENUMERATION:
            e = filter(lambda e: e['name'] == type, enumerations.values())[0]
            self.write_property("%s.%s" % (a.class_name, name),
                                "enumeration('%s', %s, '%s')" % (name, e.enumerates, default or e.enumerates[0]))

    def write_association(self, head, tail):
        """Write an association for head. False is returned if the association
        is derived.
        The head association end is enriched with the following attributes:
            derived - association is a derived union or not
            name - name of the association end (name of head is found on tail)
            class_name - name of the class this association belongs to
            opposite_class_name - name of the class at the other end of the assoc.
            lower - lower multiplicity
            upper - upper multiplicity
            subsets - derived unions that use the association
            redefines - redefines existing associations
        Returns True if the association is written. False is returned if
        the association should be handled by write_derivedunion() or
        write_redefine().
        """
        navigable = head.get('class_')
        if not navigable:
            # from this side, the association is not navigable
            return True
        try:
            derived, name = parse_association_name(head['name'])
        except KeyError:
            msg('ERROR! no name, but navigable: %s (%s.%s)' %
                (head.id, head.class_name, head.name))
            return True

        lower, upper, subsets, redefines = parse_association_multiplicity(head.lowerValue)
        # Add the values found. These are used later to generate derived unions.
        head.derived = derived
        head.name = name
        head.class_name = head.class_['name']
        head.opposite_class_name = head.type['name']
        head.lower = lower
        head.upper = upper
        head.subsets = subsets
        head.redefines = redefines

        # Derived unions and redefines are handled separately
        if derived or redefines:
            return False

        a = "association('%s', %s" % (name, head.opposite_class_name)
        if lower not in ('0', 0):
            a += ', lower=%s' % lower
        if upper != '*':
            a += ', upper=%s' % upper
        if tail.get('aggregation') == 'composite':
            a += ', composite=True'

        # Add the opposite property if the head itself is navigable:
        navigable = tail.get('class_')
        if navigable:
            try:
                o_derived, o_name = parse_association_name(tail['name'])
            except KeyError:
                msg('ERROR! no name, but navigable: %s (%s.%s)' %
                    (tail.id, tail.class_name, tail.name))
            else:
                assert not o_derived, 'One end if derived, the other end not ???'
                a += ", opposite='%s'" % o_name

        self.write_property("%s.%s" % (head.class_name, name), a + ')')
        return True

    def write_derivedunion(self, d):
        """Write a derived union. If there are no subsets a warning
        is issued. The derivedunion is still created though.
        Derived unions may be created for associations that were returned
        False by write_association()."""
        subs = ''
        for u in d.union:
            if u.derived and not u.written:
                self.write_derivedunion(u)
            if subs: subs += ', '
            subs += '%s.%s' % (u.class_name, u.name)
        if subs:
            self.write_property("%s.%s" % (d.class_name, d.name),
                                "derivedunion('%s', %s, %s, %s)" % (d.name, d.lower, d.upper == '*' and "'*'" or d.upper, subs))
        else:
            if not self.overrides.has_override('%s.%s' % (d.class_name, d.name)):
                msg('no subsets for derived union: %s.%s[%s..%s]' % (d.class_name, d.name, d.lower, d.upper))
            self.write_property("%s.%s" % (d.class_name, d.name),
                                "derivedunion('%s', %s, %s)" % (d.name, d.lower, d.upper))
        d.written = True

    def write_redefine(self, r):
        """Redefines may be created for associations that were returned
        False by write_association()."""
        self.write_property("%s.%s" % (r.class_name, r.name),
                            "redefine('%s', %s, %s)" % (r.name, r.opposite_class_name, r.redefines))


def parse_attribute(attr):
    """Returns a tuple (kind, derived, name, type, default, lower, upper)."""
    s = attr['name']
    kind = ATTRIBUTE
    derived = False
    default = None
    mult = None
    lower = 0
    upper = 1

    # First split name and type:
    try:
        name, type = map(string.strip, attr['name'].split(':'))
    except ValueError:
        name = attr['name']
        type = ''

    while not name[0].isalpha():
        if name[0] == '/':
            derived = True
        name = name[1:]
    
    if '=' in type:
        # split the type part in type and default value:
        type, default = map(string.strip, type.split('='))
    elif '[' in type:
        # split the type part in type and multiplicity:
        type, mult = map(string.strip, type.split('['))

    if default and '[' in default:
        # check if the default part has a multiplicity defined:
        default, mult = map(string.strip, default.split('['))
        
    if mult:
        if mult[-1] == ']':
            mult = mult[:-1]
        m = map(string.strip, mult.split('.'))
        lower = m[0]
        upper = m[-1]
        if upper == '*':
            upper = "'*'"

    # Make sure types are represented the Python way:
    if default and default.lower() in ('true', 'false'):
        default = default.title() # True or False...

    if type.lower() == 'boolean':
        type = 'bool'
    elif type.lower() in ('integer', 'unlimitednatural'):
        type = 'int'
    elif type.lower() == 'string':
        type = '(str, unicode)'

    if type.endswith('Kind'):
        kind = ENUMERATION

    return kind, derived, name, type, default, lower, upper

def parse_association_name(name):
    # First remove spaces
    name = name.replace(' ','')
    derived = False
    # Check if this is a derived union
    while name and not name[0].isalpha():
        if name[0] == '/':
            derived = True
        name = name[1:]
    return derived, name

def parse_association_multiplicity(mult):
    subsets = []
    redefines = None
    tag = None
    if '{' in mult:
        # we have tagged values
        mult, tag = map(string.strip, mult.split('{'))
        if tag[-1] == '}':
            tag = tag[:-1]
    else:
        mult = mult.strip()
    
    mult = mult.split('.')
    lower = mult[0]
    upper = mult[-1]
    if lower == '*':
        lower = 0
    #if upper == '*':
    #    upper = "'*'"

    if tag and tag.find('subsets') != -1:
        # find the text after 'subsets':
        subsets = tag[tag.find('subsets') + len('subsets'):]
        # remove all whitespaces and stuff
        subsets = subsets.replace(' ', '').replace('\n', '').replace('\r', '')
        subsets = subsets.split(',')
    if tag and tag.find('redefines') != -1:
        # find the text after 'redefines':
        redefines = tag[tag.find('redefines') + len('redefines'):]
        # remove all whitespaces and stuff
        redefines = redefines.replace(' ', '').replace('\n', '').replace('\r', '')
        l = redefines.split(',')
        assert len(l) == 1
        redefines = l[0]

    return lower, upper, subsets, redefines

def generate(filename, outfile=None, overridesfile=None):
    # parse the file
    all_elements = parse(filename)

    writer = Writer(outfile, overridesfile)

    # extract usable elements from all_elements. Some elements are given
    # some extra attributes.
    classes = { }
    enumerations = { }
    generalizations = { }
    associations = { }
    properties = { }
    for key, val in all_elements.items():
        # Find classes, *Kind (enumerations) are given special treatment
        if isinstance(val, element):
            if val.type == 'Class' and val.get('name'):
                if val['name'].endswith('Kind'):
                    enumerations[key] = val
                else:
                    classes[key] = val
                    # Add extra properties for easy code generation:
                    val.specialization = []
                    val.generalization = []
                    val.written = False
            elif val.type == 'Generalization':
                generalizations[key] = val
            elif val.type == 'Association':
                associations[key] = val
            elif val.type == 'Property':
                properties[key] = val

    # find inheritance relationships
    for g in generalizations.values():
        #assert g.specific and g.general
        specific = g['specific']
        general = g['general']
        classes[specific].generalization.append(classes[general])
        classes[general].specialization.append(classes[specific])

    # add values to enumerations:
    for e in enumerations.values():
        values = []
        for key in e['ownedAttribute']:
            values.append(str(properties[key]['name']))
        e.enumerates = tuple(values)

    # create file header
    writer.write(header)

    # create class definitions
    for c in classes.values():
        writer.write_classdef(c)

    # create attributes and enumerations
    for c in classes.values():
        for p in c.get('ownedAttribute') or []:
            a = properties.get(p)
            if a:
                # set class_name, since write_attribute depends on it
                a.class_name = c['name']
                if not a.get('association'):
                    writer.write_attribute(a, enumerations)

    # create associations, derivedunions are held back
    derivedunions = { } # indexed by name in stead of id
    redefines = [ ]
    for a in associations.values():
        ends = []
        for end in a['memberEnd']:
            end = properties[end]
            end.type = classes[end['type']]
            end.class_ = end.get('class_') and classes[end['class_']] or None
            #assert end.type is not end.class_
            if end.get('lowerValue'):
                end.lowerValue = all_elements[end['lowerValue']]['value']
            else:
                end.lowerValue = ''
            ends.append(end)

        for e1, e2 in ((ends[0], ends[1]), (ends[1], ends[0])):
            if not writer.write_association(e1, e2):
                # At this point the association is parsed, but not written.
                # assure that derived unions do not get overwritten
                if e1.redefines:
                    redefines.append(e1)
                else:
                    assert not derivedunions.get(e1.name)
                    derivedunions[e1.name] = e1
                    e1.union = [ ]
                    e1.written = False


    # create derived unions, first link the association ends to the d
    for a in filter(lambda e: hasattr(e, 'subsets'), properties.values()):
        for s in a.subsets:
            try:
                derivedunions[s].union.append(a)
            except KeyError:
                msg('Not a derived union: %s.%s' % (a.class_name, s))

    for d in derivedunions.values():
        writer.write_derivedunion(d)

    for r in redefines:
        msg('redefining %s -> %s.%s' % (r.redefines, r.class_name, r.name))
        writer.write_redefine(r)

    writer.close()

if __name__ == '__main__':
    generate('UML2.gaphor')
