/**
 * File:          $RCSfile: image_composite.c,v $
 * Module:        Extract/fill selected channel(s) of image
 * Part of:       Gandalf Library
 *
 * Revision:      $Revision: 1.13 $
 * Last edited:   $Date: 2004/07/14 23:29:31 $
 * Author:        $Author: pm $
 * Copyright:     (c) 2000 Imagineer Software Limited
 */

/* This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <gandalf/image/image_composite.h>
#include <gandalf/image/image_bit.h>

/**
 * \addtogroup ImagePackage
 * \{
 */

/**
 * \defgroup ImageComposite Composite one image on another
 * \{
 */

/**
 * \brief Macro: Composite images
 * \param source The input image 
 * \param source_with_alpha The input image with the alpha channel
 * \param source_mask Binary mask for source image or \c NULL
 * \param dest The destination image 
 * \return Composited \a dest image, or \c NULL on failure.
 *
 * Composites the image \a source_with_alpha onto the given image \a source,
 * writing the result onto \a dest.
 *
 * \sa gan_image_composite_i().
 */
Gan_Image *
 gan_image_composite_q ( Gan_Image *source,
                         const Gan_Image *source_with_alpha,
                         const Gan_Image *source_mask,
                         Gan_Image *dest )
{
   Gan_Pixel Pixel, SAPixel;

   if(dest == source_with_alpha)
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_image_composite_q", GAN_ERROR_INCOMPATIBLE, "");
      return NULL;
   }

   if ( dest == NULL )
      dest = gan_image_alloc ( source->format, source->type,
                               source->height, source->width );
   else if(source != dest &&
           (source->format != dest->format || source->type != dest->type ||
            source->width != dest->width || source->height != dest->height))
      dest = gan_image_set_format_type_dims(dest, source->format, source->type,
                                            source->height, source->width);

   if ( dest == NULL )
   {
      gan_err_register ( "gan_image_composite_q", GAN_ERROR_FAILURE, "");
      return NULL;
   }

   if(source_mask == NULL)
   {
      int iRow, iCol;

      for ( iRow = (int)source_with_alpha->height-1; iRow >= 0; iRow-- )
         for ( iCol = (int)source_with_alpha->width-1; iCol >= 0; iCol-- )
         {
            Pixel = gan_image_get_pix ( source, iRow+source_with_alpha->offset_y, iCol+source_with_alpha->offset_x );
            SAPixel = gan_image_get_pix ( source_with_alpha, iRow, iCol );
            if ( !gan_image_convert_pixel_i ( &Pixel,
                                              GAN_RGB_COLOUR_IMAGE,
                                              GAN_FLOAT ) ||
                 !gan_image_convert_pixel_i ( &SAPixel,
                                              GAN_RGB_COLOUR_ALPHA_IMAGE,
                                              GAN_FLOAT ) )
            {
               gan_err_register ( "gan_image_composite_q", GAN_ERROR_FAILURE, "");
               return NULL;
            }

            /* blend pixel */
            Pixel.data.rgb.f.R = SAPixel.data.rgba.f.A*SAPixel.data.rgba.f.R
                                 + (1.0F-SAPixel.data.rgba.f.A)*Pixel.data.rgb.f.R;
            Pixel.data.rgb.f.G = SAPixel.data.rgba.f.A*SAPixel.data.rgba.f.G
                                 + (1.0F-SAPixel.data.rgba.f.A)*Pixel.data.rgb.f.G;
            Pixel.data.rgb.f.B = SAPixel.data.rgba.f.A*SAPixel.data.rgba.f.B
                                 + (1.0F-SAPixel.data.rgba.f.A)*Pixel.data.rgb.f.B;

            /* replace blended pixel in image */
            if ( !gan_image_convert_pixel_i ( &Pixel, dest->format, dest->type )||
                 !gan_image_set_pix ( dest, iRow+source_with_alpha->offset_y, iCol+source_with_alpha->offset_x, &Pixel ) )
            {
               gan_err_register ( "gan_image_composite_q", GAN_ERROR_FAILURE, "");
               return NULL;
            }
         }
   }
   else
   {
      unsigned int uiRow, uiCol;
      Gan_ImageWindow window;
      unsigned int max_row, max_col;

      if(!gan_image_get_active_subwindow(source_mask, &window))
      {
         gan_err_register ( "gan_image_composite_q", GAN_ERROR_FAILURE, "");
         return NULL;
      }

      max_col = window.c0+window.width;
      max_row = window.r0+window.height;

      gan_err_test_ptr(source_mask->format == GAN_GREY_LEVEL_IMAGE, "gan_image_composite_q", GAN_ERROR_INCOMPATIBLE, "");

      if(dest != source)
      {
         if(gan_image_copy_q(source, dest) == NULL)
         {
            gan_err_register ( "gan_image_composite_q", GAN_ERROR_FAILURE, "");
            return NULL;
         }
      }
      
      switch(source_mask->type)
      {
         case GAN_BOOL:
           for ( uiRow = window.r0; uiRow < max_row; uiRow++)
              for ( uiCol = window.c0; uiCol < max_col; uiCol++)
                 if(gan_image_get_pix_b(source_mask, uiRow, uiCol))
                 {
                    Pixel = gan_image_get_pix ( source,
                                                uiRow+(unsigned int)source_with_alpha->offset_y,
                                                uiCol+(unsigned int)source_with_alpha->offset_x );
                    SAPixel = gan_image_get_pix ( source_with_alpha, uiRow, uiCol );
                    if ( !gan_image_convert_pixel_i ( &Pixel, GAN_RGB_COLOUR_IMAGE, GAN_FLOAT ) ||
                         !gan_image_convert_pixel_i ( &SAPixel, GAN_RGB_COLOUR_ALPHA_IMAGE, GAN_FLOAT ) )
                    {
                       gan_err_register ( "gan_image_composite_q", GAN_ERROR_FAILURE, "");
                       return NULL;
                    }

                    /* blend pixel */
                    Pixel.data.rgb.f.R = SAPixel.data.rgba.f.A*SAPixel.data.rgba.f.R + (1.0F-SAPixel.data.rgba.f.A)*Pixel.data.rgb.f.R;
                    Pixel.data.rgb.f.G = SAPixel.data.rgba.f.A*SAPixel.data.rgba.f.G + (1.0F-SAPixel.data.rgba.f.A)*Pixel.data.rgb.f.G;
                    Pixel.data.rgb.f.B = SAPixel.data.rgba.f.A*SAPixel.data.rgba.f.B + (1.0F-SAPixel.data.rgba.f.A)*Pixel.data.rgb.f.B;

                    /* replace blended pixel in image */
                    if ( !gan_image_convert_pixel_i ( &Pixel, dest->format, dest->type )||
                         !gan_image_set_pix ( dest,
                                              uiRow+(unsigned int)source_with_alpha->offset_y,
                                              uiCol+(unsigned int)source_with_alpha->offset_x, &Pixel ) )
                    {
                       gan_err_register ( "gan_image_composite_q", GAN_ERROR_FAILURE, "");
                       return NULL;
                    }
                 }

           break;

         default:
         {
            Gan_Pixel MaskPixel;

            for ( uiRow = window.r0; uiRow < max_row; uiRow++)
               for ( uiCol = window.c0; uiCol < max_col; uiCol++)
                  if(!gan_image_get_pix_zero(source_mask, uiRow, uiCol))
                  {
                     Pixel = gan_image_get_pix ( source,
                                                 uiRow+(unsigned int)source_with_alpha->offset_y,
                                                 uiCol+(unsigned int)source_with_alpha->offset_x );
                     SAPixel = gan_image_get_pix ( source_with_alpha, uiRow, uiCol );
                     MaskPixel = gan_image_get_pix ( source_mask, uiRow, uiCol );
                     if ( !gan_image_convert_pixel_i ( &Pixel, GAN_RGB_COLOUR_IMAGE, GAN_FLOAT ) ||
                          !gan_image_convert_pixel_i ( &SAPixel, GAN_RGB_COLOUR_IMAGE, GAN_FLOAT ) ||
                          !gan_image_convert_pixel_i ( &MaskPixel, GAN_GREY_LEVEL_IMAGE, GAN_FLOAT ) )
                     {
                        gan_err_register ( "gan_image_composite_q", GAN_ERROR_FAILURE, "");
                        return NULL;
                     }

                     /* blend pixel */
                     Pixel.data.rgb.f.R = MaskPixel.data.gl.f*SAPixel.data.rgb.f.R + (1.0F-MaskPixel.data.gl.f)*Pixel.data.rgb.f.R;
                     Pixel.data.rgb.f.G = MaskPixel.data.gl.f*SAPixel.data.rgb.f.G + (1.0F-MaskPixel.data.gl.f)*Pixel.data.rgb.f.G;
                     Pixel.data.rgb.f.B = MaskPixel.data.gl.f*SAPixel.data.rgb.f.B + (1.0F-MaskPixel.data.gl.f)*Pixel.data.rgb.f.B;

                     /* replace blended pixel in image */
                     if ( !gan_image_convert_pixel_i ( &Pixel, dest->format, dest->type )||
                          !gan_image_set_pix ( dest,
                                               uiRow+(unsigned int)source_with_alpha->offset_y,
                                               uiCol+(unsigned int)source_with_alpha->offset_x, &Pixel ) )
                     {
                        gan_err_register ( "gan_image_composite_q", GAN_ERROR_FAILURE, "");
                        return NULL;
                     }
                  }
         }
         break;
      }
   }

   /* success */
   return dest;
}

/**
 * \}
 */

/**
 * \}
 */
