/**
 * File:          $RCSfile: pbm_io.c,v $
 * Module:        PBM format image file I/O functions
 * Part of:       Gandalf Library
 *
 * Revision:      $Revision: 1.5 $
 * Last edited:   $Date: 2003/01/31 18:56:57 $
 * Author:        $Author: pm $
 * Copyright:     (c) 2000 Imagineer Software Limited
 */

/* This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <string.h>
#include <gandalf/image/io/pbm_io.h>
#include <gandalf/image/image_bit.h>
#include <gandalf/common/misc_error.h>
#include <gandalf/common/allocate.h>
#include <gandalf/common/array.h>

/**
 * \addtogroup ImagePackage
 * \{
 */

/**
 * \addtogroup ImageIO
 * \{
 */

static Gan_Bool read_line ( FILE *infile, char *s )
{
  char *news, *sptr;
  static char line[80]="";

  if ((strlen (line) == 0) || ((sptr = strtok (NULL, " \n\t")) == NULL))
  {
     while ((news = fgets (line, 80, infile)) != NULL)
        if ( (line[0] != '#') && ((sptr = strtok (line, " \n\t")) != NULL) )
           break;

     if (news == NULL)
     {
        gan_err_flush_trace();
        gan_err_register ( "read_line", GAN_ERROR_CORRUPTED_FILE,
                           "reading line");
        return GAN_FALSE;
     }
  }

  strcpy ( s, sptr );
  return GAN_TRUE;
}

/**
 * \brief Reads a binary image file in PBM format from a stream.
 * \param infile The file stream to be read
 * \param image The image structure to read the image data into or \c NULL
 * \return Pointer to image structure, or \c NULL on failure.
 *
 * Reads the PBM image from the given file stream \a infile into the given
 * \a image. If \a image is \c NULL a new image is dynamically allocated,
 * otherwise the already allocated \a image structure is reused.
 *
 * \sa gan_write_pbm_image_stream().
 */
Gan_Image *
 gan_read_pbm_image_stream ( FILE *infile, Gan_Image *image )
{
   char s[80]="";
   char *signature = "P4\n";
   int iWidth, iHeight, iRow, iCol;
   unsigned uiNoBytes;
   Gan_Bool result=GAN_TRUE;
   unsigned char *ucarr;

   fgets ( s, 3, infile );
   s[2] = '\n';
   if ( strcmp ( s, signature ) != 0 )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_pbm_image", GAN_ERROR_CORRUPTED_FILE,
                         "corrupted PBM file header (no P5)" );
      return NULL;
   }

   result = (Gan_Bool)(result & read_line ( infile, s )); iWidth  = atoi(s);
   result = (Gan_Bool)(result & read_line ( infile, s )); iHeight = atoi(s);

   /* check whether any of the header lines was corrupted */
   if ( !result )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_pbm_image", GAN_ERROR_CORRUPTED_FILE,
                         "" );
      return NULL;
   }

   if ( image == NULL )
      image = gan_image_alloc_b ( iHeight, iWidth );
   else
      image = gan_image_set_b ( image, iHeight, iWidth );

   if ( image == NULL )
   {
      gan_err_register ( "gan_read_pbm_image", GAN_ERROR_FAILURE, "" );
      return NULL;
   }

   /* if the image has zero size then we have finished */
   if ( iWidth == 0 || iHeight == 0 ) return image;

   /* build array of bits */
   uiNoBytes = (unsigned)((iWidth + 7)/8);
   ucarr = gan_malloc_array ( unsigned char, uiNoBytes );
   if ( ucarr == NULL )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_pbm_image_stream", GAN_ERROR_MALLOC_FAILED,
                         "" );
      return NULL;
   }

   /* read image data from stream */
   gan_image_fill_zero(image);
   for ( iRow = 0; iRow < iHeight; iRow++ )
   {
      if ( fread ( ucarr, sizeof(unsigned char), uiNoBytes, infile )
           != uiNoBytes )
      {
         gan_err_flush_trace();
         gan_err_register ( "gan_read_pbm_image_stream",
                            GAN_ERROR_CORRUPTED_FILE, "" );
         return NULL;
      }

      for ( iCol = (int)image->width-1; iCol >= 0; iCol-- )
         if ( ucarr[iCol/8] & (0x80 >> (iCol % 8)) )
            gan_image_set_pix_b ( image, iRow, iCol, GAN_TRUE );
   }

   /* success */
   free(ucarr);
   return image;
}

/**
 * \brief Reads a binary image file in PBM format.
 * \param filename The name of the image file
 * \param image The image structure to read the image data into or \c NULL
 * \return Pointer to image structure, or \c NULL on failure.
 *
 * Reads the PBM image with the in the file \a filename into the given
 * \a image. If \a image is \c NULL a new image is dynamically allocated;
 * otherwise the already allocated \a image structure is reused.
 *
 * \sa gan_write_pbm_image().
 */
Gan_Image *
 gan_read_pbm_image ( const char *filename, Gan_Image *image )
{
   FILE *infile;

   /* attempt to open file */
   infile = fopen ( filename, "rb" );
   if ( infile == NULL )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_pbm_image", GAN_ERROR_OPENING_FILE,
                         filename );
      return NULL;
   }

   return gan_read_pbm_image_stream ( infile, image );
}

/**
 * \brief Writes a binary image to a file stream in PBM format.
 * \param outfile The file stream
 * \param image The image structure to write to the file
 * \return #GAN_TRUE on successful write operation, or #GAN_FALSE on failure.
 *
 * Writes the \a image into the file stream \a outfile in PBM format.
 *
 * \sa gan_read_pbm_image_stream().
 */
Gan_Bool
 gan_write_pbm_image_stream ( FILE *outfile, Gan_Image *image )
{
   unsigned char *ucarr;
   int iRow, iCol, iHeight = (int)image->height;
   unsigned uiNoBytes;

   gan_err_test_bool ( image->format == GAN_GREY_LEVEL_IMAGE &&
                       image->type == GAN_BOOL,
                       "gan_write_pbm_image_stream", GAN_ERROR_INCOMPATIBLE,
                       "" );

   fprintf ( outfile, "P4\n#Constructed by Gandalf\n%d %d\n",
             (int)image->width, (int)image->height );

   /* if the image has zero size then we have finished */
   if ( image->width == 0 || image->height == 0 ) return GAN_TRUE;

   /* build array of bits */
   uiNoBytes = (image->width+7)/8;
   ucarr = gan_malloc_array ( unsigned char, uiNoBytes );
   if ( ucarr == NULL )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_write_pbm_image_stream", GAN_ERROR_FAILURE, "" );
      return GAN_FALSE;
   }


   for ( iRow = 0; iRow < iHeight; iRow++ )
   {
      gan_fill_array_uc ( ucarr, uiNoBytes, 1, 0 );
      for ( iCol = (int)image->width-1; iCol >= 0; iCol-- )
         if ( gan_image_get_pix_b ( image, iRow, iCol ) )
            ucarr[iCol/8] |= (0x80 >> (iCol % 8));

      /* write image data row to stream */
      fwrite ( ucarr, sizeof(unsigned char), uiNoBytes, outfile );
   }

   /* success */
   free(ucarr);
   return GAN_TRUE;
}

/**
 * \brief Writes a grey level image file in PBM format.
 * \param filename The name of the image file
 * \param image The image structure to write to the file
 * \return #GAN_TRUE on successful write operation, #GAN_FALSE on failure.
 *
 * Writes the \a image into PBM file \a filename.
 *
 * \sa gan_read_pbm_image().
 */
Gan_Bool
 gan_write_pbm_image ( const char *filename, Gan_Image *image )
{
   FILE *outfile;
   Gan_Bool result;

   /* attempt to open file */
   outfile = fopen ( filename, "wb" );
   if ( outfile == NULL )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_write_pbm_image", GAN_ERROR_OPENING_FILE,
                         filename );
      return GAN_FALSE;
   }

   result = gan_write_pbm_image_stream ( outfile, image );
   fclose(outfile);
   return result;
}

/**
 * \}
 */

/**
 * \}
 */
