/**
 * File:          $RCSfile: image_composite.c,v $
 * Module:        Extract/fill selected channel(s) of image
 * Part of:       Gandalf Library
 *
 * Revision:      $Revision: 1.5 $
 * Last edited:   $Date: 2003/10/13 10:26:11 $
 * Author:        $Author: pm $
 * Copyright:     (c) 2000 Imagineer Software Limited
 */

/* This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <gandalf/image/image_composite.h>

/**
 * \addtogroup ImagePackage
 * \{
 */

/**
 * \defgroup ImageComposite Composite one image on another
 * \{
 */

/**
 * \brief Macro: Composite images
 * \param source The input image 
 * \param source_with_alpha The input image with the alpha channel
 * \param dest The destination image 
 * \return Composited \a dest image, or \c NULL on failure.
 *
 * Composites the image \a source_with_alpha onto the given image \a source,
 * writing the result onto \a dest.
 *
 * \sa gan_image_composite_i().
 */
Gan_Image *
 gan_image_composite_q ( Gan_Image *source,
                         Gan_Image *source_with_alpha,
                         Gan_Image *dest )
{
   int iRow, iCol;
   Gan_Pixel Pixel, SAPixel;

   if ( dest == NULL )
   {
      dest = gan_image_alloc ( source->format, source->type,
                               source->height, source->width );
      if ( dest == NULL )
      {
         gan_err_register ( "gan_image_composite_q", GAN_ERROR_FAILURE, "");
         return NULL;
      }
   }
   else
      gan_err_test_ptr ( dest->width  == source->width &&
                         dest->height == source->height,
                         "gan_image_composite_q", GAN_ERROR_INCOMPATIBLE, "" );

   for ( iRow = (int)source_with_alpha->height-1; iRow >= 0; iRow-- )
      for ( iCol = (int)source_with_alpha->width-1; iCol >= 0; iCol-- )
      {
         Pixel = gan_image_get_pix ( source, iRow+source_with_alpha->offset_y, iCol+source_with_alpha->offset_x );
         SAPixel = gan_image_get_pix ( source_with_alpha, iRow, iCol );
         if ( !gan_image_convert_pixel_i ( &Pixel,
                                           GAN_RGB_COLOUR_IMAGE,
                                           GAN_DOUBLE ) ||
              !gan_image_convert_pixel_i ( &SAPixel,
                                           GAN_RGB_COLOUR_ALPHA_IMAGE,
                                           GAN_DOUBLE ) )
         {
            gan_err_register ( "gan_image_composite_q", GAN_ERROR_FAILURE, "");
            return NULL;
         }

         /* blend pixel */
         Pixel.data.rgb.d.R = SAPixel.data.rgba.d.A*SAPixel.data.rgba.d.R
                              + (1.0-SAPixel.data.rgba.d.A)*Pixel.data.rgb.d.R;
         Pixel.data.rgb.d.G = SAPixel.data.rgba.d.A*SAPixel.data.rgba.d.G
                              + (1.0-SAPixel.data.rgba.d.A)*Pixel.data.rgb.d.G;
         Pixel.data.rgb.d.B = SAPixel.data.rgba.d.A*SAPixel.data.rgba.d.B
                              + (1.0-SAPixel.data.rgba.d.A)*Pixel.data.rgb.d.B;

         /* replace blended pixel in image */
         if ( !gan_image_convert_pixel_i ( &Pixel, dest->format, dest->type )||
              !gan_image_set_pix ( dest, iRow+source_with_alpha->offset_y, iCol+source_with_alpha->offset_x, &Pixel ) )
         {
            gan_err_register ( "gan_image_composite_q", GAN_ERROR_FAILURE, "");
            return NULL;
         }
      }

   /* success */
   return dest;
}

/**
 * \}
 */

/**
 * \}
 */
