/*************************************************
 *
 * $GAMGI/src/gtk/plane/gamgi_gtk_plane_modify.c
 *
 * Copyright (C) 2006 Carlos Pereira
 *
 * Distributed under the terms of the GNU
 * General Public License: $GAMGI/LICENSE
 *
 */

#include "gamgi_engine.h"
#include "gamgi_gtk.h"
#include "gamgi_mesa.h"
#include "gamgi_math.h"
#include "gamgi_chem.h"
#include "gamgi_phys.h"
#include "gamgi_io.h"
#include "gamgi_global.h"

#include "gamgi_engine_create.h"
#include "gamgi_engine_start.h"
#include "gamgi_engine_remove.h"
#include "gamgi_engine_find.h"
#include "gamgi_engine_list.h"
#include "gamgi_engine_dlist.h"
#include "gamgi_engine_link.h"
#include "gamgi_engine_unlink.h"
#include "gamgi_engine_reparent.h"
#include "gamgi_gtk_dialog.h"
#include "gamgi_gtk_history.h"
#include "gamgi_gtk_object.h"
#include "gamgi_gtk_cell_symmetry.h"
#include "gamgi_gtk_plane_node.h"
#include "gamgi_gtk_plane_modify.h"
#include "gamgi_mesa_select.h"
#include "gamgi_math_vector.h"
#include "gamgi_math_matrix.h"
#include "gamgi_math_euler.h"
#include "gamgi_math_node.h"
#include "gamgi_math_position.h"
#include "gamgi_phys_projection.h"
#include "gamgi_phys_direction.h"
#include "gamgi_phys_plane.h"
#include "gamgi_io_token.h"

static gamgi_bool static_scan (gamgi_plane *plane, gamgi_dlist *start,
double *position, gamgi_atom **atom_reference)
{
gamgi_atom *atom;
gamgi_dlist *dlist;
double x, y, z;

for (dlist = start; dlist != NULL; dlist = dlist->next)
  {
  atom = GAMGI_CAST_ATOM dlist->data;
  x = atom->position[0] - position[0];
  y = atom->position[1] - position[1];
  z = atom->position[2] - position[2];
  if (x*x + y*y +z*z < GAMGI_MATH_TOLERANCE_LENGTH * GAMGI_MATH_TOLERANCE_LENGTH)
    { *atom_reference = atom; return TRUE; }
  }

return FALSE;
}

static gamgi_bool static_find (gamgi_plane *plane,
gamgi_atom **atom1, gamgi_atom **atom2, gamgi_atom **atom3)
{
gamgi_dlist *dlist;
double matrix[9], inverse[9];
double position[3], translate[3];
double euler[3];

/*********************************************
 * get list of atoms not linked to planes or *
 * directions, including the current plane   *
 *********************************************/

dlist = gamgi_engine_dlist_atom_object (plane->object.object, NULL);
dlist = gamgi_engine_dlist_atom_remove_class (dlist, GAMGI_ENGINE_PLANE);
dlist = gamgi_engine_dlist_atom_remove_class (dlist, GAMGI_ENGINE_DIRECTION);

/***************************************************
 * get rotation and translation to transform atoms *
 * back to where they would be if plane rotation   *
 * and translation were (0, 0, 0) and (0, 0, 0)    *
 ***************************************************/

gamgi_math_position_plane_rotation_get (plane, euler);
gamgi_math_euler_to_matrix (euler, matrix);
gamgi_math_matrix_inverse (matrix, inverse);

gamgi_math_matrix_vector (inverse, plane->origin, translate);
gamgi_math_position_translation (plane->object.object, position);
gamgi_math_vector_sub (position, translate, translate);

/****************************************************************
 * try to find the original atoms of the reference atoms, where *
 * plane rotation and translation are (0, 0, 0) and (0, 0, 0)   *
 ****************************************************************/

gamgi_math_matrix_vector (inverse, (*atom1)->position, position);
gamgi_math_vector_add (position, translate, position);
if (static_scan (plane, dlist, position, atom1) == FALSE) return FALSE;

gamgi_math_matrix_vector (inverse, (*atom2)->position, position);
gamgi_math_vector_add (position, translate, position);
if (static_scan (plane, dlist, position, atom2) == FALSE) return FALSE;

gamgi_math_matrix_vector (inverse, (*atom3)->position, position);
gamgi_math_vector_add (position, translate, position);
if (static_scan (plane, dlist, position, atom3) == FALSE) return FALSE;

return TRUE;
}

static void static_remove (gamgi_plane *plane)
{
gamgi_dlist *dlist;

dlist = plane->direction_start;
while (dlist != NULL)
  { gamgi_engine_remove_direction (GAMGI_CAST_DIRECTION dlist->data);
    dlist = gamgi_engine_dlist_remove_start (dlist); }
plane->direction_start = NULL;
plane->direction_end = NULL;

dlist = plane->atom_start;
while (dlist != NULL)
  { gamgi_engine_remove_atom (GAMGI_CAST_ATOM dlist->data);
    dlist = gamgi_engine_dlist_remove_start (dlist); } 
plane->atom_start = NULL;
plane->atom_end = NULL;
  
dlist = plane->orbital_start;
while (dlist != NULL)
  { gamgi_engine_remove_orbital (GAMGI_CAST_ORBITAL dlist->data);
    dlist = gamgi_engine_dlist_remove_start (dlist); }
plane->orbital_start = NULL;
plane->orbital_end = NULL;

dlist = plane->text_start;
while (dlist != NULL)
  { gamgi_engine_remove_text (GAMGI_CAST_TEXT dlist->data);
    dlist = gamgi_engine_dlist_remove_start (dlist); }
plane->text_start = NULL;
plane->text_end = NULL;
}

static GtkWidget *static_atom (GtkWidget *dialog)
{
GtkWidget *entry;
const char *name;

/**********
 * atom 1 *
 **********/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_atom1");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == FALSE) return entry;

/**********
 * atom 2 *
 **********/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_atom2");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == FALSE) return entry;

/**********
 * atom 3 *
 **********/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_atom3");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == FALSE) return entry;

return NULL;
}

static void static_end_atom (GtkWidget *dialog)
{
GtkWidget *entry;

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_atom1");
gtk_entry_set_text (GTK_ENTRY (entry), "");
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_atom2");
gtk_entry_set_text (GTK_ENTRY (entry), "");
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_atom3");
gtk_entry_set_text (GTK_ENTRY (entry), "");
}

static gamgi_enum static_class (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *entry;
GtkWidget *combo;
const char *name;
int row;

/***************************************************
 * when the plane entry is empty, look for a plane *
 ***************************************************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_plane");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == FALSE) return GAMGI_ENGINE_PLANE;

/*******************************************
 * otherwise select which object to select *
 *******************************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_reference");
row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));
if (row == GAMGI_GTK_CELL - 1) return GAMGI_ENGINE_CELL;
if (row == GAMGI_GTK_ATOMS - 1) return GAMGI_ENGINE_ATOM;

/****************************************************
 * this case can only occur when Reference is Local *
 ****************************************************/

return FALSE;
}

static void static_start_cell (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *entry;
GtkWidget *combo;

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_cell");
gtk_entry_set_text (GTK_ENTRY (entry), "");

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_h");
gtk_entry_set_text (GTK_ENTRY (entry), "");
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_k");
gtk_entry_set_text (GTK_ENTRY (entry), "");
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_l");
gtk_entry_set_text (GTK_ENTRY (entry), "");

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_vectors");
gtk_combo_box_set_active (GTK_COMBO_BOX (combo), GAMGI_PHYS_CONVENTIONAL - 1);
}

static void static_start_atom (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *entry;

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_atom1");
gtk_entry_set_text (GTK_ENTRY (entry), "");
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_atom2");
gtk_entry_set_text (GTK_ENTRY (entry), "");
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_atom3");
gtk_entry_set_text (GTK_ENTRY (entry), "");
}

static void static_instance (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog = window->dialog0;
GtkWidget *combo;
GtkWidget *hbox_order, *hbox_node;
GtkWidget *entry;
int row;

/*********************************************************
 * initialize order,node combo box (before disabling it) *
 *********************************************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_o4");
gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 7);

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_o1");
gtk_entry_set_text (GTK_ENTRY (entry), "");
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_o2");
gtk_entry_set_text (GTK_ENTRY (entry), "");
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_o3");
gtk_entry_set_text (GTK_ENTRY (entry), "");

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_order");
gtk_entry_set_text (GTK_ENTRY (entry), "");

/***********************************
 * set order,node hbox sensitivity *
 ***********************************/

hbox_order = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "hbox_order");
hbox_node = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "hbox_node");

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_reference");
row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

if (row == GAMGI_GTK_ATOMS - 1)
  {
  gtk_widget_set_sensitive (hbox_order, FALSE);
  gtk_widget_set_sensitive (hbox_node, FALSE);
  return;
  }
else gtk_widget_set_sensitive (hbox_order, TRUE);

/******************************
 * set order,node sensitivity *
 ******************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_instance");
row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

if (row == GAMGI_GTK_ORDER - 1)
  {
  gtk_widget_set_sensitive (entry, TRUE);
  gtk_widget_set_sensitive (hbox_node, FALSE);
  }
else
  {
  gtk_widget_set_sensitive (entry, FALSE);
  gtk_widget_set_sensitive (hbox_node, TRUE);
  }

}

static void static_start_position (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *entry;
char token[GAMGI_ENGINE_TOKEN];

/***********************
 * start Position page *
 ***********************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_x");
sprintf (token, "%.*f", gamgi->gamgi->length, GAMGI_PHYS_CELL_X);
gtk_entry_set_text (GTK_ENTRY (entry), token);

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_y");
sprintf (token, "%.*f", gamgi->gamgi->length, GAMGI_PHYS_CELL_Y);
gtk_entry_set_text (GTK_ENTRY (entry), token);

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_z");
sprintf (token, "%.*f", gamgi->gamgi->length, GAMGI_PHYS_CELL_Z);
gtk_entry_set_text (GTK_ENTRY (entry), token);

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_e1");
sprintf (token, "%.*f", gamgi->gamgi->angle, GAMGI_PHYS_CELL_E1);
gtk_entry_set_text (GTK_ENTRY (entry), token);

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_e2");
sprintf (token, "%.*f", gamgi->gamgi->angle, GAMGI_PHYS_CELL_E2);
gtk_entry_set_text (GTK_ENTRY (entry), token);

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_e3");
sprintf (token, "%.*f", gamgi->gamgi->angle, GAMGI_PHYS_CELL_E3);
gtk_entry_set_text (GTK_ENTRY (entry), token);
}

static void static_autonomy (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog = window->dialog0;
GtkWidget *vbox;
GtkWidget *entry;
GtkWidget *label;
GtkWidget *combo;
char token[GAMGI_ENGINE_TOKEN];
int row_reference, row_autonomy;

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_reference");
row_reference = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_autonomy");
row_autonomy = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

/************************
 * enable/disable scale *
 ************************/

vbox = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "vbox_position");

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_scale");
label = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "label_scale");

if (row_autonomy != GAMGI_GTK_ALL - 1 && row_autonomy != GAMGI_GTK_PARTIAL - 1)
  {
  gtk_widget_hide (vbox);

  gtk_entry_set_text (GTK_ENTRY (entry), "");
  if (row_autonomy == GAMGI_GTK_NONE - 1)
    {
    gtk_widget_set_sensitive (entry, FALSE);
    gtk_widget_set_sensitive (label, FALSE);
    }
  else
    {
    gtk_widget_set_sensitive (entry, TRUE);
    gtk_widget_set_sensitive (label, TRUE);
    }

  }
else
  {
  if (gtk_widget_get_visible (vbox) == FALSE)
    {
    static_start_position (window);

    gtk_widget_set_sensitive (entry, TRUE);
    gtk_widget_set_sensitive (label, TRUE);
    sprintf (token, "%.*f", GAMGI_MATH_DECIMAL_SCALE, GAMGI_MESA_SCALE);
    gtk_entry_set_text (GTK_ENTRY (entry), token);
    }
  gtk_widget_show (vbox);
  }

/**********************************
 * enable/disable thickness entry *
 **********************************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_thickness");
label = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "label_thickness");

if (row_autonomy == GAMGI_GTK_ALL - 1 || row_autonomy == GAMGI_GTK_PARTIAL - 1 ||
(row_reference == GAMGI_GTK_ATOMS - 1 && row_autonomy ==  GAMGI_GTK_NONE - 1))
  {
  gtk_widget_set_sensitive (entry, TRUE);
  gtk_widget_set_sensitive (label, TRUE);
  sprintf (token, "%.*f", GAMGI_MATH_DECIMAL_LENGTH,
  GAMGI_CHEM_PLANE_THICKNESS);
  gtk_entry_set_text (GTK_ENTRY (entry), token);
  }
else
  {
  gtk_entry_set_text (GTK_ENTRY (entry), "");
  if (row_reference == GAMGI_GTK_CELL - 1 && row_autonomy == GAMGI_GTK_NONE - 1)
    {
    gtk_widget_set_sensitive (entry, FALSE);
    gtk_widget_set_sensitive (label, FALSE);
    }
  else
    {
    gtk_widget_set_sensitive (entry, TRUE);
    gtk_widget_set_sensitive (label, TRUE);
    }
  }

}

static void static_reference (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog = window->dialog0;
GtkWidget *vbox_cell, *vbox_atom;
GtkWidget *combo;
int row;

vbox_cell = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "vbox_cell");
vbox_atom = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "vbox_atom");

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_reference");
row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

if (row == GAMGI_GTK_ATOMS - 1)
  {
  gtk_widget_hide (vbox_cell);
  gtk_widget_show (vbox_atom);
  static_start_atom (window);
  }
else
  {
  gtk_widget_hide (vbox_atom);
  gtk_widget_show (vbox_cell);
  static_start_cell (window);
  }

/**************************
 * Set combo box to Local *
 **************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_autonomy");
gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 3);
static_autonomy (NULL, window);

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_instance");
gtk_combo_box_set_active (GTK_COMBO_BOX (combo), GAMGI_GTK_ORDER - 1);
static_instance (NULL, window);
}

static void static_start_scope (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *entry;
GtkWidget *combo;
char token[GAMGI_ENGINE_TOKEN];

/*********************************************************
 * when the previous cell was a projection, initialize   *
 * with the same defaults used in gamgi_gtk_plane_create *
 *********************************************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_autonomy");
gtk_combo_box_set_active (GTK_COMBO_BOX (combo), GAMGI_GTK_NONE - 1);
static_autonomy (NULL, window);

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_instance");
gtk_combo_box_set_active (GTK_COMBO_BOX (combo), GAMGI_GTK_ORDER - 1);
static_instance (NULL, window);

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_order");
sprintf (token, "%d", GAMGI_PHYS_PLANE_RANGE_START);
gtk_entry_set_text (GTK_ENTRY (entry), token);
}

static void static_start_projection (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *entry;
GtkWidget *label;
GtkWidget *combo;

/******************************************************
 * default for projection and type combo boxes: Local *
 ******************************************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_projection");
gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 2);

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_type");
gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 2);

/*****************
 * disable scale *
 *****************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_scale");
label = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "label_scale");

gtk_widget_set_sensitive (entry, FALSE);
gtk_widget_set_sensitive (label, FALSE);

/********************
 * disable autonomy *
 ********************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_autonomy");
gtk_combo_box_set_active (GTK_COMBO_BOX (combo), GAMGI_GTK_NONE - 1);
static_autonomy (NULL, window);
}

static void static_nodes (gamgi_cell *cell, gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *combo;
int lattice, vectors;
int row;

/******************************
 * initialize node combo box  *
 * (before desabling it)      *
 ******************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_o4");
gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 7);

/********************
 * get cell lattice *
 ********************/

if (cell == NULL) lattice = 0;
else lattice = cell->lattice;

/********************
 * get cell vectors *
 ********************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_vectors");
row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

if (row == GAMGI_PHYS_PRIMITIVE - 1) vectors = GAMGI_PHYS_PRIMITIVE;
else vectors = GAMGI_PHYS_CONVENTIONAL;

/****************
 * handle nodes *
 ****************/

gamgi_gtk_cell_symmetry_nodes (lattice, vectors, dialog);
}

static void static_cell (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
gamgi_cell *cell;
GtkWidget *dialog = window->dialog0;
GtkWidget *vbox_scope, *vbox_projection;

vbox_scope = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "vbox_scope");
vbox_projection = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "vbox_projection");

cell = GAMGI_CAST_CELL gamgi_gtk_object_name_number (dialog,
"entry_cell", GAMGI_ENGINE_CELL, window);
if (cell != NULL && cell->type == GAMGI_PHYS_PROJECTION)
  {
  if (gtk_widget_get_visible (vbox_projection) == FALSE)
    static_start_projection (window);
  gtk_widget_hide (vbox_scope);
  gtk_widget_show (vbox_projection);
  }
else
  {
  if (gtk_widget_get_visible (vbox_scope) == FALSE)
    static_start_scope (window);
  gtk_widget_hide (vbox_projection);
  gtk_widget_show (vbox_scope);
  }

static_nodes (cell, window);
}

static void static_vectors (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
gamgi_cell *cell;
GtkWidget *dialog = window->dialog0;

cell = GAMGI_CAST_CELL gamgi_gtk_object_name_number (dialog,
"entry_cell", GAMGI_ENGINE_CELL, window);
if (cell != NULL && cell->type != GAMGI_PHYS_PROJECTION)
  static_nodes (cell, window);
}

static void static_reset (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *combo;
GtkWidget *entry;

/********************
 * clean plane name *
 ********************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_plane");
gtk_entry_set_text (GTK_ENTRY (entry), "");

/********************
 * reset plane data *
 ********************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_reference");
gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 2);
static_reference (NULL, window);

/****************
 * reset colors *
 ****************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_red");
gtk_entry_set_text (GTK_ENTRY (entry), "");
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_green");
gtk_entry_set_text (GTK_ENTRY (entry), "");
entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_blue");
gtk_entry_set_text (GTK_ENTRY (entry), "");
}

static gamgi_bool static_ok_cell (gamgi_cell **cell, 
int *hkl, gamgi_enum *vectors, gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *entry;
GtkWidget *combo;
const char *name;
int row;

/**********************
 * get reference cell *
 **********************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_cell");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE)
  *cell = GAMGI_CAST_CELL gamgi_gtk_object_name_number (dialog,
  "entry_cell", GAMGI_ENGINE_CELL, window);
if (*cell == NULL)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid reference cell", window);
  return FALSE;
  }

/*********************
 * Get plane indices *
 *********************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_h");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE &&
gamgi_io_token_int_scan (name, &hkl[0], -INT_MAX, INT_MAX) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid plane indices", window);
  return FALSE;
  }

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_k");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE &&
gamgi_io_token_int_scan (name, &hkl[1], -INT_MAX, INT_MAX) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid plane indices", window);
  return FALSE;
  }

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_l");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE &&
gamgi_io_token_int_scan (name, &hkl[2], -INT_MAX, INT_MAX) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid plane indices", window);
  return FALSE;
  }

if (gamgi_phys_direction_zero (hkl) == TRUE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid plane indices", window);
  return FALSE;
  }

/*********************
 * get plane vectors *
 *********************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_vectors");
row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

if (row == GAMGI_PHYS_CONVENTIONAL - 1) *vectors = GAMGI_PHYS_CONVENTIONAL;
if (row == GAMGI_PHYS_PRIMITIVE - 1) *vectors = GAMGI_PHYS_PRIMITIVE;

if (*vectors == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid plane vectors", window);
  return FALSE;
  }

return TRUE;
}

static gamgi_bool static_ok_atoms (gamgi_plane *plane,
gamgi_atom **atom1, gamgi_atom **atom2, gamgi_atom **atom3, 
gamgi_enum *inside, gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *entry;
gamgi_bool inside1, inside2, inside3;
const char *name;

/************************
 * get reference atom 1 *
 ************************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_atom1");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE)
  *atom1 = GAMGI_CAST_ATOM gamgi_gtk_object_name_number (dialog,
  "entry_atom1", GAMGI_ENGINE_ATOM, window);
if (*atom1 == NULL)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid reference atom", window);
  return FALSE;
  }

/************************
 * get reference atom 2 *
 ************************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_atom2");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE)
  *atom2 = GAMGI_CAST_ATOM gamgi_gtk_object_name_number (dialog,
  "entry_atom2", GAMGI_ENGINE_ATOM, window);
if (*atom2 == NULL)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid reference atom", window);
  return FALSE;
  }

/************************
 * get reference atom 3 *
 ************************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_atom3");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE)
  *atom3 = GAMGI_CAST_ATOM gamgi_gtk_object_name_number (dialog,
  "entry_atom3", GAMGI_ENGINE_ATOM, window);
if (*atom3 == NULL)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid reference atom", window);
  return FALSE;
  }

/**********************************************
 * check if atoms already belong to the plane *
 **********************************************/

inside1 = FALSE;
if ((*atom1)->object.object == GAMGI_CAST_OBJECT plane) inside1 = TRUE;
inside2 = FALSE;
if ((*atom2)->object.object == GAMGI_CAST_OBJECT plane) inside2 = TRUE;
inside3 = FALSE;
if ((*atom3)->object.object == GAMGI_CAST_OBJECT plane) inside3 = TRUE;

*inside = GAMGI_GTK_PARTIAL;
if (inside1 == TRUE && inside2 == TRUE && inside3 == TRUE) *inside = GAMGI_GTK_ALL;
if (inside1 == FALSE && inside2 == FALSE && inside3 == FALSE) *inside = GAMGI_GTK_NONE;

/******************************************
 * a plane cannot be modified with atoms  *
 * inside and outside, to avoid conflicts *
 ******************************************/

if (*inside == GAMGI_GTK_PARTIAL)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid atom parents", window);
  return FALSE;
  }
  
/*************************************************************
 * atoms already belonging to planes, directions cannot be   *
 * used to define new planes, directions, to avoid conflicts *
 *************************************************************/

if (*inside == GAMGI_GTK_NONE && 
gamgi_phys_plane_compatible (*atom1, *atom2, *atom3) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid atom parents", window);
  return FALSE;
  }

/**************************************
 * atoms must be far enough from each *
 * other to define a proper polygon   *
 **************************************/

if (gamgi_phys_plane_polygon (*atom1, *atom2, *atom3) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid atom parents", window);
  return FALSE;
  }

return TRUE;
}

static gamgi_bool static_ok_scope (gamgi_cell *cell, int *hkl,
gamgi_enum vectors, gamgi_enum *type, gamgi_projection *projection,
gamgi_enum *autonomy, double *thickness, int *order, gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *combo;
GtkWidget *entry;
const char *name;
int node[4];
int o1, o2, o3;
int row;

/***************************
 * this is a polygon plane *
 ***************************/

*type = GAMGI_PHYS_POLYGON;

/***********************************************
 * only pole,traces have a projection function *
 ***********************************************/

*projection = NULL;

/*********************
 * get link autonomy *
 *********************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_autonomy");
row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

if (row == GAMGI_GTK_ALL - 1) *autonomy = GAMGI_GTK_ALL;
if (row == GAMGI_GTK_PARTIAL - 1) *autonomy = GAMGI_GTK_PARTIAL;
if (row == GAMGI_GTK_NONE - 1) *autonomy = GAMGI_GTK_NONE;

/**************************
 * get scanning thickness *
 **************************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_thickness");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE &&
gamgi_io_token_double_scan (name, thickness, 
GAMGI_CHEM_PLANE_THICKNESS, DBL_MAX) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid plane thickness", window);
  return FALSE;
  }

/****************************************
 * return control for atomic planes,    *
 * continue for crystallographic planes *
 ****************************************/

if (cell == NULL) return TRUE;

/**********************
 * get plane instance *
 **********************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_instance");
row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

if (row == GAMGI_GTK_ORDER - 1)
  {
  /*******************
   * get plane order *
   *******************/

  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_order");
  name = gtk_entry_get_text (GTK_ENTRY (entry));
  if (gamgi_io_token_check (name) == TRUE &&
  gamgi_io_token_int_scan (name, order, -INT_MAX, INT_MAX) == FALSE)
    {
    gamgi_gtk_dialog_message_create ("Error", "Invalid plane order", window);
    return FALSE;
    }
  }

if (row == GAMGI_GTK_NODE - 1)
  {
  /******************
   * get plane node *
   ******************/

  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_o1");
  name = gtk_entry_get_text (GTK_ENTRY (entry));
  if (gamgi_io_token_int_scan (name, &node[0], -INT_MAX, INT_MAX) == FALSE)
    {
    gamgi_gtk_dialog_message_create ("Error", "Invalid node coordinates", window);
    return FALSE;
    }

  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_o2");
  name = gtk_entry_get_text (GTK_ENTRY (entry));
  if (gamgi_io_token_int_scan (name, &node[1], -INT_MAX, INT_MAX) == FALSE)
    {
    gamgi_gtk_dialog_message_create ("Error", "Invalid node coordinates", window);
    return FALSE;
    }

  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_o3");
  name = gtk_entry_get_text (GTK_ENTRY (entry));
  if (gamgi_io_token_int_scan (name, &node[2], -INT_MAX, INT_MAX) == FALSE)
    {
    gamgi_gtk_dialog_message_create ("Error", "Invalid node coordinates", window);
    return FALSE;
    }

  /*******************************************************************
   * if o4 is not Local, get o4 node coordinate (unless the lattice  *
   * is centered and the vectors are conventional, o4 is always 000) *
   *******************************************************************/

  combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_o4");
  row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));
  if (row != 7) node[3] = row;

  if (gamgi_math_node_check (cell->lattice, node[3]) == FALSE)
    {
    gamgi_gtk_dialog_message_create ("Error", "Invalid node coordinates", window);
    return FALSE;
    }

  /*****************************************
   * 1) get node in primitive coordinates  *
   * 2) get plane order from node and cell *
   *****************************************/

  if (vectors == GAMGI_PHYS_CONVENTIONAL)
    gamgi_math_node_primitive (cell->lattice, 
    node[0], node[1], node[2], node[3], &o1, &o2, &o3);
  else
    { o1 = node[0]; o2 = node[1]; o3 = node[2]; }
  *order = gamgi_phys_plane_order (cell, hkl, vectors, o1, o2, o3);
  }

return TRUE;
}

static gamgi_bool static_ok_projection (gamgi_enum *autonomy,
gamgi_projection *projection, gamgi_enum *type, gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *combo;
int row;

/******************************************
 * plane cannot move away from projection *
 ******************************************/

*autonomy = GAMGI_GTK_NONE;

/************************************
 * get projection: Wulff or Schmidt *
 ************************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_projection");
row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

if (row == GAMGI_PHYS_WULFF - 1) *projection = gamgi_phys_projection_wulff;
if (row == GAMGI_PHYS_SCHMIDT - 1) *projection = gamgi_phys_projection_schmidt;

if (*projection == NULL) *projection = GAMGI_PHYS_CELL_PROJECTION;

/*****************************************
 * get plane representation: pole, trace *
 *****************************************/

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_type");
row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));

if (row == GAMGI_PHYS_POLE - 1) *type = GAMGI_PHYS_POLE;
if (row == GAMGI_PHYS_TRACE - 1) *type = GAMGI_PHYS_TRACE;

if (*type != GAMGI_PHYS_POLE && *type != GAMGI_PHYS_TRACE)
  *type = GAMGI_PHYS_POLE;

return TRUE;
}

static gamgi_bool static_ok_position (gamgi_enum autonomy,
double *euler, double *origin, gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *entry;
const char *name;

if (autonomy == GAMGI_GTK_NONE)
  {
  gamgi_math_vector_zero (origin);
  gamgi_math_vector_zero (euler);
  return TRUE;
  }
  
/****************************
 * origin x,y,z coordinates *
 ****************************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_x");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE &&
gamgi_io_token_double_scan (name, &origin[0], -DBL_MAX, DBL_MAX) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid position data", window);
  return FALSE;
  }

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_y");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE &&
gamgi_io_token_double_scan (name, &origin[1], -DBL_MAX, DBL_MAX) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid position data", window);
  return FALSE;
  }

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_z");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE &&
gamgi_io_token_double_scan (name, &origin[2], -DBL_MAX, DBL_MAX) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid position data", window);
  return FALSE;
  }

/*************************
 * e1,e2,e3 euler angles *
 *************************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_e1");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE &&
gamgi_io_token_double_scan (name, &euler[0], 0.0, 180.0) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid angle data", window);
  return FALSE;
  }

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_e2");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE &&
gamgi_io_token_double_scan (name, &euler[1], 0.0, 360.0) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid angle data", window);
  return FALSE;
  }

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_e3");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE &&
gamgi_io_token_double_scan (name, &euler[2], 0.0, 360.0) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid angle data", window);
  return FALSE;
  }

return TRUE;
}

static gamgi_bool static_ok_view (float *red, float *green, 
float *blue, double *scale, gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *entry;
const char *name;

/*******
 * red *
 *******/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_red");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE &&
gamgi_io_token_float_scan (name, red, 0.0, 1.0) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid plane color", window);
  return FALSE;
  }

/*********
 * green *
 *********/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_green");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE &&
gamgi_io_token_float_scan (name, green, 0.0, 1.0) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid plane color", window);
  return FALSE;
  }

/********
 * blue *
 ********/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_blue");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE &&
gamgi_io_token_float_scan (name, blue, 0.0, 1.0) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid plane color", window);
  return FALSE;
  }

/*************
 * Get scale *
 *************/

entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_scale");
name = gtk_entry_get_text (GTK_ENTRY (entry));
if (gamgi_io_token_check (name) == TRUE &&
gamgi_io_token_double_scan (name, scale, 
GAMGI_MESA_SCALE_LOWER, GAMGI_MESA_SCALE_UPPER) == FALSE)
  {
  gamgi_gtk_dialog_message_create ("Error", "Invalid plane scale", window);
  return FALSE;
  }

return TRUE;
}

/************* internal function ************
 *                                          *
 *                STATIC_GLOBAL             *
 *                                          *
 * Makes the plane entry sensitive when the *
 * local button is pressed and insensitive  *
 * when the global button is pressed.       *
 *                                          *
 ********************************************/

static void static_global (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog = window->dialog0;
GtkWidget *hbox, *button;

hbox = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "hbox_plane");
button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_global");

if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)) == FALSE)
  {
  /*****************************
   * Choose the Plane directly *
   *****************************/

  gtk_widget_set_sensitive (hbox, TRUE);
  }
else
  {
  /********************************************
   * Choose the Cell List previously selected *
   ********************************************/

  gtk_widget_set_sensitive (hbox, FALSE);
  static_reset (window);
  }
}

/*************** internal function *******************
 *                                                   *
 *                    STATIC_OK                      *
 *                                                   *
 * This is the function that actually creates new    *
 * planes. It is triggered when the user presses Ok. *
 *                                                   *
 *****************************************************/

static void static_ok (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog = window->dialog0;
gamgi_cell *cell;
gamgi_plane *plane;
gamgi_atom *atom1, *atom2, *atom3;
gamgi_object *parent;
gamgi_dlist *dlist;
gamgi_projection projection;
GtkWidget *button;
GtkWidget *combo;
gamgi_enum vectors;
gamgi_enum reference, autonomy, type;
gamgi_enum pole, trace, inside;
gamgi_bool copy;
gamgi_bool valid;
char token[GAMGI_ENGINE_TOKEN];
double euler[3], origin[3], zero[3];
double thickness, scale;
float red, green, blue;
int hkl[3];
int order;
int row;

button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_global");
if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)) == FALSE)
  {
  /****************
   * Local method *
   ****************/

  /**************************************
   * Get name (possibly new) and number *
   **************************************/

  plane = GAMGI_CAST_PLANE gamgi_gtk_object_number (dialog,
  "entry_plane", GAMGI_ENGINE_PLANE, window, token);
  if (plane == NULL)
    {
    gamgi_gtk_dialog_message_create ("Error", "Invalid plane name", window);
    return;
    }

  /**************************
   * get initial parameters *
   **************************/

  reference = plane->reference;
  hkl[0] = plane->hkl[0];
  hkl[1] = plane->hkl[1];
  hkl[2] = plane->hkl[2];
  vectors = plane->vectors;

  type = plane->type;
  projection = plane->projection;
  autonomy = plane->autonomy;
  thickness = plane->thickness;
  order = plane->order;

  red = plane->red;
  green = plane->green;
  blue = plane->blue;
  scale = plane->object.scale;

  gamgi_math_position_plane_rotation_get (plane, euler);
  gamgi_math_position_plane_translation_get (plane, origin);

  /***********************************
   * this unneeded initialization    *
   * avoids a O3 compilation warning *
   ***********************************/

  atom1 = atom2 = atom3 = NULL; cell = NULL;
  pole = FALSE; trace = FALSE;
  inside = FALSE;

  /********************************
   * get reference: cell or atoms *
   ********************************/

  combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_reference");
  row = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));
  if (row == GAMGI_GTK_CELL - 1) reference = GAMGI_ENGINE_CELL;
  if (row == GAMGI_GTK_ATOMS - 1) reference = GAMGI_ENGINE_ATOM;

  if (reference == GAMGI_ENGINE_CELL)
    {
    /************************************
     * this is a crystallographic plane *
     ************************************/

    cell = NULL;
    parent = plane->object.object;
    if (parent->class == GAMGI_ENGINE_CELL)
      cell = GAMGI_CAST_CELL parent;

    valid = static_ok_cell (&cell, hkl, &vectors, window);
    if (valid == FALSE) return;

    if (cell->type == GAMGI_PHYS_PROJECTION)
      {
      valid = static_ok_projection (&autonomy, &projection, &type, window);
      if (valid == FALSE) return;

      if (type == GAMGI_PHYS_POLE) pole = TRUE;
      if (type == GAMGI_PHYS_TRACE) trace = TRUE;
      }
    else
      {
      valid = static_ok_scope (cell, hkl, vectors, 
      &type, &projection, &autonomy, &thickness, &order, window);
      if (valid == FALSE) return;
      }

    }
  else
    {
    /***************************
     * this is an atomic plane *
     ***************************/

    gamgi_phys_plane_reference (plane, &atom1, &atom2, &atom3);
    valid = static_ok_atoms (plane, &atom1, &atom2, &atom3, &inside, window);
    if (valid == FALSE) return;

    valid = static_ok_scope (NULL, NULL, FALSE,
    &type, &projection, &autonomy, &thickness, NULL, window);
    if (valid == FALSE) return;
    }

  valid = static_ok_position (autonomy, euler, origin, window);
  if (valid == FALSE) return;

  valid = static_ok_view (&red, &green, &blue, &scale, window);
  if (valid == FALSE) return;

  /***********************************************************************
   * atomic planes with autonomy all can be modified to autonomy partial *
   * or none, as long as reference atoms in the original positions can   *
   * replace those entered by the user, later removed with the plane     *
   ***********************************************************************/

  if (plane->reference == GAMGI_ENGINE_ATOM && reference == GAMGI_ENGINE_ATOM &&
  plane->autonomy == GAMGI_GTK_ALL && autonomy != GAMGI_GTK_ALL && 
  inside == GAMGI_GTK_ALL && static_find (plane, &atom1, &atom2, &atom3) == FALSE)
    {
    gamgi_gtk_dialog_message_create ("Error", "Invalid atom parents", window);
    return;
    }

  /***************************************
   * Save current layer as undo buffer   *
   * layer so it can be recovered later. *
   ***************************************/

  gamgi_gtk_history_update (window->layer, window->layer->undo, GAMGI_ENGINE_UNDO);

  /***************
   * reset plane *
   ***************/

  gamgi_math_vector_zero (zero);
  gamgi_math_position_plane_rotation_set (plane, zero);
  gamgi_math_position_plane_translation_set (plane, zero);

  free (plane->points);
  free (plane->loops);
  plane->points = NULL;
  plane->loops = NULL;
  plane->n_points = 0;
  plane->n_loops = 0;

  /***********************************
   * link back plane atoms to parent *
   ***********************************/

  if (plane->autonomy == GAMGI_GTK_PARTIAL ||
  (plane->autonomy == GAMGI_GTK_NONE && 
  plane->reference == GAMGI_ENGINE_ATOM))
    gamgi_engine_reparent_atom (plane->atom_start, plane->object.object);

  parent = plane->object.object;

  /*****************************************************************
   * Remove the plane initially with autonomy all except when the  *
   * same plane is modified. When reference atoms and autonomy ALL *
   * are maintained, copy is FALSE because the plane is maintained *
   *****************************************************************/

  copy = (plane->reference != GAMGI_ENGINE_ATOM || 
  reference != GAMGI_ENGINE_ATOM || plane->autonomy != GAMGI_GTK_ALL || 
  autonomy != GAMGI_GTK_ALL || inside != GAMGI_GTK_ALL);

  if (plane->autonomy == GAMGI_GTK_ALL && copy == TRUE)
    {
    /*************************************************
     * Before removing the plane, check if the       *
     * current object is affected. In this case, the *
     * current layer becomes the new current object. *
     *************************************************/

    if (gamgi_engine_find_object (window->focus,
    GAMGI_CAST_OBJECT plane) == TRUE)
      gamgi_gtk_object_focus_local (GAMGI_CAST_OBJECT window->layer, window);

    static_remove (plane);
    }

  gamgi_engine_unlink_plane (plane);

  /************************
   * set plane parameters *
   ************************/

  strcpy (plane->object.name, token);
  plane->reference = reference;
  plane->hkl[0] = hkl[0];
  plane->hkl[1] = hkl[1];
  plane->hkl[2] = hkl[2];
  plane->vectors = vectors;

  plane->type = type;
  plane->projection = projection;
  plane->autonomy = autonomy;
  plane->thickness = thickness;
  plane->order = order;

  plane->red = red;
  plane->green = green;
  plane->blue = blue;
  plane->object.scale = scale;

  /*************************************
   * create and link the plane objects *
   *************************************/

  if (plane->reference == GAMGI_ENGINE_CELL)
    {
    /************************************
     * this is a crystallographic plane *
     ************************************/
  
    if (cell->type == GAMGI_PHYS_PROJECTION)
      gamgi_phys_plane_projection (plane, cell, pole, trace);
    else
      gamgi_phys_plane_range (plane, cell, order, order);
    }

  if (plane->reference == GAMGI_ENGINE_ATOM)
    {
    if (copy == TRUE)
      {
      /********************************************************
       * this is an atomic plane: get list of atoms in parent *
       *                                                      *
       *    filter atoms linked to other plane or direction   *
       *                                                      *
       * reorder list so the three defining atoms come first  *
       ********************************************************/

      parent = gamgi_phys_plane_parent (atom1, atom2, atom3, window);
      dlist = gamgi_engine_dlist_atom_object (parent, NULL);
      dlist = gamgi_engine_dlist_atom_remove_class (dlist, GAMGI_ENGINE_PLANE);
      dlist = gamgi_engine_dlist_atom_remove_class (dlist, GAMGI_ENGINE_DIRECTION);
      }
    else
      {
      /******************************************
       * get list of atoms already in the plane *
       ******************************************/

      dlist = gamgi_engine_dlist_atom_plane (plane, NULL);
      }

    /*********************************************************
     * reset and build plane polygon AFTER linking to parent *
     *********************************************************/

    dlist = gamgi_phys_plane_reorder (dlist, atom1, atom2, atom3);
    gamgi_engine_link_object_object (GAMGI_CAST_OBJECT plane, parent);

    gamgi_math_vector_zero (zero);
    gamgi_math_position_plane_rotation_set (plane, zero);
    gamgi_math_position_plane_translation_set (plane, zero);

    gamgi_phys_plane_atoms (plane, atom1, atom2, atom3, dlist, copy);
    }

  gamgi_math_position_plane_rotation_set (plane, euler);
  gamgi_math_position_plane_translation_set (plane, origin);
  }

/*****************************************
 * Redraw gl_area image and reset dialog *
 *****************************************/
 
gtk_widget_queue_draw (window->area);
static_reset (window);
}

static void static_init (gamgi_window *window)
{
GtkWidget *dialog = window->dialog0;
GtkWidget *notebook;
GtkRequisition size;
GtkWidget *combo;

/***************************************************
 * Set notebook current size as its minimum size:  *
 * this is needed because pages are shown/hidden,  *
 * changing the currently needed size for dialog.  *
 * Fixing the dialog size only partly solves the   *
 * problem because the page size can still change. *
 ***************************************************/

notebook = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "notebook");
gtk_widget_size_request (GTK_WIDGET (notebook), &size);
gtk_widget_set_size_request (notebook, size.width, size.height);

combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_reference");
gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 2);
static_reference (NULL, window);

static_cell (NULL, window);
}

static void static_press (gamgi_object *object, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
gamgi_cell *cell;
gamgi_plane *plane;
gamgi_atom *atom;
gamgi_dlist *dlist;
GtkWidget *dialog = window->dialog0;
GtkWidget *entry;
GtkWidget *combo;
char string[2 * GAMGI_ENGINE_TOKEN];
double origin[3], euler[3];
gamgi_enum class;

class = static_class (window);
if (class == GAMGI_ENGINE_PLANE)
  {
  /****************************
   * set plane identification *
   ****************************/

  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_plane");
  sprintf (string, "%s %d", object->name, object->number);
  gtk_entry_set_text (GTK_ENTRY (entry), string);

  plane = GAMGI_CAST_PLANE object;

  combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_reference");
  if (plane->reference == GAMGI_ENGINE_CELL)
    {
    /************************************
     * This is a crystallographic plane *
     ************************************/

    gtk_combo_box_set_active (GTK_COMBO_BOX (combo), GAMGI_GTK_CELL - 1);
    static_reference (NULL, window);

    /***************************
     * set cell identification *
     ***************************/

    cell = GAMGI_CAST_CELL plane->object.object;

    entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_cell");
    sprintf (string, "%s %d", cell->object.name, cell->object.number);
    gtk_entry_set_text (GTK_ENTRY (entry), string);

    /***********************
     * Plane h,k,l indices *
     ***********************/

    entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_h");
    sprintf (string, "%d", plane->hkl[0]);
    gtk_entry_set_text (GTK_ENTRY (entry), string);

    entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_k");
    sprintf (string, "%d", plane->hkl[1]);
    gtk_entry_set_text (GTK_ENTRY (entry), string);

    entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_l");
    sprintf (string, "%d", plane->hkl[2]);
    gtk_entry_set_text (GTK_ENTRY (entry), string);

    /*****************
     * Plane vectors *
     *****************/

    combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_vectors");
    gtk_combo_box_set_active (GTK_COMBO_BOX (combo), plane->vectors - 1);

    if (cell->type == GAMGI_PHYS_PROJECTION)
      {
      combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_type");
      if (plane->type == GAMGI_PHYS_POLE)
        gtk_combo_box_set_active (GTK_COMBO_BOX (combo), GAMGI_PHYS_POLE - 1);
      if (plane->type == GAMGI_PHYS_TRACE)
        gtk_combo_box_set_active (GTK_COMBO_BOX (combo), GAMGI_PHYS_TRACE - 1);

      combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_projection");
      if (plane->projection == gamgi_phys_projection_wulff)
        gtk_combo_box_set_active (GTK_COMBO_BOX (combo), GAMGI_PHYS_WULFF - 1);
      if (plane->projection == gamgi_phys_projection_schmidt)
        gtk_combo_box_set_active (GTK_COMBO_BOX (combo), GAMGI_PHYS_SCHMIDT - 1);
      }
    else
      {
      /*********************************
       * set Plane autonomy, thickness *
       *********************************/

      combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_autonomy");
      gtk_combo_box_set_active (GTK_COMBO_BOX (combo), plane->autonomy - 1);
      static_autonomy (NULL, window);

      entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_thickness");
      if (gtk_widget_is_sensitive (entry) == TRUE)
        {
        sprintf (string, "%f", plane->thickness);
        gtk_entry_set_text (GTK_ENTRY (entry), string);
        }

      /*****************************
       * set Plane instance, order *
       *****************************/

      combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_instance");
      gtk_combo_box_set_active (GTK_COMBO_BOX (combo), GAMGI_GTK_ORDER - 1);
      static_instance (NULL, window);

      entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_order");
      sprintf (string, "%d", plane->order);
      gtk_entry_set_text (GTK_ENTRY (entry), string);
      }
    }
  else if (plane->reference == GAMGI_ENGINE_ATOM)
    {
    /***************************
     * This is an atomic plane *
     ***************************/

    gtk_combo_box_set_active (GTK_COMBO_BOX (combo), GAMGI_GTK_ATOMS - 1);
    static_reference (NULL, window);

    /******************************************
     * set Atoms identification (dlist should *
     * never be NULL: defensive programming)  *
     ******************************************/

    dlist = plane->atom_start;
    if (dlist == NULL) return;
    atom = GAMGI_CAST_ATOM dlist->data;
    entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_atom1");
    sprintf (string, "%s %d", atom->object.name, atom->object.number);
    gtk_entry_set_text (GTK_ENTRY (entry), string);

    dlist = dlist->next;
    if (dlist == NULL) return;
    atom = GAMGI_CAST_ATOM dlist->data;
    entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_atom2");
    sprintf (string, "%s %d", atom->object.name, atom->object.number);
    gtk_entry_set_text (GTK_ENTRY (entry), string);

    dlist = dlist->next;
    if (dlist == NULL) return;
    atom = GAMGI_CAST_ATOM dlist->data;
    entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_atom3");
    sprintf (string, "%s %d", atom->object.name, atom->object.number);
    gtk_entry_set_text (GTK_ENTRY (entry), string);
  
    /*********************************
     * set Plane autonomy, thickness *
     *********************************/

    combo = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "combo_autonomy");
    gtk_combo_box_set_active (GTK_COMBO_BOX (combo), plane->autonomy - 1);
    static_autonomy (NULL, window);

    entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_thickness");
    sprintf (string, "%f", plane->thickness);
    gtk_entry_set_text (GTK_ENTRY (entry), string);
    }

  /***********************************
   * set Plane orientation, position *
   ***********************************/

  if (plane->autonomy != GAMGI_GTK_NONE)
    {
    gamgi_math_position_plane_translation_get (plane, origin);

    entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_x");
    sprintf (string, "%.*f", gamgi->gamgi->length, origin[0]);
    gtk_entry_set_text (GTK_ENTRY (entry), string);

    entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_y");
    sprintf (string, "%.*f", gamgi->gamgi->length, origin[1]);
    gtk_entry_set_text (GTK_ENTRY (entry), string);

    entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_z");
    sprintf (string, "%.*f", gamgi->gamgi->length, origin[2]);
    gtk_entry_set_text (GTK_ENTRY (entry), string);

    gamgi_math_position_plane_rotation_get (plane, euler);

    entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_e1");
    sprintf (string, "%.*f", gamgi->gamgi->angle, euler[0]);
    gtk_entry_set_text (GTK_ENTRY (entry), string);
  
    entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_e2");
    sprintf (string, "%.*f", gamgi->gamgi->angle, euler[1]);
    gtk_entry_set_text (GTK_ENTRY (entry), string);

    entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_e3");
    sprintf (string, "%.*f", gamgi->gamgi->angle, euler[2]);
    gtk_entry_set_text (GTK_ENTRY (entry), string);
    }

  /**************************
   * set Plane color, scale *
   **************************/

  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_red");
  sprintf (string, "%.*f", GAMGI_MATH_DECIMAL_COLOR, plane->red);
  gtk_entry_set_text (GTK_ENTRY (entry), string);

  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_green");
  sprintf (string, "%.*f", GAMGI_MATH_DECIMAL_COLOR, plane->green);
  gtk_entry_set_text (GTK_ENTRY (entry), string);

  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_blue");
  sprintf (string, "%.*f", GAMGI_MATH_DECIMAL_COLOR, plane->blue);
  gtk_entry_set_text (GTK_ENTRY (entry), string);

  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_scale");
  if (gtk_widget_is_sensitive (entry) == TRUE)
    {
    sprintf (string, "%.*f", GAMGI_MATH_DECIMAL_SCALE, plane->object.scale);
    gtk_entry_set_text (GTK_ENTRY (entry), string);
    }
  }

else if (class == GAMGI_ENGINE_CELL)
  {
  entry = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "entry_cell");
  sprintf (string, "%s %d", object->name, object->number);
  gtk_entry_set_text (GTK_ENTRY (entry), string);
  }
else if (class == GAMGI_ENGINE_ATOM)
  {
  /***********************
   * get next atom entry *
   ***********************/

  entry = static_atom (dialog);
  if (entry == NULL)
    { static_end_atom (dialog); entry = static_atom (dialog); }
  sprintf (string, "%s %d", object->name, object->number);
  gtk_entry_set_text (GTK_ENTRY (entry), string);
  }
}

/*************** external function *************
 *                                             *
 *         GAMGI_GTK_PLANE_MODIFY_PRESS        *
 *                                             *
 *                                             *
 ***********************************************/

void gamgi_gtk_plane_modify_press (gamgi_window *window_mouse,
GdkEventButton *event, int x, int y, gamgi_window *window_dialog)
{
GtkWidget *dialog = window_dialog->dialog0;
GtkWidget *button;

/******************************
 * local mouse selection only *
 ******************************/

button = (GtkWidget *) g_object_get_data (G_OBJECT (dialog), "button_global");
if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)) == TRUE ||
window_dialog != window_mouse) return;

gamgi_mesa_select_object (window_mouse, x, y,
static_class (window_dialog), TRUE, static_press);
}

/*************** external function ******************
 *                                                  *
 *            GAMGI_GTK_PLANE_MODIFY                *
 *                                                  *
 * Creates the dialog window used to modify planes. *
 *                                                  *
 ****************************************************/

void gamgi_gtk_plane_modify (GtkWidget *widget, void *data)
{
gamgi_window *window = GAMGI_CAST_WINDOW data;
GtkWidget *dialog;
GtkWidget *notebook;
GtkWidget *table;
GtkWidget *hbox_center, *hbox_left, *hbox_left_left;
GtkWidget *vbox_dialog, *vbox_page, *vbox_center, *vbox_top, *vbox_top_top;
GtkWidget *entry;
GtkWidget *label;
GtkWidget *button;
GtkWidget *combo;
GtkListStore *store;
GtkCellRenderer *renderer;
GtkTreeIter iter;
gamgi_bool *sensitive;

/******************
 * dialog level 0 *
 ******************/

dialog = gamgi_gtk_dialog_task0_create ("Plane Modify", window);
window->action = GAMGI_GTK_PLANE_MODIFY;
gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);

/********************
 * global container *
 ********************/

vbox_dialog = gtk_vbox_new (FALSE, 5);
gtk_container_add (GTK_CONTAINER (dialog), vbox_dialog);
gtk_widget_show (vbox_dialog);

/****************
 * Cell objects *
 ****************/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_dialog), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 10);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

/*********
 * Local *
 *********/

hbox_left_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_left), hbox_left_left, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "hbox_plane", hbox_left_left);
gtk_widget_show (hbox_left_left);

label = gtk_label_new ("Plane");
gtk_box_pack_start (GTK_BOX (hbox_left_left), label, FALSE, FALSE, 0);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_box_pack_start (GTK_BOX (hbox_left_left), entry, FALSE, FALSE, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_20);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_20);
g_object_set_data (G_OBJECT (dialog), "entry_plane", entry);
gtk_widget_show (entry);

/**********
 * Global *
 **********/

button = gtk_toggle_button_new_with_label ("Global");
gtk_box_pack_start (GTK_BOX (hbox_left), button, FALSE, FALSE, 0);
g_signal_connect (button, "toggled",
G_CALLBACK (static_global), window);
g_object_set_data (G_OBJECT (dialog), "button_global", button);
gtk_widget_show (button);

/************
 * notebook *
 ************/

notebook = gtk_notebook_new ();
gtk_box_pack_start (GTK_BOX (vbox_dialog), notebook, FALSE, FALSE, 0);
gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_TOP);
g_object_set_data (G_OBJECT (dialog), "notebook", notebook);
gtk_widget_show (notebook);

/*************
 * Type page *
 *************/

vbox_page = gtk_vbox_new (FALSE, 5);
label = gtk_label_new (" Type ");
gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox_page, label);
gtk_container_set_border_width (GTK_CONTAINER (vbox_page), 10);
gtk_widget_show (vbox_page);

vbox_top = gtk_vbox_new (FALSE, 10);
gtk_box_pack_start (GTK_BOX (vbox_page), vbox_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top);

/********
 * Type *
 ********/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

label = gtk_label_new ("Reference");
gtk_box_pack_start (GTK_BOX (hbox_left), label, FALSE, FALSE, 0);
gtk_widget_show (label);

combo = gtk_combo_box_new ();
gtk_box_pack_start (GTK_BOX (hbox_left), combo, FALSE, FALSE, 0);
g_signal_connect (combo, "changed",
G_CALLBACK (static_reference), window);
g_object_set_data (G_OBJECT (dialog), "combo_reference", combo);
gtk_widget_show (combo);

store = gtk_list_store_new (1, G_TYPE_STRING);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Cell", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Atoms", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Local", -1);
gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
g_object_unref (store);

renderer = gtk_cell_renderer_text_new ();
gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), renderer, FALSE);
gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), renderer, "text", 0, NULL);

/********
 * Cell *
 ********/

vbox_top_top = gtk_vbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (vbox_top), vbox_top_top, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "vbox_cell", vbox_top_top);
gtk_widget_show (vbox_top_top);

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

table = gtk_table_new (1, 2, FALSE);
gtk_box_pack_start (GTK_BOX (hbox_center), table, FALSE, FALSE, 10);
gtk_widget_show (table);

label = gtk_label_new ("Cell");
gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_20);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_20);
g_signal_connect (entry, "changed",
G_CALLBACK (static_cell), window);
g_object_set_data (G_OBJECT (dialog), "entry_cell", entry);
gtk_widget_show (entry);

/*****************
 * Plane indices *
 *****************/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

label = gtk_label_new ("Indices");
gtk_box_pack_start (GTK_BOX (hbox_left), label, FALSE, FALSE, 0);
gtk_widget_show (label);

table = gtk_table_new (1, 6, FALSE);
gtk_box_pack_start (GTK_BOX (hbox_left), table, FALSE, FALSE, 0);
gtk_widget_show (table);

label = gtk_label_new ("H");
gtk_table_attach (GTK_TABLE (table), label, 2, 3, 0, 1, GTK_EXPAND, GTK_FILL, 5, 0);
gtk_widget_show (label);

label = gtk_label_new ("K");
gtk_table_attach (GTK_TABLE (table), label, 4, 5, 0, 1, GTK_EXPAND, GTK_FILL, 5, 0);
gtk_widget_show (label);

label = gtk_label_new ("L");
gtk_table_attach (GTK_TABLE (table), label, 6, 7, 0, 1, GTK_EXPAND, GTK_FILL, 5, 0);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 1, 2, 0, 1, GTK_EXPAND, GTK_FILL, 0, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
g_object_set_data (G_OBJECT (dialog), "entry_h", entry);
gtk_widget_show (entry);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 3, 4, 0, 1, GTK_EXPAND, GTK_FILL, 0, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
g_object_set_data (G_OBJECT (dialog), "entry_k", entry);
gtk_widget_show (entry);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 5, 6, 0, 1, GTK_EXPAND, GTK_FILL, 0, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
g_object_set_data (G_OBJECT (dialog), "entry_l", entry);
gtk_widget_show (entry);

/*********************
 * Vectors combo box *
 *********************/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

label = gtk_label_new ("Vectors");
gtk_box_pack_start (GTK_BOX (hbox_left), label, FALSE, FALSE, 0);
gtk_widget_show (label);

combo = gtk_combo_box_new ();
gtk_box_pack_start (GTK_BOX (hbox_left), combo, FALSE, FALSE, 0);
g_signal_connect (combo, "changed",
G_CALLBACK (static_vectors), window);
g_object_set_data (G_OBJECT (dialog), "combo_vectors", combo);
gtk_widget_show (combo);

store = gtk_list_store_new (1, G_TYPE_STRING);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Conventional", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Primitive", -1);
gtk_list_store_append (store, &iter); 
gtk_list_store_set (store, &iter, 0, "Local", -1);
gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
g_object_unref (store);

renderer = gtk_cell_renderer_text_new ();
gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), renderer, FALSE);
gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), renderer, "text", 0, NULL);

/*********************
 * Atom-based method *
 *********************/

vbox_top_top = gtk_vbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (vbox_top), vbox_top_top, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "vbox_atom", vbox_top_top);

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

table = gtk_table_new (3, 2, FALSE);
gtk_box_pack_start (GTK_BOX (hbox_center), table, FALSE, FALSE, 0);
gtk_container_set_border_width (GTK_CONTAINER (table), 5);
gtk_widget_show (table);

label = gtk_label_new ("Atom");
gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_widget_show (label);

label = gtk_label_new ("Atom");
gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 5, 0);
gtk_widget_show (label);

label = gtk_label_new ("Atom");
gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 5, 0);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_20);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_20);
g_object_set_data (G_OBJECT (dialog), "entry_atom1", entry);
gtk_widget_show (entry);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 0, 5);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_20);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_20);
g_object_set_data (G_OBJECT (dialog), "entry_atom2", entry);
gtk_widget_show (entry);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 1, 2, 2, 3, GTK_FILL, GTK_FILL, 0, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_20);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_20);
g_object_set_data (G_OBJECT (dialog), "entry_atom3", entry);
gtk_widget_show (entry);

/*******************
 * Projection page *
 *******************/

vbox_page = gtk_vbox_new (FALSE, 5);
label = gtk_label_new ("Projection");
gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox_page, label);
gtk_container_set_border_width (GTK_CONTAINER (vbox_page), 10);
g_object_set_data (G_OBJECT (dialog), "vbox_projection", vbox_page);
gtk_widget_show (vbox_page);

vbox_center = gtk_vbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_page), vbox_center, TRUE, TRUE, 0);
gtk_widget_show (vbox_center);

vbox_top = gtk_vbox_new (FALSE, 10);
gtk_box_pack_start (GTK_BOX (vbox_center), vbox_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top);

/**************
 * Projection *
 **************/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 40);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

/****************************
 * wulff, schmidt combo box *
 ****************************/

vbox_top_top = gtk_vbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_left), vbox_top_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top_top);

label = gtk_label_new ("Net");
gtk_box_pack_start (GTK_BOX (vbox_top_top), label, FALSE, FALSE, 0);
gtk_widget_show (label);

combo = gtk_combo_box_new ();
gtk_box_pack_start (GTK_BOX (vbox_top_top), combo, FALSE, FALSE, 0);
g_signal_connect (combo, "changed",
G_CALLBACK (static_autonomy), window);
g_object_set_data (G_OBJECT (dialog), "combo_projection", combo);
gtk_widget_show (combo);

store = gtk_list_store_new (1, G_TYPE_STRING);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Wulff", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Schmidt", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Local", -1);
gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
g_object_unref (store);

renderer = gtk_cell_renderer_text_new ();
gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), renderer, FALSE);
gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), renderer, "text", 0, NULL);

/*************************
 * pole, trace combo box *
 *************************/

vbox_top_top = gtk_vbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_left), vbox_top_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top_top);

label = gtk_label_new ("Model");
gtk_box_pack_start (GTK_BOX (vbox_top_top), label, FALSE, FALSE, 0);
gtk_widget_show (label);

combo = gtk_combo_box_new ();
gtk_box_pack_start (GTK_BOX (vbox_top_top), combo, FALSE, FALSE, 0);
g_signal_connect (combo, "changed",
G_CALLBACK (static_autonomy), window);
g_object_set_data (G_OBJECT (dialog), "combo_type", combo);
gtk_widget_show (combo);

store = gtk_list_store_new (1, G_TYPE_STRING);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Pole", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Trace", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Local", -1);
gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
g_object_unref (store);

renderer = gtk_cell_renderer_text_new ();
gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), renderer, FALSE);
gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), renderer, "text", 0, NULL);

/**************
 * Scope page *
 **************/
 
vbox_page = gtk_vbox_new (FALSE, 5);
label = gtk_label_new (" Scope ");
gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox_page, label);
gtk_container_set_border_width (GTK_CONTAINER (vbox_page), 10);
g_object_set_data (G_OBJECT (dialog), "vbox_scope", vbox_page);
gtk_widget_show (vbox_page);

vbox_center = gtk_vbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_page), vbox_center, TRUE, TRUE, 0);
gtk_widget_show (vbox_center);

vbox_top = gtk_vbox_new (FALSE, 10);
gtk_box_pack_start (GTK_BOX (vbox_center), vbox_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top);

/********
 * Link *
 ********/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

table = gtk_table_new (2, 2, FALSE);
gtk_box_pack_start (GTK_BOX (hbox_center), table, FALSE, FALSE, 0);
gtk_widget_show (table);

label = gtk_label_new ("Autonomy");
gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_widget_show (label);

combo = gtk_combo_box_new ();
gtk_table_attach (GTK_TABLE (table), combo, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 0, 5);
g_signal_connect (combo, "changed",
G_CALLBACK (static_autonomy), window);
g_object_set_data (G_OBJECT (dialog), "combo_autonomy", combo);
gtk_widget_show (combo);

store = gtk_list_store_new (1, G_TYPE_STRING);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "All", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Partial", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "None", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Local", -1);
gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
g_object_unref (store);

renderer = gtk_cell_renderer_text_new ();
gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), renderer, FALSE);
gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), renderer, "text", 0, NULL);

/*************
 * Thickness *
 *************/

label = gtk_label_new ("Thickness");
gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 5, 0);
g_object_set_data (G_OBJECT (dialog), "label_thickness", label);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_thickness", entry);
gtk_widget_show (entry);

/***************
 * Node, Order *
 ***************/

vbox_top_top = gtk_vbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (vbox_top), vbox_top_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top_top);

/*********
 * order *
 *********/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "hbox_order", hbox_left);
gtk_widget_show (hbox_left);

combo = gtk_combo_box_new ();
gtk_box_pack_start (GTK_BOX (hbox_left), combo, FALSE, FALSE, 0);
g_signal_connect (combo, "changed",
G_CALLBACK (static_instance), window);
g_object_set_data (G_OBJECT (dialog), "combo_instance", combo);
gtk_widget_show (combo);

store = gtk_list_store_new (1, G_TYPE_STRING);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Order", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Node", -1);
gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
g_object_unref (store);

renderer = gtk_cell_renderer_text_new ();
gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), renderer, FALSE);
gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), renderer, "text", 0, NULL);

entry = gtk_entry_new ();
gtk_box_pack_start (GTK_BOX (hbox_left), entry, FALSE, FALSE, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
g_object_set_data (G_OBJECT (dialog), "entry_order", entry);
gtk_widget_show (entry);

/********
 * node *
 ********/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top_top), hbox_center, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "hbox_node", hbox_center);
gtk_widget_show (hbox_center);

table = gtk_table_new (1, 8, FALSE);
gtk_box_pack_start (GTK_BOX (hbox_center), table, FALSE, FALSE, 0);
gtk_widget_show (table);

label = gtk_label_new ("O1");
gtk_table_attach (GTK_TABLE (table), label, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
g_object_set_data (G_OBJECT (dialog), "label_o1", label);
gtk_widget_show (label);

label = gtk_label_new ("O2");
gtk_table_attach (GTK_TABLE (table), label, 3, 4, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
g_object_set_data (G_OBJECT (dialog), "label_o2", label);
gtk_widget_show (label);

label = gtk_label_new ("O3");
gtk_table_attach (GTK_TABLE (table), label, 5, 6, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
g_object_set_data (G_OBJECT (dialog), "label_o3", label);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
g_object_set_data (G_OBJECT (dialog), "entry_o1", entry);
gtk_widget_show (entry);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 2, 3, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
g_object_set_data (G_OBJECT (dialog), "entry_o2", entry);
gtk_widget_show (entry);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 4, 5, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
g_object_set_data (G_OBJECT (dialog), "entry_o3", entry);
gtk_widget_show (entry);

combo = gtk_combo_box_new ();
gtk_table_attach (GTK_TABLE (table), combo, 6, 7, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
g_object_set_data (G_OBJECT (dialog), "combo_o4", combo);
gtk_widget_show (combo);

store = gtk_list_store_new (1, G_TYPE_STRING);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "000", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "011", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "101", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "110", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "111", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "122", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "211", -1);
gtk_list_store_append (store, &iter);
gtk_list_store_set (store, &iter, 0, "Local", -1);
gtk_combo_box_set_model (GTK_COMBO_BOX (combo), GTK_TREE_MODEL (store));
g_object_unref (store);

renderer = gtk_cell_renderer_text_new ();
gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo), renderer, FALSE);
gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (combo), renderer, "text", 0, NULL);

sensitive = gamgi_gtk_dialog_sensitive_create (8);
g_object_set_data (G_OBJECT (dialog), "sensitive_o4", sensitive);
gtk_cell_layout_set_cell_data_func (GTK_CELL_LAYOUT (combo),
renderer, gamgi_gtk_dialog_sensitive_scan, sensitive, free);

label = gtk_label_new ("O4");
gtk_table_attach (GTK_TABLE (table), label, 8, 9, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
g_object_set_data (G_OBJECT (dialog), "label_o4", label);
gtk_widget_show (label);

/*****************
 * Position page *
 *****************/

vbox_page = gtk_vbox_new (FALSE, 5);
label = gtk_label_new ("Position");
gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox_page, label);
gtk_container_set_border_width (GTK_CONTAINER (vbox_page), 10);
g_object_set_data (G_OBJECT (dialog), "vbox_position", vbox_page);
gtk_widget_show (vbox_page);

vbox_center = gtk_vbox_new (TRUE, 10);
gtk_box_pack_start (GTK_BOX (vbox_page), vbox_center, TRUE, TRUE, 0);
gtk_widget_show (vbox_center);

/**********
 * Origin *
 **********/

vbox_top = gtk_vbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (vbox_center), vbox_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top);

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

label = gtk_label_new ("Translation");
gtk_box_pack_start (GTK_BOX (hbox_center), label, FALSE, FALSE, 0);
gtk_widget_show (label);

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

table = gtk_table_new (1, 6, FALSE);
gtk_box_pack_start (GTK_BOX (hbox_center), table, FALSE, FALSE, 5);
gtk_widget_show (table);

label = gtk_label_new ("X");
gtk_table_attach (GTK_TABLE (table), label, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_widget_show (label);

label = gtk_label_new ("Y");
gtk_table_attach (GTK_TABLE (table), label, 3, 4, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_widget_show (label);

label = gtk_label_new ("Z");
gtk_table_attach (GTK_TABLE (table), label, 5, 6, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_x", entry);
gtk_widget_show (entry);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 2, 3, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_y", entry);
gtk_widget_show (entry);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 4, 5, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_z", entry);
gtk_widget_show (entry);

/*********
 * Angle *
 *********/

vbox_top = gtk_vbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (vbox_center), vbox_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top);

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

label = gtk_label_new ("Rotation");
gtk_box_pack_start (GTK_BOX (hbox_center), label, FALSE, FALSE, 0);
gtk_widget_show (label);

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

table = gtk_table_new (1, 6, FALSE);
gtk_box_pack_start (GTK_BOX (hbox_center), table, FALSE, FALSE, 5);
gtk_widget_show (table);

label = gtk_label_new ("E1");
gtk_table_attach (GTK_TABLE (table), label, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_widget_show (label);

label = gtk_label_new ("E2");
gtk_table_attach (GTK_TABLE (table), label, 3, 4, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_widget_show (label);

label = gtk_label_new ("E3");
gtk_table_attach (GTK_TABLE (table), label, 5, 6, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_e1", entry);
gtk_widget_show (entry);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 2, 3, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_e2", entry);
gtk_widget_show (entry);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 4, 5, 0, 1, GTK_FILL, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_7);
g_object_set_data (G_OBJECT (dialog), "entry_e3", entry);
gtk_widget_show (entry);

/*************
 * View page *
 *************/

vbox_page = gtk_vbox_new (FALSE, 5);
label = gtk_label_new (" View ");
gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox_page, label);
gtk_container_set_border_width (GTK_CONTAINER (vbox_page), 10);
gtk_widget_show (vbox_page);

vbox_center = gtk_vbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_page), vbox_center, TRUE, TRUE, 0);
gtk_widget_show (vbox_center);

vbox_top = gtk_vbox_new (FALSE, 20);
gtk_box_pack_start (GTK_BOX (vbox_center), vbox_top, FALSE, FALSE, 0);
gtk_widget_show (vbox_top);

/*********
 * Color *
 *********/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

table = gtk_table_new (1, 7, FALSE);
gtk_box_pack_start (GTK_BOX (hbox_center), table, FALSE, FALSE, 0);
gtk_widget_show (table);

label = gtk_label_new ("Color");
gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_SHRINK, GTK_FILL, 5, 0);
gtk_widget_show (label);

label = gtk_label_new ("R");
gtk_table_attach (GTK_TABLE (table), label, 2, 3, 0, 1, GTK_EXPAND, GTK_FILL, 5, 0);
gtk_widget_show (label);

label = gtk_label_new ("G");
gtk_table_attach (GTK_TABLE (table), label, 4, 5, 0, 1, GTK_EXPAND, GTK_FILL, 5, 0);
gtk_widget_show (label);

label = gtk_label_new ("B");
gtk_table_attach (GTK_TABLE (table), label, 6, 7, 0, 1, GTK_EXPAND, GTK_FILL, 5, 0);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 1, 2, 0, 1, GTK_EXPAND, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
g_object_set_data (G_OBJECT (dialog), "entry_red", entry);
gtk_widget_show (entry);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 3, 4, 0, 1, GTK_EXPAND, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
g_object_set_data (G_OBJECT (dialog), "entry_green", entry);
gtk_widget_show (entry);

entry = gtk_entry_new ();
gtk_table_attach (GTK_TABLE (table), entry, 5, 6, 0, 1, GTK_EXPAND, GTK_FILL, 5, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
g_object_set_data (G_OBJECT (dialog), "entry_blue", entry);
gtk_widget_show (entry);

/*********
 * Scale *
 *********/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_top), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

hbox_left = gtk_hbox_new (FALSE, 5);
gtk_box_pack_start (GTK_BOX (hbox_center), hbox_left, FALSE, FALSE, 0);
gtk_widget_show (hbox_left);

label = gtk_label_new ("Scale");
gtk_box_pack_start (GTK_BOX (hbox_left), label, FALSE, FALSE, 0);
g_object_set_data (G_OBJECT (dialog), "label_scale", label);
gtk_widget_show (label);

entry = gtk_entry_new ();
gtk_box_pack_start (GTK_BOX (hbox_left), entry, FALSE, FALSE, 0);
gtk_entry_set_width_chars (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
gtk_entry_set_max_length (GTK_ENTRY (entry), GAMGI_GTK_CHAR_5);
g_object_set_data (G_OBJECT (dialog), "entry_scale", entry);
gtk_widget_show (entry);

/*********************
 * Ok/Cancel buttons *
 *********************/

hbox_center = gtk_hbox_new (TRUE, 0);
gtk_box_pack_start (GTK_BOX (vbox_dialog), hbox_center, FALSE, FALSE, 0);
gtk_widget_show (hbox_center);

button = gamgi_gtk_dialog_button_create ("Ok", NULL);
gtk_widget_set_size_request (button, GAMGI_GTK_BUTTON_WIDTH, -1);
gtk_box_pack_start (GTK_BOX (hbox_center), button, FALSE, FALSE, 0);
g_signal_connect (button, "clicked",
G_CALLBACK (static_ok), window);
gtk_widget_show (button);

button = gamgi_gtk_dialog_button_create ("Cancel", "red");
gtk_widget_set_size_request (button, GAMGI_GTK_BUTTON_WIDTH, -1);
gtk_box_pack_start (GTK_BOX (hbox_center), button, FALSE, FALSE, 0);
gtk_widget_grab_focus (button);
g_signal_connect (button, "clicked",
G_CALLBACK (gamgi_gtk_dialog_task0_remove), window);
gtk_widget_show (button);

static_init (window);
gtk_widget_show (dialog);
}
