/***************************************************************************

  stream_direct.c

  The direct stream management routines

  (c) 2000-2004 Benot Minisini <gambas@users.sourceforge.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************/

#define __STREAM_IMPL_C

#include "gb_common.h"
#include "gb_error.h"
#include "gbx_value.h"
#include "gb_limit.h"

#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <errno.h>

#include "gbx_c_process.h"

#include "gbx_stream.h"


#define FDR (stream->process.fdr)
#define FDW (stream->process.fdw)


PRIVATE int stream_open(STREAM *stream, const char *path, int mode, CPROCESS *process)
{
  FDR = process->out;
  FDW = process->in;

  if (FDR >= 0)
    fcntl(FDR, F_SETFL, fcntl(FDR, F_GETFL) | O_NONBLOCK);

  return FALSE;
}


PRIVATE int stream_close(STREAM *stream)
{
  if (FDW >= 0)
  {
    close(FDW);
    FDW = -1;
    return FALSE;
  }
  else
    return TRUE;
}


PRIVATE int stream_read(STREAM *stream, char *buffer, long len)
{
  ssize_t eff_read;
  ssize_t len_read;

  if (FDR < 0)
    return TRUE;

  STREAM_eff_read = 0;

  if (stream->direct.has_car)
  {
    *buffer++ = stream->direct.car;
    len--;
    STREAM_eff_read++;
    stream->direct.has_car = FALSE;
  }

  while (len > 0)
  {
    len_read = Min(len, MAX_IO);
    eff_read = read(FDR, buffer, len_read);

    if (eff_read > 0)
    {
      len -= eff_read;
      buffer += eff_read;
      STREAM_eff_read += eff_read;
    }

    if (eff_read < len_read)
    {
      if (eff_read == 0)
        errno = 0;
      if (eff_read <= 0 && errno != EINTR)
        return TRUE;
    }
  }

  return FALSE;
}


PRIVATE int stream_write(STREAM *stream, char *buffer, long len)
{
  ssize_t eff_write;
  ssize_t len_write;

  if (FDW < 0)
    return TRUE;

  /*fprintf(stderr, "process.write: %.*s\n", len_write, buffer);*/

  while (len > 0)
  {
    len_write = Min(len, MAX_IO);
    eff_write = write(FDW, buffer, len_write);

    if (eff_write < len_write)
    {
      if (eff_write <= 0 && errno != EINTR)
        return TRUE;
    }

    len -= eff_write;
    buffer += eff_write;
  }

  return FALSE;
}


PRIVATE int stream_seek(STREAM *stream, long pos, int whence)
{
  if (FDR < 0)
    return TRUE;

  return (lseek(FDR, pos, whence) < 0);
}


PRIVATE int stream_tell(STREAM *stream, long *pos)
{
  if (FDR < 0)
    return TRUE;

  *pos = lseek(FDR, 0, SEEK_CUR);
  return (*pos < 0);
}


PRIVATE int stream_flush(STREAM *stream)
{
  return FALSE;
}


PRIVATE int stream_eof(STREAM *stream)
{
  char c;

  if (stream_read(stream, &c, 1))
    return TRUE;

  stream->direct.car = c;
  stream->direct.has_car = TRUE;
  return FALSE;

  /*
  pos = lseek(FD, 0, SEEK_CUR);
  end = lseek(FD, 0, SEEK_END);
  lseek(FD, pos, SEEK_SET);

  return (pos == end || pos < 0 || end < 0);
  */
}


PRIVATE int stream_lof(STREAM *stream, long *len)
{
  *len = stream->direct.size;
  return FALSE;
}


PRIVATE int stream_handle(STREAM *stream)
{
  return FDR;
}


DECLARE_STREAM(STREAM_process);

