/**********************************************************************
** $Id: qt/src/widgets/qeditor.h   2.3.1   edited 2001-01-26 $
**
** Definition of QEditor widget class
**
** Created : 961005
**
** Copyright (C) 1992-2000 Trolltech AS.  All rights reserved.
**
** This file is part of the widgets module of the Qt GUI Toolkit.
**
** This file may be distributed under the terms of the Q Public License
** as defined by Trolltech AS of Norway and appearing in the file
** LICENSE.QPL included in the packaging of this file.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** Licensees holding valid Qt Enterprise Edition or Qt Professional Edition
** licenses may use this file in accordance with the Qt Commercial License
** Agreement provided with the Software.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.trolltech.com/pricing.html or email sales@trolltech.com for
**   information about Qt Commercial License Agreements.
** See http://www.trolltech.com/qpl/ for QPL licensing information.
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

#ifndef __QEDITOR_H
#define __QEDITOR_H

#include <qgridview.h>
#include <qstring.h>
#include <qasciidict.h>

#include <qptrlist.h>

struct QMultiLineData;
class QEditorCommand;
class QValidator;

struct ColorData
{
  unsigned state : 4;
  unsigned len : 12;
};

typedef
  QMemArray<ColorData> ColorDataArray;

class QEditorRow
{
private:

  unsigned modify : 1;

public:

  unsigned newline : 1;
  unsigned line : 1;
  unsigned changed : 1;
  unsigned type : 4;

  QString s;
  int w;
  ColorDataArray data;

  enum {
    Background = 0,
    Normal = 1,
    Keyword = 2,
    Subr = 3,
    Operator = 4,
    Symbol = 5,
    Number = 6,
    String = 7,
    Commentary = 8,
    Breakpoint = 9,
    Current = 10,
    Datatype = 11,
    Selection = 12,
    Highlight = 13,
    Line = 14
    };

  enum {
    TypeNormal = 0,
    TypeCurrent = 1,
    TypeBreakpoint = 2
    };

  QEditorRow( QString string, int width, bool nl = true )
    :newline(nl), s(string), w(width)
  { modify = true; type = 0; line = false; changed = true; };

  void colorize(void);
  void drawBack(QPainter &p, int x, int y, int w, int h, QColor *color, bool current);
  void draw(QPainter &p, int x, int y, int w, int h, QColor color[], bool useRelief);
  void drawNormal(QPainter &p, int x, int y, int w, int h, QColor color[]);
  void change(void) { modify = true; changed = true; }

  static void analyze(QString &s, ColorDataArray &data);
  static bool isProc(QString &s);
};

#define NUM_COLORS 15

class QEditor : public QGridView
{
  Q_OBJECT

public:
  QEditor( QWidget *parent=0, const char *name=0 );
  ~QEditor();

  int topCell() const;
  void setTopCell( int row );
  void setXOffset( int );
  bool rowIsVisible( int row ) const;
  int lastRowVisible() const;
  bool rowYPos( int row, int *yPos ) const;
  void setYOffset( int );
  QRect	viewRect() const;
  int findRow(int) const;
  void ensureLineVisible( int row );

  void startUndo();
  void endUndo();

  static QAsciiDict<char> keywordDict;
  static QAsciiDict<char> subrDict;
  static QAsciiDict<char> typeDict;
  static QAsciiDict<char> firstDict;

  QColor getColor(int elt);
  void setColor(int elt, QColor col);

  void fromPos(long pos, int *row, int *col);
  long toPos(int row, int col);
  void getSelection(long *start, long *length);
  void setSelection(long start, long length);

  int lineLength( int row ) const;

  int lineType(int row) const;
  void setLineType(int row, int type);

  int findNextBreakpoint(int row);

  int posX(int, int) const;
  int posY(int, int) const;

  void stopBlink();
  void startBlink();

  void resetChangedLines(void);

    // from QMultiLineEdit

    QString textLine( int line ) const;
    void setTextLine ( int line, QString & s);
    int numLines() const;

    QSize sizeHint() const;
    QSize minimumSizeHint() const;
    QSizePolicy sizePolicy() const;

    virtual void setFont( const QFont &font );

    virtual void insertLine( const QString &s, int line = -1 );
    virtual void insertAt( const QString &s, int line, int col, bool mark = false );
    virtual void removeLine( int line );

    void cursorPosition( int *line, int *col ) const;
    virtual void setCursorPosition( int line, int col, bool mark = false );
    void getCursorPosition( int *line, int *col ) const;
    bool atBeginning() const;
    bool atEnd() const;

    virtual void setFixedVisibleLines( int lines );

    int maxLineWidth() const;

    //void setAlignment( int flags );
    //int alignment() const;

    virtual void setValidator( const QValidator * );
    const QValidator * validator() const;

    void setEdited( bool );
    bool edited() const;

    void cursorWordForward( bool mark );
    void cursorWordBackward( bool mark );

    //enum EchoMode { Normal, NoEcho, Password };
    //virtual void setEchoMode( EchoMode );
    //EchoMode echoMode() const;

    void setMaxLength(int);
    int maxLength() const;
    virtual void setMaxLineLength(int);
    int maxLineLength() const;
    virtual void setMaxLines(int);
    int maxLines() const;
    virtual void setHMargin(int);
    int hMargin() const;

    void setMarkedRegion( int row_from, int col_from, int row_to, int col_t );
    bool	hasMarkedText() const;
    QString	markedText() const;

    bool autoUpdate()	const;
    virtual void setAutoUpdate( bool );

    int tabSpace() const { return tabWidth; }
    void setTabSpace(int);

    void indent(bool);

    bool isShowProc() const { return showProc; }
    void setShowProc(bool);

    bool isUseRelief() const { return useRelief; }
    void setUseRelief(bool);

    bool isShowCurrent() const { return showCurrent; }
    void setShowCurrent(bool);

    bool isShowChange() const { return showChange; }
    void setShowChange(bool);

    bool isUseColor() const { return useColor; }
    void setUseColor(bool);

    void setUndoEnabled( bool );
    bool isUndoEnabled() const;
    void setUndoDepth( int );
    int undoDepth() const;

    bool isReadOnly() const;
    bool isOverwriteMode() const;

    QString text();

    int length() const;

    static void setDefaultTabStop( int ex );
    static int defaultTabStop();
public slots:
    virtual void       setText( const QString &);
    virtual void       setReadOnly( bool );
    virtual void       setOverwriteMode( bool );

    void       clear();
    void       append( const QString &);
    void       deselect();
    void       selectAll();
#ifndef QT_NO_CLIPBOARD
    void       paste(bool mouse = false);
    void       pasteSubType(const QCString& subtype, bool mouse = false);
    void       copyText() const;
    void       copy(bool mouse = false) const;
    void       cut();
#endif
    void       insert( const QString& );
    void       undo();
    void       redo();

signals:
    void	textChanged();
    void	returnPressed();
    void	undoAvailable( bool );
    void	redoAvailable( bool );
    void	copyAvailable( bool );
    void        cursorMoved();
    void        scrolled();

protected:
    void	paintCell( QPainter *, int row, int col );
    bool	event( QEvent * );

    void	mousePressEvent( QMouseEvent * );
    void	mouseMoveEvent( QMouseEvent * );
    void	mouseReleaseEvent( QMouseEvent * );
    void	mouseDoubleClickEvent( QMouseEvent * );
    //void 	wheelEvent( QWheelEvent * );
    void	keyPressEvent( QKeyEvent * );
    void	focusInEvent( QFocusEvent * );
    void	focusOutEvent( QFocusEvent * );
    //void	timerEvent( QTimerEvent * );
    void	leaveEvent( QEvent * );
    void	resizeEvent( QResizeEvent * );
		void imStartEvent( QIMEvent *e );
		void imComposeEvent( QIMEvent *e );
		void imEndEvent( QIMEvent *e );

    bool	focusNextPrevChild( bool );

#ifndef QT_NO_DRAGANDDROP
    void	dragMoveEvent( QDragMoveEvent* );
    void	dragEnterEvent( QDragEnterEvent * );
    void	dropEvent( QDropEvent* );
    void	dragLeaveEvent( QDragLeaveEvent* );
#endif

    int		textWidth( int );
    int		textWidth( const QString &);

    QPoint	cursorPoint() const;

    int getIndent(int, bool &);

protected:
    virtual void insert( const QString&, bool mark );
    virtual void newLine();
    virtual void tab(bool, bool = false);
    virtual void killLine();
    virtual void pageUp( bool mark=FALSE );
    virtual void pageDown( bool mark=FALSE );
    virtual void cursorLeft( bool mark=FALSE, bool wrap = TRUE );
    virtual void cursorRight( bool mark=FALSE, bool wrap = TRUE );
    virtual void cursorUp( bool mark=FALSE );
    virtual void cursorDown( bool mark=FALSE );
    virtual void backspace();
    virtual void del();
    virtual void home( bool mark=FALSE );
    virtual void end( bool mark=FALSE );

    bool getMarkedRegion( int *line1, int *col1, int *line2, int *col2 ) const;
    QString *getString( int row ) const;
    bool isEndOfParagraph( int row ) const;
    QString stringShown( int row ) const;

protected:
    bool	cursorOn;

private slots:
    void	clipboardChanged();
    void	blinkTimerTimeout();
    void	scrollTimerTimeout();
    void	dndTimeout();

private:
#ifndef QT_NO_MIME
    QCString pickSpecial(QMimeSource* ms, bool always_ask, const QPoint&);
#endif
#ifndef QT_NO_MIMECLIPBOARD
    void       pasteSpecial(const QPoint&);
#endif

    QPtrList<QEditorRow> *contents;
    QMultiLineData *d;

    static QColor default_color[NUM_COLORS];
    static bool init;
    QColor color[NUM_COLORS];

    static void initDict();

    void setY(int newY);
    void colorize(int line);

    int oldX, oldY;

    void emitCursorMoved();

   bool	readOnly;
    bool	dummy;
    bool	markIsOn;
    bool	dragScrolling ;
    bool	dragMarking;
    bool	textDirty;
    bool	wordMark;
    bool	overWrite;
    bool showProc;
    bool useRelief;
    bool showCurrent;
    bool showChange;
    bool useColor;

    int noEmitCursorMoved;

    int		cursorX;
    int		cursorY;

    int         brotherX;
    int         brotherX2;
    int         brotherY;

    int		markAnchorX;
    int		markAnchorY;
    int		markDragX;
    int		markDragY;
    int		curXPos;	// cell coord of cursor
    int		blinkTimer; // #### not used anymore - remove in 3.0
    int		scrollTimer; // #### not used anymore - remove in 3.0
    int         tabWidth;

    int		mapFromView( int xPos, int row );
    int		mapToView( int xIndex, int row ) const;

    void	pixelPosToCursorPos(QPoint p, int* x, int* y) const;
    void	setCursorPixelPosition(QPoint p, bool clear_mark=TRUE);

    //virtual int cellWidth(int) const;
    void	setWidth( int );
    void	updateCellWidth();
    bool 	partiallyInvisible( int row );
    void	makeVisible();
    void	setBottomCell( int row );

    void 	newMark( int posx, int posy, bool copy=TRUE );
    void 	markWord( int posx, int posy );
    void	extendSelectionWord( int &newX, int&newY);
    int 	charClass( QChar );
    void	turnMark( bool on );
    bool	inMark( int posx, int posy ) const;
    bool	beforeMark( int posx, int posy ) const;
    bool	afterMark( int posx, int posy ) const;
    int		setNumRowsAndTruncate();

#ifndef QT_NO_DRAGANDDROP
    void	doDrag();
#endif
    void	startAutoScroll();
    void	stopAutoScroll();

    void	cursorLeft( bool mark, bool clear_mark, bool wrap );
    void	cursorRight( bool mark, bool clear_mark, bool wrap );
    void	cursorUp( bool mark, bool clear_mark );
    void	cursorDown( bool mark, bool clear_mark );

    void	wrapLine( int line, int removed = 0);
    void	rebreakParagraph( int line, int removed = 0 );
    void	rebreakAll();
    void	insertAtAux( const QString &s, int line, int col, bool mark = FALSE );
    void	killLineAux();
    void	delAux();
    int	positionToOffsetInternal( int row, int col ) const;
    void	offsetToPositionInternal( int position, int *row, int *col ) const;
    void	deleteNextChar( int offset, int row, int col );

    void addUndoCmd( QEditorCommand* );
    void addRedoCmd( QEditorCommand* );
    void processCmd( QEditorCommand*, bool );

private:	// Disabled copy constructor and operator=
#if defined(Q_DISABLE_COPY)
    QEditor( const QEditor & );
    QEditor &operator=( const QEditor & );
#endif
};

inline bool QEditor::isReadOnly() const { return readOnly; }

inline bool QEditor::isOverwriteMode() const { return overWrite; }

inline void QEditor::setOverwriteMode( bool on )
{
    overWrite = on;
 }

inline int QEditor::lineLength( int row ) const
{
    return contents->at(row)->s.length();
}

inline int QEditor::lineType( int row ) const
{
    return contents->at(row)->type;
}

inline bool QEditor::atEnd() const
{
    return cursorY == (int)contents->count() - 1
	&& cursorX == lineLength( cursorY ) ;
}

inline bool QEditor::atBeginning() const
{
    return cursorY == 0 && cursorX == 0;
}

inline QString *QEditor::getString( int row ) const
{
    return &(contents->at( row )->s);
}

inline int QEditor::numLines() const
{
    return contents->count();
}



#endif
