/***************************************************************************

  project.c

  Project loader

  (c) 2000-2004 Benot Minisini <gambas@users.sourceforge.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************/

#define __GBX_PROJECT_C

#include "gb_common.h"
#include "gb_common_case.h"
#include "gb_alloc.h"
#include "gb_error.h"

#include <unistd.h>

#include "gb_limit.h"
#include "gb_buffer.h"
#include "gb_file.h"
#include "gbx_stream.h"
#include "gbx_library.h"
#include "gbx_archive.h"
#include "gbx_exec.h"
#include "gbx_stack.h"
#include "gb_component.h"

#include "gbx_project.h"

PUBLIC char *PROJECT_path;
PUBLIC char *PROJECT_exec_path;
PUBLIC char *PROJECT_lib_path;

PUBLIC char *PROJECT_name;
PUBLIC char *PROJECT_title;
PUBLIC char *PROJECT_startup;
PUBLIC char *PROJECT_version;

PUBLIC int PROJECT_argc;
PUBLIC char **PROJECT_argv;

PRIVATE char *project_buffer;

PRIVATE char *project_ptr;
PRIVATE int project_line;


PRIVATE void raise_error(const char *msg)
{
  static char line[6];

  sprintf(line, "%d", project_line);
  THROW(E_PROJECT, msg);
}


PRIVATE void project_name(char *name, int len)
{
  if (len == 0)
    raise_error("Bad project name");

  name[len] = 0;
  PROJECT_name = name;
}


PRIVATE void project_title(char *name, int len)
{
  name[len] = 0;
  PROJECT_title = name;
}


PRIVATE void project_version(char *name, int len)
{
  name[len] = 0;
  PROJECT_version = name;
}


PRIVATE void project_library(char *name, int len)
{
  const char *delim = ",";
  char *lib;

  name[len] = 0;

  lib = strtok(name, delim);
  while (lib != NULL)
  {
    LIBRARY_create(lib);
    lib = strtok(NULL, delim);
  }
}


#if 0
PRIVATE void project_archive(char *name, int len)
{
  name[len] = 0;
  ARCH_add(name);
}
#endif


PRIVATE void project_exec(char *name, int len)
{
  if (len == 0)
    raise_error("Bad project startup class");

  name[len] = 0;
  PROJECT_startup = name;
}


PRIVATE void project_stack(char *name, int len)
{
  int size;

  name[len] = 0;
  size = atoi(name);

  if (size >= 4 && size <= 256)
    STACK_size = size * 1024L;
}


PRIVATE void project_command(char *line, long len)
{
  static PROJECT_COMMAND command[] = {
    { "PROJECT", project_name },
    { "TITLE", project_title },
    { "LIBRARY", project_library },
    { "COMPONENT", project_library },
    { "STARTUP", project_exec },
    { "STACK", project_stack },
    { "VERSION", project_version },
    { "ARGUMENTS", NULL },
    { NULL }
    };

  PROJECT_COMMAND *pc;
  char cmd[32];
  int len_cmd;

  for (len_cmd = 0; len_cmd < len; len_cmd++)
  {
    if (line[len_cmd] == '=')
      break;
  }

  if (len_cmd >= len || len_cmd >= sizeof(cmd) || len_cmd == 0)
    raise_error("Syntax error");

  strncpy(cmd, line, len_cmd);

  for (pc = command; ; pc++)
  {
    if (pc->command == NULL)
      break;
      /*raise_error("Unknown command");*/

    if (strncasecmp(pc->command, cmd, len_cmd) == 0)
    {
      if (pc->func)
        (*pc->func)(&line[len_cmd + 1], len - len_cmd - 1);
      break;
    }
  }
}


PRIVATE void project_analyze(char *addr, long len)
{
  char *end = &addr[len];
  char c;
  char *start;


  project_ptr = addr;
  project_line = 1;
  start = project_ptr;

  for(;;)
  {
    if (project_ptr >= end)
      break;

    c = *project_ptr++;

    if (c == '\n')
    {
      project_line++;
      start = project_ptr;
      continue;
    }

    if (c == '#')
    {
      while ((project_ptr < end) && c != '\n')
        c = *project_ptr++;
      project_ptr--;
      continue;
    }

    if (c <= ' ')
      continue;

    project_ptr--;
    start = project_ptr;

    while ((project_ptr < end) && (c != '\n'))
      c = *project_ptr++;

    project_command(start, project_ptr - start - 1);
    project_ptr--;
  }

  if (!PROJECT_name || PROJECT_name[0] == 0)
    raise_error("No project name");

  if (!PROJECT_startup || PROJECT_startup[0] == 0)
    raise_error("No startup class");
    
  if (!PROJECT_title || PROJECT_title[0] == 0)
    PROJECT_title = PROJECT_name;
}


void PROJECT_init(const char *exec, const char *file)
{
  long len;
  const char *path;

  /* chemin d'installation de Gambas */

  path = FILE_find_gambas(exec);
  
  STRING_new(&PROJECT_exec_path, FILE_get_dir(FILE_get_dir(path)), -1);

  STRING_new(&PROJECT_lib_path, FILE_cat(PROJECT_exec_path, "lib/gambas", NULL), 0);

  /* fichier projet */

  if (EXEC_arch)
  {
    ARCH_init(file);

    path = FILE_get_dir(file);

    chdir(path);

    path = FILE_getcwd(NULL);
    if (path == NULL)
      goto _PANIC;
  }
  else
  {
    if (file == NULL)
      file = ".";

    if (*file == '/')
    {
      path = file;
    }
    else
    {
      if (*file == '.' && file[1] == '/')
        file += 2;

      path = FILE_getcwd(file);
      if (path == NULL)
        goto _PANIC;

      chdir(path);

      path = FILE_getcwd(NULL);
      if (path == NULL)
        goto _PANIC;
    }
  }

  len = strlen(path);

  while (len > 1)
  {
    if (path[len - 1] != '/')
      break;

    len--;
    /*path[len] = 0;*/
  }

  STRING_new(&PROJECT_path, path, len);

  chdir(PROJECT_path);

  if (EXEC_arch)
    file = ".project";
  else
    file = FILE_cat(PROJECT_path, ".project", NULL);

  STREAM_load(file, &project_buffer, &len);

  project_analyze(project_buffer, len);

  return;

_PANIC:

  ERROR_panic("Cannot initialize project: %s", strerror(errno));
}


void PROJECT_exit(void)
{
  if (project_buffer)
    FREE(&project_buffer, "PROJECT_exit");

  ARCH_exit();

  STRING_free(&PROJECT_path);
  STRING_free(&PROJECT_lib_path);
  STRING_free(&PROJECT_exec_path);
}
