namespace Galago
{
	using System;
	using System.Runtime.InteropServices;

	public class Service
		: Object
	{
		~Service()
		{
			Dispose();
		}

		public Service(IntPtr raw) : base(raw) {}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_service_new(string id, string name,
												bool native,
												ServiceFlags flags);

		public Service(string id, string name, bool native, ServiceFlags flags)
		{
			Raw = galago_service_new(id, name, native, flags);
		}

		public Service(string id, string name, bool native)
			: this(id, name, native, ServiceFlags.None)
		{
		}

		public Service(string id, string name, ServiceFlags flags)
			: this(id, name, true, flags)
		{
		}

		public Service(string id, string name)
			: this(id, name, true)
		{
		}

		public Service(string id)
			: this(id, null)
		{
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_service_get_id(IntPtr service);

		public string Id
		{
			get
			{
				IntPtr raw_ret = galago_service_get_id(Handle);
				return Marshal.PtrToStringAnsi(raw_ret);
			}
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_service_get_name(IntPtr service);

		public string Name
		{
			get
			{
				IntPtr raw_ret = galago_service_get_name(Handle);
				return Marshal.PtrToStringAnsi(raw_ret);
			}
		}

		[DllImport("libgalago.so")]
		static extern bool galago_service_is_native(IntPtr service);

		public bool Native
		{
			get { return galago_service_is_native(Handle); }
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_service_get_account(IntPtr service,
														string username,
														bool query);

		public Account GetAccount(string username, bool query)
		{
			IntPtr raw_ret = galago_service_get_account(Handle, username,
														query);
			return Object.Wrap(raw_ret) as Account;
		}

		public Account GetAccount(string username)
		{
			return GetAccount(username, true);
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_service_get_accounts(IntPtr service,
														 bool query);

		public List GetAccounts(bool query)
		{
			IntPtr raw_ret = galago_service_get_accounts(Handle, query);

			return new List(raw_ret);
		}

		public List GetAccounts()
		{
			return GetAccounts(true);
		}

		[DllImport("libgalago.so")]
		static extern void galago_service_add_account(IntPtr service,
													  IntPtr account);

		public void AddAccount(Account account)
		{
			galago_service_add_account(Handle, account.Handle);
		}

		[DllImport("libgalago.so")]
		static extern void galago_service_remove_account(IntPtr service,
														 IntPtr account);

		public void RemoveAccount(Account account)
		{
			galago_service_remove_account(Handle, account.Handle);
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_service_normalize(IntPtr service,
													  string username);

		public string Normalize(string username)
		{
			IntPtr raw_ret = galago_service_normalize(Handle, username);
			return Marshal.PtrToStringAnsi(raw_ret);
		}

		/*
		 * Signals
		 */
		[Galago.Signal("account-added")]
		public event AddedHandler OnAccountAdded
		{
			add
			{
				AddSignalHandler("account-added",
								 typeof(SignalMarshals.VoidPointerSignal),
								 typeof(AddedArgs),
								 value);
			}

			remove
			{
				RemoveSignalHandler("account-added", value);
			}
		}

		[Galago.Signal("account-removed")]
		public event RemovedHandler OnAccountRemoved
		{
			add
			{
				AddSignalHandler("account-removed",
								 typeof(SignalMarshals.VoidPointerSignal),
								 typeof(RemovedArgs),
								 value);
			}

			remove
			{
				RemoveSignalHandler("account-removed", value);
			}
		}

		[Galago.Signal("updated")]
		public event EventHandler OnUpdated
		{
			add
			{
				AddSignalHandler("updated",
								 typeof(SignalMarshals.VoidVoidSignal),
								 typeof(EventArgs),
								 value);
			}

			remove
			{
				RemoveSignalHandler("updated", value);
			}
		}
	}
}
