// ---------------------------------------------------------------------
// $Id: Input.cc,v 1.33 2006/08/11 17:25:52 daaugusto Exp $
//
//   Input.cc (created on Tue Aug 23 01:08:35 BRT 2005)
// 
//   Genetic Algorithm File Fitter (gaffitter)
//
//   Copyright (C) 2005-2006 Douglas A. Augusto
// 
// This file is part of gaffitter.
// 
// gaffitter is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2 of the License, or (at
// your option) any later version.
// 
// gaffitter is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with gaffitter; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// ---------------------------------------------------------------------

#include "Input.hh"

#include "DiskUsage.hh"
#include "util/Exception.hh"

//----------------------------------------------------------------------
Input::Input(Params& params): m_params(params) 
{
   DiskUsage::SetBlockSize(m_params.m_block_size);
}

//----------------------------------------------------------------------
void 
Input::Initialize() 
{ 
   if (m_params.m_verbose) cout << 
      "> Reading from input... " << flush << endl;

   // bypass file read?
   if (m_params.m_direct_input) { ReadInput(); } else { LoadFiles(); }

   // Check if remains files/items to process
   if (m_files.size() == 0) throw E_NoInputFiles();
}

//----------------------------------------------------------------------
void
Input::ReadInput()
{
   //--- read data from stdin (pipe)
   if (m_params.m_pipe) 
   {
      string s;
      getline(cin, s);

      if (cin) 
      {
         do 
         {
            StringToSizeID(s); //try to add the item in m_files
            getline(cin, s);

         }  while (cin);
      }
   }

   // --- read data from args too (if any)
   while (m_params.m_cmdline_items.size()>0)
   {
      StringToSizeID(m_params.m_cmdline_items.front());
      m_params.m_cmdline_items.pop_front(); // release memory
   }
}

//----------------------------------------------------------------------
void
Input::StringToSizeID(const string& s)
{
   // just a empty string, ignoring it
   if (s == "") return;

   // column separator (between 'size' and 'identifier'). At present
   // 'tab' and 'space' chars are allowed
   const char* delimiters = "\t ";

   // find the first digit of the 'size' column
   string::size_type size_begin = s.find_first_not_of(delimiters);
   if (size_begin == string::npos)
   {
      cerr << "> [Ignoring] Could not get size of: " << s << endl;

      return;
   }

   // find the first 'tab' or 'space' after the 'size' column
   string::size_type size_end = s.find_first_of(delimiters, size_begin);

   Params::Size_t size;

   // converts the 'size' column into 'size' variable
   if (!StringToDouble(size, string(s, size_begin, size_end)))
   {
      cerr << "> [Ignoring] Could not get size of: " << s << endl;

      return;
   }

   // find the first char of the 'ID' column
   string::size_type id_begin = s.find_first_not_of(delimiters, size_end);
   if (id_begin == string::npos)
   {
      cerr << "> [Ignoring] Could not get ID of: " << s << endl;

      return;
   }

   // ok!! :)
   string str_id(s, id_begin, string::npos);
         
   // size considering the allocation block
   Params::Size_t total_size = (m_params.m_no_metric) ? size : 
                   DiskUsage::AllocationSize(size*m_params.DI_Factor());

   if (CheckRange(str_id, total_size)) 
                        m_files.push_back(SizeName(str_id, total_size));
}

//----------------------------------------------------------------------
void
Input::LoadFiles()
{
   //--- read data from stdin (pipe)
   if (m_params.m_pipe) 
   {
      string s;
      getline(cin, s);

      if (cin) 
      {
         do 
         {
            AddFile(s); //try to add the given file in m_files
            getline(cin, s);

         }  while (cin);
      }
   }

   // --- read data from args too (if any)
   while (m_params.m_cmdline_items.size()>0)
   {
      AddFile(m_params.m_cmdline_items.front());

      m_params.m_cmdline_items.pop_front(); // release memory
   }
}

//---------------------------------------------------------------------
void
Input::AddFile(const string& s)
{
   Params::Size_t size = DiskUsage::GetSize(s.c_str());
   if (size == 0) { cerr << "> [Ignoring] file/dir: " << s << endl; return; }

   if (CheckRange(s,size)) { m_files.push_back(SizeName(s, size)); }
}

//---------------------------------------------------------------------
bool
Input::CheckRange(const string& s, Params::Size_t size) const
{
   if (size > m_params.m_target)
   {
      cerr << "> [Ignoring] bigger than 'target'(" 
           << m_params.PrettySize(m_params.m_target) << "): " << s
           << " (" << m_params.PrettySize(size) << ")" << endl;

      return false;
   }

   if (size > m_params.m_max_size)
   {
      cerr << "> [Ignoring] bigger than '--max-size'(" 
           << m_params.PrettySize(m_params.m_max_size) << "): " 
           << s << " (" << m_params.PrettySize(size) << ")" << endl;

      return false;
   }

   if (size <= 0.0)
   {
      cerr << "> [Ignoring] 0-sized input: " << s << endl;

      return false;
   } 

   if (size < m_params.m_min_size)
   {
      cerr << "> [Ignoring] smaller than '--min-size'(" 
           << m_params.PrettySize(m_params.m_min_size) << "): " << s
           << " (" << m_params.PrettySize(size) << ")" << endl;

      return false;
   } 

   return true;
}

//---------------------------------------------------------------------
