/* 

                          Firewall Builder

                 Copyright (C) 2000 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: PolicyListElement.cc,v 1.83 2003/03/22 09:03:43 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/



#include <iostream>

#include "config.h"

#include "fwbuilder/FWObjectDatabase.hh"
#include "fwbuilder/Host.hh"
#include "fwbuilder/Firewall.hh"
#include "fwbuilder/FWOptions.hh"
#include "fwbuilder/FWObjectReference.hh"
#include "fwbuilder/FWServiceReference.hh"
#include "fwbuilder/FWIntervalReference.hh"
#include "fwbuilder/Resources.hh"

#include "helpers.hh"
#include "platforms.hh"
#include "BuiltinDialog.hh"
#include "popupMenu.hh"

#include "Preferences.hh"
#include "CommentDialog.hh"
#include "RuleOptionsDialog.hh"

#include "FWObjectDatabaseGUI.hh"

#include "PolicyListElement.hh"
#include "PolicyListItem.hh"
#include "PolicyList.hh"

#include "main_window.hh"

#include "glademm_support.hh"

#include <iostream>

#include <assert.h>


using namespace libfwbuilder;

static string obj_n; // to store name of the object to be opened in editor


/*******************************************************************/


PolicyListElement::PolicyListElement(gint r,gint c) : Gtk::Frame()
{
    row=r;
    col=c;
    width=height=0;
    neg=false;

    set_flags( GTK_CAN_FOCUS );

    set_name("PolicyListElement");

    vbox=new Gtk::VBox();
    add(*vbox);
    vbox->show();

    popup_menu=NULL;

#ifdef __MINGW32__
/* see comment in PolicyListItem.cc about these flags */
    drag_dest_set ( GtkDestDefaults(GTK_DEST_DEFAULT_DROP | GTK_DEST_DEFAULT_MOTION),
		    target_table, n_targets , 
		    static_cast<GdkDragAction>(GDK_ACTION_COPY) );
#else
    drag_dest_set ( GTK_DEST_DEFAULT_ALL,
		    target_table, n_targets , 
		    static_cast<GdkDragAction>(GDK_ACTION_COPY) );
#endif

    rel=NULL;
}

PolicyListElement::PolicyListElement(gint r,gint c, RuleElement *re) : Gtk::Frame()
{
    FWObject          *o, *oref;

    rel=re;
    neg=rel->getNeg();

    row=r;
    col=c;
    width=height=0;

    set_flags( GTK_CAN_FOCUS );

    set_name("PolicyListElement");

    vbox=new Gtk::VBox();
    add(*vbox);
    vbox->show();

    popup_menu=NULL;

#ifdef __MINGW32__
/* see comment in PolicyListItem.cc about these flags */
    drag_dest_set ( GtkDestDefaults(GTK_DEST_DEFAULT_DROP | GTK_DEST_DEFAULT_MOTION),
		    target_table, n_targets , 
		    static_cast<GdkDragAction>(GDK_ACTION_COPY) );
#else
    drag_dest_set ( GTK_DEST_DEFAULT_ALL,
		    target_table, n_targets , 
		    static_cast<GdkDragAction>(GDK_ACTION_COPY) );
#endif

    change_any_to_orig=false;
    if (RuleElementTSrc::isA(re) ||
	RuleElementTDst::isA(re) ||
	RuleElementTSrv::isA(re) )  change_any_to_orig=true;
    

    for(list<FWObject*>::iterator  m = rel->begin(); m!= rel->end(); ++m) 
    {
	o=(*m);
	oref=NULL;
	if (FWObjectReference::isA(o) || FWServiceReference::isA(o) ||
	    FWIntervalReference::isA(o) )
	    oref=(FWReference::cast(o))->getPointer();

	if (oref!=NULL) {
	    addPolicyListItem( new PolicyListObjectItem(oref, 
                                                        rel->getNeg(), 
                                                        change_any_to_orig) );
	}

    }

    selected_item=get_first_child();
    show_all();
}

PolicyListElement::~PolicyListElement()
{
    while ( get_first_child()!=NULL )
        removePolicyListItem();

    remove();
    delete vbox;

    if (popup_menu!=NULL) delete popup_menu;
}

RuleSetList* PolicyListElement::getParentRuleSetList()
{
    Gtk::Widget *w=this;
    while (w!=NULL && dynamic_cast<RuleSetList*>(w)==NULL) 
        w=w->get_parent();

    return (RuleSetList*)w;
}

void PolicyListElement::addPolicyListItem(PolicyListItem *pi)
{
//    manage(pi);
    vbox->pack_start( *pi , true , true , 0 );
    pi->show();
//    Gdk_Window gdkw=pi->get_window();
//    cerr << "added pi " << pi << " Gdk_Window=" << gdkw << endl;
}

void PolicyListElement::removePolicyListItem(PolicyListItem *pi)
{
    if (pi==NULL)
    {
        Gtk::Widget *w=get_first_child();
        if (w) 
        { 
            w->hide();
            vbox->remove(*w); 
            delete w;
        }
    } else
    {
        pi->hide();
        vbox->remove(*pi);
        delete pi;
    }
}

Gtk::Widget* PolicyListElement::get_first_child()
{
    Gtk::Box_Helpers::Child    *ch;
    ch=*(vbox->children().begin());
    if (ch!=NULL) return ch->get_widget();
    else          return NULL;
}

Gtk::Widget* PolicyListElement::get_last_child()
{
    Gtk::Box_Helpers::Child    *ch;
    ch=*(vbox->children().rbegin());
    if (ch!=NULL) return ch->get_widget();
    else          return NULL;
}

void PolicyListElement::set_data_changed_flag(bool f)
{
    PolicyList *pl=(PolicyList*)get_user_data();
    pl->set_data_changed_flag( f );
}

/*
 *   We keep track of our physical size
 */
void PolicyListElement::size_allocate_impl(GtkAllocation* all)
{
    x=all->x;
    y=all->y;
    width=all->width;
    height=all->height;

    Gtk::Frame::size_allocate_impl(all);
}



void PolicyListElement::remove_item_from_policy(PolicyListItem* pi)
{
    assert (rel!=NULL);

    if (pi==NULL) return;
    if (rel->isAny()) return;

    FWObject *obj = pi->getObject();
    assert(obj!=NULL);

    removePolicyListItem(pi);
//    vbox->remove(*pi);   // do not need to delete because it is managed
    
    rel->removeRef(obj);  // adds "any" if obj was the last one

    if (rel->isAny()) {
	rel->setNeg(false);
	neg=false;

	obj=obj->getRoot()->getById( rel->getAnyElementId() , true );

	addPolicyListItem( new PolicyListObjectItem( obj, false,
                                                     change_any_to_orig) );
	select();
    }

    selected_item=get_first_child();
    set_data_changed_flag(true);
}

bool PolicyListElement::add_item_to_policy(FWObject *obj)
{
    if (rel==NULL) return(false);
    if (! rel->validateChild(obj) ) return(false);

    if (rel->getAnyElementId()==obj->getId()) return(false);

    string obj_id=obj->getId();
    bool present=false;
    list<FWObject*>::iterator j;
    for(j=rel->begin(); j!=rel->end(); ++j)     
    {
        FWObject *o=*j;
        string oid=o->getId();
        if(obj_id==oid) { present=true; break; }

        FWReference *ref;
        if( (ref=FWReference::cast(o))!=NULL &&
            obj_id==ref->getPointerId()) { present=true; break; }
    }
    if (present) return false;

    
    if (rel->isAny()) 
    {
	// element contains "Any". Remove corresponding PolicyListItem
	// from PolicyListElement before adding new item 
	//
        // there should be only one child widget, too
	//
	// this operation invalidates  selected_item
/*
	Gtk::Box_Helpers::BoxList::iterator bl_i;
	Gtk::Box_Helpers::Child    *ch;
	Gtk::Widget                *cc;
	bl_i=vbox->children().begin();
	if ( (ch=*bl_i)!=0 ) {
	    cc=ch->get_widget();
	    vbox->remove( *cc );
	}
*/
        removePolicyListItem();

	selected_item=NULL;

//	rel->clearChildren();
    }


    rel->addRef(obj);

    deselect();

    addPolicyListItem( new PolicyListObjectItem( obj, neg, change_any_to_orig) );
	
    selected_item=get_last_child();
	
    select();
	
    set_data_changed_flag(true);

    return(true);
}



void PolicyListElement::setNeg(bool val)
{
    neg=val;
    if (rel!=NULL) rel->setNeg(val);

    Gtk::Box_Helpers::BoxList::iterator bl_i;
    for (bl_i=vbox->children().begin(); bl_i!=vbox->children().end(); bl_i++) {
	((PolicyListItem*)(*bl_i)->get_widget())->setNeg( val );
    }

    set_data_changed_flag(true);
}

void PolicyListElement::activate()
{
    set_state(GTK_STATE_PRELIGHT);
    Gtk::Box_Helpers::BoxList::iterator bl_i;
    for (bl_i=vbox->children().begin(); bl_i!=vbox->children().end(); bl_i++) {
       ((PolicyListItem*)(*bl_i)->get_widget())->set_state(GTK_STATE_PRELIGHT);
    }
}

void PolicyListElement::deactivate()
{
    set_state(GTK_STATE_NORMAL);
    Gtk::Box_Helpers::BoxList::iterator bl_i;
    for (bl_i=vbox->children().begin(); bl_i!=vbox->children().end(); bl_i++) {
	((PolicyListItem*)(*bl_i)->get_widget()) ->set_state(GTK_STATE_NORMAL);
    }
}


void PolicyListElement::select(bool select_row)
{
    if (select_row) {
	RuleSetList* pp=getParentRuleSetList();
    
	pp->deselect_current();
	pp->activate_row(row);
	pp->set_current_selected(this);
    }

    set_state(GTK_STATE_PRELIGHT);
    request_focus();
}

void PolicyListElement::set_current_selected(Gtk::Widget *itm)
{
    selected_item=itm;
}

void PolicyListElement::select_first_child()
{
    selected_item=get_first_child();
    if (selected_item!=NULL)
	((PolicyListItem*)selected_item)->set_state(GTK_STATE_SELECTED);
}

void PolicyListElement::select_last_child()
{
    selected_item=get_last_child();
    if (selected_item!=NULL)
	((PolicyListItem*)selected_item)->set_state(GTK_STATE_SELECTED);
}

void PolicyListElement::deselect_all()
{
    Gtk::Box_Helpers::BoxList::iterator bl_i;
    Gtk::Box_Helpers::Child            *ch;
    for (bl_i=(vbox->children()).begin(); 
	 bl_i!=(vbox->children()).end(); bl_i++) {
	ch=(*bl_i);
	if (ch!=NULL) 
	    ((PolicyListItem*)ch->get_widget())->set_state(GTK_STATE_PRELIGHT);
    }
    selected_item=NULL;
}

void PolicyListElement::deselect()
{
    deselect_all();
    set_state(GTK_STATE_PRELIGHT);
}


void PolicyListElement::request_focus()
{
    if (!has_focus()) {
	grab_focus();
	draw_focus();
    }
}

void PolicyListElement::drag_data_received_impl(GdkDragContext   *context,
						gint              x,
						gint              y,
						GtkSelectionData *data,
						guint             info,
						guint32           time)
{
    void           *ptr;
    FWObject       *obj;
    PolicyListItem *itm=NULL;
    char           obj_id[64];

    Gdk_DragContext gdc ( context );
    Gtk::Widget::drag_finish ( gdc , false, false, time );

    if ( data->length >= 0 ) {
/*
 *   ObjectTree sends FWObject* via drag&drop mechanism, while
 *   PolicyListItem sends itself (PolicyListItem*, that is).  We will
 *   differentiate them using data->format here. This allows for
 *   different actions to be performed depending on where dragged
 *   object has been taken from
 *
 *   switch (data->format) {
 *
 *     case 6:    moving the whole rule, ptr is PolicyListItem*
 *     case 7:    moving object, ptr is PolicyListItem*
 *     case 8:    moving object, ptr is char* ( its Id )
 *    
 */
	obj=NULL;

	if (data->format==7) {
	    memcpy(&ptr,data->data,data->length);
	    itm=(PolicyListItem*)ptr;
	    obj=itm->getObject();
	}

	if (data->format==8 && rel!=NULL) {
	    memcpy(obj_id,(char*)data->data,data->length);
	    obj=rel->getRoot()->getById( obj_id , true );
	}

	if (obj==NULL) return;
	
	if ( add_item_to_policy(obj) ) {
/* object has successfully been added */
	    if (itm) {
		PolicyListElement* pe=itm->getParentPolicyListElement();
		if (pe)
		    pe->remove_item_from_policy(itm);
	    }
	}
    }
}

gint PolicyListElement::key_press_event_impl(GdkEventKey* ev)
{
    RuleSetList* pp=getParentRuleSetList();

    if ( selected_item==NULL ) goto another_element;

    switch (ev->keyval) {

    case GDK_space:
	show_popup_menu( (PolicyListItem*)selected_item );
	break;

    case GDK_Up:
    {
	Gtk::Box_Helpers::BoxList::reverse_iterator i;
	i=vbox->children().find( *selected_item );
	if (i!=vbox->children().rend()) {
//	    i++;
	    if (i!=vbox->children().rend())  {
		deselect_all();
		PolicyListItem *itm=(PolicyListItem*)((*i)->get_widget());
		itm->set_state(GTK_STATE_SELECTED);
		selected_item=itm;
	    }
	    else goto another_element;
	}
	else goto another_element;
	break;
    }
    case GDK_Down:
    {
	Gtk::Box_Helpers::BoxList::iterator i;
	i=vbox->children().find( *selected_item );
	if (i!=vbox->children().end()) {
	    i++;
	    if ( i!=vbox->children().end())  {
		deselect_all();
		PolicyListItem *itm=(PolicyListItem*)((*i)->get_widget());
		itm->set_state(GTK_STATE_SELECTED);
		selected_item=itm;
	    }
	    else goto another_element;
	}
	else goto another_element;
	break;
    }

    default:
	
 another_element:

	switch (ev->keyval ) {
	case GDK_Up:     pp->move_focus(GTK_DIR_UP);     break;
	case GDK_Down:   pp->move_focus(GTK_DIR_DOWN);   break;
	case GDK_Left:   pp->move_focus(GTK_DIR_LEFT);   break;
	case GDK_Right:  pp->move_focus(GTK_DIR_RIGHT);  break;
	default:         return(false);
	}
    }

    return(true);
}

void PolicyListElement::create_popup_menu()
{
    if (popup_menu==NULL)
    {
        const char *menu_items0[] = 
            { _("Edit"),
              _("Edit in another window"),
              "",
              _("Copy"),
              _("Cut"),
              _("Paste"),
              "",
              _("Negate"),
              NULL } ;

        popup_menu=new popupMenu(menu_items0);
        popup_menu->menu_select.connect(SigC::slot(this,&PolicyListElement::popup_menu_callback));
    }
}

void PolicyListElement::show_popup_menu(PolicyListItem *itm)
{
    popup_menu_pi=itm;

    create_popup_menu();

    RuleSetList        *pl=(RuleSetList*)get_user_data();
    assert(rel!=NULL);

    FWObject *f=rel;
    while (f!=NULL && !Firewall::isA(f)) f=f->getParent();
    assert(f!=NULL);

// f is a pointer at firewall object

    string cap_name;
    if (dynamic_cast<InterfacePolicyList*>(pl)!=NULL) cap_name="negation_in_interface_policy";
    if (dynamic_cast<PolicyList*>(pl)!=NULL)          cap_name="negation_in_policy";
    if (dynamic_cast<NATList*>(pl)!=NULL)             cap_name="negation_in_nat";

    bool supports_neg=false;
    try  {
        supports_neg=Resources::getTargetCapabilityBool(f->getStr("platform"),
                                                        cap_name);
    } catch (FWException &ex)
    {
        MessageDialog::Error(ex.toString());
    }

    popup_menu->set_sensitive(0,!rel->isAny());
    popup_menu->set_sensitive(1,!rel->isAny());
    popup_menu->set_sensitive(3,!rel->isAny());
    popup_menu->set_sensitive(4,!rel->isAny());

    popup_menu->set_sensitive(7,supports_neg && !rel->isAny());

    popup_menu->popup(0,0);
}

void PolicyListElement::popup_menu_callback(int menu_choice)
{
    assert(rel!=NULL);

    FWObject *obj=popup_menu_pi->getObject();

    /*
     * the problem here is that RuleElement object (pointer 'rel') may
     * change inside the call to safe_to_close_right_pane(). This
     * happens because it presents user with choice "discard
     * changes". This operation copies the top level Policy object to
     * restore data and thus recreates all its children, including this
     * Rule Element. We need to make sure we do not touch it after the call
     * to safe_to_close_right_pane()   
     *
     * TODO:
     *
     * FWObject::operaror= should really be rewritten so that it won't
     * destroy all its children before copying data, but instead copy
     * data for those that do exist and create new ones when
     * needed. This is going to be lot more complex algorithm though,
     * and most likely slow, too.
     *
     * --vk 02/09/02
     */
    bool is_any=rel->isAny();

    switch (menu_choice) {
    case 0:    // edit
/*
 *  do not edit "Any"
 */
        if (obj!=NULL && !is_any) 
        {
            main_window *mw=main_window::getMainWindowForWidget(this);
            if (mw->safe_to_close_right_pane() )
		mw->schedule_open_object(obj->getId(), obj->getLibrary() );
        }
	break;
    case 1:    // edit in another window
/*
 *  do not edit "Any"
 */
        if (obj!=NULL && !is_any) {
            main_window *mw=main_window::getAnotherMainWindowForWidget(this);
            if (mw->safe_to_close_right_pane() )
		mw->schedule_open_object(obj->getId(), obj->getLibrary() );
        }
	break;
    case 3:    // copy
	if ( obj!=NULL && ! is_any )
	    FWObjectClipboard::obj_clipboard->putObject( obj );

	break;
    case 4:    // cut
	if ( obj!=NULL && ! is_any ) {
	    FWObjectClipboard::obj_clipboard->putObject( obj );
	    remove_item_from_policy(popup_menu_pi);
	}
	break;
    case 5:    // paste
    {
        /*
         * clipboard holds a copy of the original object. Find an
         * original object, check if it still belongs to a tree and
         * add reference to it to the rule element. If user did "cut"
         * and object in the clipboard is not attached to the tree
         * anymore, then we can't paste it into the rule.
         */
	obj=FWObjectClipboard::obj_clipboard->getObject();
	if ( obj!=NULL ) {
	    FWObject *oo=FWObjectDatabaseGUI::db->getById( obj->getId() , true);
            if (oo!=NULL) add_item_to_policy(oo);
	    set_data_changed_flag(true);
	}
    }
    break;

    case 7:    // negate
	if ( ! is_any ) setNeg( ! getNeg() );
	break;
    }
}

void PolicyListElement::init() 
{
    build();
    selected_item=get_first_child();
//    drag_dest_unset();
}

void PolicyListElement::build() 
{
}

/*******************************************************************/

static int last_drag_motion_x=-1;


PolicyListRuleNum::PolicyListRuleNum(gint r,gint c,gint _num,bool _neg) :
    PolicyListElement(r,c)
{
    rule_n=_num;
    neg   =_neg;
    set_name("PolicyListRuleNum");
}

void PolicyListRuleNum::create_popup_menu()
{
    if (popup_menu==NULL)
    {
        const char *menu_items[] = 

            { _("Insert new rule above"),
              _("Add new rule below"),
              _("Remove this rule"),
              _("Move this rule up"),
              _("Move this rule down"),
              "",
              _("Copy"),
              _("Cut"),
              _("Paste above"),
              _("Paste below"),
              "",
              _("Enable"),
              _("Disable"),
              NULL } ;

        popup_menu=new popupMenu(menu_items);
        popup_menu->menu_select.connect(SigC::slot(this,&PolicyListRuleNum::popup_menu_callback));
    }
}

void PolicyListRuleNum::build() 
{
//    deselect();
    char  str[64];

    sprintf(str,"%02d",rule_n);
    addPolicyListItem(new PolicyListRuleNumItem( str , neg ));

    last_drag_motion_x=-1;
}

void PolicyListRuleNum::setNum(int rule_n)
{
    char  str[64];
    sprintf(str,"%02d",rule_n);

    Gtk::Widget *w=get_first_child();
    if (dynamic_cast<PolicyListRuleNumItem*>(w))
	dynamic_cast<PolicyListRuleNumItem*>(w) ->setNum(str);

}


gboolean PolicyListRuleNum::drag_motion_impl(GdkDragContext* context,
					     gint x,gint y,guint time)
{
    if (last_drag_motion_x == -1) last_drag_motion_x=x;
    
    if (last_drag_motion_x!=x) {

    }
    return false;
}

void PolicyListRuleNum::drag_data_received_impl(GdkDragContext   *context,
						gint              x,
						gint              y,
						GtkSelectionData *data,
						guint             info,
						guint32           time)
{
    PolicyListItem *itm=NULL;

    Gdk_DragContext gdc ( context );
    Gtk::Widget::drag_finish ( gdc , false, false, time );

    if ( data->length >= 0 ) {
/*
 *   ObjectTree sends FWObject* via drag&drop mechanism, while
 *   PolicyListItem sends itself (PolicyListItem*, that is).  We will
 *   differentiate them using data->format here. This allows for
 *   different actions to be performed depending on where dragged
 *   object has been taken from
 *
 *   switch (data->format) {
 *
 *     case 6:    moving the whole rule, ptr is PolicyListItem*
 *     case 7:    moving object, ptr is PolicyListItem*
 *     case 8:    moving object, ptr is FWObject*
 *    
 */
	void* ptr;
	memcpy(&ptr,data->data,data->length);

	if (data->format==6) {
	    itm=(PolicyListItem*)ptr;
	    PolicyListElement  *pe=itm->getParentPolicyListElement();
	    gint src_rule = PolicyList::get_rule_by_row( pe->get_row() );
	    gint dst_rule = PolicyList::get_rule_by_row( get_row() );

/* rule number src_rule moves right above rule dst_rule */
	    PolicyList         *pl=(PolicyList*)get_user_data();
	    pl->moveRuleTo(src_rule,dst_rule);
	    return;
	}
    }
}

      
void PolicyListRuleNum::show_popup_menu(PolicyListItem *itm)
{
    PolicyList             *pl=(PolicyList*)get_user_data();
    gint rule = PolicyList::get_rule_by_row( get_row() );

    create_popup_menu();

    popup_menu->set_sensitive(11, pl->isRuleDisabled( rule ));
    popup_menu->set_sensitive(12,!pl->isRuleDisabled( rule ));

    popup_menu->popup(0,0);
}

void PolicyListRuleNum::popup_menu_callback(int menu_choice)
{
    PolicyList             *pl=(PolicyList*)get_user_data();
    RuleSet                *rs=pl->getRuleSet();
    gint rule = PolicyList::get_rule_by_row( get_row() );

    deselect();

    switch (menu_choice) {
    case 0:
	pl->insertRuleBefore( rule );
	set_data_changed_flag(true);
	break;
    case 1:
	pl->appendRuleAfter( rule );
	set_data_changed_flag(true);
	break;
    case 2:
	pl->delRule( rule );
	pl->set_data_changed_flag(true);
	return;    // RETURN immediately since this rule element belongs to the rule which we delete
    case 3:
	pl->moveRuleUp( rule );
	set_data_changed_flag(true);
	break;
    case 4:
	pl->moveRuleDown( rule );
	set_data_changed_flag(true);
	break;

    case 6:  // copy
    {
	Rule *r=rs->getRuleByNum(rule);
	FWObjectClipboard::obj_clipboard->putObject( r );  // creates a copy
    }
    break;

    case 7:  // cut
    {
	Rule *r=rs->getRuleByNum(rule);
	FWObjectClipboard::obj_clipboard->putObject( r );  // creates a copy
	pl->delRule( rule );
	pl->set_data_changed_flag(true);
	return;    // RETURN immediately since this rule element belongs to the rule which we delete
    }
    break;

    case 8:  // paste above
    {
	FWObject *obj;
	if ( (obj=FWObjectClipboard::obj_clipboard->getObject())!=NULL &&
	     dynamic_cast<Rule*>(obj)!=NULL ) {
	    pl->insertRuleBefore( rule , dynamic_cast<Rule*>(obj) );
	    set_data_changed_flag(true);
	}
    }
    break;

    case 9:  // paste below
    {
	FWObject *obj;
	if ( (obj=FWObjectClipboard::obj_clipboard->getObject())!=NULL &&
	     dynamic_cast<Rule*>(obj)!=NULL ) {
	    pl->appendRuleAfter( rule , dynamic_cast<Rule*>(obj) );
	    set_data_changed_flag(true);
	}
    }
    break;

    case 11:
	pl->enableRule( rule );
	set_data_changed_flag(true);
        break;

    case 12:
        pl->disableRule( rule );
	set_data_changed_flag(true);
        break;

    }
    select();
}

/*******************************************************************/


PolicyListRuleAction::PolicyListRuleAction(gint r,gint c, Rule *ru)
  :    PolicyListElement(r,c)
{
    rule=ru;

}

void PolicyListRuleAction::create_popup_menu()
{
    FWObject     *f=((RuleSetList*)get_user_data())->getRuleSet();
    while (f!=NULL && !Firewall::isA(f)) f=f->getParent();
    assert(f!=NULL);

    bool supports_acct=false;
    try  {
        supports_acct=Resources::getTargetCapabilityBool(f->getStr("platform"),
                                                         "supports_accounting");
    } catch (FWException &ex)    {    }


    if (popup_menu==NULL)
    {
        const char *menu_items[] = 
            { _("Accept"),
              _("Deny"),
              _("Reject"),
              _("Accounting"),
              NULL } ;

        if (!supports_acct) menu_items[3]=NULL;

        popup_menu=new popupMenu(menu_items);
        popup_menu->menu_select.connect(SigC::slot(this,&PolicyListRuleAction::popup_menu_callback));
    }
}
 
void PolicyListRuleAction::show_popup_menu(PolicyListItem *itm)
{
    create_popup_menu();
    popup_menu->popup(0,0);
}

void PolicyListRuleAction::popup_menu_callback(int menu_choice)
{
    if (menu_choice>=0) {
        switch (menu_choice)
        {
        case 0: PolicyRule::cast(rule)->setAction(PolicyRule::Accept);     break;
        case 1: PolicyRule::cast(rule)->setAction(PolicyRule::Deny);       break;
        case 2: PolicyRule::cast(rule)->setAction(PolicyRule::Reject);     break;
        case 3: PolicyRule::cast(rule)->setAction(PolicyRule::Accounting); break;
        }
	build();

	set_data_changed_flag(true);
    }
}

void PolicyListRuleAction::build() 
{
    removePolicyListItem();

    string action= PolicyRule::cast(rule)->getActionAsString();

    string icn=	Resources::global_res->getIconPath(action);
    assert(icn!="");

    addPolicyListItem(new PolicyListItem( icn,"",action ));
}

/*******************************************************************/


PolicyListRuleLog::PolicyListRuleLog(gint r,gint c, Rule *ru) :
    PolicyListElement(r,c)
{
    rule=ru;

}

void PolicyListRuleLog::create_popup_menu()
{
    if (popup_menu==NULL)
    {
        const char *menu_items[] = 
            { _("Log"),
              _("Do not log"),
              NULL } ;

        popup_menu=new popupMenu(menu_items);
        popup_menu->menu_select.connect(SigC::slot(this,&PolicyListRuleLog::popup_menu_callback));
    }
}


void PolicyListRuleLog::build() 
{
    PolicyListItem *itm;

    removePolicyListItem();

    itm= new PolicyListItem( "","","" );

    RuleSetList  *pl=(RuleSetList*)get_user_data();

    if (pl->supportsLogging())
    {
        bool    log=PolicyRule::cast(rule)->getLogging();
        if (log) {
            string icn= Resources::global_res->getIconPath("Log");
            assert(icn!="");
            itm->addIcon(icn);
            itm->setQuickView(_("Logging enabled"));
        }
        else {
            itm->setQuickView(_("Logging disabled"));
        }
    }

    addPolicyListItem(itm);
}

void PolicyListRuleLog::show_popup_menu(PolicyListItem *itm)
{
    create_popup_menu();
    popup_menu->popup(0,0);
}

void PolicyListRuleLog::popup_menu_callback(int menu_choice)
{
    if (menu_choice>=0) {
	PolicyRule::cast(rule)->setLogging( (menu_choice==0) );
	build();
	set_data_changed_flag(true);
    }
}

/*******************************************************************/


PolicyListRuleComment::PolicyListRuleComment(gint r,gint c, Rule *ru) : 

    PolicyListElement(r,c)
{
    rule=ru;


    selected_item=get_first_child();

    drag_dest_unset();
}

void PolicyListRuleComment::create_popup_menu()
{
    if (popup_menu==NULL)
    {
        const char *menu_items[] = 
            { _("Edit Comment"),
              NULL } ;

        popup_menu=new popupMenu(menu_items);
        popup_menu->menu_select.connect(SigC::slot(this,&PolicyListRuleComment::popup_menu_callback));
    }
}

void PolicyListRuleComment::build() 
{
    removePolicyListItem();
    addPolicyListItem(new PolicyListCommentItem( rule->getComment() ));
}

void PolicyListRuleComment::show_popup_menu(PolicyListItem *itm)
{
    create_popup_menu();
    popup_menu->popup(0,0);
}

void PolicyListRuleComment::popup_menu_callback(int menu_choice)
{
    if (menu_choice==0) {
	CommentDialog *cmdlg=new CommentDialog( rule->getComment() );
	string str=cmdlg->run();
	delete cmdlg;
	rule->setComment(str);
	build();
	set_data_changed_flag(true);
    }
}

/*******************************************************************/

PolicyListRuleDir::PolicyListRuleDir(gint r,gint c, Rule *ru) : 
    PolicyListElement(r,c)
{
    rule=ru;


    selected_item=get_first_child();

    drag_dest_unset();
}

void PolicyListRuleDir::create_popup_menu()
{
    if (popup_menu==NULL)
    {
        const char *menu_items[] = 
            { _("Inbound"),
              _("Outbound"),
              _("Both"),
              NULL } ;

        popup_menu=new popupMenu(menu_items);
        popup_menu->menu_select.connect(SigC::slot(this,&PolicyListRuleDir::popup_menu_callback));
    }
}

void PolicyListRuleDir::build() 
{
    PolicyRule     *pr=PolicyRule::cast(rule);

    removePolicyListItem();

    if (pr->getDirection()==PolicyRule::Undefined) 
        pr->setDirection(PolicyRule::Both);

    string dir= pr->getDirectionAsString();
    string icn=	Resources::global_res->getIconPath(dir);

    addPolicyListItem(new PolicyListItem( icn,"",dir ));
}

void PolicyListRuleDir::show_popup_menu(PolicyListItem *itm)
{
    create_popup_menu();
    popup_menu->popup(0,0);
}

void PolicyListRuleDir::popup_menu_callback(int menu_choice)
{
/* direction makes sence only for PolicyRule */
    PolicyRule *prule=PolicyRule::cast(rule);
    assert(prule!=NULL);

    if (menu_choice>=0) {
        switch (menu_choice)
        {
        case 0:
            prule->setDirection( PolicyRule::Inbound );
            break;
        case 1:
            prule->setDirection( PolicyRule::Outbound );
            break;
        case 2:
            prule->setDirection( PolicyRule::Both );
            break;
        }

	build();
	set_data_changed_flag(true);
    }
}

/*******************************************************************/


PolicyListRuleOpt::PolicyListRuleOpt(gint r,gint c, Rule *ru) : 
    PolicyListElement(r,c)
{
    rule=ru;


    selected_item=get_first_child();

    drag_dest_unset();
}

void PolicyListRuleOpt::create_popup_menu()
{
    if (popup_menu==NULL)
    {
        const char *menu_items[] = 
            { _("Modify Options"),
              "",
              _("Turn logging ON"),
              _("Turn logging OFF"),
              NULL } ;

        popup_menu=new popupMenu(menu_items);
        popup_menu->menu_select.connect(SigC::slot(this,&PolicyListRuleOpt::popup_menu_callback));
    }
}

void PolicyListRuleOpt::build() 
{
    PolicyListItem *itm;

    removePolicyListItem();

    itm= new PolicyListItem( "","","" );

    string qv_text;
    string icn;

//    itm->clear();
    RuleSetList  *pl=(RuleSetList*)get_user_data();

    if (pl->supportsLogging())
    {
        bool    log=rule->getBool("log");
        if (log) {
            icn= Resources::global_res->getIconPath("Log");
            assert(icn!="");
            itm->addIcon(icn);

            qv_text+=_("Logging enabled");
        }
        else {
            qv_text+=_("Logging disabled");
        }
    }

    if (pl->supportsRuleOptions())
    {
        if (! isDefaultOptions( rule->getOptionsObject() )) {
            icn=Resources::global_res->getIconPath("Options");
            assert(icn!="");
            itm->addIcon(icn);
            qv_text+=_("\nNon-default options");
        }
    }
    itm->setQuickView( qv_text );

    addPolicyListItem(itm);
}

void PolicyListRuleOpt::show_popup_menu(PolicyListItem *itm)
{
    create_popup_menu();

    RuleSetList  *pl=(RuleSetList*)get_user_data();

    popup_menu->set_sensitive(0,pl->supportsRuleOptions());
    popup_menu->set_sensitive(2,pl->supportsLogging());
    popup_menu->set_sensitive(3,pl->supportsLogging());

    popup_menu->popup(0,0);
}

void PolicyListRuleOpt::popup_menu_callback(int menu_choice)
{
    if (menu_choice>=0) {
	switch (menu_choice) {
	case 0: 
	{
	    RuleOptionsDialog *dlg=new RuleOptionsDialog(rule->getOptionsObject());
	    dlg->run();
	    delete dlg;
	    build();
	    break;
	}
	case 2:
	case 3:
            PolicyRule *pr=PolicyRule::cast(rule);
            if (pr) pr->setLogging( (menu_choice==2) );
	    build();
	    break;
	}

	set_data_changed_flag(true);
    }
}




