/* 

                          Firewall Builder

                 Copyright (C) 2000 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: InterfaceDialog.cc,v 1.46 2003/12/05 16:19:44 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "fwbuilder/libfwbuilder-config.h"

#include <gtk--/main.h>

#include "fwbuilder/IPAddress.hh"
#include "fwbuilder/Group.hh"
#include "fwbuilder/Host.hh"
#include "fwbuilder/Firewall.hh"
#include "fwbuilder/Management.hh"

#include "FWObjectDatabaseGUI.hh"
#include "InterfaceDialog.hh"
#include "MessageDialog.hh"
#include "main_window.hh"

#include "helpers.hh"

#include <assert.h>
#include <iostream>

using namespace libfwbuilder;

#define GENERAL_PAGE         0
#define NET_ZONE_PAGE        1

InterfaceDialog::InterfaceDialog(FWObject *obj)
{
    object=obj; 

    object_parameters.push_back( DialogOption( iface_name  , "name"     ) );
    object_parameters.push_back( DialogOption( iface_label , "label"    ) );
    object_parameters.push_back( DialogOption( comment     , "comment"  ) );
    object_parameters.push_back( DialogOption( dyn_addr    , "dyn"      ) );
    object_parameters.push_back( DialogOption( unnumbered  , "unnum"    ) );
    object_parameters.push_back( DialogOption( mgmt        , "mgmt"     ) );

    FWObject *f=obj;
    while (f!=NULL && ! (Firewall::isA(f) || Host::isA(f) )) f=f->getParent();
    assert(f!=NULL);

    parent_is_host= (Host::isA(f) );

// f is a pointer at firewall object

    supports_security_levels=false;
    supports_network_zones=false;

    try  {
        supports_security_levels=Resources::getTargetCapabilityBool(f->getStr("platform"),
                                                        "security_levels");
        supports_network_zones=Resources::getTargetCapabilityBool(f->getStr("platform"),
                                                        "network_zones");
    } catch (FWException &ex)  { }

    object_parameters.push_back( DialogOption( security_level , "security_level" ) );

}

InterfaceDialog::~InterfaceDialog()
{
}

void InterfaceDialog::on_regular_toggled()
{
    on_changed();
}

/*
 *  so far "dynamic" is a property of an interface, not of its
 *  address. Therefore interface with dynamic address does not need
 *  IPv4 child element. Here we just mark an object as changed if user
 *  toggles this checkbox. This widget is part of object_parameters
 *  and its state is stored in object's attribute "dyn_addr".
 *
 *  May need to ask user whether we should delete any child Address element
 *  if dyn_addr was turned on.
 */
void InterfaceDialog::on_dyn_addr_toggled()
{
    on_changed();
}

void InterfaceDialog::on_unnum_toggled()
{
    on_changed();
}

void InterfaceDialog::on_extint_toggled()
{
    security_level->set_value( (extint->get_active())?0: 100 );
 
    on_changed();
}

void InterfaceDialog::on_changed()
{
    data_changed_flag(true);
}

void InterfaceDialog::wrk2dlg()
{
    loadPageOptions(object,  object_parameters );

    if (parent_is_host)
    {
        security_level_frame->hide();
        iface_dialog_notebook->get_nth_page(NET_ZONE_PAGE)->hide();
    } else
    {
        if (supports_security_levels)
        {
            security_level->show();
            extint->hide();
            sec_level_txt->set_text(
                _("Each interface of the firewall must have security level associated with it. \
Security level can be any number between 0 and 100, 0 being least secure \
and 100 being most secure levels. Interface with security level 0 ususally \
serves Internet connection.")
            );
        } else
        {
            security_level->hide();
            extint->show();

            extint->set_active(security_level->get_value_as_int()==0);

            sec_level_txt->set_text(
                _("One interface of the firewall must be marked as 'external'. This interface \
should be connected to the least secure network, usually the Internet")
            );
        }

        if (supports_network_zones)
        {
            iface_dialog_notebook->get_nth_page(NET_ZONE_PAGE)->show();

            net_zone_txt->set_text(
                _("Network zone consists of hosts and networks that can be reached through \
this interface of the firewall. Subnet to which this interface is directly \
attached must be part of its network zone. Other subnets reachable by \
means of routing should alse be added to the network zone.\n\n\
If network zone for this interface consists from only one subnet, you \
can simply choose that network's object in the pull-down \
below. If your network zone should include multiple subnets, \
you need to create an Object Group, then put all hosts and networks \
which are going to be part of the network zone into that group and \
finally choose this group in the pull-down below.\n")
            );

            net_zone_combo->get_entry()->set_editable(false);
            net_zone_combo->set_value_in_list(true,true);

            vector<string> obj_names;


            string lbl=string(" ")+_("Any")+string(" ");

            net_zone_objects[lbl]=FWObjectDatabaseGUI::getAnyNetworkId();
            obj_names.push_back( lbl );   // empty name

/* TODO: try to make this widget show object with appropriate icon */

            FWObject *groups=
                FWObjectDatabase::db->getById( FWObjectDatabaseGUI::getObjectGroupsId() , true );
            if (groups)
            {
                for (FWObject::iterator i=groups->begin(); i!=groups->end(); i++)
                {
                    string l1=_("Group: ")+ (*i)->getName();

                    net_zone_objects[l1]=(*i)->getId();
                    obj_names.push_back( l1 );
                }
            }

            FWObject *networks=
                FWObjectDatabase::db->getById( FWObjectDatabaseGUI::getNetworksId() , true );
            if (networks) {
                for (FWObject::iterator i=networks->begin(); i!=networks->end(); i++) {

                    string l1=_("Network: ")+ (*i)->getName();


                    net_zone_objects[l1]=(*i)->getId();
                    obj_names.push_back( l1 );
                }
            }
            net_zone_combo->set_popdown_strings( obj_names );

            string id=object->getStr("network_zone");

            if (id.empty()) id=FWObjectDatabase::getAnyNetworkId();

            FWObject *nz;
            nz=FWObjectDatabase::db->getById( id , true );

/* TODO: this code will change when we switch to a widget showing object name
 * with an icon */
            lbl= string( (Group::cast(nz)!=NULL)? _("Group: "):_("Network: ")) 
                + nz->getName();
            net_zone_combo->get_entry()->set_text( lbl );
        }
        else
            iface_dialog_notebook->get_nth_page(NET_ZONE_PAGE)->hide();
    }
    iface_name->grab_focus();
}

bool InterfaceDialog::dlg2wrk()
{
/*
 *  check if name is valid
 */
    savePageOptions(object,  object_parameters );

    if (supports_network_zones)
    {
        string nzname=net_zone_combo->get_entry()->get_text();
        if ( ! nzname.empty() && !net_zone_objects[nzname].empty() ) {
            object->setStr("network_zone",net_zone_objects[nzname]);
        }
    }

    return(true);
}


/**
 *  enable and disable menu items which make sence only for this object
 */
void InterfaceDialog::updateMainMenu()
{
    Gtk::MenuBar *main_menubar=main_w->getMainMenuBar();
    Gtk::Widget *itm;

    DialogPlugin::updateMainMenu();

    Interface *intf=Interface::cast(object);
    if ( intf && ! intf->isDyn() && ! intf->isUnnumbered() )
    {    
        itm=find_widget("insert_ipv4",main_menubar);
        if (itm) itm->set_sensitive(true);
    }

    if ( intf && intf->getPhysicalAddress()==NULL )
    {
        itm=find_widget("insert_physaddress",main_menubar);
        if (itm) itm->set_sensitive(true);
    }

    FWObject *p=object;
    while (p!=NULL && !Firewall::isA(p)) p=p->getParent();

    if (p!=NULL && Firewall::isA(p) )
    {
        itm=find_widget("policy",main_menubar);
        if (itm) itm->set_sensitive(true);

        itm=find_widget("rules_druid",main_menubar);
        if (itm) itm->set_sensitive(true);

        itm=find_widget("compile_policy",main_menubar);
        if (itm) itm->set_sensitive(true);

        Management *mgmt=(Firewall::cast(p))->getManagementObject();
        assert(mgmt!=NULL);
        PolicyInstallScript *pis   = mgmt->getPolicyInstallScript();

        itm=find_widget("install_policy",main_menubar);
        if (itm) itm->set_sensitive( pis->isEnabled() && pis->getCommand()!="" );
    }
}


