/* 

                          Firewall Builder

                 Copyright (C) 2000 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: FirewallDialog.cc,v 1.158 2003/09/29 05:17:09 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"

#include "fwbuilder/Firewall.hh"
#include "fwbuilder/FWOptions.hh"
#include "fwbuilder/Management.hh"
#include "fwbuilder/Policy.hh"
#include "fwbuilder/InterfacePolicy.hh"
#include "fwbuilder/NAT.hh"
#include "fwbuilder/FWObjectDatabase.hh"
#include "fwbuilder/snmp.hh"
#include "fwbuilder/dns.hh"

#include "FirewallDialog.hh"
#include "GenericBackgroundOpDialog.hh"
#include "MessageDialog.hh"
#include "PolicyDialog.hh"
#include "IPAddressWidget.hh"
#include "helpers.hh"
#include "FileSel.hh"
#include "Preferences.hh"
#include "main_window.hh"
#include "DialogFactory.hh"

#include <gtk--.h>
#include <iostream>
#include <assert.h>



#define GENERAL_NOTEBOOK_PAGE          0
#define SNMP_NOTEBOOK_PAGE             1
#define COMPILER_NOTEBOOK_PAGE         2

#define FIREWALL_NOTEBOOK_PAGE         3
#define OS_NOTEBOOK_PAGE               4

using namespace libfwbuilder;


int FirewallDialog::current_page=0;

FirewallDialog::FirewallDialog(FWObject *obj) 
{ 
    set_name("firewall_dialog");

    object=obj;

    options=NULL;

    targetFWDialog = NULL;
    targetOSDialog = NULL;



//    fw_addr=manage(new IPAddressWidget());
//    table16->attach(*fw_addr, 1, 2, 1, 2, GTK_FILL, 0, 0, 0);
//    fw_addr->changed.connect(SigC::slot(this,&FirewallDialog::on_changed));

    firewall_dialog_notebook->switch_page.connect(SigC::slot(this, 
					   &FirewallDialog::on_switch_page));

    firewall_dialog_notebook->set_page(current_page);

    host_os_menu=Resources::getOS();

    host_os->set_menu(host_os_menu);
    host_os->changed.connect(SigC::slot(this,&FirewallDialog::on_host_os_changed));
    host_os->show_all();

    platforms=Resources::getPlatforms();

    fw_platform->set_menu(platforms);
    fw_platform->changed.connect(SigC::slot(this,&FirewallDialog::on_fw_platform_changed));
    fw_platform->show_all();

    on_off_menu["1"]=_("On");
    on_off_menu["0"]=_("Off");
    on_off_menu[DialogPlugin::magic_empty_option]=_("No change");

/***********************************/

    object_parameters.push_back( DialogOption( fw_name          , "name"                 ) );
    object_parameters.push_back( DialogOption( comment          , "comment"              ) );
//    object_parameters.push_back( DialogOption( fw_addr          , "address"              ) );
    object_parameters.push_back( DialogOption( fw_platform      , "platform"             ) );
    object_parameters.push_back( DialogOption( host_os          , "host_OS"              ) );

//    common_options.push_back(    DialogOption( dyn_addr         , "dyn_addr"             ) );
    common_options.push_back(    DialogOption( snmp_description , "snmp_description"     ) );
    common_options.push_back(    DialogOption( snmp_location    , "snmp_location"        ) );
    common_options.push_back(    DialogOption( snmp_contact     , "snmp_contact"         ) );

    common_options.push_back(    DialogOption( cmdline          , "cmdline"              ) );
    common_options.push_back(    DialogOption( compiler         , "compiler"             ) );

    mgmt_txt1->set_line_wrap(true);
    mgmt_txt2->set_line_wrap(true);
    mgmt_txt3->set_line_wrap(true);


}

FirewallDialog::~FirewallDialog() 
{ 
}

void FirewallDialog::on_mgmt_use_install_script_clicked()
{
#ifndef __MINGW32__
    disableFWBDoptions();
    on_changed();
#endif
}

void FirewallDialog::on_mgmt_use_fwbd_clicked()
{
    enableFWBDoptions();
    on_changed();
}

void FirewallDialog::enableFWBDoptions()
{
#ifdef HAVE_LIBSSL
    mgmt_fwbd_port->set_sensitive(true);
    mgmt_keys->set_sensitive(true);
    mgmt_fw_key->set_sensitive(true);
    mgmt_clear_fw_key->set_sensitive(true);

    inst_script->set_sensitive(false);
    inst_cmdline->set_sensitive(false);
#endif
}

void FirewallDialog::disableFWBDoptions()
{
    mgmt_fwbd_port->set_sensitive(false);
    mgmt_keys->set_sensitive(false);
    mgmt_fw_key->set_sensitive(false);
    mgmt_clear_fw_key->set_sensitive(false);

    inst_script->set_sensitive(true);
    inst_cmdline->set_sensitive(true);
}

void FirewallDialog::clearTargetFWDialog()
{
    if (targetFWDialog!=NULL)
    {
        targetFWDialog->hide();
        firewall_tab->remove();
        delete targetFWDialog;
        targetFWDialog=NULL;
    }
}

void FirewallDialog::clearTargetOSDialog()
{
    if (targetOSDialog!=NULL)
    {
        targetOSDialog->hide();
        network_tab->remove();
        delete targetOSDialog;
        targetOSDialog=NULL;
    }
}

void FirewallDialog::showDefaultCompiler(const std::string &platform)
{
    char str[256];

    string compiler=Resources::platform_res[platform]->getCompiler();

    sprintf(str,_("Compiler (default: '%s')"),
            compiler.c_str());

    compiler_label->set_text(str);
//    compiler_label->set_line_wrap(true);
    compiler_label->set_justify(GTK_JUSTIFY_LEFT);
}

void FirewallDialog::showFirewallTab(const std::string &platform)
{
    targetFWDialog = DialogFactory::createDialog(object,platform);
    firewall_tab->add(*targetFWDialog);
    targetFWDialog->show();
}

void FirewallDialog::showOSTab(const std::string &os)
{
    targetOSDialog = DialogFactory::createDialog(object,os);
    network_tab->add(*targetOSDialog);
    targetOSDialog->show();
}

void FirewallDialog::showChosenFirewallTab()
{
    clearTargetFWDialog();
    showFirewallTab( fw_platform->get_value() );
}

void FirewallDialog::showChosenOSTab()
{
    clearTargetOSDialog();
    showOSTab( host_os->get_value() );
}

/**
 *   this callback is called when user changes host OS
 */
void FirewallDialog::on_host_os_changed()
{
    showChosenOSTab();

    string new_os=host_os->get_value();

    saveOptionMenu(object, host_os , "host_OS");

#if 0
    bool   err=false;
    string err_str;
    try {
        Resources::setDefaultTargetOptions(new_os,Firewall::cast(object));
    } catch (FWException &ex)
    {
        err=true;
        err_str=ex.toString();
    }
    if (err)  MessageDialog::Error(err_str,main_w);
#endif

    targetOSDialog->wrk2dlg();
    targetOSDialog->readDefaults();

    on_changed();
}

/**
 *   this callback is called when user changes firewall platform
 */
void FirewallDialog::on_fw_platform_changed()
{
    showChosenFirewallTab();

    string new_platform=fw_platform->get_value();

    saveOptionMenu(object, fw_platform , "platform");
#if 0
    bool err=false;
    string err_str;
    try {
        Resources::setDefaultTargetOptions(new_platform,Firewall::cast(object));  // set default parameters for chosen platform
    } catch (FWException &ex)
    {
        err=true;
        err_str=ex.toString();
    }
    if (err)  MessageDialog::Error(err_str,main_w);
#endif

    targetFWDialog->wrk2dlg();
    targetFWDialog->readDefaults();

    showDefaultCompiler( fw_platform->get_value() );

    on_changed();
}

void FirewallDialog::on_switch_page(Gtk::Notebook_Helpers::Page* p0,guint p1)
{
    // do nothing if dialog is not visible
    // it turns out, "switch_page" signal gets invoked 
    // when notebook pages are destroyed
    if(!is_visible()) 
        return;

    updateMainMenu();
    current_page=p1;
}

string  FirewallDialog::getCertDescString(int no, const string& d)
{
    char buf[32];
    sprintf(buf,"%d: ",no);
    return  string(buf)+ d;
}

void    FirewallDialog::fillListOfCertificates()
{
    Management *mgmt=(Firewall::cast(object))->getManagementObject();
    assert(mgmt!=NULL);

    map<string, pair<string, string> > certs= Preferences::global_prefs->getCerificates();
    map<string, pair<string, string> >::iterator i;
    vector<string>     cert_list;

    string fw_cert_id=mgmt->getFWBDManagement()->getIdentityId();
    string fw_cert_desc=certs[fw_cert_id].first;
    int n=0;
    for(i=certs.begin(); i!=certs.end(); i++,n++) {	
	if ( ! (*i).first.empty() )
	    cert_list.push_back( getCertDescString(n,(*i).second.first ) );
    }

    mgmt_keys->set_popdown_strings( cert_list );
    mgmt_keys->get_entry()->set_text( fw_cert_desc );
}

/*
 * an (unsuccessfull) attempt to add ID's of certificates as
 * user_data. This won't work because widget Gtk::Combo returns data
 * as a string in Gtk::Entry, not as an item in the list, so we can't
 * really pull user_data from there

    Gtk::List_Helpers::ItemList           ilist=mgmt_keys->get_list()->items();
    Gtk::List_Helpers::ItemList::iterator j;

    for(i=certs.begin(), j=ilist.begin(); 
	i!=certs.end() && j!=ilist.end(); 
	i++,j++) {
	(*j)->set_data_full("cert_id", new string( (*i).first ), &_cert_list_destroy_string );
	cerr << "*** " << (*i).first << "   " << (*i).second.first << endl;
    }

}

static void _cert_list_destroy_string(gpointer data)
{
    string *sp= (string*)data;
    delete sp;
}
*/


/*
 *  Regarding FirewallOptions *options
 *
 *  Unfortunately we have to find this objects every time we load data
 *  into dialog because it may change. In particular, it changes when
 *  user clicks "Undo". This happens because dialog actually works
 *  with a copy of original object in tm. When user clicks Undo this
 *  copy is overriden with data from original; when this happens, all
 *  child objects are recreated.  (vk)
 */

/**
 *  This method is called to load data from the object into dialog
 *  This happens at the very beginning and when user clicks "Undo"
 *  (vk)
 */
void FirewallDialog::wrk2dlg()
{
    options=(Firewall::cast(object))->getOptionsObject();
    assert(options!=NULL);

    Management *mgmt=(Firewall::cast(object))->getManagementObject();
    assert(mgmt!=NULL);

    fw_name->grab_focus();


#ifndef HAVE_LIBSNMP
    button16->set_sensitive(false);
#endif

    loadPageOptions(object,  object_parameters );
    loadPageOptions(options, common_options );

    showChosenFirewallTab();
    showChosenOSTab();

    showDefaultCompiler( fw_platform->get_value() );

    targetFWDialog->wrk2dlg();
    targetOSDialog->wrk2dlg();

/*********************  data for fwbd and install script **************/
    FWBDManagement      *fwbdm = mgmt->getFWBDManagement();
    PolicyInstallScript *pis   = mgmt->getPolicyInstallScript();
    SNMPManagement      *snmpm = mgmt->getSNMPManagement();
    
    snmp_r_community->set_text(snmpm->getReadCommunity());
    snmp_w_community->set_text(snmpm->getWriteCommunity());

    inst_script->set_text(  pis->getCommand() );
    inst_cmdline->set_text( pis->getArguments() );

#ifdef HAVE_LIBSSL

    int port=fwbdm->getPort();
    if (port==-1) 
	port= Resources::global_res->getResourceInt("/FWBuilderResources/FWBD/port");
    mgmt_fwbd_port->set_value( port );

    fillListOfCertificates();

    const Key *key=fwbdm->getPublicKey();
    if (key) mgmt_fw_key->set_text( key->getFingerprint() );

#ifdef __MINGW32__
    mgmt_use_fwbd->set_active(true);
#else
    if (pis->isEnabled()) mgmt_use_install_script->set_active(true);
    else                  mgmt_use_fwbd->set_active(true);
#endif

#else
    mgmt_use_fwbd->set_sensitive(false);
    mgmt_use_install_script->set_active(true);

    disableFWBDoptions();
/*
    mgmt_keys->hide();
    mgmt_fwbd_port->hide();
    mgmt_fw_key->hide();
    mgmt_clear_fw_key->hide();
    mgmt_txt1->hide();
    mgmt_txt2->hide();
    mgmt_txt3->hide();


    mgmt_txt1->set_text(_("Installation via fwbd daemon is disabled because program has been compiled without SSL support"));


    mgmt_txt1->show();
*/
#endif
}

/**
 *    This method is called to copy dialog data into the object (when
 *    user clicks "Apply" ) (vk)
 */
bool FirewallDialog::dlg2wrk()
{
    Management *mgmt=(Firewall::cast(object))->getManagementObject();
    assert(mgmt!=NULL);

    savePageOptions(object, object_parameters );

    savePageOptions(options, common_options );

    targetFWDialog->dlg2wrk();
    targetOSDialog->dlg2wrk();

/*********************  management data *******************8*/

    FWBDManagement      *fwbdm = mgmt->getFWBDManagement();
    PolicyInstallScript *pis   = mgmt->getPolicyInstallScript();
    SNMPManagement      *snmpm = mgmt->getSNMPManagement();

    mgmt->setAddress( (Firewall::cast(object))->getAddress() );

    snmpm->setReadCommunity( snmp_r_community->get_text() );
    snmpm->setWriteCommunity( snmp_w_community->get_text() );

    fwbdm->setPort( mgmt_fwbd_port->get_value_as_int() );

    map<string, pair<string, string> > certs= Preferences::global_prefs->getCerificates();
    map<string, pair<string, string> >::iterator i;

    string cert=mgmt_keys->get_entry()->get_text();
    int n=0;
    for(i=certs.begin(); i!=certs.end(); i++,n++) 
	if ( ! (*i).first.empty() && cert == getCertDescString(n , (*i).second.first ) ) {
	    fwbdm->setIdentityId( (*i).first );
	    break;
	}

    pis->setCommand( inst_script->get_text() );
    pis->setArguments( inst_cmdline->get_text() );

    pis->setEnabled( mgmt_use_install_script->get_active() );
    fwbdm->setEnabled( mgmt_use_fwbd->get_active() );

    return(true);
}

void FirewallDialog::on_mgmt_clear_fw_key_clicked()
{
    Management *mgmt=(Firewall::cast(object))->getManagementObject();
    assert(mgmt!=NULL);

    mgmt->getFWBDManagement()->removePublicKey();
    mgmt_fw_key->set_text("");
    on_changed();
}

void FirewallDialog::on_changed()
{
    data_changed_flag(true);
}

void FirewallDialog::on_snmp_get_descr_released()
{
#ifdef HAVE_LIBSNMP

    string addr;
    string rcomm=snmp_r_community->get_text();

    addr=Address::cast(object)->getAddress().toString();

    if (rcomm.empty())
    {
	MessageDialog::Error(_("missing SNMP community string"), main_w);
	return;
    }

    if(addr=="" || addr=="0.0.0.0") 
    {
        vector<IPAddress> addrs;
        try {
            addrs=DNS::getHostByName( fw_name->get_text() );
        } catch (FWException &e)
        {
            MessageDialog::Error( e.toString(), this );
            return;
        }
        addr=addrs.front().toString();
    }

    int t=Preferences::global_prefs->getOptInt("/FWBuilderPreferences/Network/SNMPTimeout", -1);

    SNMP_sysdesc_query *q=new SNMP_sysdesc_query();

    q->init(addr , rcomm,
	    Preferences::global_prefs->getOptInt("/FWBuilderPreferences/Network/SNMPRetries",  SNMP_DEFAULT_RETRIES),
	    t==-1?SNMP_DEFAULT_TIMEOUT:(1000000L*t)
    );


    GenericBackgroundOpDialog bdisp(q);
    
    try
    {
        bdisp.execute();
        snmp_description->set_point(0);
        snmp_description->forward_delete( snmp_description->get_length() );
        snmp_description->insert( XMLTools::cleanForNVTASCII( q->getDescr()) );
        snmp_location->set_text( XMLTools::cleanForNVTASCII( q->getLocation()) );
        snmp_contact->set_text( XMLTools::cleanForNVTASCII( q->getContact()) );

        data_changed_flag(true);
    } catch(const FWException &ex)
    {
        //do nothing
    }

    bdisp.disconnect();
    delete q;

#endif
}

void FirewallDialog::on_find_compiler_clicked()
{   

    FileSel *fs=new FileSel(_("Find compiler"),


                            Preferences::global_prefs->getWdir(),
                            "");
    string cmp=fs->run();
    delete fs;
    if(cmp!="")
	compiler->set_text( cmp );
}

void FirewallDialog::on_find_install_script_clicked()
{   

    FileSel *fs=new FileSel(_("Find install script"),
                            Preferences::global_prefs->getWdir(),
                            "");
    string ins=fs->run();
    delete fs;
    if(ins!="")
	inst_script->set_text( ins );
}

/**
 *  enable and disable menu items which make sence only for this object
 */
void FirewallDialog::updateMainMenu()
{
    Gtk::MenuBar *main_menubar=main_w->getMainMenuBar();
    Gtk::Widget *itm;

    DialogPlugin::updateMainMenu();

    itm=find_widget("policy",main_menubar);
    if (itm) itm->set_sensitive(true);

    itm=find_widget("rules_druid",main_menubar);
    if (itm) itm->set_sensitive(true);

    itm=find_widget("compile_policy",main_menubar);
    if (itm) itm->set_sensitive(true);

    Management *mgmt=(Firewall::cast(object))->getManagementObject();
    assert(mgmt!=NULL);
    PolicyInstallScript *pis   = mgmt->getPolicyInstallScript();

    itm=find_widget("install_policy",main_menubar);
    if (itm) itm->set_sensitive( pis->isEnabled() && pis->getCommand()!="" );

    itm=find_widget("insert_interface",main_menubar);
    if (itm) itm->set_sensitive(true);

#ifdef HAVE_LIBSNMP
    itm=find_widget("insert_interface_via_snmp",main_menubar);
    if (itm) itm->set_sensitive(true);
#endif
}



