/*
 *
 * Copyright (c) 2003 The Regents of the University of California.  All 
 * rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * - Neither the name of the University nor the names of its
 *   contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR  PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
 

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <ctype.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <signal.h>

#include "fusd.h"

#define MAX 1000

#define MULTIPLE 1000

int do_open(struct fusd_file_info *file)
{
  file->private_data = 0;
  return 0;
}

int do_close(struct fusd_file_info *file)
{
  return 0;
}


ssize_t do_read(struct fusd_file_info *file, char *buffer, size_t length,
	     loff_t *offset)
{
  char buf[100];
  sprintf(buf, "%s %d\n",
	  (char *) file->device_info,
	  (int) file->private_data++);
  if (length < strlen(buf)) buf[length-1] = '\0';
  strcpy(buffer, buf);
  return strlen(buffer)+1; /* include \0 in what's returned */
}


int main(int argc, char *argv[])
{
  int i, n;
  int retval;
  int errors;
  pid_t *pids;

  if (argc != 2 || (n = atoi(argv[1])) < 1 || n > MAX) {
    printf("usage: %s <num-of-devices>\n", argv[0]);
    exit(1);
  }

  pids = (pid_t *) malloc(sizeof(pid_t) * n);

  if (!fork()) {
    /* server */
    struct fusd_file_operations f = { open: do_close, close: do_close,
				      read: do_read };
    char *buf;
    char *devname;
    int i;
    
    for (i = 0; i < n; i++) {
      buf = malloc(100);
      devname = malloc(100);
      sprintf(buf, "/dev/fusd-test/state%03d", i);
      sprintf(devname, "state%03d", i);
      if (fusd_register(buf, "fusd-test", devname, 0666, buf+strlen("/dev/fusd-test/state"), &f) < 0)
	perror(buf);
    }
    printf("server starting\n");
    fusd_run();

  }

  /* clients */
  sleep(3);
  for (i = 0; i < n; i++) {
    retval = fork();

    if (retval > 0) {
      pids[i] = retval;
    } else if (retval == 0) {
      char buf[200];
      int last[MAX], fd[MAX];
      struct timeval tv;
      int devnum, ok = 0;
      
      sleep(1 + i/5);
      printf("pid %d: starting\n", getpid());
      fflush(stdout);
      /* randomize */
      gettimeofday(&tv, NULL);
      srandom(tv.tv_usec);
      
      for (i = 0; i < n; i++) {
	last[i] = 0;
	fd[i] = -1;
      }
      
      for (i = 0; i < n*MULTIPLE; i++) {
	int open_err = 0;
	devnum = rand() % n;
	while (fd[devnum] < 0) {
	  sprintf(buf, "/dev/fusd-test/state%03d", devnum);
	  /* printf("pid %d: trying to open %s\n", getpid(), buf); */
	  if ((fd[devnum] = open(buf, O_RDWR)) < 0) {
	    printf("pid %d: opening %s: %m, retrying...\n", getpid(), buf);
	    sleep(5);
	    open_err = 1;
	  } else {
	    if (open_err)
	      printf("pid %d: opening %s: success\n", getpid(), buf);
	    break;
	  }
	}
	if (fd[devnum] >= 0) {
	  int num, count, n;
	  if ((n = read(fd[devnum], buf, 100)) < 0)
	    printf("pid %d: reading from %s: %s\n", getpid(), buf,
		   strerror(errno));
	  else if (sscanf(buf, "%d %d", &num, &count) != 2)
	    printf("pid %d: uh oh!\n", getpid());
	  else if (num != devnum || count != last[devnum])
	    printf("pid %d: OUT OF ORDER %s", getpid(), buf);
	  else {
	    //	    printf("pid %d: OK %s", getpid(), buf);
	    last[devnum]++;
	    ok++;
	  }
	}
      }

      for (i = 0; i < n; i++) {
	if (fd[i] >= 0) {
	  if (close(fd[i]) < 0) {
	    fprintf(stderr, "pid %d: error closing: %m\n", getpid());
	  } else {
	    ok++;
	  }
	}
      }

      errors = (n * (MULTIPLE+1)) - ok;
      printf("pid %d: done, %d errors%s\n", getpid(), errors,
	     errors ? " (DANGER WILL ROBINSON!!!!)" : "");
      exit(errors ? 1 : 0);
    }
  }

  /* master waits for children after having spawned them */
  retval = 1;

  for (;;) {
    int pid, status;
    int i, j;

    for (i = 0; i < n; i++) {
      pid = wait(&status);

      for (j = 0; j < n; j++) {
	if (pids[j] == pid) {
	  pids[j] = -1;
	  fprintf(stderr, "master: pid %d finished\n", pid);
	  break;
	}
      }

      if (j >= n) {
	fprintf(stderr, "master: nonclient pid %d exited\n", pid);
	goto done;
      }

      if (WEXITSTATUS(status)) {
	fprintf(stderr, "master: uh oh, client %d reported retval %d\n",
		pid, WEXITSTATUS(status));
	goto done;
      }

    }

    fprintf(stderr, "master: all clients exited successfully\n");
    retval = 0;
    goto done;
  }

 done:
  signal(SIGINT, SIG_IGN);
  kill(0, SIGINT);
  exit(retval);
}
