/*
 *
 * Copyright (c) 2003 The Regents of the University of California.  All 
 * rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * - Neither the name of the University nor the names of its
 *   contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR  PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
 

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <errno.h>
#include <sys/types.h>
#include <fcntl.h>
#include <string.h>
#include <asm/poll.h>

#include "fusd.h"

/* This function makes a file descriptor non-blocking */
int nonblock(int fd)
{
  int val;

  if ((val = fcntl(fd, F_GETFL, 0)) < 0)
    return val;
  val |= O_NONBLOCK;
  if (fcntl(fd, F_SETFL, val) < 0)
    return val;

  return 0;
}



void run_client(char *file)
{
  FILE *f;
  int fd, i, retval;
  fd_set s;
  char buf[4096];

  if ((f = fopen(file, "r")) == NULL) {
    perror(file);
    exit(1);
  }

  fd = fileno(f);

  if (nonblock(fd) < 0) {
    perror("Oops: setting fd to nonblocking mode had a problem");
    exit(1);
  }

  for (;;) {
    FD_ZERO(&s);
    FD_SET(fd, &s);
    printf("Going to sleep using select...\n");
    retval = select(fd+1, &s, NULL, NULL, NULL);
    printf("Select has awakened!  Joy!\n");

    if (retval < 0) {
      perror("...but select had an error");
      exit(1);
    }

    if (retval == 0 || !FD_ISSET(fd, &s)) {
      printf("...but the descriptor is not readable.  Sigh.\n");
      exit(1);
    }

    retval = read(fd, buf, 4096);

    if (retval < 0) {
      perror("...but trying to read gave us an error");
      exit(1);
    }

    if (retval == 0) {
      printf("...but the retval of read was 0 for some odd reason!\n");
      exit(1);
    }

    printf("And we successfully got data!  %d bytes worth!  Here it is...\n",
           retval);
    buf[retval] = '\0';

    for (i = 0; i < retval; i++)
      if (!isprint(buf[i]) && !isspace(buf[i]))
          buf[i] = '.';

    printf("%s\n", buf);
  }
}

int zeroreturn(struct fusd_file_info *file) { return 0; }

char *bufs[100];
int num_bufs = 0;


ssize_t do_read(struct fusd_file_info *file, char *buffer, size_t len,
	     loff_t *offset)
{
  int i;

  if (num_bufs) {
    if (strlen(bufs[0]) < len) len = strlen(bufs[0]);
    strncpy(buffer, bufs[0], len);
    free(bufs[0]);
    for (i=1; i<num_bufs; i++) bufs[i-1] = bufs[i];
    num_bufs--;
    return len;
  } else {
    return -EWOULDBLOCK;
  }
}


struct fusd_file_info *outstanding_poll = NULL;

ssize_t do_write(struct fusd_file_info *file, const char *buffer,
                 size_t len, loff_t *offset)
{
  if (num_bufs >= 100)
    return -EWOULDBLOCK;

  bufs[num_bufs] = malloc(strlen(buffer) + 1);
  memcpy(bufs[num_bufs], buffer, len);
  bufs[num_bufs][len] = '\0';
  num_bufs++;

  if (outstanding_poll) {
    fusd_return(outstanding_poll, FUSD_NOTIFY_INPUT);
    outstanding_poll = NULL;
  }
  return len;
}

ssize_t do_poll_diff(struct fusd_file_info *file, unsigned int flags)
{
  int curr_flags = (num_bufs ? FUSD_NOTIFY_INPUT : 0);

  if (outstanding_poll != NULL) {
    fusd_return(outstanding_poll, curr_flags);
    outstanding_poll = NULL;
  }

  if (curr_flags != flags)
    return curr_flags;
  else {
    outstanding_poll = file;
    return -FUSD_NOREPLY;
  }
}

int main(int argc, char *argv[])
{
  if (argc != 5) {
    fprintf(stderr, "usage: %s <file-to-read-using-select> <dev-class> <dev-name> <client|server>\n",
	    argv[0]);
    exit(1);
  }

  if (!strcmp(argv[4], "server")) {
    /* server */
    struct fusd_file_operations f = { open: zeroreturn, close: zeroreturn,
				      read: do_read, write: do_write,
				      poll_diff: do_poll_diff};
    printf("server starting\n");
    if (fusd_register(argv[1], argv[2], argv[3], 0666, NULL, &f) < 0)
      perror(argv[1]);
    else
      fusd_run();
  } else if (!strcmp(argv[2], "client")) {
    /* client */
    sleep(1);
    printf("client starting\n");
    run_client(argv[1]);
  } else {
    fprintf(stderr, "2nd arg must be 'server' or 'client'\n");
    exit(0);
  }

  return 0;
}

