/*
 *
 * Copyright (c) 2003 The Regents of the University of California.  All 
 * rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * - Neither the name of the University nor the names of its
 *   contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR  PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
 

/*
 *
 * openrace.c: A test program as part of FUSD.
 *
 * This regression test program tries to exercise the FUSD race
 * condition between a device provider closing (and unregistering a
 * device), and clients trying to open that device.
 *
 * It should run for less than 30 seconds, then exit, saying ALL TESTS
 * PASSED.  If it freezes, or processes get stuck in the D state, or
 * you end up with zombie devices (in /dev/fusd/status), then there is
 * a problem.
 *
 * $Id: openrace2.c,v 1.3 2003/07/11 22:29:40 cerpa Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/wait.h>
#include <signal.h>
#include <string.h>

#include "fusd.h"

#define DEVNAME "/dev/testfusd/openrace"


#define FORK \
  tmp = fork(); \
  if (tmp < 0) { perror("can't fork"); exit(1); } \
  if (tmp == 0)

#define ITERATIONS 1000
#define MAX_BEFORE_EXIT ((n/2) * ITERATIONS)

int n;

void done(int signo)
{
  static int num_done = 0;
  int status;
  pid_t pid;

  while ((pid = waitpid(-1, &status, WNOHANG)) > 0) {
    printf("pid [%d] done\n", pid);
    num_done++;
  }

  /* if all N clients, plus the server, are done... we're done */
  if (num_done == n+1) {
    printf("test complete\n");
    exit(1);
  }
}

int _open(struct fusd_file_info *file)
{
  static int count = 0;

  /* exit abruptly in the middle of the request stream */
  if (count++ >= MAX_BEFORE_EXIT)
    exit(1);

  return -ENOMEM;
}

int _close(struct fusd_file_info *file)
{
  return 0;
}


int main(int argc, char *argv[])
{
  int tmp, i;

  if (argc != 2 || (n = atoi(argv[1])) <= 0) {
    fprintf(stderr, "usage: %s <number-of-clients> (use at least 10 or 20)\n", argv[0]);
    exit(0);
  }

  printf("master is pid %d\n", getpid());
  signal(SIGCHLD, done);

  /* fork server */
  FORK {
    fusd_file_operations_t
      fops = {
	open: _open, 
	close: _close
      };

    if (fusd_register(DEVNAME, 0666, 0, &fops) < 0) {
      perror("Register failed");
      exit(1);
    } 
    printf("server is pid %d, will respond to %d before exiting\n",
	   getpid(), MAX_BEFORE_EXIT);
    fusd_run();
  }

  sleep(1);

  for (i = 0; i < n; i++) {
    FORK {
      int j;
      int enomem=0, enoent=0, epipe=0, eother=0;

      printf("launched client %d, pid %d\n", i, getpid());

      for (j = 0; j < ITERATIONS; j++) {
	open(DEVNAME, O_RDWR);

	switch (errno) {
	case ENOMEM:
	  enomem++;
	  break;
	case ENOENT:
	  enoent++;
	  break;
	case EPIPE:
	  epipe++;
	  break;
	default:
	  eother++;
	  printf("[%d] got strange retval %d (%m)\n", getpid(), errno);
	  break;
	}
      }

      printf("[%d] done; %d nomem, %d noent, %d pipe, %d other\n", getpid(),
	     enomem, enoent, epipe, eother);
      exit(1);
    }
  }

  /* parent: wait for children to all exit */
  while (1)
    pause();
}



