
/*
    FUNIONFS: UNIONFS over FUSE Usermode filesystem
    Copyright (C) 2005-2006  Stephane APIOU <stephane.apiou@free.fr>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <fuse.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/mman.h>

#include "funionfs.h"

extern f_opt_t f_opt;

unsigned long maxpath;
struct unionfs_desc *unionfs_list;

int
issubpath(char *abspath, char *basepath)
{
	if (*basepath != '/')
		return 0;

	if (*abspath != '/')
		return 0;

	while (*++abspath == *++basepath)
	{
		/* abspath and base path are equal */
		if (*abspath == '\0')
			return 1;
	}
	if (*abspath == '\0')
		return 1;
	return 0;
}

char *
abs2relpath(char *abspath, char *basepath)
{
	char *result;

	if (*basepath != '/')
		return NULL;

	if (*abspath != '/')
		return NULL;

	while (*++abspath == *++basepath)
	{
		/* abspath and base path are equal */
		if (*abspath == '\0')
			return NULL;
	}
	/* return just after the last / */
	while (abspath[-1] != '/')
		abspath--;
	while (basepath[-1] != '/')
		basepath--;

	result = (char *) malloc(strlen(basepath) * 2 + strlen(abspath) + 1);
	if (result == NULL)
		return NULL;

	while (*basepath != '\0')
	{
		if (*basepath == '/')
			strcat(result, "../");
	}
	if (basepath[-1] != '/')
		strcat(result, "../");
	strcat(result, abspath);

	return result;
}

/* warning basepath must start with a slash */

char *
rel2abspath(char *relpath, char *basepath)
{
	char *copy_path;
	char *copy_bpath;
	char *orig_cpypath;
	char *temp_path;
	char link_path[MAX_LINKPATH];
	char *new_path;
	char *max_path;
	int readlinks = 0;
	int n;

	//DEBUG_PRINTF("rel2abspath( %s , %s );\n",relpath,basepath);

	/* test if basepath is an absolute path */

	if (*basepath != '/')
		return NULL;

	/* Make a copy of the source path since we may need to modify it. */
	copy_path = strdup(relpath);
	if (copy_path == NULL)
		return NULL;
	orig_cpypath = copy_path;

	/*
	 * Make a copy of basepath and allocate a buffer of sizeof(relpath)+sizeof(basepath) since
	 * the resulting path can't be larger
	 */
	copy_bpath = (char *) malloc(strlen(basepath) + strlen(relpath) + 3);
	if (copy_bpath == NULL)
	{
		free(orig_cpypath);
		return NULL;
	}
	strcpy(copy_bpath, basepath);
	new_path = copy_bpath;
	max_path = copy_bpath + strlen(basepath) + strlen(relpath);

	/* If it's a relative pathname use getwd for starters. */
	if (*copy_path != '/')
	{
		new_path += strlen(new_path);
		if (new_path[-1] != '/')
			*new_path++ = '/';
	}
	else
	{
		*new_path++ = '/';
		copy_path++;
	}
	/* Expand each slash-separated pathname component. */
	while (*copy_path != '\0')
	{
		/* Ignore stray "/". */
		if (*copy_path == '/')
		{
			copy_path++;
			continue;
		}
		if (*copy_path == '.')
		{
			/* Ignore ".". */
			if (copy_path[1] == '\0' || copy_path[1] == '/')
			{
				copy_path++;
				continue;
			}
			if (copy_path[1] == '.')
			{
				if (copy_path[2] == '\0' || copy_path[2] == '/')
				{
					copy_path += 2;
					/* Ignore ".." at root. */
					if (new_path == copy_bpath + 1)
						continue;
					/* Handle ".." by backing up. */
					while ((--new_path)[-1] != '/')
						;
					continue;
				}
			}
		}
		/* Safely copy the next pathname component. */
		while (*copy_path != '\0' && *copy_path != '/')
		{
			if (new_path > max_path)
			{
				free(copy_bpath);
				free(orig_cpypath);
				return NULL;
			}
			*new_path++ = *copy_path++;
		}

		/* Protect against infinite loops. */
		if (readlinks++ > MAX_READLINKS)
		{
			free(copy_bpath);
			free(orig_cpypath);
			return NULL;
		}
		/* See if latest pathname component is a symlink. */
		*new_path = '\0';
		n = readlink(copy_bpath, link_path, MAX_LINKPATH - 1);
		if (n < 0)
		{
			/* EINVAL means the file exists but isn't a symlink. */
			if (errno != EINVAL)
			{
				free(copy_bpath);
				free(orig_cpypath);
				return NULL;
			}
		}
		else
		{
			/* Note: readlink doesn't add the null byte. */
			link_path[n] = '\0';
			if (*link_path == '/')
			{
				/* Start over for an absolute symlink. */
				*copy_bpath = '\0';
			}
			else
			{
				/* Otherwise back up over this component. */
				while (*(--new_path) != '/')
					;
				new_path[1] = '\0';
			}
			temp_path =
				(char *) malloc(strlen(copy_bpath) +
						strlen(copy_path) +
						strlen(link_path) + 4);
			if (temp_path == NULL)
			{
				free(copy_bpath);
				free(orig_cpypath);
				return NULL;
			}
			strcpy(temp_path, copy_bpath);
			free(copy_bpath);
			copy_bpath = temp_path;
			new_path = copy_bpath;
			max_path =
				copy_bpath + strlen(copy_bpath) +
				strlen(copy_path) + strlen(link_path) + 3;

			/* Insert symlink contents into copy_path. */
			temp_path =
				(char *) malloc(strlen(copy_path) +
						strlen(link_path) + 1);
			if (temp_path == NULL)
			{
				free(copy_bpath);
				free(orig_cpypath);
				return NULL;
			}
			strcpy(temp_path, link_path);
			strcat(temp_path, copy_path);
			free(orig_cpypath);
			orig_cpypath = temp_path;
			copy_path = temp_path;
		}
		*new_path++ = '/';
	}
	/* Delete trailing slash but don't whomp a lone slash. */
	if (new_path != copy_bpath + 1 && new_path[-1] == '/')
		new_path--;
	/* Make sure it's null terminated. */
	*new_path = '\0';

	temp_path = strdup(copy_bpath);
	if (temp_path == NULL)
	{
		free(copy_bpath);
		free(orig_cpypath);
		return NULL;
	}
	free(copy_bpath);
	free(orig_cpypath);

	DEBUG_PRINTF("rel2abspath( %s , %s )= %s\n", relpath, basepath,
		     temp_path);

	return temp_path;
}

int
funionfs_addpath(char *path, int ro)
{
	struct unionfs_desc *pdesc, *plast;
	char *copy_path;
	int res;
	struct stat statfile;

	res = lstat(path, &statfile);
	if (res < 0)
		return -errno;
	if (strlen(path) > maxpath)
		maxpath = strlen(path);

	copy_path = strdup(path);
	if (copy_path == NULL)
		return (-ENOMEM);

	pdesc = (struct unionfs_desc *) malloc(sizeof(struct unionfs_desc));
	if (pdesc == NULL)
	{
		free(copy_path);
		return (-ENOMEM);
	}
	if (unionfs_list != NULL)
	{
		plast = unionfs_list;
		while (plast->pnext != NULL)
			plast = plast->pnext;
	}

	pdesc->path = copy_path;
	pdesc->ro = ro;
	pdesc->pnext = NULL;
	if (unionfs_list != NULL)
		plast->pnext = pdesc;
	else
		unionfs_list = pdesc;

	return 1;

}

/* get path for read and write 
 * return is:
 *  negative: error (-errno)
 *  bits are as follow up
 *  01234
 *  ^^^^^
 *  |||||
 *  ||||\_ file exist in read branch, deletion is done in write branch with a deleted file
 *  |||\__ file can be created in this path
 *  ||\___ file is delete in write branch
 *  |\____ file exist in write branch
 *  \_____ file exist in read branch
 */

int
funionfs_realpath(const char *path, char **get_path, char **put_path,
		  char **new_path, struct unionfs_desc *tab[3])
{
	int len;
	struct unionfs_desc *pdesc, *pnew;
	int res;
	char *cur_path, *pend;
	struct stat statfile;
	int deleted, to_delete;

	len = strlen(path) + maxpath + f_opt.del_len + 1;
	cur_path = (char *) malloc(len);
	if (cur_path == NULL)
		return -ENOMEM;

	*get_path = NULL;
	*put_path = NULL;
	*new_path = NULL;
	deleted = 0;
	to_delete = 0;
	pnew = NULL;

	for (pdesc = unionfs_list; pdesc != NULL; pdesc = pdesc->pnext)
	{
		strncpy(cur_path, pdesc->path, len);
		pend = cur_path + strlen(cur_path);
		while (*--pend == '/') ;
		pend[1] = '\0';
		strncat(cur_path, path, len);
		/* delete trailing slashs */
		pend = cur_path + strlen(cur_path);
		while (*--pend == '/') ;
		pend[1] = '\0';

		res = 0;
		DEBUG_PRINTF("%s\n", cur_path);
		if (lstat(cur_path, &statfile) >= 0)
			res = 1;

		strcat(cur_path, f_opt.del_string);
		DEBUG_PRINTF("%s\n", cur_path);
		if (lstat(cur_path, &statfile) >= 0)
			res |= 2;
		pend[1] = '\0';

		switch (res)
		{
		case 3:
			deleted = 1;
			DEBUG_PRINTF("DELETED BUT EXISTS\n");
			if (*get_path != NULL)
				free(*get_path);
			*get_path = NULL;
			if (*put_path != NULL)
				free(*put_path);
			*put_path = NULL;
			break;

		case 2:
			deleted = 1;
			DEBUG_PRINTF("DELETED\n");
			if (*get_path != NULL)
				free(*get_path);
			*get_path = NULL;
			if (*put_path != NULL)
				free(*put_path);
			*put_path = NULL;
			break;
		case 1:
			if (*get_path != NULL)
			{
				DEBUG_PRINTF("TO DELETE\n");
				to_delete = 1;
			}
			deleted = 0;
			if (pdesc->ro)
			{
				if (*get_path != NULL)
					free(*get_path);
				*get_path = cur_path;
				tab[IDX_GET] = pdesc;
			}
			else
			{
				if (*get_path != NULL)
					free(*get_path);
				*get_path = cur_path;
				tab[IDX_GET] = pdesc;
				cur_path = (char *) malloc(len);
				if (cur_path == NULL)
				{
					free(*get_path);
					return -ENOMEM;
				}
				strcpy(cur_path, *get_path);
				*put_path = cur_path;
				tab[IDX_PUT] = pdesc;
			}
			cur_path = (char *) malloc(len);
			if (cur_path == NULL)
			{
				if (*get_path != NULL)
					free(*get_path);
				if (*put_path != NULL)
					free(*put_path);
				return -ENOMEM;
			}
			break;
		}
		if (!pdesc->ro)
			pnew = pdesc;
	}

	if (pnew != NULL)
	{
		*new_path = cur_path;
		strncpy(cur_path, pnew->path, len);
		pend = cur_path + strlen(cur_path);
		while (*--pend == '/') ;
		pend[1] = '\0';
		strncat(cur_path, path, len);
		/* delete trailing slashs */
		pend = cur_path + strlen(cur_path);
		while (*--pend == '/') ;
		pend[1] = '\0';
		tab[IDX_NEW] = pnew;
	}
	else
		free(cur_path);

	res = (*put_path != NULL) << 1 | (*get_path !=
					  NULL) | deleted << 2 | (*new_path !=
								  NULL) << 3 |
		to_delete << 4;

	if (*put_path == NULL)
	{
		*put_path = (char *) malloc(1);
		**put_path = '\0';
	}
	if (*get_path == NULL)
	{
		*get_path = (char *) malloc(1);
		**get_path = '\0';
	}
	if (*new_path == NULL)
	{
		*new_path = (char *) malloc(1);
		**new_path = '\0';
	}
	return res;
}
int
old_funionfs_realpath(const char *path, char **get_path, char **put_path,
		  char **new_path, struct unionfs_desc *tab[3])
{
	int len;
	struct unionfs_desc *pdesc, *pnew;
	int res;
	char *cur_path, *pend;
	struct stat statfile;
	int deleted, to_delete;

	len = strlen(path) + maxpath + f_opt.del_len + 1;
	cur_path = (char *) malloc(len);
	if (cur_path == NULL)
		return -ENOMEM;

	*get_path = NULL;
	*put_path = NULL;
	*new_path = NULL;
	deleted = 0;
	to_delete = 0;
	pnew = NULL;

	for (pdesc = unionfs_list; pdesc != NULL; pdesc = pdesc->pnext)
	{
		strncpy(cur_path, pdesc->path, len);
		pend = cur_path + strlen(cur_path);
		while (*--pend == '/') ;
		pend[1] = '\0';
		strncat(cur_path, path, len);
		/* delete trailing slashs */
		pend = cur_path + strlen(cur_path);
		while (*--pend == '/') ;
		pend[1] = '\0';

		res = 0;
		DEBUG_PRINTF("%s\n", cur_path);
		if (lstat(cur_path, &statfile) >= 0)
			res = 1;

		strcat(cur_path, f_opt.del_string);
		DEBUG_PRINTF("%s\n", cur_path);
		if (lstat(cur_path, &statfile) >= 0)
			res |= 2;
		pend[1] = '\0';

		switch (res)
		{
		case 3:
			deleted = 1;
			DEBUG_PRINTF("DELETED BUT EXISTS\n");
			if (*get_path != NULL)
				free(*get_path);
			*get_path = NULL;
			if (*put_path != NULL)
				free(*put_path);
			*put_path = NULL;
			break;

		case 2:
			deleted = 1;
			DEBUG_PRINTF("DELETED\n");
			if (*get_path != NULL)
				free(*get_path);
			*get_path = NULL;
			if (*put_path != NULL)
				free(*put_path);
			*put_path = NULL;
			break;
		case 1:
			if (*get_path != NULL)
			{
				DEBUG_PRINTF("TO DELETE\n");
				to_delete = 1;
			}
			deleted = 0;
			if (pdesc->ro)
			{
				if (*get_path != NULL)
					free(*get_path);
				*get_path = cur_path;
				tab[IDX_GET] = pdesc;
			}
			else
			{
				if (*get_path != NULL)
					free(*get_path);
				*get_path = cur_path;
				tab[IDX_GET] = pdesc;
				cur_path = (char *) malloc(len);
				if (cur_path == NULL)
				{
					free(*get_path);
					return -ENOMEM;
				}
				strcpy(cur_path, *get_path);
				*put_path = cur_path;
				tab[IDX_PUT] = pdesc;
			}
			cur_path = (char *) malloc(len);
			if (cur_path == NULL)
			{
				if (*get_path != NULL)
					free(*get_path);
				if (*put_path != NULL)
					free(*put_path);
				return -ENOMEM;
			}
			break;
		}
		if (!pdesc->ro)
			pnew = pdesc;
	}

	if (pnew != NULL)
	{
		*new_path = cur_path;
		strncpy(cur_path, pnew->path, len);
		pend = cur_path + strlen(cur_path);
		while (*--pend == '/') ;
		pend[1] = '\0';
		strncat(cur_path, path, len);
		/* delete trailing slashs */
		pend = cur_path + strlen(cur_path);
		while (*--pend == '/') ;
		pend[1] = '\0';
		tab[IDX_NEW] = pnew;
	}
	else
		free(cur_path);

	res = (*put_path != NULL) << 1 | (*get_path !=
					  NULL) | deleted << 2 | (*new_path !=
								  NULL) << 3 |
		to_delete << 4;

	if (*put_path == NULL)
	{
		*put_path = (char *) malloc(1);
		**put_path = '\0';
	}
	if (*get_path == NULL)
	{
		*get_path = (char *) malloc(1);
		**get_path = '\0';
	}
	if (*new_path == NULL)
	{
		*new_path = (char *) malloc(1);
		**new_path = '\0';
	}
	return res;
}

int
funionfs_pathtowrite(const char *path)
{
	struct unionfs_desc *pdesc_tab[3];

	char *getpath, *setpath, *newpath;
	char *str, *pend;
	int res;

	/* create the path to file */
	str = strdup(path);
	if (str == NULL)
	{
		errno = ENOMEM;
		return -1;
	}

	/* detect last element of pathname */
	pend = str + strlen(str);
	while ((*--pend != '/') && (pend >= str)) ;
	pend[0] = '\0';

	res = funionfs_realpath(str, &getpath, &setpath, &newpath, pdesc_tab);
	if (res < 0)
	{
		free(str);
		return res;
	}


	if (*setpath == '\0')
	{
		res = funionfs_copytowrite(getpath, newpath);
	}
	else
		res = 0;

	free(str);
	free(getpath);
	free(setpath);
	free(newpath);

	if (res < 0)
		return -errno;

	return 0;
}

int
funionfs_copytowrite(char *readpath, char *writepath)
{
	char *wstr, *rstr;
	char *readfile, *writefile;
	char *wpend, *rpend;
	struct stat sb, rsb;
	int readfd, writefd;
	char link_path[MAX_LINKPATH];
	struct utimbuf buf;
	int endflag;

	DEBUG_PRINTF("copytowrite %s %s\n", readpath, writepath);

	/* create the path to file */
	wstr = strdup(writepath);
	if (wstr == NULL)
	{
		errno = ENOMEM;
		return -1;
	}

	rstr = strdup(readpath);
	if (rstr == NULL)
	{
		free(wstr);
		errno = ENOMEM;
		return -1;
	}
	wpend = wstr + strlen(wstr);
	rpend = rstr + strlen(rstr);

	/* find start of variable element of both pathnames */
	while ((*wpend == *rpend) && (wpend > wstr) && (rpend > rstr))
	{
		wpend--;
		rpend--;
	}

	//DEBUG_PRINTF("Identical part: %s = %s\n",rpend,wpend);

	while ((*wpend != '/') && (*wpend != '\0'))
		wpend++;
	while ((*rpend != '/') && (*rpend != '\0'))
		rpend++;
	endflag = -1;

	/* walk thru the path and create each element to the last element */
	/* if the element does'nt exist, clone it from the readonly path */

	//DEBUG_PRINTF("Left as constant: %s : %s\n",rpend,wpend);
	while (endflag < 0)
	{
		/* suppress leading slash */
		while (*wpend == '/')
			wpend++;
		while (*rpend == '/')
			rpend++;
		/* detect pathname element */
		while ((*wpend != '/') && (*wpend != '\0'))
			wpend++;
		if (*wpend == '\0')
			endflag = 1;
		*wpend = '\0';
		while ((*rpend != '/') && (*rpend != '\0'))
			rpend++;
		if (*rpend == '\0')
			endflag = 1;
		*rpend = '\0';

		//DEBUG_PRINTF("Currently: %s %s\n",rstr,wstr);
		/* get element attributes from the read path */
		if (lstat(rstr, &rsb) < 0)
		{
			break;
		}
		/* does the file exist in the writing path ? */
		if (lstat(wstr, &sb) < 0)
		{
			if (S_ISDIR(rsb.st_mode))
			{	/* a directory ? : create it */
				if (mkdir(wstr, rsb.st_mode & 07777) < 0)
					break;
			}
			if (S_ISSOCK(rsb.st_mode) || S_ISCHR(rsb.st_mode)
			    || S_ISBLK(rsb.st_mode) || S_ISFIFO(rsb.st_mode))
			{
				/* a device or a pipe ? : create it */
				if (mknod(wstr, rsb.st_mode, rsb.st_rdev) < 0)
					break;
			}
			if (S_ISLNK(rsb.st_mode))
			{	/* a link ?:create it */
				if (readlink
				    (readpath, link_path, MAX_LINKPATH - 1) < 0)
					break;
				if (symlink(wstr, link_path) < 0)
					break;
			}
			if (S_ISREG(rsb.st_mode))
			{	/* a file ?: copy it */
				if ((readfd = open(readpath, O_RDONLY)) < 0)
					break;

				if ((writefd =
				     open(wstr, O_CREAT | O_RDWR, S_IRWXU)) < 0)
				{
					close(readfd);
					break;
				}

				if (ftruncate(writefd, rsb.st_size) < 0)
				{
					close(readfd);
					close(writefd);
					break;
				}

				if ((readfile =
				     (char *) mmap(NULL, rsb.st_size, PROT_READ,
						   MAP_SHARED, readfd,
						   0)) == (char *) -1)
				{
					close(readfd);
					close(writefd);
					break;
				}

				if ((writefile =
				     (char *) mmap(NULL, rsb.st_size,
						   PROT_WRITE, MAP_SHARED,
						   writefd, 0)) == (char *) -1)
				{
					close(readfd);
					close(writefd);
					break;
				}

				memcpy(writefile, readfile, rsb.st_size);

				if (munmap(readfile, rsb.st_size) < 0)
				{
					close(readfd);
					close(writefd);
					break;
				}

				if (munmap(writefile, rsb.st_size) < 0)
				{
					close(readfd);
					close(writefd);
					break;
				}
				close(readfd);
				close(writefd);
			}
			/* set permissions, group, user and time of the readonly file */
			if (chmod(wstr, rsb.st_mode & 07777) < 0)
				break;
			if (chown(wstr, rsb.st_uid, rsb.st_gid) < 0)
				break;
			buf.actime = rsb.st_atime;
			buf.modtime = rsb.st_mtime;
			if (utime(wstr, &buf) < 0)
				break;
		}
		if (endflag > 0)
			break;
		*wpend = '/';
		*rpend = '/';
	}
	free(wstr);
	free(rstr);

	return endflag;
}

int
parse_and_add_dirs(void)
{
	char *ptr, *end;
	char *path;
	char basepath[4096];
	int type, ret;

	ptr = getcwd(basepath, 4096);
	if (ptr == NULL)
		return -errno;

	// first add dirlist to the hierarchy

	ptr = f_opt.dirlist;
	while (*ptr)
	{
		end = ptr;
		while (*end && (*end != ':') && (*end != '='))
		{
			end++;
		}

		type = 0;	// default type is Read/Write

		if (*end == '=')
		{
			*end++ = '\0';
			if ((*end != 'r') && (*end != 'R'))
				return -1;
			end++;
			switch (*end++)
			{
			case 'o':
			case 'O':
				type = 1;
				break;
			case 'w':
			case 'W':
				break;
			default:
				return -1;
			}
			if (*end != ':' && *end)
				return -1;
		}
		if (*end == ':')
			*end++ = '\0';

		path = rel2abspath(ptr, basepath);
		ret = funionfs_addpath(path, type);
		free(path);
		if (ret < 0)
		{
			return -ret;
		}
		ptr = end;
	}

	// then add the main path to the list

	if (*f_opt.firstdir != '\0')
	{
		if (strcmp(f_opt.firstdir, "NONE")
		    && strcmp(f_opt.firstdir, "none")
		    && strcmp(f_opt.firstdir, "None"))
		{
			path = rel2abspath(f_opt.firstdir, basepath);
			ret = funionfs_addpath(path, 0);
			free(path);
			if (ret < 0)
			{
				return -ret;
			}
		}
	}

	return 0;
}

struct unionfs_walk *
funionfs_openpath(const char *path)
{
	struct unionfs_walk *pwalk;

	pwalk = (struct unionfs_walk *) malloc(sizeof(struct unionfs_walk));
	if (pwalk == NULL)
	{
		errno = ENOMEM;
		return NULL;
	}
	pwalk->path = strdup(path);
	if (pwalk == NULL)
	{
		free(pwalk);
		errno = ENOMEM;
		return NULL;
	}
	pwalk->curdir = unionfs_list;
	pwalk->str = NULL;

	return pwalk;
}

char *
funionfs_readpath(struct unionfs_walk *pwalk, long *type)
{
	struct unionfs_desc *pdesc;
	char *str;
	char *pend;
	int res;
	struct stat statfile;

	while (1)
	{
		if (pwalk->str != NULL)
		{
			free(pwalk->str);
			pwalk->str = NULL;
		}

		pdesc = pwalk->curdir;

		if (pdesc == NULL)
		{
			*type = 0;
			return NULL;
		}
		pwalk->curdir = pdesc->pnext;

		str = (char *) malloc(strlen(pdesc->path) +
				      strlen(pwalk->path) + 1);
		if (str == NULL)
		{
			*type = 0;
			errno = ENOMEM;
			return NULL;
		}
		pwalk->str = str;

		strcpy(str, pdesc->path);

		/* delete trailing slashs */
		pend = str + strlen(str);
		while (*--pend == '/') ;
		pend[1] = '\0';

		strcat(str, pwalk->path);

		/* delete trailing slashs */
		pend = str + strlen(str);
		while (*--pend == '/') ;
		pend[1] = '\0';

		res = 0;
		DEBUG_PRINTF("%s\n", str);
		if (lstat(str, &statfile) >= 0)
			res = 1;
		if (res)
		{
			if (pdesc->ro)
			{
				*type = 1;
			}
			else
			{
				*type = 3;
			}
			return str;
		}
	}

	return NULL;
}

int
funionfs_closepath(struct unionfs_walk *pwalk)
{
	if (pwalk != NULL)
	{
		if (pwalk->str != NULL)
			free(pwalk->str);
		if (pwalk->path != NULL)
			free(pwalk->path);
		free(pwalk);
	}
	return 1;
}
