/*
 * Copyright 2012 Canonical Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import QtQuick 2.0
import QtGraphicalEffects 1.0
import Ubuntu.Components 0.1
import Ubuntu.Components.ListItems 0.1 as ListItem
import Ubuntu.Components.Popups 0.1
import Friends 0.2

Item {
    id: tile
    anchors {
        left: parent.left
        right: parent.right        
    }
    property real detailsOpacity : 0
    property string timeString: friendsUtils.createTimeString(timestamp)
    property int likeCount: likes > 0 ? likes : liked ? 1 : 0
    property string likesString: likeCount > 1 ? i18n.tr("people liked this") : i18n.tr("person liked this")
    property string icon: "images/"+service+".png"

    function recalculateTimeString() {
        tile.timeString = friendsUtils.createTimeString(timestamp);
    }

    Item {
        id: tileRect
        anchors {
            right: parent.right
            left: parent.left
        }
        height: status_update_content.height + detailsWrapper.height

        FriendsUtils {
            id: friendsUtils
        }

        FriendsDispatcher {
            id: friends
            onLikeComplete: {
                favoriteSpinner.visible = false;
                if (success) {
                    console.log ("Like succeeded");
                } else {
                    console.log ("Like failed: " + errorMessage);
                }
            }
            onUnlikeComplete: {
                favoriteSpinner.visible = false;
                 if (success) {
                     console.log ("Like succeeded");
                 } else {
                     console.log ("UnLike failed: " + errorMessage);
                 }
             }
            onRetweetComplete: {
                retweetSpinner.visible = false;
                if (success) {
                     console.log ("Retweet completed successfully");
                } else {
                     console.log ("Retweet failed: " + errorMessage);
                }
             }
        }

        UbuntuShape {
            id: avatarImage
            height: units.dp(48)
            width: units.dp(48)
            anchors {
                left: parent.left
                top: parent.top
                leftMargin: units.gu(1)
                topMargin: units.gu(1)
            }
            image: Image {
                source: Qt.resolvedUrl(avatar)
                asynchronous: true
                smooth: true
                fillMode: Image.PreserveAspectCrop
                sourceSize.width: units.dp(48)
            }

            MouseArea {
                anchors.fill: avatarImage
                onClicked: {
                    Qt.openUrlExternally(url);
                }
            }
        }

        Image {
            id: privateIcon
            anchors {
                right: avatarImage.right
                bottom: avatarImage.bottom
                bottomMargin: units.dp(3)
                rightMargin: units.dp(3)
            }
            source: "images/private.png"
            asynchronous: true
            visible: stream == "private"
        }

        Image {
            id: replyIcon
            anchors {
                right: avatarImage.right
                bottom: avatarImage.bottom
                bottomMargin: units.dp(3)
                rightMargin: units.dp(3)
            }
            source: "images/replies.png"
            asynchronous: true
            visible: stream == "mentions"
        }

        Image {
            id: serviceIcon
            anchors {
                left: parent.left
                bottom: status_update_content.bottom
                topMargin: units.gu(1)
                leftMargin: units.gu(1)
                bottomMargin: units.gu(1)
            }
            source: icon
            width: units.gu(2)
            height: units.gu(2)
            asynchronous: true
            smooth: true
        }

        Item {
            id: status_update_content
            anchors {
                left: avatarImage.right
                right: parent.right
                top: parent.top
                topMargin: units.gu(1)
                leftMargin: units.gu(1)
                bottomMargin: units.gu(1)
                rightMargin: units.gu(1)
            }
            height: childrenRect.height

            MouseArea {
                anchors.fill: status_update_content
                onClicked: {
                    if (tile.state != 'Details') {
                        tile.state = 'Details';
                    } else {
                        tile.state = 'List';
                    }
                }
            }

            Column {
                spacing: units.gu(1)
                anchors {
                    left: parent.left;
                    right: parent.right
                    bottomMargin: units.gu(2)
                }
                Item {
                    anchors {
                        left: parent.left
                        right: parent.right
                        rightMargin: units.gu(1)
                    }
                    height: childrenRect.height
                    Label {
                        text: sender
                        fontSize: "medium"
                        font.bold: true
                        anchors {
                            left: parent.left
                            right: time.left
                        }
                        wrapMode: Text.WrapAtWordBoundaryOrAnywhere
                    }
                    Label {
                        id: time
                        anchors {
                            right: parent.right
                        }

                        text: timeString
                        fontSize: "small"
                        color: Theme.palette.normal.baseText
                    }
                }

                Label {
                    id: messageText
                    anchors {
                        right: parent.right
                        left: parent.left
                    }
                    text: message
                    maximumLineCount: tile.state != 'Details' ? 10 : 0
                    elide: Text.ElideRight
                    visible: message.length > 0 && message != linkDescription
                    wrapMode: Text.WrapAtWordBoundaryOrAnywhere
                    textFormat: Text.StyledText
                    linkColor: Theme.palette.normal.baseText
                    onLinkActivated: Qt.openUrlExternally(link)
                }
                UbuntuShape {
                    height: locationName.height + units.gu(1)
                    width: locationName.width + units.gu(2)
                    visible: location.length > 0
                    Label {
                        id: locationName
                        anchors.centerIn: parent
                        text: location
                        wrapMode: Text.WrapAtWordBoundaryOrAnywhere
                    }
                    MouseArea {
                        anchors.fill: parent
                        onClicked: {
                            if (url.length > 0)
                            {
                                Qt.openUrlExternally(url);
                            }
                        }
                    }
                }
                UbuntuShape {
                    height: linkNameShape.height + units.gu(1)
                    width: linkNameShape.width + units.gu(2)
                    visible: linkName.length > 0
                    Label {
                        id: linkNameShape
                        anchors.centerIn: parent
                        text: linkName
                        wrapMode: Text.WrapAtWordBoundaryOrAnywhere
                    }
                    MouseArea {
                        anchors.fill: parent
                        onClicked: {
                            if (linkUrl.length > 0)
                                Qt.openUrlExternally(linkUrl);

                        }
                    }
                }
                Label {
                    id: linkDesc
                    anchors {
                        right: parent.right
                        left: parent.left
                    }
                    text: linkDescription
                    maximumLineCount: tile.state != 'Details' ? 10 : 0
                    elide: Text.ElideRight
                    visible: linkDescription.length > 0
                    wrapMode: Text.WrapAtWordBoundaryOrAnywhere
                    //width: parent.width - units.gu(3)
                    textFormat: Text.StyledText
                    onLinkActivated: Qt.openUrlExternally(link)
                }

                Row {
                    id: favorite
                    height: favoriteIcon.height + units.gu(1)
                    anchors {
                        left: parent.left
                        right: parent.right
                        bottomMargin: units.gu(1)
                    }
                    spacing: units.dp(3)
                    Image {
                        id: favoriteIcon
                        width: units.gu(3)
                        height: units.gu(3)
                        opacity: liked ? 1.0 : 0.1

                        source: "images/favorite.png"
                        asynchronous: true
                        MouseArea {
                            anchors.fill: favoriteIcon
                            onClicked: {
                                favoriteSpinner.visible = true;
                                if (liked) {
                                    friends.unlikeAsync(accountId, messageId);
                                }
                                else {
                                    friends.likeAsync(accountId, messageId);
                                }
                            }
                        }
                        ActivityIndicator {
                            id: favoriteSpinner
                            anchors.centerIn: parent
                            width: parent.width
                            visible: false
                            running: visible
                        }
                    }
                    Label {
                        id: likesLabel
                        text: (likeCount > 0) ? likeCount + " " + likesString : ""
                        fontSize: "small"
                        color: Theme.palette.normal.baseText
                    }
                }

                Item {
                    id: repeatIcon
                    anchors {
                        right: parent.right
                        bottom: parent.bottom
                        rightMargin: units.gu(2)
                        bottomMargin: units.gu(1)
                    }

                    opacity: detailsOpacity
                    height: fromMe ? 0 : childrenRect.height
                    // FIXME: Use features API instead
                    visible: (service == "twitter" || service == "identica") && !fromMe && stream != "private"
                    Image {
                        id: repeatIconImage
                        width: units.gu(3)
                        height: units.gu(3)
                        source: "images/share.png"
                        asynchronous: true
                        visible: retweetSpinner.visble ? false : true
                        MouseArea {
                            anchors.fill: parent
                            onClicked: {
                                console.log ("Retweeting " + messageId + " from " + accountId);
                                retweetSpinner.visible = true
                                friends.retweetAsync(accountId, messageId);
                            }
                        }
                        ActivityIndicator {
                            id: retweetSpinner
                            anchors.centerIn: parent
                            //width: parent.width
                            visible: false
                            running: visible
                        }
                    }
                }

            }
        }

        Item {
            id: detailsWrapper
            height: detailsLoader.height + units.gu(1)
            opacity: detailsOpacity
            visible:  detailsOpacity > 0
            anchors {
                top: status_update_content.bottom
                left: parent.left
                right: parent.right
            }

            Loader {
                id: detailsLoader
                anchors {
                    right: parent.right
                    left: parent.left
                }
                visible: true
            }

            onVisibleChanged: {
                if (visible && detailsLoader.state != Loader.Ready) {
                    detailsLoader.source = "StatusUpdateTileDetails.qml";
                }
            }
        }
    }

    states: [State {
                name: "Details"
                PropertyChanges { target: tile; detailsOpacity: 1; x: 0 }
                StateChangeScript {
                    script: {
                        listView.positionViewAtIndex(listView.indexAt(tile.x, tile.y), ListView.Contain);
                    }
                }

            },
            State {
                name: "List"
                PropertyChanges { target: tile; detailsOpacity: 0; x: 0 }
            }]


    transitions: Transition {
        NumberAnimation {
            duration: 300; properties: "detailsOpacity,x,height,width"
        }
    }
}
