/*
   control is such a trick--

   we can guide the ship
   but are we ever really in control
   of where we land?
*/

#include <sys/time.h>

#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>

#include <math.h>
#include <string.h>

#include <pthread.h>
#include <sched.h>
#include <sys/mman.h>

#include "fweelin_midiio.h"
#include "fweelin_core.h"

// ******** MIDI 

int MidiIO::activate() {
  printf("MIDI: Starting MIDI thread..\n");

  midithreadgo = 1;
  int ret = pthread_create(&midi_thread,
			   0,
			   run_midi_thread,
			   this);
  if (ret != 0) {
    printf("(start) pthread_create failed, exiting");
    return 1;
  }

  // Setup high priority threads
  struct sched_param schp;
  memset(&schp, 0, sizeof(schp));
  // MIDI thread at SCHED_FIFO?
  //schp.sched_priority = sched_get_priority_max(SCHED_OTHER);
  schp.sched_priority = sched_get_priority_min(SCHED_FIFO);
  printf("MIDI: HiPri Thread %d\n",schp.sched_priority);
  if (pthread_setschedparam(midi_thread, SCHED_FIFO, &schp) != 0) {    
    printf("MIDI: Can't set realtime thread, will use nonRT!\n");
  }

  return 0;
}

void MidiIO::close() {
  printf("MIDI: begin close...\n");
  midithreadgo = 0;
  pthread_join(midi_thread,0);
  printf("MIDI: end\n");
}

MidiIO::~MidiIO() {
  delete[] note_port;
  if (in_ports != 0)
    delete[] in_ports;
  if (out_ports != 0)
    delete[] out_ports;
}

/* Open ALSA sequencer wit num_in writeable ports and num_out readable ports. */
/* The sequencer handle and the port IDs are returned.                        */  
int MidiIO::open_seq(int num_in, int num_out) {

  int l1;
  char portname[64];
  
  numins = num_in;
  numouts = num_out;
  in_ports = new int[num_in];
  out_ports = new int[num_out];

  if (snd_seq_open(&seq_handle, "default", SND_SEQ_OPEN_DUPLEX, 0) < 0) {
    fprintf(stderr, "MIDI: Error opening ALSA sequencer.\n");
    return(-1);
  }
  snd_seq_set_client_name(seq_handle, "FreeWheeling");
  for (l1 = 0; l1 < num_in; l1++) {
    sprintf(portname, "FreeWheeling IN %d", l1+1);
    if ((in_ports[l1] = snd_seq_create_simple_port(seq_handle, portname,
              SND_SEQ_PORT_CAP_WRITE|SND_SEQ_PORT_CAP_SUBS_WRITE,
              SND_SEQ_PORT_TYPE_APPLICATION)) < 0) {
      fprintf(stderr, "MIDI: Error creating sequencer port.\n");
      return(-1);
    }
  }
  for (l1 = 0; l1 < num_out; l1++) {
    sprintf(portname, "FreeWheeling OUT %d", l1+1);
    if ((out_ports[l1] = snd_seq_create_simple_port(seq_handle, portname,
              SND_SEQ_PORT_CAP_READ|SND_SEQ_PORT_CAP_SUBS_READ,
              SND_SEQ_PORT_TYPE_APPLICATION)) < 0) {
      fprintf(stderr, "MIDI: Error creating sequencer port.\n");
      return(-1);
    }
  }
  return(0);
}

void MidiIO::ReceiveEvent(Event *ev, EventProducer *from) {
  // Handle special events for MIDI
  switch (ev->GetType()) {
  case T_EV_SetMidiTuning :
    {
      SetMidiTuningEvent *tev = (SetMidiTuningEvent *) ev;
      
      // OK!
      if (CRITTERS)
	printf("MIDI: Received SetMidiTuning "
	       "(new tuning: %d)\n", tev->tuning);
      SetBenderTune(tev->tuning);
    }
    return;

  case T_EV_SetMidiEchoPort :
    {
      SetMidiEchoPortEvent *pev = (SetMidiEchoPortEvent *) ev;
      
      // OK!
      if (CRITTERS)
	printf("MIDI: Received SetMidiEchoPort "
	       "(port #: %d)\n", pev->echoport);
      if (pev->echoport >= 0 && pev->echoport <= numouts)
	echoport = pev->echoport;
      else
	printf("MIDI: Invalid port #%d (valid range 0-%d)\n",
	       pev->echoport,numouts);
    }
    return;
    
  default:
    break;
  }

  // Handle MIDI event echos..
  snd_seq_event_t outev;
  char echo = 1;

  // Setup echo event
  snd_seq_ev_set_subs(&outev);
  snd_seq_ev_set_direct(&outev);

  FloConfig *fs = app->getCFG();

  switch (ev->GetType()) {
  case T_EV_Input_MIDIController :
    {
      MIDIControllerInputEvent *mcev = (MIDIControllerInputEvent *) ev;

      if (mcev->outport > 0 && mcev->outport <= numouts) {
	snd_seq_ev_set_source(&outev,out_ports[mcev->outport-1]);
	snd_seq_ev_set_controller(&outev,
				  mcev->channel,mcev->ctrl,mcev->val);
      } else
	echo = 0;
    }
    break;

  case T_EV_Input_MIDIProgramChange :
    {
      MIDIProgramChangeInputEvent *mpev = (MIDIProgramChangeInputEvent *) ev;

      if (mpev->outport > 0 && mpev->outport <= numouts) {
	snd_seq_ev_set_source(&outev,out_ports[mpev->outport-1]);
	snd_seq_ev_set_pgmchange(&outev,
				 mpev->channel,mpev->val);
      } else
	echo = 0;
    }
    break;

  case T_EV_Input_MIDIPitchBend :
    {
      MIDIPitchBendInputEvent *mpev = (MIDIPitchBendInputEvent *) ev;

      if (mpev->outport > 0 && mpev->outport <= numouts) {
	snd_seq_ev_set_source(&outev,out_ports[mpev->outport-1]);
	snd_seq_ev_set_pitchbend(&outev,
				 mpev->channel,mpev->val);
      } else
	echo = 0;
    }
    break;
      
  case T_EV_Input_MIDIKey :
    {
      MIDIKeyInputEvent *mkev = (MIDIKeyInputEvent *) ev;

      if (mkev->outport > 0 && mkev->outport <= numouts) {
	snd_seq_ev_set_source(&outev,out_ports[mkev->outport-1]);
	if (mkev->down)
	  snd_seq_ev_set_noteon(&outev,
				mkev->channel,
				mkev->notenum+fs->transpose,
				mkev->vel);
	else
	  snd_seq_ev_set_noteoff(&outev,
				 mkev->channel,
				 mkev->notenum+fs->transpose,
				 mkev->vel);
      } else
	echo = 0;
    }
    break;

  default:
    break;
  }

  if (echo) {
    // Echo the event to MIDI out
    snd_seq_event_output_direct(seq_handle, &outev);
  }
}

void *MidiIO::run_midi_thread(void *ptr)
{
  MidiIO *inst = static_cast<MidiIO *>(ptr);
  int npfd;
  struct pollfd *pfd;

  FloConfig *fs = inst->app->getCFG();

  printf("MIDIthread start..\n");

  // Open up ALSA sequencer client
  // Note hardwired one input & output midi port-- for now!
  if (inst->open_seq(1, fs->GetNumMIDIOuts()) < 0) {
    fprintf(stderr, "MIDI: Can't open ALSA sequencer.\n");
    exit(1);
  }
  npfd = snd_seq_poll_descriptors_count(inst->seq_handle, POLLIN);
  pfd = (struct pollfd *)alloca(npfd * sizeof(struct pollfd));
  snd_seq_poll_descriptors(inst->seq_handle, pfd, npfd, POLLIN);
  
  inst->app->getEMG()->ListenEvent(inst,0,T_EV_SetMidiTuning);
  inst->app->getEMG()->ListenEvent(inst,0,T_EV_SetMidiEchoPort);
  inst->app->getEMG()->ListenEvent(inst,fs->GetInputMatrix(),
				   T_EV_Input_MIDIKey);
  inst->app->getEMG()->ListenEvent(inst,fs->GetInputMatrix(),
				   T_EV_Input_MIDIController);
  inst->app->getEMG()->ListenEvent(inst,fs->GetInputMatrix(),
				   T_EV_Input_MIDIProgramChange);
  inst->app->getEMG()->ListenEvent(inst,fs->GetInputMatrix(),
				   T_EV_Input_MIDIPitchBend);
  
  // Main MIDI loop
  while (inst->midithreadgo) {
    // Every second, come out of poll-- so we don't get locked up
    // if no MIDI events come in
    if (poll(pfd, npfd, 1000) > 0) {
      // There's an event here! Read!
      snd_seq_event_t *ev;
      //      static int cnt = 0;
      do {
	snd_seq_event_input(inst->seq_handle, &ev);
	switch (ev->type) {
#if 0
	case SND_SEQ_EVENT_QFRAME:
	  printf("MTC quarterframe\n");
	  break;

	case SND_SEQ_EVENT_CLOCK:
	  if (cnt % 24 == 0)
	    printf("clock: %d\n",cnt);
	  cnt++;
	  //printf("clock: %d\n",ev->data.time.tick);
	  //printf("clock: %d\n",ev->data.time.time.tv_sec);
	  //printf("clock: %d\n",ev->data.queue.param.time.time.tv_sec);
	  break;

	case SND_SEQ_EVENT_TICK:
	  printf("MIDI: 'tick' not yet implemented\n");
	  break;

	case SND_SEQ_EVENT_TEMPO:
	  printf("MIDI: 'tempo' not yet implemented\n");
	  break;
#endif

	case SND_SEQ_EVENT_CONTROLLER: 
	  // OK MIDI event!
	  {
	    MIDIControllerInputEvent *mevt = (MIDIControllerInputEvent *)
	      Event::GetEventByType(T_EV_Input_MIDIController);
	    mevt->outport = inst->echoport;
	    mevt->channel = ev->data.control.channel;
	    mevt->ctrl = ev->data.control.param;
	    mevt->val = ev->data.control.value;
	    if (inst->app->getCFG()->IsDebugInfo())
	      printf("MIDI: Controller %d channel %d value %d\n",mevt->ctrl,
	    	     mevt->channel,mevt->val);
	    inst->app->getEMG()->BroadcastEventNow(mevt, inst);
	  }
	  break;

	case SND_SEQ_EVENT_PGMCHANGE:
	  // OK MIDI event!
	  {
	    MIDIProgramChangeInputEvent *mevt = (MIDIProgramChangeInputEvent *)
	      Event::GetEventByType(T_EV_Input_MIDIProgramChange);
	    mevt->outport = inst->echoport;
	    mevt->channel = ev->data.control.channel;
	    mevt->val = ev->data.control.value;
	    if (inst->app->getCFG()->IsDebugInfo())
	      printf("MIDI: Program change channel %d value %d\n",
	    	     mevt->channel,mevt->val);
	    inst->app->getEMG()->BroadcastEventNow(mevt, inst);
	  }	  
	  break;

	case SND_SEQ_EVENT_PITCHBEND:
	  // OK MIDI event!
	  {
	    // Store incoming bender value
	    inst->curbender = ev->data.control.value;
	    // Perform tune
	    ev->data.control.value += inst->bendertune;

	    MIDIPitchBendInputEvent *mevt = (MIDIPitchBendInputEvent *)
	      Event::GetEventByType(T_EV_Input_MIDIPitchBend);
	    mevt->outport = inst->echoport;
	    mevt->channel = ev->data.control.channel;
	    mevt->val = ev->data.control.value;
	    if (inst->app->getCFG()->IsDebugInfo())
	      printf("MIDI: Pitchbend channel %d value %d\n",
	    	     mevt->channel,mevt->val);
	    inst->app->getEMG()->BroadcastEventNow(mevt, inst);
	  }	  
	  break;

	case SND_SEQ_EVENT_NOTEON:
	  // OK MIDI event!
	  {
	    MIDIKeyInputEvent *mevt = (MIDIKeyInputEvent *)
	      Event::GetEventByType(T_EV_Input_MIDIKey);
	    mevt->channel = ev->data.control.channel;
	    mevt->notenum = ev->data.note.note;
	    if (mevt->notenum < 0 || mevt->notenum >= MAX_MIDI_NOTES) {
	      printf("MIDI: Bad MIDI note #%d!\n",mevt->notenum);
	      mevt->notenum = 0;
	    }
	    mevt->vel = ev->data.note.velocity;
	    if (ev->data.note.velocity > 0) {
	      mevt->down = 1;
	      inst->note_port[mevt->notenum] = mevt->outport = inst->echoport;
	    }
	    else {
	      mevt->down = 0;
	      mevt->outport = inst->note_port[mevt->notenum];
	    }
	    if (inst->app->getCFG()->IsDebugInfo())
	      printf("MIDI: Note %d %s, channel %d velocity %d\n",
		     mevt->notenum,
		     (mevt->down ? "on" : "off"),
		     mevt->channel, mevt->vel);
	    inst->app->getEMG()->BroadcastEventNow(mevt, inst);
	  }	  
	  break;

	case SND_SEQ_EVENT_NOTEOFF: 
	  // OK MIDI event!
	  {
	    MIDIKeyInputEvent *mevt = (MIDIKeyInputEvent *)
	      Event::GetEventByType(T_EV_Input_MIDIKey);
	    mevt->down = 0;
	    mevt->channel = ev->data.control.channel;
	    mevt->notenum = ev->data.note.note;
	    if (mevt->notenum < 0 || mevt->notenum >= MAX_MIDI_NOTES) {
	      printf("MIDI: Bad MIDI note #%d!\n",mevt->notenum);
	      mevt->notenum = 0;
	    }
	    mevt->vel = ev->data.note.velocity;
	    mevt->outport = inst->note_port[mevt->notenum];
	    if (inst->app->getCFG()->IsDebugInfo())
	      printf("MIDI: Note %d off, channel %d velocity %d\n",
		     mevt->notenum, mevt->channel, mevt->vel);
	    inst->app->getEMG()->BroadcastEventNow(mevt, inst);
	  }	  
	  break;
	}
	
	snd_seq_free_event(ev);
      } while (snd_seq_event_input_pending(inst->seq_handle, 0) > 0);
    }
  }

  inst->app->getEMG()->UnlistenEvent(inst,0,T_EV_SetMidiTuning);
  inst->app->getEMG()->UnlistenEvent(inst,0,T_EV_SetMidiEchoPort);
  inst->app->getEMG()->UnlistenEvent(inst,fs->GetInputMatrix(),
				     T_EV_Input_MIDIKey);
  inst->app->getEMG()->UnlistenEvent(inst,fs->GetInputMatrix(),
				     T_EV_Input_MIDIController);
  inst->app->getEMG()->UnlistenEvent(inst,fs->GetInputMatrix(),
				     T_EV_Input_MIDIProgramChange);
  inst->app->getEMG()->UnlistenEvent(inst,fs->GetInputMatrix(),
				     T_EV_Input_MIDIPitchBend);

  printf("MIDI: thread done\n");
  return 0;
}
