/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright (C) 1991-2010 OpenCFD Ltd.
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Description

\*---------------------------------------------------------------------------*/

#include <unistd.h>

#include <OpenFOAM/error.H>
#include "timer.H"

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

defineTypeNameAndDebug(Foam::timer, 0);

jmp_buf Foam::timer::envAlarm;

struct sigaction Foam::timer::oldAction_;

unsigned int Foam::timer::oldTimeOut_ = 0;

// * * * * * * * * * * * * * Static Member Functions * * * * * * * * * * * * //

void Foam::timer::signalHandler(int)
{ 
    if (debug)
    {
        Info<< "Foam::timer::signalHandler(int sig) : "
            << " timed out. Jumping."
            << endl;
    }
    longjmp(envAlarm, 1);
}

// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //


// Construct from components
Foam::timer::timer(const unsigned int newTimeOut)
:
    newTimeOut_(newTimeOut)
{

    if (newTimeOut > 0)
    {
        // Is singleton since handler is static function
        if (oldTimeOut_ != 0)
        {
            FatalErrorIn
            (
                "Foam::timer::timer(const unsigned int)"
            )   << "timer already used."
                << abort(FatalError);    
        }

        // Install alarm signal handler:
        // - do not block any signals while in it
        // - clear list of signals to mask
        struct sigaction newAction;
        newAction.sa_handler = timer::signalHandler;
	    newAction.sa_flags = SA_NODEFER;
        sigemptyset(&newAction.sa_mask);

        if (sigaction(SIGALRM, &newAction, &oldAction_) < 0)
        {
            FatalErrorIn
            (
                "Foam::timer::timer(const unsigned int)"
            )   << "sigaction(SIGALRM) error"
                << abort(FatalError);    
        }

        oldTimeOut_ = ::alarm(newTimeOut);

        if (debug)
        {
            Info<< "Foam::timer::timer(const unsigned int) : "
                << " installing timeout " << int(newTimeOut_)
                << " seconds"
                << " (overriding old timeout " << int(oldTimeOut_)
                << ")." << endl;
        }
    }
}


// * * * * * * * * * * * * * * * * Destructor  * * * * * * * * * * * * * * * //

Foam::timer::~timer()
{
    if (newTimeOut_ > 0)
    {
        if (debug)
        {
            Info<< "Foam::timer::~timer(const unsigned int) : timeOut="
                << int(newTimeOut_)
                << " : resetting timeOut to " << int(oldTimeOut_) << endl;
        }

        // Reset timer
        ::alarm(oldTimeOut_);
        oldTimeOut_ = 0;

        // Restore signal handler
        if (sigaction(SIGALRM, &oldAction_, NULL) < 0)
        {
            FatalErrorIn
            (
                "Foam::timer::~timer(const struct sigaction&"
                "const struct sigaction&)"
            )   << "sigaction(SIGALRM) error"
                << abort(FatalError);    
        }
    }
}

// ************************ vim: set sw=4 sts=4 et: ************************ //
