/* $Id: system_logon.c,v 1.10 2001/04/24 17:18:55 japh Exp $
 * system_logon.c  -- fancylogin system-logon procedure.
 *
 *             fancylogin uses ncurses to display a colorful login-
 *             screen with input-masks.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Written by Richard Bergmair.
 * ANSI-conformance testing by Andreas Krennmair
 */


#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "environment.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <pwd.h>
#include <grp.h>
#include <utmp.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <time.h>
#include <string.h>

#include "macros.h"
#include "limits.h"
#include "log_message.h"
#include "emergency.h"
#include "after_login.h"
#include "pam_login.h"
#include "conf.h"

#define _WTMP_FILE "/var/log/wtmp"


static char **getnewenv (struct passwd *user);
static int write_lastlog_entry(uid_t user_uid, char *rmthost,
                               struct lastlog *oldlog);
int system_logon (struct passwd *user, char *rmthost, int preserve);


char * emergency_env[] = 
{ "PATH=/bin:/usr/bin:", 
  "SHELL=/bin/sh" ,
  NULL
};


struct utmp utent;

void
checkutmp(int picky)
{
  char *line;
  struct utmp *ut;
  pid_t pid = getpid();

  setutent();

  /* First, try to find a valid utmp entry for this process.  */
  while ((ut = getutent()))
    if (ut->ut_pid == pid && ut->ut_line[0] && ut->ut_id[0] &&
	(ut->ut_type==LOGIN_PROCESS || ut->ut_type==USER_PROCESS))
      break;

  /* If there is one, just use it, otherwise create a new one.  */
  if (ut) {
    utent = *ut;
  } else {
    line = ttyname(0);
    if (!line) {
      exit(1);
    }
    if (strncmp(line, "/dev/", 5) == 0)
      line += 5;
    memset((void *) &utent, 0, sizeof utent);
    utent.ut_type = LOGIN_PROCESS;
    utent.ut_pid = pid;
    strncpy(utent.ut_line, line, sizeof utent.ut_line);
    /* XXX - assumes /dev/tty?? */
    strncpy(utent.ut_id, utent.ut_line + 3, sizeof utent.ut_id);
    strcpy(utent.ut_user, "LOGIN");
    utent.ut_time = time(NULL);
  }
}

void
setutmp(const char *name, const char *line, const char *host)
{
  utent.ut_type = USER_PROCESS;
  strncpy(utent.ut_user, name, sizeof utent.ut_user);
  utent.ut_time = time(NULL);
  /* other fields already filled in by checkutmp above */
  setutent();
  pututline(&utent);
  endutent();
#if HAVE_UPDWTMP
  updwtmp(_WTMP_FILE, &utent);
#endif
}


  
/*****************************************************************************/
/* getnewenv: returns the environment, the shell needs from us, to give      */
/*            username, our PID, home-directory, ...                         */
/*****************************************************************************/

static char **
getnewenv (struct passwd *user)
{
  #define __NUM_VARIABLES__ 7 


  int i;
  char **ne;


  /*
   * allocate memory for the environment
   */

  ne = (char **)malloc( sizeof(char *) * (__NUM_VARIABLES__));
  if (ne==NULL)
    return ne;

  for (i=0;i<__NUM_VARIABLES__;i++)
    {
      ne[i] = (char *)malloc(sizeof(char) * (__MAX_STR_LEN__));
      if (ne[i]==NULL)
	{
	  return NULL;
	}
    }

  i=0;

  /*
   * put some values in there.
   *
   * if you add something here, don't forget to increment
   * __NUM_VARIABLES__!
   */

  sprintf(ne[i++], "HOME=%s", user->pw_dir);
  sprintf(ne[i++], "SHELL=%s", user->pw_shell);
  sprintf(ne[i++], "USER=%s", user->pw_name);
  sprintf(ne[i++], "LOGNAME=%s", user->pw_name);
  sprintf(ne[i++], "TERM=%s", getenv("TERM")?getenv("TERM"):"linux");

  if (get_set_path_env())
    {
      if ((user->pw_uid) == 0)
        sprintf(ne[i++], "PATH=/sbin:/bin:/usr/sbin:/usr/bin");
      else
        sprintf(ne[i++], "PATH=/usr/local/bin:/bin:/usr/bin:.");
    }

  ne[i] = NULL;

  return ne;
}



/*****************************************************************************/
/* write_lastlog_entry: writes an entry of the current login into the lastlog*/
/*****************************************************************************/

static int
write_lastlog_entry(uid_t user_uid, char *rmthost, struct lastlog *oldlog)
{
  struct lastlog lastlog;       /* structure for parsing lastlog */
  int lastlog_file;             /* file-handler for /var/log/lastlog */

  /* open lastlog for writing */
  if ((lastlog_file = open (FILENAME_LASTLOG, O_RDWR)) == -1)
    {
      log_message (10001, "couldn't open FILENAME_LASTLOG for O_RDWR!");
      return -1;
    }

  /* what is our line? */
  sscanf (ttyname(0), "/dev/%s", lastlog.ll_line);

  /* what host do we log on from? */
  strcpy (lastlog.ll_host, rmthost);

  /* what is the time? */
  time (&lastlog.ll_time);

  /* what user logged on? */
  lseek (lastlog_file, (unsigned long)user_uid, SEEK_SET);

  /* read old data */
  read (lastlog_file, oldlog, sizeof *oldlog);

  /* "reseek" to where we just read from */
  lseek (lastlog_file, (unsigned long)user_uid, SEEK_SET);

  /* write data */
  write (lastlog_file, (char *)&lastlog, sizeof lastlog);

  /* close file */
  close (lastlog_file);

  /* all is groovy */
  return 0;
}



/*****************************************************************************/
/* system_logon: log user on to system                                       */
/*****************************************************************************/

int
system_logon (struct passwd *user, char *rmthost, int preserve)
{
  char **new_environment;
  struct utmp utmpentry, failentry;
  struct group *grp;
  struct lastlog oldlog;
  char * line;
  int i=0;
  int rc;

#if HAVE_LIBPAM
  pam_startpam();
  pam_startpamsession();
#endif

  checkutmp(strcmp(user->pw_name,"root"));

  /*
   * Update utmp and wtmp entries
   */

  line = malloc(strlen(ttyname(0))-5/* length of /dev/ */+1/* NUL-byte */);
  if (line!=NULL)
    {
      sscanf(ttyname(0),"/dev/%s",line);
      setutmp(user->pw_name,line,rmthost?rmthost:"");
      free(line);
    }

  /*
   * Get user's environment, and should that fail, default to
   * the emergency environment.
   */
  
  if ((new_environment = getnewenv (user)) == NULL)
    new_environment = emergency_env;
   
  /*
   * cd to user's home
   */

  chdir (user->pw_dir);


  /*
   * Change permissions to terminal
   * thanks, Thomas Wernitz.
   */

  grp = getgrnam("tty");
  chown (ttyname(0), user->pw_uid, grp->gr_gid);
  chmod (ttyname(0), 0620);

  write_lastlog_entry(user->pw_uid, rmthost?rmthost:"", &oldlog);

  /*
   * execute user's shell
   */

  rc=fork();
  if (rc==0)
    {
      char * login_shell;
    
     /*
      * Change to user's uid and gid
      */

      setgid (user->pw_gid);
      initgroups (user->pw_name, user->pw_gid);
      setuid (user->pw_uid);
  
      after_login (user->pw_uid, rmthost?rmthost:"", &oldlog);

     /*
      * Execute the user's shell
      */
      login_shell = malloc(strlen(user->pw_shell)+2);
      if (login_shell==NULL)
        {
          log_message(49998,"Error: virtual memory exhausted! "
          		    "Can't execute shell!\n");
          _exit(1);
        }
      strcpy(login_shell,"-");
      strcat(login_shell,user->pw_shell);

      if (!(preserve))
        execle(user->pw_shell, login_shell , NULL,
               new_environment);
      else
        {
          while (new_environment[i]!=NULL)
            putenv(new_environment[i++]);
          execl (user->pw_shell, login_shell , NULL);
        }
    }
  else
    if (rc == -1)
      {
        char errormessage[__MAX_STR_LEN__];

        sprintf(errormessage,"couldn't fork off process for user-environment!");
        log_message (1, errormessage);
        return -1;
      }
    else {
      wait (&i);
#if HAVE_LIBPAM
      pam_endpamsession();
      pam_endpam();
#endif
    }

  return 0;
}



/*****************************************************************************/
/* (c) Copyright 1999-2000 Richard Bergmair, 2000-2001 Andreas Krennmair     */
/*****************************************************************************/
